// Copyright 2016 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#include "device/gamepad/gamepad_pad_state_provider.h"

#include <cmath>

#include "device/gamepad/gamepad_data_fetcher.h"
#include "third_party/WebKit/public/platform/WebGamepads.h"

using blink::WebGamepad;
using blink::WebGamepads;

namespace device {

namespace {

const float kMinAxisResetValue = 0.1f;

}  // namespace

GamepadPadStateProvider::GamepadPadStateProvider() {
  pad_states_.reset(new PadState[WebGamepads::itemsLengthCap]);

  for (unsigned i = 0; i < WebGamepads::itemsLengthCap; ++i)
    ClearPadState(pad_states_.get()[i]);
}

GamepadPadStateProvider::~GamepadPadStateProvider() {}

PadState* GamepadPadStateProvider::GetPadState(GamepadSource source,
                                               int source_id) {
  // Check to see if the device already has a reserved slot
  PadState* empty_slot = nullptr;
  for (size_t i = 0; i < WebGamepads::itemsLengthCap; ++i) {
    PadState& state = pad_states_.get()[i];
    if (state.source == source && state.source_id == source_id) {
      // Retrieving the pad state marks this gamepad as active.
      state.active_state = GAMEPAD_ACTIVE;
      return &state;
    }
    if (!empty_slot && state.source == GAMEPAD_SOURCE_NONE)
      empty_slot = &state;
  }
  if (empty_slot) {
    empty_slot->source = source;
    empty_slot->source_id = source_id;
    empty_slot->active_state = GAMEPAD_NEWLY_ACTIVE;
  }
  return empty_slot;
}

void GamepadPadStateProvider::ClearPadState(PadState& state) {
  memset(&state, 0, sizeof(PadState));
}

void GamepadPadStateProvider::InitializeDataFetcher(
    GamepadDataFetcher* fetcher) {
  fetcher->InitializeProvider(this);
}

void GamepadPadStateProvider::MapAndSanitizeGamepadData(PadState* pad_state,
                                                        WebGamepad* pad,
                                                        bool sanitize) {
  DCHECK(pad_state);
  DCHECK(pad);

  if (!pad_state->data.connected) {
    memset(pad, 0, sizeof(WebGamepad));
    return;
  }

  // Copy the current state to the output buffer, using the mapping
  // function, if there is one available.
  if (pad_state->mapper)
    pad_state->mapper(pad_state->data, pad);
  else
    *pad = pad_state->data;

  pad->connected = true;

  if (!sanitize)
    return;

  // About sanitization: Gamepads may report input event if the user is not
  // interacting with it, due to hardware problems or environmental ones (pad
  // has something heavy leaning against an axis.) This may cause user gestures
  // to be detected erroniously, exposing gamepad information when the user had
  // no intention of doing so. To avoid this we require that each button or axis
  // report being at rest (zero) at least once before exposing its value to the
  // Gamepad API. This state is tracked by the axis_mask and button_mask
  // bitfields. If the bit for an axis or button is 0 it means the axis has
  // never reported being at rest, and the value will be forced to zero.

  // We can skip axis sanitation if all available axes have been masked.
  uint32_t full_axis_mask = (1 << pad->axesLength) - 1;
  if (pad_state->axis_mask != full_axis_mask) {
    for (size_t axis = 0; axis < pad->axesLength; ++axis) {
      if (!(pad_state->axis_mask & 1 << axis)) {
        if (fabs(pad->axes[axis]) < kMinAxisResetValue) {
          pad_state->axis_mask |= 1 << axis;
        } else {
          pad->axes[axis] = 0.0f;
        }
      }
    }
  }

  // We can skip button sanitation if all available buttons have been masked.
  uint32_t full_button_mask = (1 << pad->buttonsLength) - 1;
  if (pad_state->button_mask != full_button_mask) {
    for (size_t button = 0; button < pad->buttonsLength; ++button) {
      if (!(pad_state->button_mask & 1 << button)) {
        if (!pad->buttons[button].pressed) {
          pad_state->button_mask |= 1 << button;
        } else {
          pad->buttons[button].pressed = false;
          pad->buttons[button].value = 0.0f;
        }
      }
    }
  }
}

}  // namespace device