/* Copyright 2001-2005 Matt Flax <flatmax@ieee.org>
   This file is part of MFFM Time Scale Modification for Audio.

   MFFM Time Scale Modification for Audio is free software; you can redistribute
 it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.
   
   MFFM Time Scale Modification for Audio is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
   
   You have received a copy of the GNU General Public License
   along with MFFM Time Scale Modification for Audio
 */
#ifndef WSOLA4AUDIERE_H
#define WSOLA4AUDIERE_H


#include "FileBuffer.H"

#include <audiere.h>
#include "../WSOLA.H"

using namespace audiere;

#define DEF_TAU 1.0
#define MAX_TAU 4
#define MIN_TAU 0.25

namespace audiere {

  /**
  This class interfaces WSOLA and Audiere.
      By interfacing WSOLA and Audiere, time scale modificaiton of audio
      (speeding and slowing of audio in real time) is accomplished. Audiere
      is responsible for file reading and decomression as well as sound card
      device operation. WSOLA is responsible for the time scale modification
      in real time.
      
      To use this class one must :
      //Start Audiere as normal ...
      
      //Construct a WSOLA based sound source...
      WSOLA4SoxNAudiere mWsolaSource = new WSOLA4SoxNAudiere(mSampleSource, fileName, mTau, mFrameStartIndex, mFrameEndIndex);
      
      //create a stream using WSOLA4SoxNAudiere as the source
      OutputStreamPtr  mOutputStream(OpenSound(mAudioDevice, mWsolaSource , true));
      //play the stream to hear it.
      mOutputStream->play();
      
      //You may now alter the speed of audio playback in real time ...
      mWsolaSource->setTau(float tau); //Where float is the speed of playback where : fast < tau=1.0 < slow
      
      Upon finishing, the callback function is executed so WSOLA4SoxNAudiere can
      signal the parent application that the stream has finished playing.
  */
  class WSOLA4SoxNAudiere : public RefImplementation<SampleSource> {
    
    ///This private function reads, scales and writes audio to/from Audiere
    int WSOLARead(int frame_count){
      //cout<<"WSOLARead enter"<<endl;
		lastFrame=0;
      int framesProcessed=(int)((double)windowSize/2.0/(double)channels);
	  int res=0;
      //exit strategy
      if ((wsola->getSourceIndex()/channels+framesProcessed)>input.getEnd()){
			std::cout<<"last frame : exit as you wish"<<endl;
			lastFrame=1;
			//std::cout<<"last frame : "<<framesProcessed<<" frames dropped"<<endl;
			//return framesProcessed;
			//return 0;
		//getLastData(int srcLoc, int srcCnt, st_sample_t* source, int desiredLoc, int desiredCnt, st_sample_t* desired){
      }

	  res=fileBuffer->getData(wsola->getSourceIndex(), wsola->getSourceLength(), &(*input.window)[0],
							 wsola->getDesiredIndex(), wsola->getDesiredLength(), &(*desiredInput.window)[0]);

      if (firstFrame){
		std::cout<<"firstFrame"<<endl;
		//load the input data to the buffer and then to wsola
		wsola->initProcess(&(*input.window)[0], tau);
		firstFrame=0;
      }
      
      //Process outputting to the local memory buffer
      if (tau!=1.0){
	      //load the input data to the buffer and then to wsola
		  wsola->loadSourceInput(&(*input.window)[0]);
		   //Load desired input data
			wsola->loadDesiredInput(&(*desiredInput.window)[0]);
		int ret=wsola->processFrame(output, tau); //process halfWindow
		if (ret<0){
			if (ret==WSOLA::INPUT_READ2DF_ERR)
			    std::cout<<"Destination frame read error, most probably using a factor >=1.0 and at the end of the input file"<<endl;
			else if (ret==WSOLA::INPUT_READ2SF_ERR)
			    std::cout<<"Source frame read error, most probably using a factor <1.0, and at the end of the input file"<<endl;
			else if (ret==WSOLA::WRITE_ERR)
				std::cout<<"Is your disk full, we encountered a write error"<<endl;
			else if (ret==WSOLA::PROCFFT_ERR)
				std::cout<<"Please contact the author - this error shouldn't occur."<<endl;
			else if (ret==WSOLA::DEFAULT_ERR)
				std::cout<<"Un-known default error encountered"<<endl;
			else //Handle other errors here
				std::cout<<"Unknown error - unreliable failure"<<endl;
		}
	  } else {
		memcpy(output,&(*input.window)[0],framesProcessed*channels*sizeof(ATYPE));
		#ifdef W_DEBUG_OA
			wsola->tempOutput->write((char*)&(*input.window)[0],framesProcessed*channels*sizeof(ATYPE));
		#endif
		wsola->shiftOn(tau);
	  }
      
      if (lastFrame){
		if (lastFrameCallback){
			lastFrameCallback(callBackData);
			lastFrameCallback=NULL;
		}
		return 0;
	  }
      return framesProcessed;
    }
  public:
    
    /**The constructor allows one to pass the audiere file (source) and
       specify the initial speed (ta) as well as the start point (startPoint)
       and end point (endPoint). The callback function is operational in non
       WIN32 environments, where it is called upon hitting the last frame
       lastFrameCallback_in(callBackData_in);
    */
    WSOLA4SoxNAudiere(SampleSourcePtr source, const char * fileName, double ta=DEF_TAU, int startPoint=0, int endPoint=0,
		void (*lastFrameCallback_in)(void *)=NULL, void *callBackData_in=NULL) {
      fileBuffer=NULL;
	  lastFrame=0;
      m_source=source; //Audiere instantiation
      // Look to see whats happening with channels, sample rate and format
      source->getFormat(channels,fs,mSampleFormat);
      printf("channels=%d, rate=%d, Sample Format = %d\n",channels,fs,mSampleFormat);
      //Work out the necessary sample count
      windowSize =channels*HANNING_LENGTH(fs);
      std::cout<<"windowSize="<<windowSize<<endl;
      // Make sure that the half window size is an integer number of channels
      while (remainder((double)windowSize /2.0/(double)channels, floor((double)windowSize /2.0/(double)channels)) != 0.0){
		windowSize ++;
      }
      std::cout<<"windowSize="<<windowSize<<endl;
      
      //Begin buffering the file
	  try {
      fileBuffer= new FileBuffer(fileName, windowSize, channels, fs);
	  } catch (int ret) {
		  throw ret;
	  }
      if (!fileBuffer){
		cerr<<"WSOLA4SoxNAudiere: Counld't instantiate the FileBuffer ... are you out of memory ?"<<endl;
		//TOIDO: put throw here
		throw -1;
      }
      
      tau = ta;
      firstFrame = 1;
      wsola=0;
      cout<<"FS="<<fs<<'\n'<<"channels="<<channels<<endl;
	  try {
		wsola = new WSOLA (windowSize, fs, channels);//Version 2 instantiation;	
	  } catch (int ret) {
		  throw ret;
	  }
      if (!wsola){
	std::cout<<"WSOLA4SoxNAudiere(SampleSourcePtr source) : couldn't instantiate WSOLA"<<endl;
		//TODO: put throw here
		throw -1;
      }
      
      output=NULL; //Null the input / output buffers
      
      //Reserve local memory stores
      
      //set the input time code which holds the length of the audio
      // as well as the beginning, current and end locations
      // as well as a window for the input data
      input.init(startPoint,m_source->getLength()+1); //the point we want to start from
      desiredInput.init(startPoint,m_source->getLength()+1); //the point we want to start from
      setPosition(startPoint);
      (*input.window)=wsola->getSourceLength(); //Set the WSOLA window size
      (*desiredInput.window)=wsola->getDesiredLength(); //Set the WSOLA window size
      input.setFinish(m_source->getLength()+1); //Set the absolute length
      desiredInput.setFinish(MAXINT-2); //Set the absolute length

      if (endPoint!=0)//Guard against dummy entry
	if (endPoint<=input.getFinish()-1)
	  input.setEnd(endPoint); //the point we want to end at
	else
	  input.setEnd(input.getFinish()-1); //Force end at end of stream
     desiredInput.setEnd(MAXINT-3); //Set the end large !
      
	 //Set up the output array
      output = new ATYPE[windowSize]; //The output store
      if (output==NULL){
		std::cout<<"error defining output memory"<<endl;
		// TODO: put throw here
		throw -2;
      }

//	input.setFinish(input.getFinish()+windowSize);
//	  input.setEnd(input.getEnd()+windowSize);

      //set up the callback funciton
      lastFrameCallback=lastFrameCallback_in;
      callBackData=callBackData_in;

	  //Seek to start position before returning
  	  fileBuffer->getData(wsola->getSourceIndex(), wsola->getSourceLength(), &(*input.window)[0],
							 wsola->getDesiredIndex(), wsola->getDesiredLength(), &(*desiredInput.window)[0]);
	}
    
    ///Deconstructor
    ~WSOLA4SoxNAudiere(){
      if (fileBuffer) delete fileBuffer; fileBuffer=NULL;
      if (wsola) delete wsola; wsola=NULL;
      if (output) delete [] output; output=NULL;
    }
    
    ///Used to get various format information like channel count, sample rate and sample format
    void ADR_CALL getFormat(int& channel_count, int& sample_rate, SampleFormat& sample_format){
      m_source->getFormat(channel_count,sample_rate,sample_format);
      if (GetSampleSize(sample_format)!=wsola->getFrameSize())
	printf("Difference between Audiere (%d bytes) and WSOLA (%d bytes) frame sizes ... should be OK as long as WSOLA > Audiere",GetSampleSize(sample_format),wsola->getFrameSize());
    }

    ///This overloaded function is called by Audiere to fill the output buffer
    int ADR_CALL read(int frame_count, void* buffer) {
		if (lastFrame)
			return 0;
			//cout<<"read enter"<<endl;
      //std::cout<<"request frame_count "<<frame_count<<endl;
      int framesProcessed=0, realSize=(int)((double)windowSize/2.0/(double)channels);
      //std::cout<<"framesProcessed "<<framesProcessed<<" realSize "<<realSize<<endl;

      while ((framesProcessed+realSize)<frame_count){
		int oneCycleCnt=WSOLARead(realSize);
		int i;
		if (oneCycleCnt>0){
			unsigned char* tempUC=&((unsigned char*)buffer)[framesProcessed*channels];
			short int* tempSS=&((short int*)buffer)[framesProcessed*channels];
	    	  switch (mSampleFormat){
				case SF_U8:
					  for (i=0;i<oneCycleCnt*channels;i++)
						tempUC[i]=ST_SAMPLE_TO_SIGNED_BYTE(output[i]);
					break;
				case SF_S16:
					for (i=0;i<oneCycleCnt*channels;i++)
						tempSS[i]=ST_SAMPLE_TO_SIGNED_WORD(output[i]);
					break;
				default:
					cout<<"sample format not found error"<<endl;
					exit(-1);
				}
				framesProcessed+=oneCycleCnt;
		} else {
			break;
		}
		}

      //std::cout<<"returning "<<framesProcessed<<" wanted "<<frame_count<<endl;
      //cout<<"read exit"<<endl;
      return framesProcessed;
    }
    
    ///Resets the source and WSOLA
    void ADR_CALL reset() {
      m_source->reset();
      wsola->reset();
    }
    
    ///Checks whether the source is seekable
    bool ADR_CALL isSeekable() {
      //In this version the file-buffer is used and removes the possibility of
      //seeking.
      return 0;
    }

    ///Returns the length of the source
    int  ADR_CALL getLength() {
      return m_source->getLength();
    }
    
    ///Set the position of both the source and WSOLA accordingly
    void ADR_CALL setPosition(int position){
      //m_source->setPosition(position);
      wsola->setPosition(position*channels);
      input=position;
      desiredInput=position;
    }
    
    ///Position is indexed to the original stream index, as tau is variable
    int  ADR_CALL getPosition(){
      std::cout<<"getpos"<<endl;
      //return input.getCount();
      //return m_source->getPosition();
      return fileBuffer->getPosition();
    }
    
    ///Tells whether the source is set to repeat
    bool ADR_CALL getRepeat() {
      return m_source->getRepeat();
    }
    ///Sets the state of the source repeat
    void ADR_CALL setRepeat(bool repeat){
      m_source->setRepeat(repeat);
    }
    ///Sets the play back speed
    void  setTau(float ta) {
      std::cout<<"setTau "<<ta<<endl;
      this->tau = ta;
      std::cout<<"tau is now "<<this->tau<<endl;
    }
    ///Get the playback speed
    float getTau(){
      return tau;
    }
    
    ///Conformity with new Audiere API which supports tags : getTagCount
    ADR_METHOD(int) getTagCount() {return m_source->getTagCount();}
    ///Conformity with new Audiere API which supports tags : getTagKey
    const char* ADR_CALL getTagKey(int i) {return m_source->getTagKey(i);}
    ///Conformity with new Audiere API which supports tags : getTagValue
    const char* ADR_CALL getTagValue(int i) {return m_source->getTagValue(i);}
    ///Conformity with new Audiere API which supports tags : getTagType
    const char* ADR_CALL getTagType(int i) {return m_source->getTagType(i);}
    
  private:
      int lastFrame;

  ///The input source (file)
    SampleSourcePtr m_source;
    
    ///The time scale modification class
    WSOLA *wsola;
    
     ///The number of channels in the source
    int channels;
    /// The sample rate of the source (Hz)
    int fs;
    /// Audiere related sample format
    SampleFormat mSampleFormat;
    /// The size of the hanning window
    int windowSize;
    
    ///Locally stored speed variable
    float tau;
    ///Indication that this is the first frame
    int firstFrame;
    
    /// Input file buffer
    FileBuffer *fileBuffer;
    /// Input WSOLA buffer
    TIMECODETYPE_W input, desiredInput;
    /// Output buffer
    ATYPE *output;
    
  public:
    /** A callback function which notifies a parent app. that the last frame is being processed.
	This function is passed some user data (if necessary)
    */
    void (*lastFrameCallback)(void *);
    ///The user data to pass in the callback function argument
    void *callBackData;
};
}
#endif
