//=============================================================================
//   This file is part of VTKEdge. See vtkedge.org for more information.
//
//   Copyright (c) 2008 Kitware, Inc.
//
//   VTKEdge may be used under the terms of the GNU General Public License 
//   version 3 as published by the Free Software Foundation and appearing in 
//   the file LICENSE.txt included in the top level directory of this source
//   code distribution. Alternatively you may (at your option) use any later 
//   version of the GNU General Public License if such license has been 
//   publicly approved by Kitware, Inc. (or its successors, if any).
//
//   VTKEdge is distributed "AS IS" with NO WARRANTY OF ANY KIND, INCLUDING
//   THE WARRANTIES OF DESIGN, MERCHANTABILITY, AND FITNESS FOR A PARTICULAR
//   PURPOSE. See LICENSE.txt for additional details.
//
//   VTKEdge is available under alternative license terms. Please visit
//   vtkedge.org or contact us at kitware@kitware.com for further information.
//
//=============================================================================
// .NAME vtkKWEImageDataLIC2D
// .SECTION Description
// GPU implementation of a Line Integral Convolution, a technique for
// imaging  vector fields.
//
// The input on port 0 is an vtkImageData with extents of a 2D image. It needs
// a vector field on point data.
// Port 1 is a special port for customized noise input. It is an optional port.
// If not present, noise is generated by the filter. Even if none-power-of-two
// texture are supported, giving a power-of-two image may result in faster
// execution on the GPU.
// If noise input is not specified, then the filter using vtkImageNoiseSource to
// generate a 128x128 noise texture.
// This filter only works on point vectors. One can use a
// vtkCellDataToPointData filter to convert cell vectors to point vectors.
//
// .SECTION Required OpenGL Extensins
// GL_ARB_texture_non_power_of_two
// GL_VERSION_2_0
// GL_ARB_texture_float
// GL_ARB_draw_buffers
// GL_EXT_framebuffer_object
// GL_ARB_pixel_buffer_object

#ifndef __vtkKWEImageDataLIC2D_h
#define __vtkKWEImageDataLIC2D_h

#include "vtkImageAlgorithm.h"
#include "VTKEdgeConfigure.h" // include configuration header
#include "vtkWeakPointer.h" // needed for vtkWeakPointer.

class vtkRenderWindow;
class vtkImageNoiseSource;

class VTKEdge_HYBRID_EXPORT vtkKWEImageDataLIC2D : public vtkImageAlgorithm
{
public:
  static vtkKWEImageDataLIC2D* New();
  vtkTypeRevisionMacro(vtkKWEImageDataLIC2D, vtkImageAlgorithm);
  void PrintSelf(ostream& os, vtkIndent indent);

  // Description:
  // Get/Set the context. Context must be a vtkOpenGLRenderWindow. 
  // This does not increase the reference count of the 
  // context to avoid reference loops.
  // SetContext() may raise an error is the OpenGL context does not support the
  // required OpenGL extensions.
  void SetContext(vtkRenderWindow* context);
  vtkRenderWindow* GetContext();

  // Description:
  // Number of steps. Initial value is 20.
  // class invariant: Steps>0.
  // In term of visual quality, the greater the better.
  vtkSetMacro(Steps,int);
  vtkGetMacro(Steps,int);
  
  // Description:
  // Step size.
  // Specify the step size as a unit of the cell length of the input vector
  // field. Cell lenghth is the length of the diagonal of a cell.
  // Initial value is 1.0.
  // class invariant: StepSize>0.0.
  // In term of visual quality, the smaller the better.
  // The type for the interface is double as VTK interface is double
  // but GPU only supports float. This value will be converted to
  // float in the execution of the algorithm.
  vtkSetClampMacro(StepSize, double, 0.0, 100.0);
  vtkGetMacro(StepSize, double);

  // Description:
  // The the magnification factor. Default is 1
  vtkSetClampMacro(Magnification, int, 1, VTK_INT_MAX);
  vtkGetMacro(Magnification, int);


  void TranslateInputExtent(const int* inExt, 
    const int* inWholeExtent, int *outExt);

//BTX
protected:
  vtkKWEImageDataLIC2D();
  ~vtkKWEImageDataLIC2D();

  virtual int RequestInformation(vtkInformation *, 
    vtkInformationVector **, vtkInformationVector *);

  // Description:
  // Fill the input port information objects for this algorithm.  This
  // is invoked by the first call to GetInputPortInformation for each
  // port so subclasses can specify what they can handle.
  // Redefined from the superclass.
  virtual int FillInputPortInformation(int port,
                                       vtkInformation *info);
  
  int RequestUpdateExtent (vtkInformation * vtkNotUsed(request),
                           vtkInformationVector **inputVector,
                           vtkInformationVector *vtkNotUsed( outputVector ));
  
  // Description:
  // This is called by the superclass.
  // This is the method you should override.
  virtual int RequestData(vtkInformation *request,
                          vtkInformationVector **inputVector,
                          vtkInformationVector *outputVector);

  int Steps;
  double StepSize;
  int Magnification;
  vtkWeakPointer<vtkRenderWindow> Context;
  vtkImageNoiseSource* NoiseSource;

  bool ARBColorBufferFloatSupported;
  bool OwnWindow;
private:
  vtkKWEImageDataLIC2D(const vtkKWEImageDataLIC2D&); // Not implemented.
  void operator=(const vtkKWEImageDataLIC2D&); // Not implemented.
//ETX
};

#endif


