/*
** This file is part of the ViTE project.
**
** This software is governed by the CeCILL-A license under French law
** and abiding by the rules of distribution of free software. You can
** use, modify and/or redistribute the software under the terms of the
** CeCILL-A license as circulated by CEA, CNRS and INRIA at the following
** URL: "http://www.cecill.info".
** 
** As a counterpart to the access to the source code and rights to copy,
** modify and redistribute granted by the license, users are provided
** only with a limited warranty and the software's author, the holder of
** the economic rights, and the successive licensors have only limited
** liability.
** 
** In this respect, the user's attention is drawn to the risks associated
** with loading, using, modifying and/or developing or reproducing the
** software by the user in light of its specific status of free software,
** that may mean that it is complicated to manipulate, and that also
** therefore means that it is reserved for developers and experienced
** professionals having in-depth computer knowledge. Users are therefore
** encouraged to load and test the software's suitability as regards
** their requirements in conditions enabling the security of their
** systems and/or data to be ensured and, more generally, to use and
** operate it in the same conditions as regards security.
** 
** The fact that you are presently reading this means that you have had
** knowledge of the CeCILL-A license and that you accept its terms.
**
**
** ViTE developers are (for version 0.* to 1.0):
**
**        - COULOMB Kevin
**        - FAVERGE Mathieu
**        - JAZEIX Johnny
**        - LAGRASSE Olivier
**        - MARCOUEILLE Jule
**        - NOISETTE Pascal
**        - REDONDY Arthur
**        - VUCHENER Clément 
**
*/

#include <fstream>
#include <string>
#include <set>
#include <map>
#include <queue>
#include <list>
/* -- */
#include "common/Errors.hpp"
/* -- */
#include "trace/values/Values.hpp"
#include "trace/Trace.hpp"
/* -- */
#include "parser/Line.hpp"
#include "parser/File.hpp"
#include "parser/Definition.hpp"
#include "parser/Parser.hpp"
#include "parser/ParserDefinitionPaje.hpp"
#include "parser/ParserEventPaje.hpp"
#include "parser/ParserPaje.hpp"
/* -- */
using namespace std;

ParserPaje::ParserPaje() : parserdefinition(new ParserDefinitionPaje()), parserevent(new ParserEventPaje()) {
    _is_finished = false;
    _is_canceled = false;
}

ParserPaje::~ParserPaje() {
    delete parserdefinition;
    delete parserevent;
}

void ParserPaje::parse(const string &filename, Trace &trace, bool finish_trace_after_parse) {

    set_file_to_parse(filename);

    Line line;

    Definition::init_definitions("Paje");

    static const string PERCENT = "%";
    string event_identity_string;
    unsigned int event_identity;

    _file.open(_file_to_parse);

    if(!_file.is_opened() || !_file.is_a_file()) {
        _is_canceled = true;
        finish();
        trace.finish();
        Error::set(Error::_OPEN, line.get_line_count(), Error::_WARNING);
        return;
    }

    while(!(_file.is_eof() || _is_canceled)) {
        
        try {
            _file.get_line(line);
        }
	catch(char *){
	  continue;
	}

    	if(line.starts_with(PERCENT)) {
            parserdefinition->store_definition(line);
        }
    	else if (!line.item(0, event_identity_string)) {
            continue; // We have \n 
    	}
    	else {
            // We check if we have an event identifier
            if(sscanf(event_identity_string.c_str(), "%u", &event_identity) != 1){
                Error::set(Error::_EXPECT_ID_DEF, line.get_line_count(), Error::_WARNING);
                continue;
            }
            Definition current_definition;
            try {
                current_definition = parserdefinition->get_definition(event_identity);
            }
            catch (unsigned int) {
                Error::set(Error::_UNKNOWN_ID_DEF + event_identity_string, line.get_line_count(), Error::_WARNING);
                continue;
            }
            parserevent->store_event(current_definition,
                                     line,
                                     trace);
        }
    }
    finish();

    if(finish_trace_after_parse) {
        trace.finish();
    }
}


void ParserPaje::set_file_to_parse(const string &filename) {
    _file_to_parse = filename;
}


float ParserPaje::get_size_loaded() const {
    return _file.get_size_loaded();
}

bool ParserPaje::is_end_of_parsing() const {
    return _is_finished;
}

void ParserPaje::finish() {
    _is_finished = true;
}

void ParserPaje::set_canceled() {
    _is_canceled = true;
}

const map<unsigned int, Definition> *ParserPaje::get_definitions() const {
    return parserdefinition->get_hash_table_of_definitions();
}
