/*
 *
 *   Copyright (C) 2005-2010 by Raymond Huang
 *   plushuang at users.sourceforge.net
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *
 *  ---
 *
 *  In addition, as a special exception, the copyright holders give
 *  permission to link the code of portions of this program with the
 *  OpenSSL library under certain conditions as described in each
 *  individual source file, and distribute linked combinations
 *  including the two.
 *  You must obey the GNU Lesser General Public License in all respects
 *  for all of the code used other than OpenSSL.  If you modify
 *  file(s) with this exception, you may extend this exception to your
 *  version of the file(s), but you are not obligated to do so.  If you
 *  do not wish to do so, delete this exception statement from your
 *  version.  If you delete this exception statement from all source
 *  files in the program, then also delete it here.
 *
 */

#include <ug_selector.h>
#include <ug_list_view.h>
#include <ug_item.h>
#include <ug_url.h>

#include <glib/gi18n.h>

enum ELEMENT_INDEX {
	ELEMENT_VIEW,
	ELEMENT_STORE,
	ELEMENT_HASH,
	ELEMENT_HOST_STORE,
	ELEMENT_EXT_STORE,
	ELEMENT_SIZE
};

//	signal handler
static void on_mark_all (GtkWidget* button, UgSelector* selector);
static void on_mark_none (GtkWidget* button, UgSelector* selector);
static void on_mark_by_filter (GtkWidget* button, UgSelector* selector);
static void	on_message_dialog_response (GtkDialog* dlg, gint response, UgSelector* selector);
static void on_filter_dialog_response  (GtkDialog* dlg, gint response, UgSelector* selector);
static void on_filter_host_all (GtkWidget* widget, UgSelector* selector);
static void on_filter_host_none (GtkWidget* widget, UgSelector* selector);
static void on_filter_ext_all (GtkWidget* widget, UgSelector* selector);
static void on_filter_ext_none (GtkWidget* widget, UgSelector* selector);

void	ug_selector_init (UgSelector* selector, GtkWindow* container)
{
	GtkBox*		vbox;
	GtkBox*		hbox;
	GtkWidget*	widget;
	gchar*		string;

	selector->page_array = g_ptr_array_sized_new (ELEMENT_SIZE * 2);
	selector->parent = container;
	selector->self = gtk_vbox_new (FALSE, 2);
	vbox = (GtkBox*) selector->self;

	hbox = (GtkBox*) gtk_hbox_new (FALSE, 2);
	gtk_box_pack_start (vbox, (GtkWidget*) hbox, FALSE, FALSE, 1);
	string = g_strconcat (_("Base hypertext reference"), " <base href> :", NULL);
	selector->href_label = gtk_label_new (string);
	g_free (string);
	gtk_box_pack_start (hbox, selector->href_label, FALSE, TRUE, 1);

	selector->href_entry = (GtkEntry*) gtk_entry_new ();
	gtk_box_pack_start (vbox, (GtkWidget*) selector->href_entry, FALSE, FALSE, 1);
	selector->href_separator = gtk_hseparator_new ();
	gtk_box_pack_start (vbox, selector->href_separator, FALSE, FALSE, 1);

	selector->notebook = (GtkNotebook*) gtk_notebook_new ();
	gtk_box_pack_start (vbox, (GtkWidget*) selector->notebook, TRUE, TRUE, 1);

	hbox = (GtkBox*) gtk_hbox_new (FALSE, 2);
	gtk_box_pack_start (vbox, (GtkWidget*) hbox, FALSE, FALSE, 1);
	// mark all
	widget = gtk_button_new_with_mnemonic (_("Mark _All"));
	g_signal_connect (widget, "clicked", G_CALLBACK (on_mark_all), selector);
	gtk_box_pack_start (hbox, widget, TRUE, TRUE, 1);
	selector->mark_all = widget;
	// mark none
	widget = gtk_button_new_with_mnemonic (_("Mark _None"));
	g_signal_connect (widget, "clicked", G_CALLBACK (on_mark_none), selector);
	gtk_box_pack_start (hbox, widget, TRUE, TRUE, 1);
	selector->mark_none = widget;
	// mark by filter
	widget = gtk_button_new_with_mnemonic (_("_Mark by filter..."));
	g_signal_connect (widget, "clicked", G_CALLBACK (on_mark_by_filter), selector);
	gtk_box_pack_start (hbox, widget, TRUE, TRUE, 1);
	selector->mark_by_filter = widget;

	gtk_widget_show_all ((GtkWidget*) vbox);
}

static gboolean hash_table_foreach (gpointer key, gpointer value, gpointer user_data)
{
	g_list_free (value);
	return TRUE;
}

void	ug_selector_finalize (UgSelector* selector)
{
	GtkListStore*	store;
	GHashTable*		hash;
	GPtrArray*		array;
	guint			index;

	array = selector->page_array;
	for (index=0; index < array->len; index+=ELEMENT_SIZE) {
		hash  = g_ptr_array_index (array, index+ELEMENT_HASH);
		if (hash) {
			g_hash_table_foreach_remove (hash, hash_table_foreach, NULL);
			g_hash_table_destroy (hash);
		}
		// ELEMENT_STORE
		store = g_ptr_array_index (array, index+ELEMENT_STORE);
		ug_item_store_clear_after (store, NULL);
		g_object_unref (store);
		// ELEMENT_HOST_STORE
		store = g_ptr_array_index (array, index+ELEMENT_HOST_STORE);
		ug_item_store_clear_after (store, NULL);
		g_object_unref (store);
		// ELEMENT_EXT_STORE
		store = g_ptr_array_index (array, index+ELEMENT_EXT_STORE);
		ug_item_store_clear_after (store, NULL);
		g_object_unref (store);
	}
	g_ptr_array_free (selector->page_array, TRUE);
}

void	ug_selector_hide_href (UgSelector* selector)
{
	gtk_widget_hide ((GtkWidget*) selector->href_label);
	gtk_widget_hide ((GtkWidget*) selector->href_entry);
	gtk_widget_hide ((GtkWidget*) selector->href_separator);
}

gboolean	ug_selector_show_error (UgSelector* selector)
{
	UgItem*			item;
	GtkTreeModel*	model;
	GtkTreeIter		iter;
	GPtrArray*		array;
	GtkWidget*		dialog;
	gboolean		valid;
	guint			index;
	gchar*			title;

	// hide or destroy current dialog
	if (selector->filter_dialog) {
		gtk_widget_destroy ((GtkWidget*) selector->filter_dialog);
		selector->filter_dialog = NULL;
	}
	if (selector->message_dialog) {
		gtk_widget_destroy ((GtkWidget*) selector->message_dialog);
		selector->message_dialog = NULL;
	}
	// has marked item?
	array = selector->page_array;
	for (index=0; index < array->len; index+=ELEMENT_SIZE) {
		model = g_ptr_array_index (array, index+ELEMENT_STORE);
		valid = gtk_tree_model_get_iter_first (model, &iter);
		while (valid) {
			gtk_tree_model_get (model, &iter, 0, &item, -1);
			if (item->mark) {
				gtk_widget_set_sensitive (selector->self, TRUE);
				return FALSE;
			}
			valid = gtk_tree_model_iter_next (model, &iter);
		}
	}
	// disable sensitive of parent window
	// enable sensitive in function on_message_dialog_response()
	if (selector->parent)
		gtk_widget_set_sensitive ((GtkWidget*) selector->parent, FALSE);
	// show error message
	dialog = gtk_message_dialog_new (selector->parent,
	                                 GTK_DIALOG_DESTROY_WITH_PARENT,
	                                 GTK_MESSAGE_ERROR, GTK_BUTTONS_OK,
	                                 _("No URL marked."));
	title = g_strconcat ("Uget - ", _("Error"), NULL);
	gtk_window_set_title ((GtkWindow*) dialog, title);
	g_free (title);
	gtk_window_set_transient_for ((GtkWindow*) dialog, selector->parent);
	g_signal_connect (dialog, "response", G_CALLBACK (on_message_dialog_response), selector);
	selector->message_dialog = (GtkDialog*) dialog;

	if (gtk_window_get_modal (selector->parent))
		gtk_dialog_run ((GtkDialog*) dialog);
	else
		gtk_widget_show ((GtkWidget*) dialog);

	return TRUE;
}

GtkListStore*	ug_selector_add_page (UgSelector* selector, const gchar* title)
{
	GtkTreeView*		item_view;
	GtkListStore*		item_list;
	GtkWidget*			scrolled;

	item_view = ug_item_view_new (NULL, "URL");
	item_list = gtk_list_store_new (1, G_TYPE_POINTER);
	ug_item_view_for_selector (item_view);
	gtk_tree_view_set_model (item_view, (GtkTreeModel*) item_list);

	g_ptr_array_add (selector->page_array, item_view);	// ELEMENT_VIEW
	g_ptr_array_add (selector->page_array, item_list);	// ELEMENT_STORE
	g_ptr_array_add (selector->page_array, NULL);		// ELEMENT_HASH
	g_ptr_array_add (selector->page_array, gtk_list_store_new (1, G_TYPE_POINTER));	// ELEMENT_HOST_STORE
	g_ptr_array_add (selector->page_array, gtk_list_store_new (1, G_TYPE_POINTER));	// ELEMENT_EXT_STORE
	scrolled = gtk_scrolled_window_new (NULL, NULL);
	gtk_scrolled_window_set_shadow_type (GTK_SCROLLED_WINDOW (scrolled), GTK_SHADOW_IN);
	gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (scrolled),
	                                GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
	gtk_container_add (GTK_CONTAINER(scrolled), GTK_WIDGET(item_view));
	gtk_widget_show ((GtkWidget*) scrolled);

	gtk_notebook_append_page (selector->notebook, scrolled, gtk_label_new (title));
	return item_list;
}

GtkListStore*	ug_selector_get_store (UgSelector* selector, guint nth_page)
{
	nth_page *= ELEMENT_SIZE;
	if (nth_page < selector->page_array->len)
		return g_ptr_array_index (selector->page_array, nth_page + ELEMENT_STORE);
	return NULL;
}

GList*	ug_selector_get_marked (UgSelector* selector)
{
	UgItem*			item;
	UgUrlPart*		urlpart;
	GtkTreeModel*	model;
	GtkTreeIter		iter;
	GList*			result;
	GPtrArray*		array;
	GString*		gstr;
	const gchar*	base_href;
	gboolean		valid;
	guint			index;

	result = NULL;
	array = selector->page_array;
	urlpart = g_malloc (sizeof (UgUrlPart));
	base_href = gtk_entry_get_text (selector->href_entry);
	if (base_href[0] == 0)
		base_href = NULL;
	for (index=0; index < array->len; index+=ELEMENT_SIZE) {
		model = g_ptr_array_index (array, index+ELEMENT_STORE);
		valid = gtk_tree_model_get_iter_first (model, &iter);
		while (valid) {
			gtk_tree_model_get (model, &iter, 0, &item, -1);
			valid = gtk_tree_model_iter_next (model, &iter);
			if (item->mark == FALSE)
				continue;
			ug_url_part (urlpart, item->value, -1);
			if (urlpart->url_scheme_len || base_href == NULL) {
				result = g_list_prepend (result, item->value);
				item->value = NULL;		// clear
			}
			else {
				gstr = g_string_new (base_href);
				if (gstr->str[gstr->len -1] == '/') {
					if (item->value[0] == '/')
						g_string_truncate (gstr, gstr->len -1);
				}
				else if (item->value[0] != '/') {
					g_string_append_c (gstr, '/');
				}
				g_string_append (gstr, item->value);
				result = g_list_prepend (result, g_string_free (gstr, FALSE));
			}
		}
	}
	g_free (urlpart);
	return result;
}

static void	ug_selector_make_filter (UgSelector* selector, guint index)
{
	GtkTreeModel*	model;
	GtkTreeIter		iter;
	GtkListStore*	store;
	GtkTreeIter		store_iter;
	GHashTable*		hash;
	UgItem*			item;
	UgItem*			item_f;
	gboolean		valid;
	UgUrlPart*		urlpart;
	gchar*			old_key;
	gchar*			key;
	GList*			list;	// value

	hash = g_ptr_array_index (selector->page_array, index + ELEMENT_HASH);
	if (hash)
		return;
	hash = g_hash_table_new (g_str_hash, g_str_equal);
	g_ptr_array_index (selector->page_array, index + ELEMENT_HASH) = hash;

	model = g_ptr_array_index (selector->page_array, index + ELEMENT_STORE);
	valid = gtk_tree_model_get_iter_first (model, &iter);
	urlpart = g_malloc (sizeof (UgUrlPart));
	while (valid) {
		gtk_tree_model_get (model, &iter, 0, &item, -1);
		valid = gtk_tree_model_iter_next (model, &iter);

		ug_url_part (urlpart, item->value, -1);
		// create filter by host ----------------
		if (urlpart->host_len)
			key = g_strndup (urlpart->url, urlpart->url_location_len - urlpart->folder_len);
		else
			key = g_strdup ("(none)");
		if (g_hash_table_lookup_extended (hash, key, (gpointer*)&old_key, (gpointer*)&list)) {
			g_free (key);
			key = old_key;
		}
		else {
			item_f = g_malloc0 (sizeof (UgItem));
			store = g_ptr_array_index (selector->page_array, index + ELEMENT_HOST_STORE);
			gtk_list_store_append (store, &store_iter);
			gtk_list_store_set (store, &store_iter, 0, item_f, -1);
			item_f->value = key;
			item_f->mark = TRUE;
			list = NULL;
		}
		list = g_list_prepend (list, item);
		g_hash_table_insert (hash, key, list);
		// create filter by ext -----------------
		if (urlpart->file_ext_len)
			key = g_strdup_printf (".%.*s", urlpart->file_ext_len, urlpart->file_ext);
		else
			key = g_strdup (".");
		if (g_hash_table_lookup_extended (hash, key, (gpointer*)&old_key, (gpointer*)&list)) {
			g_free (key);
			key = old_key;
		}
		else {
			item_f = g_malloc0 (sizeof (UgItem));
			store = g_ptr_array_index (selector->page_array, index + ELEMENT_EXT_STORE);
			gtk_list_store_append (store, &store_iter);
			gtk_list_store_set (store, &store_iter, 0, item_f, -1);
			item_f->value = key;
			item_f->mark = TRUE;
			list = NULL;
		}
		list = g_list_prepend (list, item);
		g_hash_table_insert (hash, key, list);
	}
	ug_url_part_free (urlpart);
}

static void	ug_selector_show_filter (UgSelector* selector, guint index)
{
	GtkBox*			vbox;
	GtkBox*			hbox;
	GtkBox*			hbox_button;
	GtkWidget*		scrolled;
	GtkWidget*		button;
	GtkTreeModel*	model;
	GtkSizeGroup*	sizegroup;
	gchar*			title;

	// hide or destroy current dialog
	if (selector->filter_dialog) {
		gtk_widget_destroy ((GtkWidget*) selector->filter_dialog);
		selector->filter_dialog = NULL;
	}
	if (selector->message_dialog) {
		gtk_widget_destroy ((GtkWidget*) selector->message_dialog);
		selector->message_dialog = NULL;
	}
	// disable sensitive of parent window
	// enable sensitive in function on_filter_dialog_response()
	if (selector->parent)
		gtk_widget_set_sensitive ((GtkWidget*) selector->parent, FALSE);
	// create filter dialog
	title = g_strconcat ("Uget - ", _("Mark by filter"), NULL);
	selector->filter_dialog = (GtkDialog*) gtk_dialog_new_with_buttons (title,
	                  selector->parent,
	         (GTK_DIALOG_NO_SEPARATOR | GTK_DIALOG_DESTROY_WITH_PARENT),
	                  GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
	                  GTK_STOCK_OK,     GTK_RESPONSE_OK,
	                  NULL);
	g_free (title);
	gtk_window_set_modal ((GtkWindow*) selector->filter_dialog, FALSE);
//	gtk_window_set_resizable ((GtkWindow*) selector->filter_dialog, FALSE);
	gtk_window_set_destroy_with_parent ((GtkWindow*) selector->filter_dialog, TRUE);
	gtk_window_set_transient_for ((GtkWindow*) selector->filter_dialog, selector->parent);
	gtk_window_resize ((GtkWindow*) selector->filter_dialog, 480, 330);
	g_signal_connect (selector->filter_dialog, "response", G_CALLBACK (on_filter_dialog_response), selector);

	gtk_box_pack_start ((GtkBox*) selector->filter_dialog->vbox, gtk_label_new (_("Mark URLs by host AND filename extension.")), FALSE, FALSE, 3);
	gtk_box_pack_start ((GtkBox*) selector->filter_dialog->vbox, gtk_label_new (_("This will reset all marks of URLs.")), FALSE, FALSE, 3);

	hbox = (GtkBox*) gtk_hbox_new (FALSE, 2);
	gtk_box_pack_start ((GtkBox*) selector->filter_dialog->vbox, (GtkWidget*) hbox, TRUE, TRUE, 1);

	// left side ----------------------------
	vbox = (GtkBox*) gtk_vbox_new (FALSE, 2);
	gtk_box_pack_start (hbox, (GtkWidget*) vbox, TRUE, TRUE, 2);
	// filter
	selector->filter_host = ug_item_view_new (NULL, _("Host"));
	gtk_widget_set_size_request ((GtkWidget*) selector->filter_host, 120, 120);
	ug_item_view_for_selector (selector->filter_host);
	scrolled = gtk_scrolled_window_new (NULL, NULL);
	gtk_scrolled_window_set_shadow_type (GTK_SCROLLED_WINDOW (scrolled), GTK_SHADOW_IN);
	gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (scrolled),
	                                GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
	gtk_container_add (GTK_CONTAINER (scrolled), GTK_WIDGET(selector->filter_host));
	gtk_box_pack_start (vbox, scrolled, TRUE, TRUE, 2);
	// button
	sizegroup = gtk_size_group_new (GTK_SIZE_GROUP_HORIZONTAL);
	hbox_button = (GtkBox*) gtk_hbox_new (FALSE, 2);
	gtk_box_pack_start (vbox, (GtkWidget*) hbox_button, FALSE, FALSE, 2);
	button = gtk_button_new_with_mnemonic (_("_All"));
	gtk_size_group_add_widget (sizegroup, button);
	gtk_box_pack_start (hbox_button, button, FALSE, FALSE, 1);
	g_signal_connect (button, "clicked", G_CALLBACK (on_filter_host_all), selector);
	button = gtk_button_new_with_mnemonic (_("_None"));
	gtk_size_group_add_widget (sizegroup, button);
	gtk_box_pack_start (hbox_button, button, FALSE, FALSE, 1);
	g_signal_connect (button, "clicked", G_CALLBACK (on_filter_host_none), selector);

	// right side (filename extension)
	vbox = (GtkBox*) gtk_vbox_new (FALSE, 2);
	gtk_box_pack_start (hbox, (GtkWidget*) vbox, FALSE, TRUE, 2);
	// filter
	selector->filter_ext = ug_item_view_new (NULL, _("File Ext."));
	gtk_widget_set_size_request ((GtkWidget*) selector->filter_ext, 120, 120);
	ug_item_view_for_selector (selector->filter_ext);
	scrolled = gtk_scrolled_window_new (NULL, NULL);
	gtk_scrolled_window_set_shadow_type (GTK_SCROLLED_WINDOW (scrolled), GTK_SHADOW_IN);
	gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (scrolled),
		                                GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
	gtk_container_add (GTK_CONTAINER (scrolled), GTK_WIDGET(selector->filter_ext));
	gtk_box_pack_start (vbox, scrolled, TRUE, TRUE, 2);
	// button
	sizegroup = gtk_size_group_new (GTK_SIZE_GROUP_HORIZONTAL);
	hbox_button = (GtkBox*) gtk_hbox_new (FALSE, 2);
	gtk_box_pack_start (vbox, (GtkWidget*) hbox_button, FALSE, FALSE, 2);
	button = gtk_button_new_with_mnemonic (_("A_ll"));
	gtk_size_group_add_widget (sizegroup, button);
	gtk_box_pack_start (hbox_button, button, FALSE, FALSE, 1);
	g_signal_connect (button, "clicked", G_CALLBACK (on_filter_ext_all), selector);
	button = gtk_button_new_with_mnemonic (_("N_one"));
	gtk_size_group_add_widget (sizegroup, button);
	gtk_box_pack_start (hbox_button, button, FALSE, FALSE, 1);
	g_signal_connect (button, "clicked", G_CALLBACK (on_filter_ext_none), selector);

	gtk_widget_show_all (selector->filter_dialog->vbox);

	model = g_ptr_array_index (selector->page_array, index + ELEMENT_HOST_STORE);
	gtk_tree_view_set_model (selector->filter_host, model);
	model = g_ptr_array_index (selector->page_array, index + ELEMENT_EXT_STORE);
	gtk_tree_view_set_model (selector->filter_ext, model);

	if (gtk_window_get_modal (selector->parent))
		gtk_dialog_run (selector->filter_dialog);
	else
		gtk_widget_show ((GtkWidget*) selector->filter_dialog);
}

//	signal handler ------------------------------------------------------------
static void on_mark_all (GtkWidget* button, UgSelector* selector)
{
	gint			nth_page;
	GtkListStore*	liststore;
	GtkWidget*		widget;

	nth_page = gtk_notebook_get_current_page (selector->notebook);
	if (nth_page < 0)
		return;
	liststore = ug_selector_get_store (selector, nth_page);
	ug_item_store_set_mark_all (liststore, TRUE);
	widget = g_ptr_array_index (selector->page_array, nth_page * ELEMENT_SIZE + ELEMENT_VIEW);
	gtk_widget_queue_draw (widget);
}

static void on_mark_none (GtkWidget* button, UgSelector* selector)
{
	gint			nth_page;
	GtkListStore*	liststore;
	GtkWidget*		widget;

	nth_page = gtk_notebook_get_current_page (selector->notebook);
	if (nth_page < 0)
		return;
	liststore = ug_selector_get_store (selector, nth_page);
	ug_item_store_set_mark_all (liststore, FALSE);
	widget = g_ptr_array_index (selector->page_array, nth_page * ELEMENT_SIZE + ELEMENT_VIEW);
	gtk_widget_queue_draw (widget);
}

static void on_mark_by_filter (GtkWidget* button, UgSelector* selector)
{
	gint		index;		// begin index

	index = gtk_notebook_get_current_page (selector->notebook) * ELEMENT_SIZE;
	if (index < 0)
		return;
	ug_selector_make_filter (selector, index);
	ug_selector_show_filter (selector, index);
}

static void	on_message_dialog_response (GtkDialog *dialog, gint response, UgSelector* selector)
{
	gtk_widget_set_sensitive (selector->self, TRUE);	// set sensitive FALSE in ug_selector_show_error()
	gtk_widget_destroy ((GtkWidget*) dialog);
	selector->message_dialog = NULL;

	if (selector->parent)
		gtk_widget_set_sensitive ((GtkWidget*) selector->parent, TRUE);
}

static gboolean  item_store_foreach (GtkTreeModel *model, GtkTreePath *path, GtkTreeIter *iter, gpointer data)
{
	UgItem*	item;

	// decrease mark count
	gtk_tree_model_get (model, iter, 0, &item, -1);
	if (item->mark > 0)
		item->mark--;
	return FALSE;
}

static void on_filter_dialog_response (GtkDialog* dialog, gint response, UgSelector* selector)
{
	GtkListStore*	store;
	GHashTable*		hash;
	GList*			marked;
	GList*			link;
	GList*			related;
	guint			index;

	if (selector->parent)
		gtk_widget_set_sensitive ((GtkWidget*) selector->parent, TRUE);
	gtk_widget_hide ((GtkWidget*) dialog);
	if (response == GTK_RESPONSE_OK) {
		index = gtk_notebook_get_current_page (selector->notebook) * ELEMENT_SIZE;
		hash  = g_ptr_array_index (selector->page_array, index + ELEMENT_HASH);
		// clear all mark
		store = g_ptr_array_index (selector->page_array, index + ELEMENT_STORE);
		ug_item_store_set_mark_all (store, FALSE);
		// host
		store = (GtkListStore*) gtk_tree_view_get_model (selector->filter_host);
		marked = ug_item_store_get_marked (store);
		for (link = marked;  link;  link = link->next) {
			related = g_hash_table_lookup (hash, ((UgItem*)link->data)->value);
			for (; related; related=related->next)
				((UgItem*)related->data)->mark++;	// increase mark count
		}
		g_list_free (marked);
		// file ext
		store = (GtkListStore*) gtk_tree_view_get_model (selector->filter_ext);
		marked = ug_item_store_get_marked (store);
		for (link = marked;  link;  link = link->next) {
			related = g_hash_table_lookup (hash, ((UgItem*)link->data)->value);
			for (; related; related=related->next)
				((UgItem*)related->data)->mark++;	// increase mark count
		}
		g_list_free (marked);
		// remark
		store = g_ptr_array_index (selector->page_array, index + ELEMENT_STORE);
		gtk_tree_model_foreach ((GtkTreeModel*) store, item_store_foreach, NULL);
		gtk_widget_queue_draw ((GtkWidget*) selector->notebook);
	}

	gtk_widget_destroy ((GtkWidget*) dialog);
	selector->filter_dialog = NULL;
	return;
}

static void on_filter_host_all (GtkWidget* widget, UgSelector* selector)
{
	GtkListStore*	store;
	guint			index;

	index = gtk_notebook_get_current_page (selector->notebook) * ELEMENT_SIZE;
	store = g_ptr_array_index (selector->page_array, index + ELEMENT_HOST_STORE);
	ug_item_store_set_mark_all (store, TRUE);
	gtk_widget_queue_draw ((GtkWidget*) selector->filter_host);
}

static void on_filter_host_none (GtkWidget* widget, UgSelector* selector)
{
	GtkListStore*	store;
	guint			index;

	index = gtk_notebook_get_current_page (selector->notebook) * ELEMENT_SIZE;
	store = g_ptr_array_index (selector->page_array, index + ELEMENT_HOST_STORE);
	ug_item_store_set_mark_all (store, FALSE);
	gtk_widget_queue_draw ((GtkWidget*) selector->filter_host);
}

static void on_filter_ext_all (GtkWidget* widget, UgSelector* selector)
{
	GtkListStore*	store;
	guint			index;

	index = gtk_notebook_get_current_page (selector->notebook) * ELEMENT_SIZE;
	store = g_ptr_array_index (selector->page_array, index + ELEMENT_EXT_STORE);
	ug_item_store_set_mark_all (store, TRUE);
	gtk_widget_queue_draw ((GtkWidget*) selector->filter_ext);
}

static void on_filter_ext_none (GtkWidget* widget, UgSelector* selector)
{
	GtkListStore*	store;
	guint			index;

	index = gtk_notebook_get_current_page (selector->notebook) * ELEMENT_SIZE;
	store = g_ptr_array_index (selector->page_array, index + ELEMENT_EXT_STORE);
	ug_item_store_set_mark_all (store, FALSE);
	gtk_widget_queue_draw ((GtkWidget*) selector->filter_ext);
}

