/*
 *
 *   Copyright (C) 2005-2010 by Raymond Huang
 *   plushuang at users.sourceforge.net
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *
 *  ---
 *
 *  In addition, as a special exception, the copyright holders give
 *  permission to link the code of portions of this program with the
 *  OpenSSL library under certain conditions as described in each
 *  individual source file, and distribute linked combinations
 *  including the two.
 *  You must obey the GNU Lesser General Public License in all respects
 *  for all of the code used other than OpenSSL.  If you modify
 *  file(s) with this exception, you may extend this exception to your
 *  version of the file(s), but you are not obligated to do so.  If you
 *  do not wish to do so, delete this exception statement from your
 *  version.  If you delete this exception statement from all source
 *  files in the program, then also delete it here.
 *
 */

#include <ug_batch_form.h>
#include <ug_utils.h>
#include <ug_url.h>

#include <glib/gi18n.h>

static void on_radio1_clicked (GtkWidget* widget, UgBatchForm* bform);
static void on_radio2_clicked (GtkWidget* widget, UgBatchForm* bform);

void	ug_batch_form_init (UgBatchForm* bform)
{
	GtkTable*		table;
//	GtkWidget*		vbox;
	GtkWidget*		label;
	GtkAdjustment*	spinner_adj;
	GtkEditable*	editable;
	GtkWidget*			scrolled;
	GtkCellRenderer*	renderer;
	GtkTreeViewColumn*	column;
	GtkTreeSelection*	selection;

	bform->self = gtk_table_new (6, 6, FALSE);
	table = (GtkTable*) bform->self;

	// URL entry
	label = gtk_label_new_with_mnemonic (_("_URL:"));
	bform->entry = gtk_entry_new ();
	editable = GTK_EDITABLE (bform->entry);

	gtk_table_attach (table, label, 0, 1, 0, 1, GTK_SHRINK, GTK_SHRINK, 3, 3);
	gtk_table_attach (table, bform->entry, 1, 6, 0, 1, GTK_FILL | GTK_EXPAND, GTK_SHRINK, 3, 3);

	label = gtk_label_new (_("e.g."));
	gtk_table_attach (table, label, 0, 1, 1, 2, GTK_SHRINK, GTK_SHRINK, 3, 3);
	label = gtk_label_new ("http://for.example/path/pre*.jpg");
	gtk_table_attach (table, label, 1, 6, 1, 2, GTK_SHRINK, GTK_SHRINK, 3, 3);
	gtk_table_attach (table, gtk_hseparator_new (), 0, 6, 2, 3, GTK_FILL | GTK_EXPAND, GTK_SHRINK, 3, 3);

	// spin "From"
	bform->radio_button = gtk_radio_button_new_with_mnemonic (NULL, _("_From:"));
	g_signal_connect (bform->radio_button, "clicked", G_CALLBACK(on_radio1_clicked), bform);
	gtk_table_attach (table, bform->radio_button, 0, 1, 3, 4,
	                  GTK_SHRINK, GTK_SHRINK, 3, 3);
	spinner_adj = (GtkAdjustment *) gtk_adjustment_new (0.0, 0.0,
	                                                    99999.0, 1.0,
	                                                    5.0, 0.0);
	bform->spin_from = gtk_spin_button_new (spinner_adj, 1.0, 0);
	gtk_table_attach (table, bform->spin_from, 1, 2, 3, 4, GTK_FILL | GTK_EXPAND, GTK_SHRINK, 3, 3);
	g_signal_connect_swapped (bform->spin_from, "value-changed", G_CALLBACK(ug_batch_form_update_preview), bform);

	// spin "To"
	label = gtk_label_new_with_mnemonic (_("_To:"));
	gtk_table_attach (table, label, 2, 3, 3, 4, GTK_SHRINK, GTK_SHRINK, 3, 3);
	bform->label_to = label;
	spinner_adj = (GtkAdjustment *) gtk_adjustment_new (10.0, 1.0,
	                                                    99999.0, 1.0,
	                                                    5.0, 0.0);
	bform->spin_to = gtk_spin_button_new (spinner_adj, 1.0, 0);
	gtk_label_set_mnemonic_widget (GTK_LABEL(label), bform->spin_to);
	gtk_table_attach (table, bform->spin_to, 3, 4, 3, 4, GTK_FILL | GTK_EXPAND, GTK_SHRINK, 3, 3);
	g_signal_connect_swapped (bform->spin_to, "value-changed", G_CALLBACK(ug_batch_form_update_preview), bform);

	// spin "digits"
	label = gtk_label_new_with_mnemonic (_("digits:"));
	gtk_table_attach (table, label, 4, 5, 3, 4, GTK_SHRINK, GTK_SHRINK, 3, 3);
	bform->digits_label = label;
	spinner_adj = (GtkAdjustment *) gtk_adjustment_new (2.0, 1.0,
	                                                    20.0, 1.0,
	                                                    5.0, 0.0);
	bform->digits_spin = gtk_spin_button_new (spinner_adj, 1.0, 0);
	gtk_label_set_mnemonic_widget (GTK_LABEL(label), bform->digits_spin);
	gtk_table_attach (table, bform->digits_spin, 5, 6, 3, 4, GTK_FILL | GTK_EXPAND, GTK_SHRINK, 3, 3);
	g_signal_connect_swapped (bform->digits_spin, "value-changed", G_CALLBACK(ug_batch_form_update_preview), bform);

	// a - z
	bform->radio_button = gtk_radio_button_new_with_mnemonic_from_widget (GTK_RADIO_BUTTON(bform->radio_button),
	                                                                   _("F_rom:"));
	g_signal_connect (bform->radio_button, "clicked", G_CALLBACK(on_radio2_clicked), bform);
	gtk_table_attach (table, bform->radio_button, 0, 1, 4, 5, GTK_SHRINK, GTK_SHRINK, 3, 3);
	bform->entry_a = gtk_entry_new ();
	gtk_entry_set_text (GTK_ENTRY(bform->entry_a), "a");
	gtk_entry_set_max_length (GTK_ENTRY(bform->entry_a), 1);
	gtk_entry_set_width_chars (GTK_ENTRY(bform->entry_a), 2);
	gtk_widget_set_sensitive (bform->entry_a, FALSE);
	gtk_table_attach (table, bform->entry_a, 1, 2, 4, 5, GTK_FILL | GTK_EXPAND, GTK_SHRINK, 3, 3);
	g_signal_connect_swapped (GTK_EDITABLE (bform->entry_a), "changed", G_CALLBACK(ug_batch_form_update_preview), bform);

	bform->entry_z = gtk_entry_new ();
	gtk_entry_set_text (GTK_ENTRY(bform->entry_z), "z");
	gtk_entry_set_max_length (GTK_ENTRY(bform->entry_z), 1);
	gtk_entry_set_width_chars (GTK_ENTRY(bform->entry_z), 2);
	gtk_widget_set_sensitive (bform->entry_z, FALSE);
	gtk_table_attach (table, bform->entry_z, 3, 4, 4, 5, GTK_FILL | GTK_EXPAND, GTK_SHRINK, 3, 3);
	g_signal_connect_swapped (GTK_EDITABLE (bform->entry_z), "changed", G_CALLBACK(ug_batch_form_update_preview), bform);

	label = gtk_label_new (_("case-sensitive"));
	gtk_widget_set_sensitive (label, FALSE);
	gtk_table_attach (table, label, 4, 6, 4, 5, GTK_FILL, GTK_SHRINK, 3, 3);
	bform->label_case = label;

	// preview
	bform->preview = (GtkTreeView*) gtk_tree_view_new ();
	bform->preview_list = gtk_list_store_new (1, G_TYPE_STRING);
	gtk_tree_view_set_model (bform->preview, (GtkTreeModel*) bform->preview_list);
//	gtk_tree_view_set_fixed_height_mode (bform->preview, TRUE);
	gtk_widget_set_size_request ((GtkWidget*) bform->preview, 140, 140);
	selection = gtk_tree_view_get_selection (bform->preview);
	gtk_tree_selection_set_mode (selection, GTK_SELECTION_NONE);
	// It will free UgBatchForm.preview_list when UgBatchForm.preview destroy.
	g_object_unref (bform->preview_list);

	renderer = gtk_cell_renderer_text_new ();
	column   = gtk_tree_view_column_new_with_attributes (_("Preview"), renderer, "text", 0, NULL);
//	gtk_tree_view_column_set_sizing (column, GTK_TREE_VIEW_COLUMN_FIXED);
	gtk_tree_view_append_column (bform->preview, column);
	scrolled = gtk_scrolled_window_new (NULL, NULL);
	gtk_scrolled_window_set_shadow_type (GTK_SCROLLED_WINDOW (scrolled), GTK_SHADOW_IN);
	gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (scrolled),
	                                GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
	gtk_container_add (GTK_CONTAINER(scrolled), GTK_WIDGET(bform->preview));
	gtk_table_attach (table, scrolled, 0, 6, 7, 8, GTK_FILL, GTK_FILL | GTK_EXPAND, 3, 3);

	ug_batch_form_update_preview (bform);
	gtk_widget_show_all (bform->self);
}

gboolean	ug_batch_form_update_preview (UgBatchForm* bform)
{
	GtkTreeIter		iter;
	GList*			string_list;
	GList*			link;
	UgUrlPart*		urlpart;

	string_list = ug_batch_form_get_list (bform, TRUE);

	gtk_list_store_clear (bform->preview_list);
	// character mode
	if (gtk_toggle_button_get_active ((GtkToggleButton*) bform->radio_button)) {
		if (*gtk_entry_get_text ((GtkEntry*) bform->entry_a) == 0 ||
		    *gtk_entry_get_text ((GtkEntry*) bform->entry_z) == 0)
		{
			gtk_list_store_append (bform->preview_list, &iter);	// skip one row
			gtk_list_store_append (bform->preview_list, &iter);
			gtk_list_store_set (bform->preview_list, &iter, 0, _("No character in 'From' or 'To' entry."), -1);
			return FALSE;
		}
	}
	// character mode and digit mode
	if (string_list == NULL) {
		gtk_list_store_append (bform->preview_list, &iter);	// skip one row
		gtk_list_store_append (bform->preview_list, &iter);
		gtk_list_store_set (bform->preview_list, &iter, 0, _("No wildcard(*) character in URL entry."), -1);
		return FALSE;
	}
	else {
		urlpart = ug_url_part_new (string_list->data, -1);
		if (urlpart->url_scheme_len == 0) {
			gtk_list_store_append (bform->preview_list, &iter);	// skip one row
			gtk_list_store_append (bform->preview_list, &iter);
			gtk_list_store_set (bform->preview_list, &iter, 0, _("URL is not valid."), -1);
			ug_url_part_free (urlpart);
			return FALSE;
		}
		ug_url_part_free (urlpart);
	}

	for (link = g_list_last (string_list);  link;  link = link->prev) {
		gtk_list_store_append (bform->preview_list, &iter);
		gtk_list_store_set (bform->preview_list, &iter, 0, link->data, -1);
	}
	g_list_foreach (string_list, (GFunc) g_free, NULL);
	g_list_free (string_list);
	return TRUE;
}

GList*		ug_batch_form_get_list (UgBatchForm* bform, gboolean preview_only)
{
	GString*	gstr;
	guint		from, to, cur, digits;
	gboolean	char_mode;
	const gchar*	string;
	gint			string_len;
	guint			string_offset;
	GList*			result;

	result = NULL;
	string = gtk_entry_get_text (GTK_ENTRY(bform->entry));
	string_len = strlen (string);
	string_offset = ug_str_find_charset (string, string_len, 0, "*");
	if (string_offset == string_len)
		return NULL;

	char_mode = gtk_toggle_button_get_active ((GtkToggleButton*) bform->radio_button);
	if (char_mode) {
		from   = *gtk_entry_get_text ((GtkEntry*) bform->entry_a);
		to     = *gtk_entry_get_text ((GtkEntry*) bform->entry_z);
		digits = 1;
		if (from == 0 || to == 0)
			return NULL;
	}
	else {
		from   = gtk_spin_button_get_value_as_int ((GtkSpinButton*) bform->spin_from);
		to     = gtk_spin_button_get_value_as_int ((GtkSpinButton*) bform->spin_to);
		digits = gtk_spin_button_get_value_as_int ((GtkSpinButton*) bform->digits_spin);
	}
	// swap from & to
	if (from > to) {
		cur = from;
		from = to;
		to = cur;
	}

	gstr = g_string_sized_new (80);
	g_string_append_len (gstr, string, string_offset);
	for (cur = from;  cur <= to;  cur++) {
		if (preview_only && cur == from + 3 && cur + 1 < to) {
			result = g_list_prepend (result, g_strdup (" ..."));
			cur = to;
		}
		if (char_mode)
			g_string_append_printf (gstr, "%c", cur);
		else
			g_string_append_printf (gstr, "%.*d", digits, cur);
		g_string_append (gstr, string + string_offset + 1);
		result = g_list_prepend (result, g_strdup (gstr->str));
		g_string_set_size (gstr, string_offset);
	}
	g_string_free (gstr, TRUE);
	return result;
}

// ----------------------------------------------------------------------------
//	signal handler
static void on_radio1_clicked(GtkWidget* widget, UgBatchForm* bform)
{
	gtk_widget_set_sensitive (bform->spin_from, TRUE);
	gtk_widget_set_sensitive (bform->spin_to, TRUE);
	gtk_widget_set_sensitive (bform->label_to, TRUE);
	gtk_widget_set_sensitive (bform->digits_label, TRUE);
	gtk_widget_set_sensitive (bform->digits_spin, TRUE);
	gtk_widget_set_sensitive (bform->label_case, FALSE);
	gtk_widget_set_sensitive (bform->entry_a, FALSE);
	gtk_widget_set_sensitive (bform->entry_z, FALSE);
	ug_batch_form_update_preview (bform);
}

static void on_radio2_clicked(GtkWidget* widget, UgBatchForm* bform)
{
	gtk_widget_set_sensitive (bform->spin_from, FALSE);
	gtk_widget_set_sensitive (bform->spin_to, FALSE);
	gtk_widget_set_sensitive (bform->label_to, FALSE);
	gtk_widget_set_sensitive (bform->digits_label, FALSE);
	gtk_widget_set_sensitive (bform->digits_spin, FALSE);
	gtk_widget_set_sensitive (bform->label_case, TRUE);
	gtk_widget_set_sensitive (bform->entry_a, TRUE);
	gtk_widget_set_sensitive (bform->entry_z, TRUE);
	ug_batch_form_update_preview (bform);
}

