#ifndef INCLUDED_BASE32_H
#define INCLUDED_BASE32_H
/* vim:ts=4:sw=4:noet
 * (tabspace=4)
 * 
 * Copyright (C) 2004, 2005 Walter Doekes, <walter@djcvt.net>.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

/**
 * Conversion routines to and from BASE32.
 */

#include "types.h"

#ifdef __cplusplus
extern "C" {
#endif /* __cplusplus */

/**
 * Convert an octet-stream to base32.
 * The octet-stream is supplied through src, the size of the octet-stream
 * through len.
 * The base32 will be put in dest, along with a terminating zero, make
 * sure there is room for ceil(len*8/5)+1 characters.
 */
int uint8tobase32(unsigned char* dest, const uint8_t* src, unsigned len);

/**
 * Convert an uint64_t-stream to base32.
 * See uint8tobase32(). dest must be at least ceil(len*8*8/5)+1 characters
 * long.
 * Use this if you're feeding 64bit numbers into the stream. If you use the
 * 8bit equivalent, BIG_ENDIAN machines will get things wrong.
 * This may actually fail if the malloc fails.
 */
int uint64tobase32(unsigned char* dest, const uint64_t* src, unsigned len);

/**
 * Convert a base32 octet-stream to exactly len uint8's.
 * Make sure your dest memory is initialized to zero!
 * If the conversion fails (due to invalid characters in src), it will return
 * -1 instead of 0.
 */
int base32touint8(uint8_t* dest, const unsigned char* src, unsigned len);

/**
 * Convert a base32 octet-stream to exactly len uint64's.
 * Make sure your dest memory is initialized to zero!
 * If the conversion fails (due to invalid characters in src), it will return
 * -1 instead of 0.
 */
int base32touint64(uint64_t* dest, const unsigned char* src, unsigned len);

#ifdef __cplusplus
} /* extern "C" */
#endif /* __cplusplus */

#endif /* INCLUDED_BASE32_H */
