/**
 * Copyright (C) 2007-2012 Lawrence Murray
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the Free
 * Software Foundation; either version 2 of the License, or (at your option)
 * any later version.
 * 
 * @author Lawrence Murray <lawrence@indii.org>
 * $Rev: 364 $
 * $Date: 2012-01-22 20:27:14 +0800 (Sun, 22 Jan 2012) $
 */
#include "SpectrumImage.hpp"

#include "../../image/ImageManipulation.hpp"

#include "wx/dcmemory.h"
#include "wx/dcbuffer.h"

#define ICON_HEIGHT 4
#define ICON_WIDTH 256

using namespace indii;

SpectrumImage::SpectrumImage(wxWindow *parent, const unsigned int cluster,
    const Type type) :
    wxPanel(parent), model(NULL), cluster(cluster), type(type),
    fg(ICON_WIDTH,ICON_HEIGHT) {
  //SetBackgroundStyle(wxBG_STYLE_CUSTOM);
  SetSize(ICON_WIDTH+2,ICON_HEIGHT+2);
  SetMinSize(wxSize(ICON_WIDTH+2,ICON_HEIGHT+2));
  
  fgPrepared = false;
}

void SpectrumImage::setModel(ClusterModel* model) {
  this->model = model;
  ignore();

  if (model != NULL) {
    watch(model);
    fgPrepared = false;
    Refresh();
  }
}

void SpectrumImage::notifyNumClustersChange() {
  /* pre-condition */
  assert (model != NULL);

  if (cluster < model->getNumClusters()) {
    fgPrepared = false;
    Refresh();
  }
}

void SpectrumImage::notifyClusterChange(const unsigned int i) {
  //
}

void SpectrumImage::OnPaint(wxPaintEvent& evt) {
  /* pre-condition */
  assert (model != NULL);

  if (cluster < model->getNumClusters()) {
    if (!fgPrepared) {
      prepareForeground();
    }
    
    wxPaintDC dc(this);
    if (IsEnabled()) {
      dc.SetBackground(wxBrush(wxColour(99,99,99)));
      dc.Clear();
      dc.DrawBitmap(fg, 1, 1);
    } else {
      dc.SetBackground(wxBrush(wxColour(200,200,200)));
      dc.Clear();
    }
  }
}

void SpectrumImage::prepareForeground() {
  ColourSpace cs;
  ColourSpace::rgb_t rgb(3);
  ColourSpace::hsl_t hsl1(3), hsl2(3);
  int x, y;
  float adj, val;
  
  wxMemoryDC dc(fg);  
  wxColour c(model->getColour(cluster));
  wxImage img(ICON_WIDTH, ICON_HEIGHT);
    
  rgb(0) = c.Red();
  rgb(1) = c.Green();
  rgb(2) = c.Blue();
  
  cs.rgb2hsl(rgb, hsl1);
  
  #pragma omp for schedule(dynamic)
  for (x = 0; x < ICON_WIDTH; x++) {
    switch (type) {
      case H:
        adj = -0.5f + static_cast<float>(x)/ICON_WIDTH;
        val = fmod(6.0f + hsl1(0) + 6.0f*adj, 6.0f);
        hsl2(0) = val;
        hsl2(1) = 0.8f;
        hsl2(2) = 190.0f;
        break;
      case S:
        adj = (2.0f*x - ICON_WIDTH)/ICON_WIDTH;
        val = cs.bound(0.0f, adj + hsl1(1), 1.0f);
        hsl2(0) = hsl1(0);
        hsl2(1) = val;
        hsl2(2) = 190.0f;
        break;
      case L:
        adj = 255.0f*(2.0f*x - ICON_WIDTH)/ICON_WIDTH;
        val = cs.bound(0.0f, adj + hsl1(2), 255.0f);
        hsl2(0) = hsl1(0);
        hsl2(1) = 0.0f;
        hsl2(2) = val;
        break;
    }
    cs.hsl2rgb(hsl2, rgb);
    
    for (y = 0; y < ICON_HEIGHT; y++) {
      img.SetRGB(x, y, rgb(0), rgb(1), rgb(2));
    }
  }
  
  dc.DrawBitmap(img, 0, 0, true);
  fgPrepared = true;
}

BEGIN_EVENT_TABLE(SpectrumImage, wxPanel)
EVT_PAINT(SpectrumImage::OnPaint)
END_EVENT_TABLE()
