/*
 * Serialize a tagged collection to a text file
 *
 * Copyright (C) 2003,2004,2005  Enrico Zini <enrico@debian.org>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 */

#include <tagcoll/TextFormat.h>
#include <tagcoll/patch.h>

using namespace std;
using namespace wibble;

namespace tagcoll {
namespace textformat {

// Parse an element
// Return the trailing separating char, that can be:
//  input::Input::Eof
//  '\n'
//  ':'
//  ','
// Return the item in `item'

// element: \s*[^ \t,:]\s*([.:])\s*
// or
// element: \s*[^ \t,:].*?[^ \t,:]\s*([.:])\s+
int parseElement(input::Input& in, string& item)
{
	item = string();
	string sep;
	int c;
	char sepchar = 0;
	enum {LSPACE, ITEM, ISPACE, ISEP, TSPACE} state = LSPACE;
	while ((c = in.nextChar()) != input::Input::Eof)
	{
		if (c == '\n')
		{
			if (sepchar && sepchar != ':')
				throw tagcoll::exception::Input("separator character ends the line");
			else
				return '\n';
		}
		switch (state)
		{
			// Optional leading space
			case LSPACE:
				switch (c)
				{
					case ' ':
					case '\t':
						break;
					case ':':
					case ',':
						throw tagcoll::exception::Input("element cannot start with a separation character");
						break;
					default:
						item += c;
						state = ITEM;
						break;
				}
				break;
			// Non-separating characters
			case ITEM:
				switch (c)
				{
					case ' ':
					case '\t':
						sep += c;
						state = ISPACE;
						break;
					case ':':
					case ',':
						sepchar = c;
						sep += c;
						state = ISEP;
						break;
					default:
						item += c;
						break;
				}
				break;
			// Space inside item or at the end of item
			case ISPACE:
				switch (c)
				{
					case ' ':
					case '\t':
						sep += c;
						break;
					case ':':
					case ',':
						sepchar = c;
						state = TSPACE;
						break;
					default:
						item += sep;
						item += c;
						sep = string();
						state = ITEM;
						break;
				}
				break;
			// Separator inside item or at the end of item
			case ISEP:
				switch (c)
				{
					case ' ':
					case '\t':
						if (sep.size() > 1)
							throw tagcoll::exception::Input("item is followed by more than one separator characters");
						state = TSPACE;
						break;
					case ':':
					case ',':
						sep += c;
						break;
					default:
						item += sep;
						item += c;
						sepchar = 0;
						sep = string();
						state = ITEM;
						break;
				}
				break;
			case TSPACE:
				switch (c)
				{
					case ' ':
					case '\t':
						break;
					default:
						in.pushChar(c);
						return sepchar;
				}
				break;
		}
	}
	return input::Input::Eof;
}


void outputPatch(const PatchList<std::string, std::string>& patch, FILE* out)
{
	for (PatchList<std::string, std::string>::const_iterator i = patch.begin();
			i != patch.end(); i++)
	{
		if (fprintf(out, "%s: ", i->first.c_str()) < 0)
			throw wibble::exception::System("writing item");

		bool start = true;

		std::set<string> stags;
		for (std::set<std::string>::const_iterator j = i->second.added.begin();
				j != i->second.added.end(); j++)
		{
			if (start)
			{
				if (fprintf(out, "+%s", j->c_str()) < 0)
					throw wibble::exception::System("writing patched tag");
				start = false;
			} else
				if (fprintf(out, ", +%s", j->c_str()) < 0)
					throw wibble::exception::System("writing patched tag");
		}
		for (std::set<std::string>::const_iterator j = i->second.removed.begin();
				j != i->second.removed.end(); j++)
		{
			if (start)
			{
				if (fprintf(out, "-%s", j->c_str()) < 0)
					throw wibble::exception::System("writing patched tag");
				start = false;
			} else
				if (fprintf(out, ", -%s", j->c_str()) < 0)
					throw wibble::exception::System("writing patched tag");
		}

		if (fprintf(out, "\n") < 0)
			throw wibble::exception::System("writing newline after tagset");
	}
}

PatchList<std::string, std::string> parsePatch(input::Input& in)
{
	PatchList<string, string> patch;
	parse(in, patchAssembler(inserter(patch)));
	return patch;
}

}
}

#include <tagcoll/TextFormat.tcc>

// vim:set ts=4 sw=4:
