/*
 * synaptiks -- a touchpad control tool
 *
 *
 * Copyright (C) 2009, 2010 Sebastian Wiesner <basti.wiesner@gmx.net>
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */


#ifndef MOUSEDEVICESMONITOR_H
#define MOUSEDEVICESMONITOR_H

/**
 * @file
 *
 * Provide a class to monitor and query mouse devices.
 */

#include <QtCore/QObject>

namespace synaptiks {

    class MouseDevicesMonitorPrivate;

    /**
     * @brief Monitor and query plugged mouse devices.
     *
     * This class constantly monitors mouse devices.  It emits signals, if
     * mouse devices are plugged or unplugged, and provides rudimentary
     * information about plugged mouse devices.
     *
     * \section usage Usage
     *
     * To be informed about plugged and unplugged mouses, just connect to
     * the corresponding signals:
     *
     * @code
     * MouseDevicesMonitor *monitor = new MouseDevicesMonitor(this);
     * this->connect(monitor, SIGNAL(mousePlugged(const QString&)),
     *               SLOT(doSomethingWithPluggedMouse(const QString&)));
     * this->connect(monitor, SIGNAL(mouseUnplugged(const QString&)),
     *               SLOT(doSomethingWithUnluggedMouse(const QString&)));
     * @endcode
     *
     * You can also use this class to enumerate plugged devices:
     *
     * @code
     * foreach (QString mouse, monitor->pluggedMouseDevices()) {
     *     kDebug() << monitor->productName(mouse);
     * }
     * @endcode
     *
     * \section dbus D-Bus
     *
     * All members marked as @c Q_SCRIPTABLE are exported on D-Bus on the
     * interface @c org.kde.MouseDevicesManager.
     */
    class MouseDevicesMonitor: public QObject {
        Q_OBJECT
        Q_CLASSINFO("D-Bus Interface", "org.kde.MouseDevicesMonitor")

    public:
        /**
         * @brief Create a new object.
         *
         * @param parent same as for QObject(QObject*)
         */
        MouseDevicesMonitor(QObject *parent=0);

        /**
         * @brief Destroy this object.
         */
        virtual ~MouseDevicesMonitor();

    public Q_SLOTS:
        /**
         * @brief List all currently plugged mouse devices.
         *
         * @return a list containing the ids of all plugged mouse devices
         */
        Q_SCRIPTABLE QStringList pluggedMouseDevices() const;

        /**
         * @brief Query the product name of a device.
         *
         * @warning
         *
         * Only use this method with @em plugged devices.  This class cannot
         * obtain information about devices, which are not currently
         * plugged.
         *
         * @param id the device id
         * @return the product name
         */
        Q_SCRIPTABLE QString productName(const QString &id) const;

        /**
         * @brief Is the given device a touchpad?
         *
         * @warning
         *
         * Only use this method with @em plugged devices.  This class cannot
         * obtain information about devices, which are not currently
         * plugged.
         *
         * @param id the device id
         * @return @c true, if the given device is a touchpad, @c false
         *         otherwise
         */
        Q_SCRIPTABLE bool isTouchpad(const QString &id) const;

    Q_SIGNALS:
        /**
         * @brief A mouse device was plugged.
         *
         * @param id the id of this device
         */
        Q_SCRIPTABLE void mousePlugged(const QString &id);

        /**
         * @brief A mouse device was unplugged.
         *
         * @warning
         *
         * Do @em not pass the given @p id to methods of this class.  This
         * slot is invoked @em after the mouse has been unplugged, therefore
         * this class cannot any longer obtain information about the device.
         *
         * @param id the id of this device
         */
        Q_SCRIPTABLE void mouseUnplugged(const QString &id);

    private:
        Q_DECLARE_PRIVATE(MouseDevicesMonitor)
        Q_DISABLE_COPY(MouseDevicesMonitor)

        Q_PRIVATE_SLOT(d_func(), void _k_deviceAdded(const QString&))
        Q_PRIVATE_SLOT(d_func(), void _k_deviceRemoved(const QString&))

        MouseDevicesMonitorPrivate * const d_ptr;
    };
}

#endif /* MOUSEDEVICESMONITOR_H */
