-------------------------------------------------------------------------------
-- (C) Altran Praxis Limited
-------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--=============================================================================

with ContextManager;
with E_Strings;

separate (DAG)
procedure CreateStructConstraint
  (OutputFile       : in     SPARK_IO.File_Type;
   StartTypeSym     : in     Dictionary.Symbol;
   VarSym           : in     Cells.Cell;
   Scope            : in     Dictionary.Scopes;
   AssocVar         : in     Dictionary.Symbol;
   VCGHeap          : in out Cells.Heap_Record;
   ContainsReals    : in out Boolean;
   VCGFailure       : in out Boolean;
   StructConstraint :    out Cells.Cell) is
   QuantIdentStack, ConstraintStack : CStacks.Stack;

   RecCompType, InitialVariable, IVarSym, -- symbol for variable associated with current type
     Field_Name_Sym, TypeSym : Dictionary.Symbol;

   TrueCell, OpCell, Result, TOS, TOS2, RecordElem : Cells.Cell;

   exitUp : Boolean;

   QuantNum : Natural := 0;

   -- Front end for access to ConstraintStack
   procedure PopOffConstraintStack (Result : out Cells.Cell)
   --# global in out ConstraintStack;
   --#        in out VCGHeap;
   --# derives ConstraintStack,
   --#         Result,
   --#         VCGHeap         from ConstraintStack,
   --#                              VCGHeap;
   is
   begin
      CStacks.PopOff (VCGHeap, ConstraintStack, Result);
   end PopOffConstraintStack;

   -- Front end for access to ConstraintStack
   procedure PushOnConstraintStack (Input : in Cells.Cell)
   --# global in out ConstraintStack;
   --#        in out Statistics.TableUsage;
   --#        in out VCGHeap;
   --# derives ConstraintStack       from VCGHeap &
   --#         Statistics.TableUsage from *,
   --#                                    VCGHeap &
   --#         VCGHeap               from *,
   --#                                    ConstraintStack,
   --#                                    Input;
   is
   begin
      CStacks.Push (VCGHeap, Input, ConstraintStack);
   end PushOnConstraintStack;

   -- Front end for access to QuantIdentStack
   procedure PopOffQuantIdentStack (Result : out Cells.Cell)
   --# global in out QuantIdentStack;
   --#        in out VCGHeap;
   --# derives QuantIdentStack,
   --#         Result,
   --#         VCGHeap         from QuantIdentStack,
   --#                              VCGHeap;
   is
   begin
      CStacks.PopOff (VCGHeap, QuantIdentStack, Result);
   end PopOffQuantIdentStack;

   -- Front end for access to QuantIdentStack
   procedure PushOnQuantIdentStack (Input : in Cells.Cell)
   --# global in out QuantIdentStack;
   --#        in out Statistics.TableUsage;
   --#        in out VCGHeap;
   --# derives QuantIdentStack       from VCGHeap &
   --#         Statistics.TableUsage from *,
   --#                                    VCGHeap &
   --#         VCGHeap               from *,
   --#                                    Input,
   --#                                    QuantIdentStack;
   is
   begin
      CStacks.Push (VCGHeap, Input, QuantIdentStack);
   end PushOnQuantIdentStack;

   procedure Create_Ident_Str (Ident_Str :    out LexTokenManager.Lex_String;
                               Suffix    : in     Natural)
   --# global in out LexTokenManager.State;
   --# derives Ident_Str,
   --#         LexTokenManager.State from LexTokenManager.State,
   --#                                    Suffix;
   is
      Ex_Lin : E_Strings.T;
      Ex_Str : E_Strings.T;
   begin
      -- Add I___ to start of Ex_Lin
      -- Changed to aggregate to avoid flow error
      Ex_Str := E_Strings.Copy_String (Str => "I___");
      E_Strings.Put_Int_To_String (Dest     => Ex_Lin,
                                   Item     => Suffix,
                                   Start_Pt => 1,
                                   Base     => 10);
      E_Strings.Append_Examiner_String (E_Str1 => Ex_Str,
                                        E_Str2 => Ex_Lin);
      LexTokenManager.Insert_Examiner_String (Str     => Ex_Str,
                                              Lex_Str => Ident_Str);
   end Create_Ident_Str;

   procedure Create_Quant_Ident
     (Quant_Ident   :    out Dictionary.Symbol;
      This_Type_Sym : in     Dictionary.Symbol;
      Scope         : in     Dictionary.Scopes;
      Quant_Num     : in out Natural)
   --# global in out Dictionary.Dict;
   --#        in out LexTokenManager.State;
   --#        in out SPARK_IO.File_Sys;
   --# derives Dictionary.Dict,
   --#         SPARK_IO.File_Sys     from *,
   --#                                    Dictionary.Dict,
   --#                                    LexTokenManager.State,
   --#                                    Quant_Num,
   --#                                    Scope,
   --#                                    This_Type_Sym &
   --#         LexTokenManager.State,
   --#         Quant_Num             from *,
   --#                                    Quant_Num &
   --#         Quant_Ident           from Dictionary.Dict;
   is
      Ident_Str : LexTokenManager.Lex_String;
   begin
      -- increment counter for uniqueness
      Quant_Num := Quant_Num + 1;

      Create_Ident_Str (Ident_Str => Ident_Str,
                        Suffix    => Quant_Num);
      Dictionary.AddQuantifiedVariable
        (Name        => Ident_Str,
         Comp_Unit   => ContextManager.NullUnit,
         Declaration => Dictionary.Null_Location,
         TypeMark    => This_Type_Sym,
         Region      => Dictionary.GetRegion (Scope),
         Variable    => Quant_Ident);
   end Create_Quant_Ident;

   -- Creates a 1-d array access
   procedure Create_Array_Access (Array_Elem               :    out Cells.Cell;
                                  Array_Ident, Quant_Ident : in     Cells.Cell)
   --# global in out Statistics.TableUsage;
   --#        in out VCGHeap;
   --# derives Array_Elem,
   --#         VCGHeap               from Array_Ident,
   --#                                    Quant_Ident,
   --#                                    VCGHeap &
   --#         Statistics.TableUsage from *,
   --#                                    Array_Ident,
   --#                                    Quant_Ident,
   --#                                    VCGHeap;
   is
      Param, DAG_Cell, Access_List : Cells.Cell;
   begin
      CreateCellKind (Access_List, VCGHeap, Cells.List_Function);
      SetRightArgument (Access_List, Quant_Ident, VCGHeap);
      SetLeftArgument (Quant_Ident, Quant_Ident, VCGHeap);

      CreateOpCell (Param, VCGHeap, SP_Symbols.comma);
      SetLeftArgument (Param, Array_Ident, VCGHeap);
      SetRightArgument (Param, Access_List, VCGHeap);

      CreateCellKind (DAG_Cell, VCGHeap, Cells.Element_Function);
      SetRightArgument (DAG_Cell, Param, VCGHeap);

      Array_Elem := DAG_Cell;
   end Create_Array_Access;

   -- Add an extra array quantifier to an
   -- "element (r,[i1,...,iM]" expression
   procedure ConcatArrayQuant (HeadDAG, IdentCell : in Cells.Cell)
   --# global in out Statistics.TableUsage;
   --#        in out VCGHeap;
   --# derives Statistics.TableUsage,
   --#         VCGHeap               from *,
   --#                                    HeadDAG,
   --#                                    IdentCell,
   --#                                    VCGHeap;
   is
      HeadBracket, Param, HeadVar : Cells.Cell;
   begin
      -- Starts with "element (r,[i1,i2,..,iM]" in HeadDAG
      -- and "i_N" in IdentCell
      HeadBracket := RightPtr (VCGHeap, HeadDAG); -- points to ,
      HeadBracket := RightPtr (VCGHeap, HeadBracket); -- points to [
      HeadVar     := RightPtr (VCGHeap, HeadBracket); -- points to i1 or ,
      while Cells.Get_Kind (VCGHeap, HeadVar) = Cells.Op loop
         HeadBracket := HeadVar;
         HeadVar     := RightPtr (VCGHeap, HeadVar);
      end loop;
      -- HeadBracket now points to [ or , and B-links to HeadVar
      -- HeadVar points to iM and B-links to ]

      -- Set up the pointers in the Ident cell 'iN'
      SetLeftArgument (IdentCell, IdentCell, VCGHeap);
      SetRightArgument (IdentCell, RightPtr (VCGHeap, HeadVar), VCGHeap);

      -- Get a new comma cell and set up pointers
      -- to iM (left) and iN (right)
      CreateOpCell (Param, VCGHeap, SP_Symbols.comma);
      SetLeftArgument (Param, HeadVar, VCGHeap);
      SetRightArgument (Param, IdentCell, VCGHeap);

      -- Now insert the comma into the expression
      SetRightArgument (HeadBracket, Param, VCGHeap);
      -- Should result in "element (r,[i1,i2,..,iM,iN]"
   end ConcatArrayQuant;

   -- Create a 'forall' without an explicit predicate
   procedure CreateShortQuantExpr (QuantExpr   :    out Cells.Cell;
                                   QuantIdent  : in     Cells.Cell;
                                   ThisTypeSym : in     Dictionary.Symbol)
   --# global in     Dictionary.Dict;
   --#        in out Statistics.TableUsage;
   --#        in out VCGHeap;
   --# derives QuantExpr,
   --#         VCGHeap               from Dictionary.Dict,
   --#                                    QuantIdent,
   --#                                    ThisTypeSym,
   --#                                    VCGHeap &
   --#         Statistics.TableUsage from *,
   --#                                    Dictionary.Dict,
   --#                                    QuantIdent,
   --#                                    ThisTypeSym,
   --#                                    VCGHeap;
   is
      BaseTypeSym                                                   : Dictionary.Symbol;
      TypeRange, VarDecl, Predicate, QuantBody, QuantDag, TypeIdent : Cells.Cell;
   begin
      -- create cell for base type sym
      BaseTypeSym := Dictionary.GetRootType (ThisTypeSym);
      CreateFixedVarCell (TypeIdent, VCGHeap, BaseTypeSym);

      -- create VarDecl as QuantIdent : TypeIdent
      CreateOpCell (VarDecl, VCGHeap, SP_Symbols.colon);
      SetLeftArgument (VarDecl, QuantIdent, VCGHeap);
      SetRightArgument (VarDecl, TypeIdent, VCGHeap);

      -- prefix True expression with implication of type range,
      -- if not Boolean
      CreateTrueCell (VCGHeap, Predicate);
      if not Dictionary.TypeIsBoolean (ThisTypeSym) then
         CreateRangeConstraint (QuantIdent, ThisTypeSym, VCGHeap, TypeRange);
         Imply (TypeRange, VCGHeap, Predicate);
      end if;

      -- create QuantBody as (VarDecl, --)
      CreateOpCell (QuantBody, VCGHeap, SP_Symbols.comma);
      SetLeftArgument (QuantBody, VarDecl, VCGHeap);
      SetRightArgument (QuantBody, Predicate, VCGHeap);

      CreateOpCell (QuantDag, VCGHeap, SP_Symbols.RWforall);
      SetRightArgument (QuantDag, QuantBody, VCGHeap);

      QuantExpr := QuantDag;
      -- Should return
      --   "forall (QI:TI,Range->True)" or
      --   "forall (QI:TI,True)"
   end CreateShortQuantExpr;

   procedure CreateQuantExpr
     (QuantExpr   :    out Cells.Cell;
      QuantIdent  : in     Cells.Cell;
      ThisTypeSym : in     Dictionary.Symbol;
      BoolExpr    : in     Cells.Cell)
   --# global in     Dictionary.Dict;
   --#        in out Statistics.TableUsage;
   --#        in out VCGHeap;
   --# derives QuantExpr,
   --#         VCGHeap               from BoolExpr,
   --#                                    Dictionary.Dict,
   --#                                    QuantIdent,
   --#                                    ThisTypeSym,
   --#                                    VCGHeap &
   --#         Statistics.TableUsage from *,
   --#                                    BoolExpr,
   --#                                    Dictionary.Dict,
   --#                                    QuantIdent,
   --#                                    ThisTypeSym,
   --#                                    VCGHeap;
   is
      BaseTypeSym                                                   : Dictionary.Symbol;
      TypeRange, VarDecl, Predicate, QuantBody, QuantDag, TypeIdent : Cells.Cell;
   begin
      -- create cell for base type sym
      BaseTypeSym := Dictionary.GetRootType (ThisTypeSym);
      CreateFixedVarCell (TypeIdent, VCGHeap, BaseTypeSym);

      -- create VarDecl as QuantIdent : TypeIdent
      CreateOpCell (VarDecl, VCGHeap, SP_Symbols.colon);
      SetLeftArgument (VarDecl, QuantIdent, VCGHeap);
      SetRightArgument (VarDecl, TypeIdent, VCGHeap);

      -- prefix BoolExpr with implication of [sub]type range, but
      -- suppress range for Boolean since Boolean subtypes must be full range
      Predicate := BoolExpr;
      if not Dictionary.TypeIsBoolean (ThisTypeSym) then
         CreateRangeConstraint (QuantIdent, ThisTypeSym, VCGHeap, TypeRange);
         Imply (TypeRange, VCGHeap, Predicate);
      end if;

      -- create QuantBody as (VarDecl, Predicate)
      CreateOpCell (QuantBody, VCGHeap, SP_Symbols.comma);
      SetLeftArgument (QuantBody, VarDecl, VCGHeap);
      SetRightArgument (QuantBody, Predicate, VCGHeap);

      CreateOpCell (QuantDag, VCGHeap, SP_Symbols.RWforall);
      SetRightArgument (QuantDag, QuantBody, VCGHeap);

      QuantExpr := QuantDag;
      --Should return "forall (QI:TI,Range->Bool)"
   end CreateQuantExpr;

   -- Compose two 'forall' quantifiers into a composite
   procedure ConcatQuantExpr (HeadExpr, TailExpr : in Cells.Cell)
   --# global in     Dictionary.Dict;
   --#        in     OutputFile;
   --#        in out SPARK_IO.File_Sys;
   --#        in out VCGFailure;
   --#        in out VCGHeap;
   --# derives SPARK_IO.File_Sys from *,
   --#                                Dictionary.Dict,
   --#                                OutputFile,
   --#                                TailExpr,
   --#                                VCGHeap &
   --#         VCGFailure        from *,
   --#                                Dictionary.Dict,
   --#                                TailExpr,
   --#                                VCGHeap &
   --#         VCGHeap           from *,
   --#                                Dictionary.Dict,
   --#                                HeadExpr,
   --#                                TailExpr;
   is
      TailComma, TailImp : Cells.Cell;
   begin
      -- HeadExpr is (forall y:T, inrange-> (forall x:T, ...)
      -- TailExpr is (forall z:T, inrange->P)
      TailComma := RightPtr (VCGHeap, TailExpr);  -- point to ,
      TailImp   := RightPtr (VCGHeap, TailComma);  -- point to -> or True

      if (Cells.Get_Symbol_Value (VCGHeap, TailImp) = Dictionary.GetTrue) then
         -- It's a True Cell, so dispose of it
         Cells.Dispose_Of_Cell (VCGHeap, TailImp);
         -- Now tag the predicate after ,
         SetRightArgument (TailComma, HeadExpr, VCGHeap);
      elsif (Cells.Get_Kind (VCGHeap, TailImp) = Cells.Op and Cells.Get_Op_Symbol (VCGHeap, TailImp) = SP_Symbols.implies) then
         -- Need to garbage-collect what's on the
         -- B arrow of TailImp already; should be a True cell
         Cells.Dispose_Of_Cell (VCGHeap, RightPtr (VCGHeap, TailImp));
         -- Tag the predicate after ->
         SetRightArgument (TailImp, HeadExpr, VCGHeap);
      else
         -- We are in error here: generate a
         -- a warning that the VC may be corrupted
         SPARK_IO.New_Line (OutputFile, 1);
         SPARK_IO.Put_Line (OutputFile, "!!!        'forall' structure in VC is corrupt.", 0);
         VCGFailure := True;
      end if;

      -- Result is now in TailExpr as
      -- "forall z:T, inrange-> (forall y:T, ...)"
   end ConcatQuantExpr;

   function GetNextRecordComponent (RecComp : Dictionary.Symbol) return Dictionary.Symbol
   --# global in Dictionary.Dict;
   is
      NextComp, RecType : Dictionary.Symbol;
      Num               : Positive;
   begin
      RecType := Dictionary.GetRecordType (RecComp);
      Num     := 1;
      loop
         exit when RecComp = Dictionary.GetNonExtendedRecordComponent (RecType, Num);
         Num := Num + 1;
      end loop;
      if Num < Dictionary.GetNumberOfNonExtendedComponents (RecType) then
         NextComp := Dictionary.GetNonExtendedRecordComponent (RecType, Num + 1);
      else
         NextComp := Dictionary.NullSymbol;
      end if;
      return NextComp;
   end GetNextRecordComponent;

   procedure CreateRecordAccess (RecordElem  :    out Cells.Cell;
                                 RecCompType : in     Dictionary.Symbol;
                                 RecIdent    : in     Cells.Cell)
   --# global in     Dictionary.Dict;
   --#        in out Statistics.TableUsage;
   --#        in out VCGHeap;
   --# derives RecordElem            from VCGHeap &
   --#         Statistics.TableUsage from *,
   --#                                    VCGHeap &
   --#         VCGHeap               from *,
   --#                                    Dictionary.Dict,
   --#                                    RecCompType,
   --#                                    RecIdent;
   is
      DAGCell : Cells.Cell;
   begin

      CreateCellKind (DAGCell, VCGHeap, Cells.Field_Access_Function);
      Cells.Set_Symbol_Value (VCGHeap, DAGCell, RecCompType);
      Cells.Set_Lex_Str (VCGHeap, DAGCell, Dictionary.GetSimpleName (RecCompType));
      SetRightArgument (DAGCell, RecIdent, VCGHeap);

      RecordElem := DAGCell;
   end CreateRecordAccess;

   -- Changed to use a stack to do a depth-first search,
   -- and altered from a function to a procedure to allow
   -- debugging side effects
   procedure ContainsQuantIdent (DataElem, QuantIdent : in     Cells.Cell;
                                 Result               :    out Boolean)
   --# global in out Statistics.TableUsage;
   --#        in out VCGHeap;
   --# derives Result,
   --#         VCGHeap               from DataElem,
   --#                                    QuantIdent,
   --#                                    VCGHeap &
   --#         Statistics.TableUsage from *,
   --#                                    DataElem,
   --#                                    QuantIdent,
   --#                                    VCGHeap;
   is
      CurrElem        : Cells.Cell;
      FoundQuantIdent : Boolean;
      MyTempStack     : CStacks.Stack;
   begin
      CurrElem        := DataElem;
      FoundQuantIdent := False;
      CStacks.CreateStack (MyTempStack);
      CStacks.Push (VCGHeap, CurrElem, MyTempStack);
      while not (CStacks.IsEmpty (MyTempStack) or FoundQuantIdent) loop
         CStacks.PopOff (VCGHeap, MyTempStack, CurrElem);
         if Cells.Get_Kind (VCGHeap, CurrElem) = Cells.Field_Access_Function then
            -- a record field
            CStacks.Push (VCGHeap, RightPtr (VCGHeap, CurrElem), MyTempStack);
         elsif Cells.Get_Kind (VCGHeap, CurrElem) = Cells.Fixed_Var then
            -- a fixed identifier
            if Cells.Get_Symbol_Value (VCGHeap, CurrElem) = Cells.Get_Symbol_Value (VCGHeap, QuantIdent) then
               FoundQuantIdent := True;
            end if;
         elsif Cells.Get_Kind (VCGHeap, CurrElem) = Cells.Element_Function then
            -- An array element
            -- Look in the array name itself
            CStacks.Push (VCGHeap, LeftPtr (VCGHeap, RightPtr (VCGHeap, CurrElem)), MyTempStack);
            -- Look at the comma or ident
            CStacks.Push (VCGHeap, Cells.Get_B_Ptr (VCGHeap, RightPtr (VCGHeap, RightPtr (VCGHeap, CurrElem))), MyTempStack);
         elsif Cells.Get_Kind (VCGHeap, CurrElem) = Cells.Op then
            -- Explore both sides
            CStacks.Push (VCGHeap, LeftPtr (VCGHeap, CurrElem), MyTempStack);
            CStacks.Push (VCGHeap, RightPtr (VCGHeap, CurrElem), MyTempStack);
         end if;

      end loop;
      Result := FoundQuantIdent;
   end ContainsQuantIdent;

   -- Adopted to concatenate forall structures
   -- if multi-dim arrays are present
   procedure QuantifyConstraint (ThisTypeSym : Dictionary.Symbol)
   --# global in     Dictionary.Dict;
   --#        in     OutputFile;
   --#        in out ConstraintStack;
   --#        in out QuantIdentStack;
   --#        in out SPARK_IO.File_Sys;
   --#        in out Statistics.TableUsage;
   --#        in out VCGFailure;
   --#        in out VCGHeap;
   --# derives ConstraintStack,
   --#         QuantIdentStack,
   --#         Statistics.TableUsage,
   --#         VCGFailure,
   --#         VCGHeap               from *,
   --#                                    ConstraintStack,
   --#                                    Dictionary.Dict,
   --#                                    QuantIdentStack,
   --#                                    ThisTypeSym,
   --#                                    VCGHeap &
   --#         SPARK_IO.File_Sys     from *,
   --#                                    ConstraintStack,
   --#                                    Dictionary.Dict,
   --#                                    OutputFile,
   --#                                    QuantIdentStack,
   --#                                    ThisTypeSym,
   --#                                    VCGHeap;
   is
      NonArrayStack, MyTempStack                                      : CStacks.Stack;
      QuantIdent, QuantExpr, LastElement, DataElem, MyRangeConstraint : Cells.Cell;
      LoopCount                                                       : Positive;
      Emptied, QuantFound, ElemFound, FirstLoop                       : Boolean;
   begin
      CStacks.CreateStack (MyTempStack);
      CStacks.CreateStack (NonArrayStack);

      -- Create a range constraint for the top element
      PopOffConstraintStack (DataElem);
      CreateRangeConstraint (DataElem, ThisTypeSym, VCGHeap, MyRangeConstraint);

      -- Find the first 'element (', if it exists,
      -- and set "LastElement" to point to it
      ElemFound   := False;
      LastElement := DataElem; -- to avoid flow error
      FirstLoop   := True;
      loop  -- findelt
         if Cells.Get_Kind (VCGHeap, DataElem) = Cells.Element_Function then
            -- Found.  If this is the first one, mark it as such
            if not ElemFound then
               LastElement := DataElem;
            end if;
            ElemFound := True;
         end if;
         if FirstLoop then
            FirstLoop := False;
         else
            CStacks.Push (VCGHeap, DataElem, NonArrayStack);
         end if;
         Emptied := CStacks.IsEmpty (ConstraintStack);
         if not Emptied then
            PopOffConstraintStack (DataElem);
         end if;
         exit when Emptied;
      end loop;  -- findelt

      -- Should now have all of initial ConstraintStack on NAS
      -- in reverse order

      if ElemFound then
         -- There must be identifiers around since
         -- we've found an 'element ()' reference
         -- which is in 'LastElement'
         -- Push identifiers onto MyTempStack to ensure they
         -- are available in same order as array elems
         while not CStacks.IsEmpty (QuantIdentStack) loop
            PopOffQuantIdentStack (QuantIdent);
            CStacks.Push (VCGHeap, QuantIdent, MyTempStack);
         end loop;

         -- Now construct and compose the forall structures
         LoopCount := 1;
         while not CStacks.IsEmpty (MyTempStack) loop -- compose

            -- Shuffle round the identifier onto QIS
            CStacks.PopOff (VCGHeap, MyTempStack, QuantIdent);
            PushOnQuantIdentStack (QuantIdent);

            -- Look to see if LastElement contains the given identifier
            ContainsQuantIdent (LastElement, QuantIdent, QuantFound);
            -- Create the forall structure, if appropriate
            if QuantFound then
               -- All things considered, it *should* find an ident!
               if (LoopCount = 1) then
                  -- Quantify over the range constraint
                  CreateQuantExpr
                    (QuantExpr,
                     QuantIdent,
                     Dictionary.GetType (Cells.Get_Symbol_Value (VCGHeap, QuantIdent)),
                     MyRangeConstraint);
               else
                  CreateShortQuantExpr (QuantExpr, QuantIdent, Dictionary.GetType (Cells.Get_Symbol_Value (VCGHeap, QuantIdent)));
                  ConcatQuantExpr (LastElement, QuantExpr);
               end if;
               LastElement := QuantExpr;
               -- And handle the looping
               LoopCount := LoopCount + 1;
            end if;
         end loop; -- compose
      else
         -- ElemFound not True
         LastElement := MyRangeConstraint;
      end if;

      -- Return the nonarray cells to the stack
      while not CStacks.IsEmpty (NonArrayStack) loop
         CStacks.PopOff (VCGHeap, NonArrayStack, DataElem);
         PushOnConstraintStack (DataElem);
      end loop;

      if not ElemFound then
         -- Put range constraint on head of CS
         PushOnConstraintStack (MyRangeConstraint);
      else
         -- Put the 'forall' on top
         PushOnConstraintStack (LastElement);
      end if;

   end QuantifyConstraint;

   -----------------------

   function SafeAdvanceVariable (NewRec : Boolean) return Dictionary.Symbol
   --# global in ConstraintStack;
   --#        in Dictionary.Dict;
   --#        in VCGHeap;
   is
      Sym : Dictionary.Symbol;
   begin
      Sym := Cells.Get_Assoc_Var (VCGHeap, CStacks.Top (VCGHeap, ConstraintStack));
      if Sym /= Dictionary.NullSymbol then
         if NewRec then
            Sym := Dictionary.GetFirstRecordSubcomponent (Sym);
         else
            Sym := Dictionary.GetNextRecordSubcomponent (Sym);
         end if;
      end if;
      return Sym;
   end SafeAdvanceVariable;

   -----------------------

   function Get_Array_Index
     (Type_Sym    : Dictionary.Symbol;
      Dimension   : Positive;
      Initial_Var : Dictionary.Symbol)
     return        Dictionary.Symbol
   --# global in Dictionary.Dict;
   is
      Result : Dictionary.Symbol;
   begin
      -- If Type_Sym is the symbol of a constrained array (sub)type then return
      -- the Dimension'th index of that type.
      -- If TypeSym is unconstrained
      -- (and this can only be true at the outermost level the quantifier we
      -- are constructing) then return the Dimension'th ParameterConstraintSymbol
      -- associated with the object Initial_Var.
      if Dictionary.IsUnconstrainedArrayType (Type_Sym) then
         Result := Dictionary.GetSubprogramParameterConstraint (Initial_Var, Dimension);
      else
         Result := Dictionary.GetArrayIndex (Type_Sym, Dimension);
      end if;
      return Result;
   end Get_Array_Index;

   function Is_An_External_Input (Sym : Dictionary.Symbol) return Boolean
   --# global in Dictionary.Dict;
   is
      Result : Boolean := False;
      Entire : Dictionary.Symbol;
      Enc    : Dictionary.Symbol;
   begin
      if Dictionary.IsVariable (Sym) then
         Result := Dictionary.GetOwnVariableOrConstituentMode (Sym) = Dictionary.InMode;
      elsif Dictionary.IsVariableOrSubcomponent (Sym) then
         Entire := Sym;
         loop
            Enc := Dictionary.GetEnclosingObject (Entire);
            exit when Enc = Dictionary.NullSymbol;
            Entire := Enc;
            exit when not Dictionary.IsVariableOrSubcomponent (Entire);
            exit when Dictionary.IsVariable (Entire);
         end loop;
         Result := Dictionary.GetOwnVariableOrConstituentMode (Entire) = Dictionary.InMode;
      end if;
      return Result;
   end Is_An_External_Input;

   -- Create type'always_valid(variable) on constraint stack
   procedure Create_Always_Valid_Assumption
     (Type_Sym         : in     Dictionary.Symbol;
      Var_Sym          : in     Dictionary.Symbol;
      Var_Ref          : in     Cells.Cell;
      Constraint_Stack : in out CStacks.Stack)
   --# global in out Statistics.TableUsage;
   --#        in out VCGHeap;
   --# derives Constraint_Stack,
   --#         Statistics.TableUsage,
   --#         VCGHeap               from *,
   --#                                    Constraint_Stack,
   --#                                    Type_Sym,
   --#                                    Var_Ref,
   --#                                    Var_Sym,
   --#                                    VCGHeap;
   is
      Fun_Cell, Var_Type_Cell : Cells.Cell;
   begin
      CreateCellKind (Var_Type_Cell, VCGHeap, Cells.Reference);
      Cells.Set_Symbol_Value (VCGHeap, Var_Type_Cell, Type_Sym); -- Set the type
      Cells.Set_Assoc_Var (VCGHeap, Var_Type_Cell, Var_Sym);     -- Set the variable
      CStacks.Push (VCGHeap, Var_Type_Cell, Constraint_Stack);   -- type/variable is LH operand
      CreateCellKind (Fun_Cell, VCGHeap, Cells.Attrib_Function); -- Cell for Always_Valid token
      Cells.Set_Lex_Str (VCGHeap, Fun_Cell, LexTokenManager.Always_Valid_Token);
      Cells.Set_Assoc_Var (VCGHeap, Fun_Cell, Var_Sym);          -- the always_valid variable symbol
      SetRightArgument (Fun_Cell, Var_Ref, VCGHeap);              -- The always_valid variable
      CStacks.Push (VCGHeap, Fun_Cell, Constraint_Stack);        -- always_valid (variable) is RH operand
      PushOperator (Binary, SP_Symbols.apostrophe, VCGHeap, Constraint_Stack); -- the "'" operator
   end Create_Always_Valid_Assumption;

   -- Create for_all elements in-type on constraint stack
   procedure Create_All_Elements_In_Type
     (Type_Sym         : in     Dictionary.Symbol;
      Var_Ref          : in     Cells.Cell;
      Scope            : in     Dictionary.Scopes;
      Initial_Var      : in     Dictionary.Symbol;
      Quant_Num        : in out Natural;
      Quant_Id_Stack   : in out CStacks.Stack;
      Constraint_Stack : in out CStacks.Stack)
   --# global in out Dictionary.Dict;
   --#        in out LexTokenManager.State;
   --#        in out SPARK_IO.File_Sys;
   --#        in out Statistics.TableUsage;
   --#        in out VCGHeap;
   --# derives Constraint_Stack,
   --#         Quant_Id_Stack,
   --#         Statistics.TableUsage from *,
   --#                                    Dictionary.Dict,
   --#                                    Initial_Var,
   --#                                    LexTokenManager.State,
   --#                                    Quant_Id_Stack,
   --#                                    Quant_Num,
   --#                                    Scope,
   --#                                    Type_Sym,
   --#                                    Var_Ref,
   --#                                    VCGHeap &
   --#         Dictionary.Dict,
   --#         SPARK_IO.File_Sys     from *,
   --#                                    Dictionary.Dict,
   --#                                    Initial_Var,
   --#                                    LexTokenManager.State,
   --#                                    Quant_Num,
   --#                                    Scope,
   --#                                    Type_Sym &
   --#         LexTokenManager.State,
   --#         Quant_Num             from *,
   --#                                    Dictionary.Dict,
   --#                                    Quant_Num,
   --#                                    Type_Sym &
   --#         VCGHeap               from *,
   --#                                    Constraint_Stack,
   --#                                    Dictionary.Dict,
   --#                                    Initial_Var,
   --#                                    LexTokenManager.State,
   --#                                    Quant_Id_Stack,
   --#                                    Quant_Num,
   --#                                    Scope,
   --#                                    Type_Sym,
   --#                                    Var_Ref;
   is
      Quant_Id_Sym            : Dictionary.Symbol;
      Dim_Max, Loop_Count     : Positive;
      Array_Elem, Quant_Ident : Cells.Cell;
   begin
      Dim_Max := Dictionary.GetNumberOfDimensions (Type_Sym);
      -- Create the initial "element (Var_Ref,[i1" in Array_Elem
      Create_Quant_Ident
        (Quant_Ident   => Quant_Id_Sym,
         This_Type_Sym => Get_Array_Index (Type_Sym, 1, Initial_Var),
         Scope         => Scope,
         Quant_Num     => Quant_Num);
      CreateFixedVarCell (Quant_Ident, VCGHeap, Quant_Id_Sym);
      CStacks.Push (VCGHeap, Quant_Ident, Quant_Id_Stack);
      Create_Array_Access (Array_Elem  => Array_Elem,
                           Array_Ident => Var_Ref,
                           Quant_Ident => Quant_Ident);

      -- Cope with multi-dimensions
      Loop_Count := 2;
      while (Loop_Count <= Dim_Max) loop
         -- Generate a new identifier and put it onto the QI stack
         Create_Quant_Ident
           (Quant_Ident   => Quant_Id_Sym,
            This_Type_Sym => Get_Array_Index (Type_Sym, Loop_Count, Initial_Var),
            Scope         => Scope,
            Quant_Num     => Quant_Num);
         CreateFixedVarCell (Quant_Ident, VCGHeap, Quant_Id_Sym);
         CStacks.Push (VCGHeap, Quant_Ident, Quant_Id_Stack);
         -- join onto reference
         ConcatArrayQuant (Array_Elem, Quant_Ident);
         Loop_Count := Loop_Count + 1;
      end loop;

      -- Push the array structure back onto ConstraintStack
      CStacks.Push (Heap     => VCGHeap,
                    CellName => Array_Elem,
                    S        => Constraint_Stack);
      --# accept F, 50, Constraint_Stack, Constraint_Stack,
      --#        "Constraint_Stack definitely depends on itself. CStacks.Push dependency relation is unnatural.";
   end Create_All_Elements_In_Type;

begin -- CreateStructConstraint;

   CStacks.CreateStack (QuantIdentStack);
   CStacks.CreateStack (ConstraintStack);

   InitialVariable := Cells.Get_Symbol_Value (VCGHeap, VarSym);

   TypeSym := StartTypeSym;
   -- The TOS also keeps a reference to the variable associated with
   -- the type being processed.
   Cells.Set_Assoc_Var (VCGHeap, VarSym, AssocVar);
   PushOnConstraintStack (VarSym);

   -- ConstraintStack = VarSym(I)
   loop -- down
      if DiscreteTypeWithCheck (TypeSym, Scope) then
         -- ConstraintStack = VarSym(I) -> ...
         IVarSym := Cells.Get_Assoc_Var (VCGHeap, CStacks.Top (VCGHeap, ConstraintStack));

         if IVarSym /= Dictionary.NullSymbol and then Is_An_External_Input (IVarSym) then
            -- Only generate in-type assumptions if the variable is marked as Always_Valid.
            if Dictionary.VariableOrSubcomponentIsMarkedValid (IVarSym) then
               -- The variable associated with this type is marked
               -- always valid. Hence generate a type_always_valid(X) hypothesis

               -- ConstraintStack = VarSym(I) -> ...

               PopOffConstraintStack (TOS);                     -- TOS=variable being accessed

               -- Create type'always_valid(variable) on constraint stack.
               Create_Always_Valid_Assumption
                 (Type_Sym         => TypeSym,
                  Var_Sym          => IVarSym,
                  Var_Ref          => TOS,
                  Constraint_Stack => ConstraintStack);

            else
               -- The variable associated with this type is in fact not marked
               -- always valid. At this stage the easiest thing to do is to generate a
               -- true hypothesis instead of the in-type hypothesis
               CStacks.Pop (VCGHeap, ConstraintStack);
               CreateTrueCell (VCGHeap, TOS);
               PushOnConstraintStack (TOS);
               -- ConstraintStack = TrueCell -> ...
            end if;
         else
            -- This is not an external inupt variable.
            -- Get the quantifiers into the dictionary
            -- and replaced types with constrainted on stack
            QuantifyConstraint (TypeSym);
            -- ConstraintStack = Constraint -> ...
         end if;
         TypeSym := Dictionary.NullSymbol;

      elsif ArrayTypeWithCheck (TypeSym, Scope) then
         IVarSym := Cells.Get_Assoc_Var (VCGHeap, CStacks.Top (VCGHeap, ConstraintStack));

         if IVarSym /= Dictionary.NullSymbol and then Is_An_External_Input (IVarSym) then
            -- The array variable is an external input.
            -- Do generate in-type assumptions.

            if Dictionary.VariableOrSubcomponentIsMarkedValid (IVarSym) then
               -- The array is marked as always valid.  Insert always_valid
               -- assumption into hypotheses.

               -- ConstraintStack = VarSym(I) -> ...
               PopOffConstraintStack (TOS);      -- TOS=variable being accessed

               -- Create type'always_valid(array_variable) on constraint stack.
               Create_Always_Valid_Assumption
                 (Type_Sym         => TypeSym,
                  Var_Sym          => IVarSym,
                  Var_Ref          => TOS,
                  Constraint_Stack => ConstraintStack);

               -- Complete: The entire array is assumed to be always_Valid;
               -- terminate the loop.
               TypeSym := Dictionary.NullSymbol;

            else
               -- The array variable is an in mode own variable (external input)
               -- and not marked as Always_Valid: suppress in-type assumptions.
               -- At this stage the easiest thing to do is to generate a
               -- true hypothesis instead of the in-type hypothesis
               CStacks.Pop (VCGHeap, ConstraintStack);
               CreateTrueCell (VCGHeap, TOS);
               PushOnConstraintStack (TOS);
               -- ConstraintStack = TrueCell -> ...

               -- Complete: terminate the loop.
               TypeSym := Dictionary.NullSymbol;
            end if;
         else
            -- The array object is not an external variable generate
            -- in-type hypotheses for its elements.

            -- ConstraintStack = VarSym(I) -> ...
            PopOffConstraintStack (TOS);

            -- Create all elements are in-type assumption on constraint stack.
            Create_All_Elements_In_Type
              (Type_Sym         => TypeSym,
               Var_Ref          => TOS,
               Scope            => Scope,
               Initial_Var      => InitialVariable,
               Quant_Num        => QuantNum,
               Quant_Id_Stack   => QuantIdentStack,
               Constraint_Stack => ConstraintStack);

            -- Prepare for next iteration
            TypeSym := Dictionary.GetArrayComponent (TypeSym);
         end if;

      elsif RecordTypeWithCheck (TypeSym, Scope) then
         IVarSym := Cells.Get_Assoc_Var (VCGHeap, CStacks.Top (VCGHeap, ConstraintStack));
         if IVarSym /= Dictionary.NullSymbol
           and then Is_An_External_Input (IVarSym)
           and then Dictionary.VariableOrSubcomponentIsMarkedValid (IVarSym) then
            -- The entire record is an external input and is marked as always_valid
            -- ConstraintStack = VarSym(I) -> ...
            PopOffConstraintStack (TOS);      -- TOS=variable being accessed

            -- Create type'always_valid(recod_variable) on constraint stack.
            Create_Always_Valid_Assumption
              (Type_Sym         => TypeSym,
               Var_Sym          => IVarSym,
               Var_Ref          => TOS,
               Constraint_Stack => ConstraintStack);

            -- Complete: all of the components of the record are assumed to
            -- be always_valid; terminate the loop.
            TypeSym := Dictionary.NullSymbol;
         else
            -- The entire record is not marked as always_valid.
            -- Perhaps some of its components are marked?

            -- This is the first time this record has been seen.
            -- ConstraintStack = VarSym(I) -> ...

            IVarSym := Cells.Get_Assoc_Var (VCGHeap, CStacks.Top (VCGHeap, ConstraintStack));

            -- RecCompType = The first record component (NB in the type, not value, realm)
            TypeSym     := Dictionary.GetRootType (TypeSym);
            RecCompType := Dictionary.CurrentSymbol (Dictionary.FirstRecordComponent (TypeSym));

            -- Get the current variable reference, its type should be TypeSym
            Field_Name_Sym := SafeAdvanceVariable (True);
            -- Thus, Field_Name_Sym = The first record component in the value realm

            TypeSym := Dictionary.GetType (RecCompType);

            -- If the first record component denotes a tagged null record,
            -- then skip it and move onto the next component.
            if Dictionary.TypeIsRecord (TypeSym) and then not Dictionary.RecordHasSomeFields (TypeSym) then

               RecCompType    := GetNextRecordComponent (RecCompType);
               Field_Name_Sym := SafeAdvanceVariable (False);
               TypeSym        := Dictionary.GetType (RecCompType);
            end if;

            if Field_Name_Sym = Dictionary.NullSymbol
              and then Dictionary.IsVariable (IVarSym)
              and then Dictionary.GetOwnVariableOrConstituentMode (IVarSym) = Dictionary.InMode then
               -- The variable associated with this type is in fact not marked
               -- always valid. At this stage the easiest thing to do is to generate a
               -- true hypothesis instead of the in-type hypothesis
               CStacks.Pop (VCGHeap, ConstraintStack);
               CreateTrueCell (VCGHeap, TOS);
               PushOnConstraintStack (TOS);
               -- ConstraintStack = TrueCell -> ...
               TypeSym := Dictionary.NullSymbol;
            else
               -- put in a record cell marker for this RecordComponent (type)
               -- including a ref to its subcomponent (value)
               CreateOpCell (OpCell, VCGHeap, SP_Symbols.RWrecord);
               Cells.Set_Symbol_Value (VCGHeap, OpCell, RecCompType);
               Cells.Set_Assoc_Var (VCGHeap, OpCell, Field_Name_Sym);

               -- just read TOS and don't pop it off as it will be
               -- needed by the other fields of the record
               TOS := CStacks.Top (VCGHeap, ConstraintStack);
               CreateRecordAccess (RecordElem, RecCompType, TOS);

               -- GOK
               Cells.Set_Assoc_Var (VCGHeap, RecordElem, Field_Name_Sym);

               PushOnConstraintStack (OpCell);
               PushOnConstraintStack (RecordElem);
               -- ConstraintStack = RecordElem(I') -> OpCell(I') -> VarSym(I) -> ...
            end if;
         end if;
      else
         -- encountered a type with no check
         -- push TRUE to be able to deal with other fields of any records
         -- ConstraintStack = VarSym(I) -> ...
         CStacks.Pop (VCGHeap, ConstraintStack);
         CreateTrueCell (VCGHeap, TrueCell);
         PushOnConstraintStack (TrueCell);
         TypeSym := Dictionary.NullSymbol;
         -- ConstraintStack = TrueCell -> ...
      end if;

      --447--check for reals moved here in support of optional real RTCs
      if IsRealType (TypeSym) then
         ContainsReals := True;
      end if;

      if TypeSym = Dictionary.NullSymbol then
         exitUp := False;
         loop -- up

            -- ConstraintStack should contain at least one element
            -- at this point
            -- ConstraintStack = RES(I') -> OpCell(I') -> VarSym(I) -> ...
            --             or  = RES -> <empty>
            PopOffConstraintStack (TOS);
            -- TOS = RES
            -- ConstraintStack = OpCell -> VarSym -> ...
            --             or  = <empty>

            if CStacks.IsEmpty (ConstraintStack) then
               -- ConstraintStack had only one element left
               -- set this to Result and exit up & down loops
               Result := TOS;
               exitUp := True;

            elsif Cells.Get_Kind (VCGHeap, CStacks.Top (VCGHeap, ConstraintStack)) = Cells.Op
              and then Cells.Get_Op_Symbol (VCGHeap, CStacks.Top (VCGHeap, ConstraintStack)) = SP_Symbols.RWrecord then
               -- found a record marker cell
               -- TOS = RES
               -- ConstraintStack = OpCell(Rec) -> VarSym -> ...

               RecCompType := GetNextRecordComponent (Cells.Get_Symbol_Value (VCGHeap, CStacks.Top (VCGHeap, ConstraintStack)));
               if RecCompType = Dictionary.NullSymbol then
                  -- all the fields of the record have been processed
                  -- remove record marker from stack
                  -- ConstraintStack = OpCell(Rec) -> VarSym -> ...
                  CStacks.Pop (VCGHeap, ConstraintStack);
                  -- remove True top-of-stack which is no longer needed
                  -- ConstraintStack = VarSym -> ...
                  CStacks.Pop (VCGHeap, ConstraintStack);

                  -- ConstraintStack = ...
                  PushOnConstraintStack (TOS);
                  -- ConstraintStack = RES -> ...
               else
                  -- process next field of record
                  -- ConstraintStack = OpCell -> VarSym -> ...
                  -- Thus, advance the ivar to the next field
                  IVarSym := SafeAdvanceVariable (False);
                  PopOffConstraintStack (OpCell);
                  -- ConstraintStack = VarSym -> ...
                  Cells.Set_Symbol_Value (VCGHeap, OpCell, RecCompType);
                  Cells.Set_Assoc_Var (VCGHeap, OpCell, IVarSym);

                  -- just read TOS and don't pop it off as it will be
                  -- needed by the other fields of the record
                  CreateRecordAccess (RecordElem, RecCompType, CStacks.Top (VCGHeap, ConstraintStack));
                  TypeSym := Dictionary.GetType (RecCompType);
                  exitUp  := True; -- exit up loop and go back to down loop

                  -- Push back record marker cell
                  PushOnConstraintStack (OpCell);
                  -- ConstraintStack = OpCell -> VarSym -> ...

                  -- Push back TOS which contains constraints from
                  -- earlier fields of the record
                  PushOnConstraintStack (TOS);
                  -- ConstraintStack = RES -> OpCell -> VarSym -> ...

                  -- GOK
                  Cells.Set_Assoc_Var (VCGHeap, RecordElem, IVarSym);

                  PushOnConstraintStack (RecordElem);
                  -- ConstraintStack = RecordElem -> RES -> OpCell -> VarSym -> ...
                  --                        I                  I

               end if; -- RecCompType = Dictionary.NullSymbol
            else
               -- must have just finished dealing with a non-first
               -- field of a record
               -- ConstraintStack = OpCell(~Rec) -> VarSym -> ...
               PopOffConstraintStack (TOS2);
               -- ConstraintStack = VarSym -> ...
               Conjoin (TOS2, VCGHeap, TOS);
               PushOnConstraintStack (TOS);
               -- ConstraintStack = RES&OpCell(~Rec) -> VarSym -> ...
            end if; -- CStacks.IsEmpty (ConstraintStack)

            exit when exitUp;
         end loop; -- up

      end if; -- TypeSym = Dictionary.NullSymbol

      exit when CStacks.IsEmpty (ConstraintStack);
   end loop; -- down

   -- At this point the ConstraintStack is empty
   -- (the up loop has been exited through its first exit)
   -- and Result contains the last TOS.

   --# accept F, 501, Result, "Result always defined here" &
   --#        F, 602, StructConstraint, Result, "Result always defined here";
   StructConstraint := Result;
   --Debug.PrintDAG ("Result = ", Result, VCGHeap, Scope);
end CreateStructConstraint;
