/***************************************************************************
    smb4kpreviewer  -  This class queries a remote share for a preview
                             -------------------
    begin                : Mo Mai 28 2007
    copyright            : (C) 2007-2008 by Alexander Reinholdt
    email                : dustpuppy@users.berlios.de
 ***************************************************************************/

/***************************************************************************
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful, but   *
 *   WITHOUT ANY WARRANTY; without even the implied warranty of            *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU     *
 *   General Public License for more details.                              *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc., 59 Temple Place, Suite 330, Boston,   *
 *   MA  02111-1307 USA                                                    *
 ***************************************************************************/

#ifndef SMB4KPREVIEWER_H
#define SMB4KPREVIEWER_H

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

// Qt includes
#include <QObject>
#include <QString>
#include <QQueue>

// KDE includes
#include <kprocess.h>
#include <kdemacros.h>

// forward declarations
class Smb4KPreviewItem;

/**
 * This class is part of the core of Smb4K. It queries a remote SMB share for
 * a preview and returns the result.
 *
 * @author Alexander Reinholdt <dustpuppy@users.berlios.de>
 */

class KDE_EXPORT Smb4KPreviewer : public QObject
{
  Q_OBJECT

  public:
    /**
     * The constructor
     *
     * @param parent            The parent object
     */
    Smb4KPreviewer( QObject *parent = 0 );

    /**
     * The destructor
     */
    ~Smb4KPreviewer();

    /**
     * Get a preview of the contents of @p item.
     *
     * In the case that @p item represents a 'homes' share, the user will be
     * prompted for the user name with which he wants to log in and the share
     * name of @p item will be set to the result.
     *
     * @param item              The item for which a preview should be
     *                          requested.
     *
     * @returns TRUE if einter the share name is not 'homes' or if it could
     * successfully be set to a user name. Otherwise it returns FALSE.
     */
    bool preview( Smb4KPreviewItem *item );

    /**
     * Using this function, you can find out whether the previewer is running
     * at the moment.
     *
     * @returns TRUE if the previewer is running or FALSE otherwise.
     */
    bool isRunning() { return m_working; }

    /**
     * Abort any action the previewer is performing at the moment and clear
     * the queue.
     */
    void abort();

  signals:
    /**
     * Emits the state the previewer is in
     *
     * @param state             The state
     */
    void state( int state );

    /**
     * Emits the preview after the process exited successfully. Get the contents
     * of the remote share by looping through the Smb4KPreviewItem::contents() list.
     *
     * @param item              The item for which the preview was received.
     */
    void result( Smb4KPreviewItem *item );

    /**
     * This signal is emitted if an error occurred.
     */
    void failed();

  protected:
    /**
     * Reimplemented from QObject to process the incoming data.
     *
     * @param event             The QTimerEvent event
     */
    void timerEvent( QTimerEvent *event );

  protected slots:
    /**
     * This slot is called, when the process exited.
     *
     * @param exitCode           The exit code of the process
     *
     * @param exitStatus         The exit status of the process (@see QProcess::ExitStatus)
     */
    void slotProcessFinished( int exitCode,
                              QProcess::ExitStatus exitStatus );

    /**
     * This slot is invoked when an error occurred with the process.
     *
     * @param errorCode       The error code returned by the process
     */
    void slotProcessError( QProcess::ProcessError errorCode );

  private:
    /**
     * The KProcess object
     */
    KProcess *m_proc;

    /**
     * This is the pointer to the Smb4KPreviewItem that's
     * processed.
     */
    Smb4KPreviewItem *m_item;

    /**
     * Indicates whether the previewer is running or not.
     */
    bool m_working;

    /**
     * This pointer queue holds the pointers to the Smb4KPreviewItem
     * objects that are to be processed.
     */
    QQueue<Smb4KPreviewItem *> m_queue;

    /**
     * If the process was aborted, this property is TRUE and FALSE
     * otherwise
     */
    bool m_aborted;

    /**
     * The process error code
     */
    QProcess::ProcessError m_process_error;
};

#endif
