/***************************************************************************
 *   Copyright (C) 2008 by S. MANKOWSKI / G. DE BURE skrooge@mankowski.fr  *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>  *
 ***************************************************************************/
/** @file
 * This file is Skrooge plugin for bank management.
 *
 * @author Stephane MANKOWSKI / Guillaume DE BURE
 */
#include "skgaccountboardwidget.h"
#include "skgdocumentbank.h"
#include "skgtraces.h"
#include "skgservices.h"
#include "skgaccountobject.h"
#include "skgunitobject.h"
#include "skgmainpanel.h"

#include <kaction.h>

#include <QDomDocument>

SKGAccountBoardWidget::SKGAccountBoardWidget ( SKGDocument* iDocument )
        : SKGWidget ( iDocument )
{
    SKGTRACEIN ( 10, "SKGAccountBoardWidget::SKGAccountBoardWidget" );

    ui.setupUi ( this );

    //Create menu
    setContextMenuPolicy ( Qt::ActionsContextMenu );

    menuFavorite=new KAction ( KIcon("rating"), i18nc ("Noun, a type of account", "Favorites only" ), this );
    menuFavorite->setCheckable ( true );
    menuFavorite->setChecked ( false );
    connect ( menuFavorite, SIGNAL ( triggered ( bool ) ), this, SLOT ( dataModified() ) );
    addAction ( menuFavorite );

    {
        QAction* sep=new QAction ( this );
        sep->setSeparator ( true );
        addAction ( sep );
    }

    menuAssets=new KAction ( i18nc ("Noun, a type of account", "Assets" ), this );
    menuAssets->setCheckable ( true );
    menuAssets->setChecked ( true );
    connect ( menuAssets, SIGNAL ( triggered ( bool ) ), this, SLOT ( dataModified() ) );
    addAction ( menuAssets );

    menuCurrent=new KAction ( i18nc ("Noun, a type of account", "Current" ), this );
    menuCurrent->setCheckable ( true );
    menuCurrent->setChecked ( true );
    connect ( menuCurrent, SIGNAL ( triggered ( bool ) ), this, SLOT ( dataModified() ) );
    addAction ( menuCurrent );

    menuCreditCard=new KAction ( i18nc ("Noun, a type of account", "Credit card" ), this );
    menuCreditCard->setCheckable ( true );
    menuCreditCard->setChecked ( true );
    connect ( menuCreditCard, SIGNAL ( triggered ( bool ) ), this, SLOT ( dataModified() ) );
    addAction ( menuCreditCard );

    menuInvestment=new KAction ( i18nc ("Noun, a type of account", "Investment" ), this );
    menuInvestment->setCheckable ( true );
    menuInvestment->setChecked ( true );
    connect ( menuInvestment, SIGNAL ( triggered ( bool ) ), this, SLOT ( dataModified() ) );
    addAction ( menuInvestment );

    menuOther=new KAction ( i18nc ("Noun, a type of account","Other" ), this );
    menuOther->setCheckable ( true );
    menuOther->setChecked ( true );
    connect ( menuOther, SIGNAL ( triggered ( bool ) ), this, SLOT ( dataModified() ) );
    addAction ( menuOther );

    //Refresh
    connect ( ( const QObject* ) getDocument(), SIGNAL ( tableModified ( QString, int ) ), this, SLOT ( dataModified ( QString, int ) ), Qt::QueuedConnection );
}

SKGAccountBoardWidget::~SKGAccountBoardWidget()
{
    SKGTRACEIN ( 10, "SKGAccountBoardWidget::~SKGAccountBoardWidget" );
    menuAssets=NULL;
    menuCurrent=NULL;
    menuCreditCard=NULL;
    menuInvestment=NULL;
    menuOther=NULL;
    menuFavorite=NULL;
}

QString SKGAccountBoardWidget::getState()
{
    QDomDocument doc ( "SKGML" );
    QDomElement root = doc.createElement ( "parameters" );
    doc.appendChild ( root );

    root.setAttribute ( "menuFavorite", menuFavorite && menuFavorite->isChecked() ? "Y" : "N" );
    root.setAttribute ( "menuAssets", menuAssets && menuAssets->isChecked() ? "Y" : "N" );
    root.setAttribute ( "menuCurrent", menuCurrent && menuCurrent->isChecked() ? "Y" : "N" );
    root.setAttribute ( "menuCreditCard", menuCreditCard && menuCreditCard->isChecked() ? "Y" : "N" );
    root.setAttribute ( "menuInvestment", menuInvestment && menuInvestment->isChecked() ? "Y" : "N" );
    root.setAttribute ( "menuOther", menuOther && menuOther->isChecked() ? "Y" : "N" );
    return doc.toString();
}

void SKGAccountBoardWidget::setState ( const QString& iState )
{
    QDomDocument doc ( "SKGML" );
    if ( doc.setContent ( iState ) )
    {
        QDomElement root = doc.documentElement();

        if (menuFavorite) menuFavorite->setChecked ( root.attribute ( "menuFavorite" ) =="Y" );
        if (menuAssets) menuAssets->setChecked ( root.attribute ( "menuAssets" ) =="Y" );
        if (menuCurrent) menuCurrent->setChecked ( root.attribute ( "menuCurrent" ) =="Y" );
        if (menuCreditCard) menuCreditCard->setChecked ( root.attribute ( "menuCreditCard" ) =="Y" );
        if (menuInvestment) menuInvestment->setChecked ( root.attribute ( "menuInvestment" ) =="Y" );
        if (menuOther) menuOther->setChecked ( root.attribute ( "menuOther" ) =="Y" );
    }

    dataModified ( "", 0 );
}

void SKGAccountBoardWidget::dataModified ( const QString& iTableName, int iIdTransaction )
{
    SKGTRACEIN ( 10, "SKGAccountBoardWidget::dataModified" );
    Q_UNUSED ( iIdTransaction );

    if ( iTableName=="account" || iTableName.isEmpty() )
    {
        KLocale* locale=KGlobal::locale();
        SKGDocumentBank* doc= dynamic_cast<SKGDocumentBank*> ( getDocument() );
        if ( doc )
        {
            SKGServices::SKGUnitInfo primary=doc->getPrimaryUnit();
            int nbAccount=0;
            SKGError err=SKGObjectBase::getNbObjects ( doc, "account", "", nbAccount );
            if ( err.isSucceeded() )
            {
                QString html;
                if ( !nbAccount )
                {
                    html="<html><body>"+i18nc ("Message", "First, you have to create at least one account<br>from <a href=\"___\">\"Bank and Account\"</a> page or <a href=\"__I\">import</a> operations." ) +"</body></html>";
                }
                else
                {
                    //Build where clause
                    QString wc;
                    if ( menuAssets && menuAssets->isChecked() )
                    {
                        wc="t_type='A'";
                    }
                    if ( menuCurrent && menuCurrent->isChecked() )
                    {
                        if ( !wc.isEmpty() ) wc+=" OR ";
                        wc+="t_type='C'";
                    }
                    if ( menuCreditCard && menuCreditCard->isChecked() )
                    {
                        if ( !wc.isEmpty() ) wc+=" OR ";
                        wc+="t_type='D'";
                    }
                    if ( menuInvestment && menuInvestment->isChecked() )
                    {
                        if ( !wc.isEmpty() ) wc+=" OR ";
                        wc+="t_type='I'";
                    }
                    if ( menuOther && menuOther->isChecked() )
                    {
                        if ( !wc.isEmpty() ) wc+=" OR ";
                        wc+="t_type='O'";
                    }

                    if ( wc.isEmpty() ) wc="1=0";

                    if ( menuFavorite && menuFavorite->isChecked() )
                    {
                        wc="t_bookmarked='Y' AND ("+wc+')';
                    }

                    //Build display
                    SKGStringListList listTmp;
                    err=SKGServices::executeSelectSqliteOrder ( doc,
                            "SELECT t_name, t_TYPENLS, f_CURRENTAMOUNT, t_close  from v_account_display WHERE ("+wc+") ORDER BY t_TYPENLS, t_name",
                            listTmp );
                    if ( err.isSucceeded() )
                    {
                        html+="<html><body><table class=\"table\">";
                        double sumTypeV1=0;
                        double sumV1=0;
                        QString currentType;
                        int nb=listTmp.count();
                        for ( int i=1; i<nb; ++i ) //Ignore header
                        {
                            QString name=listTmp.at ( i ).at ( 0 );
                            QString type=listTmp.at ( i ).at ( 1 );
                            double v1=SKGServices::stringToDouble ( listTmp.at ( i ).at ( 2 ) );
                            bool closed= ( listTmp.at ( i ).at ( 3 ) =="Y" );

                            if ( type!=currentType )
                            {
                                if ( !currentType.isEmpty() )
                                {
                                    html+="<tr class=\"tabletotal\"><td><b>"+SKGServices::stringToHtml ( i18nc ("Message", "Total of %1", currentType ) ) +"</b></td>"
                                          "<td align=\"right\"><b>"+doc->formatMoney ( sumTypeV1, primary ) +"</b></td></tr>";
                                    sumTypeV1=0;
                                }
                                currentType=type;
                            }
                            if ( !closed || abs ( v1 ) >0.1 )
                            {
                                SKGAccountObject account ( getDocument() );
                                account.setName ( name );
                                account.load();

                                SKGUnitObject unitAccount;
                                account.getUnit ( unitAccount );
                                QString unitAccountSymbol=unitAccount.getSymbol();

                                html+=QString ( "<tr><td><a href=\""+SKGServices::stringToHtml ( name ) +"\">" ) + ( closed ? "<i>" : "" ) +SKGServices::stringToHtml ( name ) + ( closed ? "</i>" : "" ) +"</a></td>"
                                      "<td align=\"right\"><font color=\"" + ( v1<0 ? "red" : "black" ) +"\">";

                                if ( !unitAccountSymbol.isEmpty() && primary.Symbol!=unitAccountSymbol )
                                {
                                    double unitAccountValue=SKGServices::stringToDouble ( unitAccount.getAttribute ( "f_CURRENTAMOUNT" ) );
                                    html+=SKGServices::stringToHtml ( locale->formatMoney ( v1/unitAccountValue, unitAccountSymbol,unitAccount.getNumberDecimal() ) );
                                    html+='=';
                                }
                                html+=SKGServices::stringToHtml ( locale->formatMoney ( v1, primary.Symbol, primary.NbDecimal ) );
                                html+="</font></td></tr>";
                            }
                            sumTypeV1+=v1;
                            sumV1+=v1;
                        }

                        if ( !currentType.isEmpty() )
                        {
                            html+="<tr class=\"tabletotal\"><td><b>"+SKGServices::stringToHtml ( i18nc ("the numerical total of a sum of values", "Total of %1", currentType ) ) +"</b></td>"
                                  "<td align=\"right\"><b>"+doc->formatMoney ( sumTypeV1, primary ) +"</b></font></td>"
                                  "</tr>";
                        }
                        html+="<tr class=\"tabletotal\"><td><b><a href=\""+SKGServices::stringToHtml ( i18nc ( "Noun, all items in a list","All" ) ) +
                              "\">"+SKGServices::stringToHtml ( i18nc ( "Noun, the numerical total of a sum of values","Total" ) ) +"</a></b></td>"
                              "<td align=\"right\"><b>"+doc->formatMoney ( sumV1, primary ) +"</b></font></td>"
                              "</tr></b>";
                        html+="</table></body></html>";
                    }
                }
                ui.kLabel->setText ( html );
            }
        }
    }
}

void SKGAccountBoardWidget::onOpen ( const QString& iLink )
{
    //Call operation plugin
    QDomDocument doc ( "SKGML" );

    if ( iLink=="__I" )
    {
        QAction* action=SKGMainPanel::getMainPanel()->getGlobalAction ( "import_operation" );
        if (action) action->trigger();
    }
    else if ( iLink!="___" )
    {
        doc.setContent ( getDocument()->getParameter ( "SKGOPERATION_DEFAULT_PARAMETERS" ) );
        QDomElement root = doc.documentElement();
        if ( root.isNull() )
        {
            root=doc.createElement ( "parameters" );
            doc.appendChild ( root );
        }

        root.setAttribute ( "account", iLink );
        SKGMainPanel::getMainPanel()->openPage ( SKGMainPanel::getMainPanel()->getPluginByName ( "Skrooge operation plugin" ), -1, doc.toString() );
    }
    else
    {
        doc.setContent ( getDocument()->getParameter ( "SKGBANK_DEFAULT_PARAMETERS" ) );
        SKGMainPanel::getMainPanel()->openPage ( SKGMainPanel::getMainPanel()->getPluginByName ( "Skrooge bank plugin" ), -1, doc.toString() );
    }
}
#include "skgaccountboardwidget.moc"
