/***************************************************************************
 *   Copyright (C) 2008 by S. MANKOWSKI / G. DE BURE skrooge@mankowski.fr  *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>  *
 ***************************************************************************/
/** @file
* This file defines classes SKGImportExportManager.
*
* @author Stephane MANKOWSKI / Guillaume DE BURE
 */
#include "skgimportexportmanager.h"
#include "skgtraces.h"
#include "skgservices.h"
#include "skgbankincludes.h"
#include "skgobjectbase.h"
#include "skgruleobject.h"
#include "skgpayeeobject.h"
#include "skgrecurrentoperationobject.h"
#include "skgimportskg.h"
#include "skgimportkmy.h"
#include "skgimportgnc.h"
#include "skgimportgsb.h"
#include "skgimportxhb.h"

#include <klocale.h>
#include <ksavefile.h>

#include <QRegExp>
#include <QFileInfo>
#include <QTextCodec>
#include <QCryptographicHash>

/**
* Opening balance string
 */
#define OPENINGBALANCE "Opening Balance"

int SKGImportExportManager::nbOperationsNotImported=0;
int SKGImportExportManager::nbOperationsImported=0;
SKGError SKGImportExportManager::ofxError;
#ifdef SKG_ENABLE_OFX
QList<OfxStatementData> SKGImportExportManager::ofxInitialBalance;
#endif

SKGImportExportManager::SKGImportExportManager ( SKGDocumentBank* iDocument,
        const QString& iFileName,
        ImportExportMode iMode)
        :QObject(), document ( iDocument ), mode ( iMode ), fileName ( iFileName ),
        csvSeparator ( ';' ),csvHeaderIndex ( -1 ),
        defaultAccount ( NULL ), defaultUnit ( NULL )
{
    setAutomaticValidation ( true );
    setAutomaticApplyRules ( false );

    csvMapper["date"]="^date";
    csvMapper["account"]="^account";
    csvMapper["number"]="^number|^num?ro";
    csvMapper["mode"]="^mode|^type";
    csvMapper["payee"]="^payee|^tiers";
    csvMapper["comment"]="^comment|^libell?|^d?tail|^info";
    csvMapper["status"]="^status|^pointage";
    csvMapper["bookmarked"]="^bookmarked";
    csvMapper["category"]="^cat\\w*gor\\w*";
    csvMapper["amount"]="^value|^amount|^valeur|^montant";
    csvMapper["quantity"]="^quantity";
    csvMapper["unit"]="^unit";
    csvMapper["sign"]="^sign|^sens";
    csvMapper["idgroup"]="^idgroup";
    csvMapper["idtransaction"]="^idtransaction";

    if (mode==AUTO)
    {
        QString extension=QFileInfo ( fileName ).suffix().toUpper();
        if ( extension=="QIF" ) mode=SKGImportExportManager::QIF;
        else if ( extension=="OFX" || extension=="QFX" ) mode=SKGImportExportManager::OFX;
        else if ( extension=="UNCOMPRESSED" || extension=="GNUCASH" || extension=="GNC" ) mode=SKGImportExportManager::GNUCASH;
        else if ( extension=="KMY" ) mode=SKGImportExportManager::KMY;
        else if ( extension=="SKG" ) mode=SKGImportExportManager::SKG;
        else if ( extension=="SQLITE" ) mode=SKGImportExportManager::SQLITE;
        else if ( extension=="GSB" ) mode=SKGImportExportManager::GSB;
        else if ( extension=="XHB" ) mode=SKGImportExportManager::XHB;
        else if ( extension=="XML" ) mode=SKGImportExportManager::XML;
        else mode=SKGImportExportManager::CSV;
    }
}

SKGImportExportManager::~SKGImportExportManager()
{
    setDefaultAccount ( NULL );
    setDefaultUnit ( NULL );
    document=NULL;
    defaultAccount=NULL;
    defaultUnit=NULL;
}

SKGImportExportManager::ImportExportMode SKGImportExportManager::getMode()
{
    return mode;
}

SKGDocumentBank* SKGImportExportManager::getDocument()
{
    return document;
}

QString SKGImportExportManager::getFileName()
{
    return fileName;
}

SKGError SKGImportExportManager::setCodec ( const QString& iCodec )
{
    SKGError err;
    SKGTRACEINRC ( 10, "SKGImportExportManager::setCodec", err );
    codec=iCodec;
    return err;
}
SKGError SKGImportExportManager::setDefaultAccount ( SKGAccountObject* iAccount )
{
    SKGError err;
    SKGTRACEINRC ( 10, "SKGImportExportManager::setDefaultAccount", err );
    if ( defaultAccount )
    {
        delete defaultAccount;
        defaultAccount=NULL;
    }
    if ( iAccount ) defaultAccount=new SKGAccountObject ( *iAccount );
    return err;
}

SKGError SKGImportExportManager::getDefaultAccount ( SKGAccountObject& oAccount )
{
    SKGError err;
    SKGTRACEINRC ( 10, "SKGImportExportManager::getDefaultAccount", err );
    if ( defaultAccount==NULL && document )
    {
        QFileInfo fInfo ( fileName );
        QString name=fInfo.baseName();
        name.replace ( '_', ' ' );

        //Searching if an account exist
        QString whereClause="t_name='"+name+'\'';
        foreach ( const QString& val, name.split ( ' ' ) )
        {
            if ( !whereClause.isEmpty() ) whereClause+=" OR ";
            whereClause+="t_number='"+val+'\'';
        }
        if ( !whereClause.isEmpty() )
        {
            SKGObjectBase::SKGListSKGObjectBase listAccount;
            err=SKGObjectBase::getObjects ( document, "v_account", whereClause, listAccount );
            if ( err.isSucceeded() )
            {
                if ( listAccount.count() ==1 )
                {
                    //Yes ! Only one account found
                    SKGAccountObject account=listAccount.at ( 0 );
                    defaultAccount=new SKGAccountObject ( account );
                    err=document->sendMessage ( i18nc ( "An information message",  "Using account '%1' for import",account.getName() ) );
                }
                else
                {
                    err=document->sendMessage ( i18nc ( "An information message",  "More than one possible account found." ) );
                }
            }
        }

        //If better account not found, then we must create one
        if ( defaultAccount==NULL )
        {
            SKGAccountObject account;
            SKGBankObject bank ( document );
            if ( err.isSucceeded() ) err=bank.setName ( name );
            if ( err.isSucceeded() && bank.load().isFailed() ) err=bank.save ( false ); //Save only
            if ( err.isSucceeded() ) err=bank.addAccount ( account );
            if ( err.isSucceeded() ) err=account.setName ( name );
            if ( err.isSucceeded() && account.load().isFailed() ) err=account.save ( false ); //Save only
            if ( err.isSucceeded() ) defaultAccount=new SKGAccountObject ( account );
            if ( err.isSucceeded() ) err=document->sendMessage ( i18nc ( "An information message",  "Default account '%1' created for import",name ) );
        }
    }

    if ( defaultAccount!=NULL )  oAccount=*defaultAccount;

    return err;
}

SKGError SKGImportExportManager::setDefaultUnit ( SKGUnitObject* iUnit )
{
    SKGError err;
    SKGTRACEINRC ( 10, "SKGImportExportManager::setDefaultUnit", err );
    if ( defaultUnit )
    {
        delete defaultUnit;
        defaultUnit=NULL;
    }
    if ( iUnit ) defaultUnit=new SKGUnitObject ( *iUnit );
    return err;
}

SKGError SKGImportExportManager::getDefaultUnit ( SKGUnitObject& oUnit, const QDate* iDate )
{
    SKGError err;
    SKGTRACEINRC ( 10, "SKGImportExportManager::getDefaultUnit", err );
    if ( document && ( defaultUnit==NULL || iDate ) )
    {
        if ( defaultUnit==NULL )
        {
            delete defaultUnit;
            defaultUnit=NULL;
        }

        //Do we have to found the best unit for a date ?
        QString wc="t_type IN ('1', '2', 'C')";
        if ( iDate )
        {
            //Yes
            wc+=" AND d_MINDATE<'"+SKGServices::dateToSqlString ( QDateTime ( *iDate ) ) +'\'';
        }

        //Check if a unit exist
        SKGObjectBase::SKGListSKGObjectBase listUnits;
        err=SKGObjectBase::getObjects ( document, "v_unit",wc+" ORDER BY ABS(f_CURRENTAMOUNT-1) ASC" , listUnits );
        if ( err.isSucceeded() )
        {
            if ( listUnits.count() ==0 )
            {
                //Not found, we have to create one
                QDateTime now=QDateTime::currentDateTime();
                QString postFix=SKGServices::dateToSqlString ( now );

                SKGUnitObject unit ( document );
                QString name=i18nc ( "Noun",  "Unit for import %1",postFix );
                err=unit.setName ( name );
                if ( unit.load().isFailed() )
                {
                    if ( err.isSucceeded() ) err=unit.setSymbol ( name );
                    if ( err.isSucceeded() ) err=unit.save ( false ); //Save only

                    SKGUnitValueObject unitval;
                    if ( err.isSucceeded() ) err=unit.addUnitValue ( unitval );
                    if ( err.isSucceeded() ) err=unitval.setQuantity ( 1 );
                    if ( err.isSucceeded() ) err=unitval.setDate ( now.date() );
                    if ( err.isSucceeded() ) err=unitval.save ( false, false ); //Save only without reload

                    if ( err.isSucceeded() ) err=document->sendMessage ( i18nc ( "An information message",  "Default unit '%1' created for import",name ) );
                }

                if ( err.isSucceeded() ) defaultUnit=new SKGUnitObject ( unit );
            }
            else
            {
                //Found, we can use it
                defaultUnit=new SKGUnitObject ( ( SKGUnitObject ) listUnits.at ( 0 ) );
            }
        }
    }

    if ( defaultUnit!=NULL )
    {
        oUnit=*defaultUnit;
    }

    return err;
}

void SKGImportExportManager::setAutomaticValidation ( bool iValidation )
{
    automaticValidationOfImportedOperation=iValidation;
}

void SKGImportExportManager::setAutomaticApplyRules ( bool iValidation )
{
    automaticApplyRulesOfImportedOperation=iValidation;
}

SKGError SKGImportExportManager::finalizedImportedOperations()
{
    SKGError err;
    SKGTRACEINRC ( 2, "SKGImportExportManager::finalizedImportedOperations", err );

    //Apply rules
    if ( automaticApplyRulesOfImportedOperation && document )
    {
        //Get rules
        SKGObjectBase::SKGListSKGObjectBase rules;
        err=SKGObjectBase::getObjects ( document, "v_rule", "1=1 ORDER BY f_sortorder", rules );

        int nb=rules.count();
        for ( int i=0; err.isSucceeded() && i<nb; ++i )
        {
            SKGRuleObject rule=rules.at ( i );
            err=rule.execute ( SKGRuleObject::IMPORTING );
        }
    }

    //Change imported status
    if ( err.isSucceeded() ) err = SKGServices::executeSqliteOrder ( document, QString ( "UPDATE operation SET t_imported='" ) +
                                       ( automaticValidationOfImportedOperation ? "Y" : "P" ) +
                                       "' WHERE t_imported='T'" );
    return err;
}

SKGError SKGImportExportManager::importFile()
{
    SKGError err;
    SKGTRACEINRC ( 2, "SKGImportExportManager::importFile", err );
    err=SKGServices::executeSqliteOrder ( document, "ANALYZE" );
    if ( err.isSucceeded() )
    {
        if ( mode==SKGImportExportManager::QIF ) err=SKGImportExportManager::importQIF();
        else if ( mode==SKGImportExportManager::CSV ) err=SKGImportExportManager::importCSV();
        else if ( mode==SKGImportExportManager::CSVUNIT ) err=SKGImportExportManager::importCSVUnit();
        else if ( mode==SKGImportExportManager::GNUCASH ) err=SKGImportGnc::importFile(this, document);
        else if ( mode==SKGImportExportManager::GSB ) err=SKGImportGsb::importFile(this, document);
        else if ( mode==SKGImportExportManager::XHB ) err=SKGImportXhb::importFile(this, document);
        else if ( mode==SKGImportExportManager::KMY ) err=SKGImportKmy::importFile(this, document);
        else if ( mode==SKGImportExportManager::SKG ) err=SKGImportSkg::importFile(this, document);
        else if ( mode==SKGImportExportManager::SQLITE ) err=SKGImportSkg::importFile(this, document);
#ifdef SKG_ENABLE_OFX
        else if ( mode==SKGImportExportManager::OFX ) err=SKGImportExportManager::importOFX();
#endif
        else
        {
            err.setReturnCode ( ERR_NOTIMPL );
            err.setMessage ( i18nc ( "Error message",  "This import mode is not yet implemented" ) );
        }
    }

    if ( err.isSucceeded() ) err = finalizedImportedOperations();

    return err;
}

SKGError SKGImportExportManager::exportFile()
{
    SKGError err;
    SKGTRACEINRC ( 2, "SKGImportExportManager::exportFile", err );
    err=SKGServices::executeSqliteOrder ( document, "ANALYZE" );
    if ( err.isSucceeded() )
    {
        if ( mode==SKGImportExportManager::QIF ) err=SKGImportExportManager::exportQIF();
        else if ( mode==SKGImportExportManager::CSV ) err=SKGImportExportManager::exportCSV();
        else if ( mode==SKGImportExportManager::KMY ) err=SKGImportKmy::exportFile(this, document);
        else if ( mode==SKGImportExportManager::SKG ) err=SKGImportSkg::exportFile(this, document, false);
        else if ( mode==SKGImportExportManager::SQLITE ) err=SKGImportSkg::exportFile(this, document, true);
        else if ( mode==SKGImportExportManager::XML )
        {
            QDomDocument doc;
            err=SKGServices::copySqliteDatabaseToXml(getDocument()->getDatabase(), doc);
            if (err.isSucceeded())
            {
                KSaveFile file ( fileName );
                if ( !file.open () )
                {
                    err.setReturnCode ( ERR_INVALIDARG );
                    err.setMessage ( i18nc ( "Error message",  "Save file '%1' failed",fileName ) );
                }
                else
                {
                    QTextStream stream ( &file );
                    stream << doc.toString() << endl;
                }

                //Close file
                file.finalize();
                file.close();
            }
        }
        else
        {
            err.setReturnCode ( ERR_NOTIMPL );
            err.setMessage ( i18nc ( "Error message",  "This export mode is not yet implemented" ) );
        }
    }

    return err;
}


SKGError SKGImportExportManager::exportQIF()
{
    SKGError err;
    SKGTRACEINRC ( 2, "SKGImportExportManager::exportQIF", err );
    SKGTRACEL ( 10 ) << "Input filename=" << fileName << endl;

    if ( document )
    {
        //Open file
        KSaveFile file ( fileName );
        if ( !file.open () )
        {
            err.setReturnCode ( ERR_INVALIDARG );
            err.setMessage ( i18nc ( "Error message",  "Save file '%1' failed",fileName ) );
        }
        else
        {
            QTextStream stream ( &file );
            SKGTRACEL ( 2 ) << "Text codec=" << QTextCodec::codecForLocale ()->name() << endl;

            err=document->beginTransaction ( "#INTERNAL#", 2 );
            if ( err.isSucceeded() )
            {

                //Export categories
                SKGObjectBase::SKGListSKGObjectBase categories;
                if ( err.isSucceeded() ) err=SKGObjectBase::getObjects ( document, "v_category_display_tmp", "1=1 ORDER BY t_fullname, id", categories );
                int nbcat=categories.count();
                if ( err.isSucceeded() && nbcat )
                {
                    stream << "!Type:Cat\n";
                    err=document->beginTransaction ( "#INTERNAL#", nbcat );
                    for ( int i=0; err.isSucceeded() && i<nbcat; ++i )
                    {
                        SKGCategoryObject cat=categories.at ( i );
                        QString catName=cat.getFullName();
                        if ( !catName.isEmpty() )
                        {
                            stream << "N" << catName.replace ( OBJECTSEPARATOR, ":" ) << endl;
                            if ( SKGServices::stringToDouble ( cat.getAttribute ( "f_REALCURRENTAMOUNT" ) ) <0 )
                            {
                                stream << "E" << endl;
                            }
                            else
                            {
                                stream << "I" << endl;
                            }
                            stream << "^" << endl;
                        }
                        if ( err.isSucceeded() ) err=document->stepForward ( i+1 );
                    }

                    if ( err.isSucceeded() ) err=document->endTransaction ( true );
                    else  document->endTransaction ( false );
                }
                if ( err.isSucceeded() ) err=document->stepForward ( 1 );

                SKGServices::SKGUnitInfo primaryUnit=document->getPrimaryUnit();

                //Get operations
                QString currentAccountName;
                SKGObjectBase::SKGListSKGObjectBase operations;
                if ( err.isSucceeded() ) err=SKGObjectBase::getObjects ( document, "v_operation_display_all", "t_template='N' ORDER BY t_ACCOUNT, d_date, id", operations );
                int nb=operations.count();
                if ( err.isSucceeded() )
                {
                    err=document->beginTransaction ( "#INTERNAL#", nb );
                    for ( int i=0; err.isSucceeded() && i<nb; ++i )
                    {
                        SKGOperationObject operation=operations.at ( i );

                        //Get account name
                        QString accountName=operation.getAttribute ( "t_ACCOUNT" );

                        //In the same account ?
                        if ( accountName!=currentAccountName )
                        {
                            SKGAccountObject account ( document );
                            account.setName ( accountName );
                            account.load();

                            SKGBankObject bank;
                            account.getBank ( bank );

                            //Write header
                            stream << "!Account\n";
                            stream << 'N' << accountName << endl;
                            QString type= ( account.getType() ==SKGAccountObject::CURRENT ? "Bank": ( account.getType() ==SKGAccountObject::CREDITCARD ? "CCard": ( account.getType() ==SKGAccountObject::INVESTMENT ? "Invst": ( account.getType() ==SKGAccountObject::ASSETS ? "Oth A": "Cash" ) ) ) );
                            stream << 'T' << type << endl;
                            QString number=bank.getNumber();
                            QString bnumber=account.getAgencyNumber();
                            QString cnumber=account.getNumber();
                            if ( !bnumber.isEmpty() )
                            {
                                if ( !number.isEmpty() ) number+='-';
                                number+=bnumber;
                            }
                            if ( !cnumber.isEmpty() )
                            {
                                if ( !number.isEmpty() ) number+='-';
                                number+=cnumber;
                            }
                            stream << 'D' << number << endl;
                            //stream << "/"      Statement balance date
                            //stream << "$"      Statement balance amount
                            stream << '^' << endl;
                            currentAccountName=accountName;

                            stream << "!Type:" << type << "\n";
                        }


                        //Write operation
                        /*
                        DONE	D      Date
                        DONE	T      Amount
                        N/A	U      Transaction amount (higher possible value than T)
                        DONE	C      Cleared status
                        DONE	N      Number (check or reference number)
                        DONE	P      Payee/description
                        DONE	M      Memo
                        N/A	A      Address (up to 5 lines; 6th line is an optional message)
                        DONE	L      Category (category/class or transfer/class)
                        DONE	S      Category in split (category/class or transfer/class)
                        DONE	E      Memo in split
                        DONE	$      Dollar amount of split
                        N/A	%      Percentage of split if percentages are used
                        N/A	F      Reimbursable business expense flag
                        N/A	X      Small Business extensions
                        DONE    Y      Security
                        DONE    I      Price
                        DONE    Q      Quantity (number of shares or split ratio)
                        N/A    O      Commission
                        DONE	^      End of entry
                        */
                        SKGUnitObject unit;
                        operation.getUnit ( unit );
                        bool investment=false;
                        bool unitExported=false;
                        if ( unit.getSymbol() !=primaryUnit.Symbol && !primaryUnit.Symbol.isEmpty() ) unitExported=true;
                        if ( unit.getType() ==SKGUnitObject::SHARE )
                        {
                            unitExported=true;
                            investment=true;
                        }

                        QString date=SKGServices::dateToSqlString ( QDateTime ( operation.getDate() ) );
                        if ( date.isEmpty() )
                        {
                            //This is an opening balance
                            date="0000-00-00";
                        }
                        stream << 'D' << date << endl;
                        if ( !unitExported )
                        {
                            stream << 'T' << SKGServices::doubleToString ( operation.getCurrentAmount() ) << endl;
                        }

                        if ( !investment )
                        {
                            int number=operation.getNumber();
                            if ( number ) stream << 'N' << operation.getNumber() << endl;
                        }
                        else
                        {
                            stream << 'N' << ( operation.getCurrentAmount() >0 ? "Buy" : "Sell" ) << endl;
                        }

                        if ( unitExported )
                        {
                            stream << 'Y' << unit.getSymbol() << endl;
                        }

                        SKGPayeeObject payeeObj;
                        operation.getPayee(payeeObj);
                        QString payee=payeeObj.getName();
                        QString address=payeeObj.getAddress();
                        if ( date=="0000-00-00" ) payee=OPENINGBALANCE;
                        if ( payee.length() ) stream << 'P' << payee << endl;
                        if ( address.length() ) stream << 'A' << address << endl;

                        QString memo=operation.getMode() +"  "+operation.getComment();
                        memo=memo.trimmed();
                        if ( memo.length() ) stream << 'M' << memo << endl;

                        SKGOperationObject::OperationStatus status=operation.getStatus();
                        stream << 'C' << ( status==SKGOperationObject::POINTED ? "C": ( status==SKGOperationObject::CHECKED ? "R": "" ) ) << endl;

                        //Get sub operations
                        SKGObjectBase::SKGListSKGObjectBase suboperations;
                        err=operation.getSubOperations ( suboperations );
                        if ( err.isSucceeded() )
                        {
                            int nb=suboperations.size();
                            QString category;
                            if ( nb==1 )
                            {
                                SKGSubOperationObject suboperation=suboperations.at ( 0 );
                                //Dump quantity
                                if ( unitExported )
                                {
                                    stream << 'Q' << SKGServices::doubleToString ( abs ( suboperation.getQuantity() ) ) << endl;
                                    stream << 'I' << SKGServices::doubleToString ( abs ( operation.getCurrentAmount() /suboperation.getQuantity() ) ) << endl;
                                }

                                //Get category of this simple operation
                                SKGCategoryObject cat;
                                suboperation.getCategory ( cat );
                                category=cat.getFullName().replace ( OBJECTSEPARATOR, ":" );
                            }

                            //Is it a transfer
                            SKGOperationObject transfer;
                            if ( operation.isTransfer ( transfer ) )
                            {
                                if ( !category.isEmpty() ) category.prepend ( '/' );

                                SKGAccountObject transferAccount;
                                err=transfer.getParentAccount ( transferAccount );
                                if ( err.isSucceeded() ) category.prepend ( '['+transferAccount.getName() +']' );
                            }
                            if ( !category.isEmpty() ) stream << 'L' << category << endl;

                            if ( nb>1 )
                            {
                                //Split operation
                                for ( int i=0; i<nb; ++i )
                                {
                                    SKGSubOperationObject suboperation=suboperations.at ( i );
                                    SKGCategoryObject cat;
                                    suboperation.getCategory ( cat );

                                    QString category=cat.getFullName().replace ( OBJECTSEPARATOR, ":" );
                                    if ( !category.isEmpty() ) stream << 'S' << category << endl;
                                    QString memo=suboperation.getComment();
                                    memo=memo.trimmed();
                                    if ( memo.length() ) stream << 'E' << memo << endl;
                                    stream << '$' << SKGServices::doubleToString ( suboperation.getQuantity() ) << endl;
                                }
                            }
                        }

                        stream << '^' << endl;
                        if ( err.isSucceeded() ) err=document->stepForward ( i+1 );
                    }

                    if ( err.isSucceeded() ) err=document->endTransaction ( true );
                    else  document->endTransaction ( false );
                }
                if ( err.isSucceeded() ) err=document->stepForward ( 2 );
                if ( err.isSucceeded() ) err=document->endTransaction ( true );
                else  document->endTransaction ( false );
            }
        }

        //Close file
        file.finalize();
        file.close();
    }
    return err;
}

SKGError SKGImportExportManager::exportCSV()
{
    SKGError err;
    SKGTRACEINRC ( 2, "SKGDocumentBank::exportCSV", err );
    SKGTRACEL ( 10 ) << "Input filename=" << fileName << endl;

    if ( document )
    {
        //Open file
        KSaveFile file ( fileName );
        if ( !file.open () )
        {
            err.setReturnCode ( ERR_INVALIDARG );
            err.setMessage ( i18nc ( "Error message",  "Save file '%1' failed",fileName ) );
        }
        else
        {
            QTextStream out ( &file );
            err=SKGServices::dumpSelectSqliteOrder ( ( SKGDocument* ) document,
                    "SELECT d_date as date, t_ACCOUNT as account, i_number as number, t_mode as mode, "
                    "t_PAYEE as payee, t_REALCOMMENT as comment, f_REALQUANTITY as quantity, "
                    "t_UNIT as unit, f_REALCURRENTAMOUNT as amount, t_TYPEEXPENSE as sign, t_REALCATEGORY as category, t_status as status, "
                    "t_bookmarked as bookmarked, id idtransaction, i_group_id idgroup "
                    "FROM v_operation_consolidated ORDER BY d_date, id, i_SUBOPID", &out, SKGServices::DUMP_CSV );
        }

        //Close file
        file.finalize();
        file.close();
    }

    return err;
}

SKGError SKGImportExportManager::exportOFX()
{
    SKGError err;
    SKGTRACEINRC ( 2, "SKGDocumentBank::exportOFX", err );
    SKGTRACEL ( 10 ) << "Input filename=" << fileName << endl;

    return err;
}

QStringList SKGImportExportManager::getCSVMappingFromLine ( const QString& iLine )
{
    QStringList output;
    QString line=iLine.trimmed();

    //Split first line
    QStringList csvAttributes=SKGServices::splitCSVLine ( line, ';', true, &csvSeparator );
    int nb=csvAttributes.count();
    for ( int i=0; i<nb; ++i )
    {
        QString att=csvAttributes[i].toLower();

        QMapIterator<QString, QString> csvMapperIterator ( csvMapper );
        bool found=false;
        while ( !found && csvMapperIterator.hasNext() )
        {
            csvMapperIterator.next();
            if ( QRegExp ( csvMapperIterator.value(), Qt::CaseInsensitive ).indexIn ( att ) !=-1 && !output.contains ( csvMapperIterator.key() ) )
            {
                output.push_back ( csvMapperIterator.key() );
                found=true;
            }
        }

        if ( !found )
        {
            output.push_back ( "" ); //To ignore this column
        }
    }
    return output;
}

SKGError SKGImportExportManager::setCSVMappingRules ( const QMap<QString, QString>& iCSVMappingRules )
{
    SKGError err;
    SKGTRACEINRC ( 10, "SKGImportExportManager::setCSVMappingRules", err );
    csvMapper=iCSVMappingRules;
    return err;
}

SKGError SKGImportExportManager::setCSVMapping ( const QStringList* iCSVMapping )
{
    SKGError err;
    SKGTRACEINRC ( 10, "SKGImportExportManager::setCSVMapping", err );

    csvMapping.clear();

    if ( iCSVMapping==NULL )
    {
        //Automatic build
        //Open file
        QFile file ( fileName );
        if ( !file.open ( QIODevice::ReadOnly ) )
        {
            err.setReturnCode ( ERR_INVALIDARG );
            err.setMessage ( i18nc ( "Error message",  "Open file '%1' failed",fileName ) );
        }
        else
        {
            QTextStream stream ( &file );
            if ( !codec.isEmpty() ) stream.setCodec ( codec.toAscii().constData() );

            //Ignore useless lines
            int headerIndex=getCSVHeaderIndex();
            for ( int i=0; i<headerIndex; ++i )
                stream.readLine();

            //Get mapping
            if ( !stream.atEnd() )
            {
                csvMapping=getCSVMappingFromLine ( stream.readLine() );
            }
            else
            {
                err.setReturnCode ( ERR_INVALIDARG );
            }

            //close file
            file.close();
        }
    }
    else
    {
        //Manual build
        csvMapping=*iCSVMapping;
    }

    if ( err.isSucceeded() )
    {
        //Check if mandatory attributes have been found
        if ( !csvMapping.contains ( "date" ) || !csvMapping.contains ( "amount" ) )
            err=SKGError ( ERR_FAIL, i18nc ( "Error message",  "Columns date and amount not found. Set import parameters in settings." ) );
    }

    return err;
}

QStringList SKGImportExportManager::getCSVMapping() const
{
    SKGTRACEIN ( 10, "SKGImportExportManager::getCSVMapping" );
    return csvMapping;
}

SKGError SKGImportExportManager::setCSVHeaderIndex ( int iIndex )
{
    SKGError err;
    SKGTRACEINRC ( 10, "SKGImportExportManager::setCSVHeaderIndex", err );

    if ( iIndex==-1 )
    {
        //Automatic build
        //Open file
        QFile file ( fileName );
        if ( !file.open ( QIODevice::ReadOnly ) )
        {
            err.setReturnCode ( ERR_INVALIDARG );
            err.setMessage ( i18nc ( "Error message",  "Open file '%1' failed",fileName ) );
        }
        else
        {
            QTextStream stream ( &file );
            if ( !codec.isEmpty() ) stream.setCodec ( codec.toAscii().constData() );

            int i=0;
            csvHeaderIndex=-1;
            while ( !stream.atEnd() && csvHeaderIndex==-1 )
            {
                //Read line
                QStringList map=getCSVMappingFromLine ( stream.readLine() );
                if ( map.contains ( "date" ) && map.contains ( "amount" ) ) csvHeaderIndex=i;

                ++i;
            }

            //close file
            file.close();
        }
    }
    else
    {
        //Manual build
        csvHeaderIndex=iIndex;
    }

    return err;
}

int SKGImportExportManager::getCSVHeaderIndex()
{
    SKGTRACEIN ( 10, "SKGImportExportManager::getCSVHeaderIndex" );
    if ( csvHeaderIndex==-1 ) setCSVHeaderIndex ( -1 );
    return csvHeaderIndex;
}

SKGError SKGImportExportManager::importCSVUnit()
{
    SKGError err;
    SKGTRACEINRC ( 2, "SKGImportExportManager::importCSVUnit", err );
    SKGTRACEL ( 10 ) << "Input filename=" << fileName << endl;

    if ( document )
    {
        //Begin transaction
        err=document->beginTransaction ( "#INTERNAL#", 3 );
        if ( err.isSucceeded() )
        {
            //File name is the name of the unit
            QFileInfo fInfo ( fileName );
            QString unitName=fInfo.baseName();

            //Default mapping
            if ( getCSVMapping().count() ==0 )
            {
                err= setCSVMapping ( NULL );
                if ( err.isSucceeded() ) err=document->sendMessage ( i18nc ( "An information message",  "Use automatic search of the columns" ) );
            }

            //Step 1 done
            if ( err.isSucceeded() ) err=document->stepForward ( 1 );

            //Open file
            if ( err.isSucceeded() )
            {
                QFile file ( fileName );
                if ( !file.open ( QIODevice::ReadOnly | QIODevice::Text ) )
                {
                    err.setReturnCode ( ERR_INVALIDARG );
                    err.setMessage ( i18nc ( "Error message",  "Open file '%1' failed",fileName ) );
                }
                else
                {
                    QTextStream stream ( &file );
                    if ( !codec.isEmpty() ) stream.setCodec ( codec.toAscii().constData() );

                    //Ignore useless lines
                    int headerIndex=getCSVHeaderIndex();
                    for ( int i=0; i<=headerIndex; ++i )
                        stream.readLine();

                    //Get data column
                    QStringList dates;
                    QStringList lines;
                    int posdate=csvMapping.indexOf ( "date" );
                    if ( posdate!=-1 )
                    {
                        while ( !stream.atEnd() )
                        {
                            //Read line
                            QString line=stream.readLine().trimmed();
                            if ( !line.isEmpty() )
                            {
                                lines.push_back ( line );

                                //Get date
                                QStringList field=SKGServices::splitCSVLine ( line );
                                if ( posdate<field.count() ) dates.push_back ( field.at ( posdate ) );
                            }
                        }
                    }

                    //close file
                    file.close();

                    //Select dateformat
                    QString dateFormat=SKGServices::getDateFormat ( dates );
                    if ( dateFormat.isEmpty() )
                    {
                        err.setReturnCode ( ERR_FAIL );
                        err.setMessage ( i18nc ( "Error message",  "Date format not supported" ) );
                    }
                    if ( err.isSucceeded() )
                        err=document->sendMessage ( i18nc ( "An information message",  "Import of '%1' with code '%2' and date format '%3'",fileName,codec,dateFormat ) );

                    //Step 2 done
                    if ( err.isSucceeded() ) err=document->stepForward ( 2 );

                    //Treat all lines
                    if ( err.isSucceeded() )
                    {
                        int nb=lines.size();
                        err=document->beginTransaction ( "#INTERNAL#", nb );

                        //Save last mapping used in a settings
                        QString mappingDesc;
                        int nbMap=csvMapping.count();
                        for ( int i=0; i<nbMap; ++i )
                        {
                            if ( i ) mappingDesc+='|';
                            mappingDesc+=csvMapping.at ( i );
                        }
                        if ( err.isSucceeded() ) err=document->setParameter ( "SKG_LAST_CSV_UNIT_MAPPING_USED", mappingDesc );

                        int posdate=csvMapping.indexOf ( "date" );
                        int posvalue=csvMapping.indexOf ( "amount" );
                        if ( posdate!=-1 && posvalue!=-1 )
                        {
                            for ( int i = 0; err.isSucceeded() && i < nb; ++i )
                            {
                                QStringList atts=SKGServices::splitCSVLine ( lines.at ( i ) );
                                err=document->addOrModifyUnitValue ( unitName,
                                                                     SKGServices::stringToTime ( SKGServices::dateToSqlString ( atts.at ( posdate ), dateFormat ) ).date(),
                                                                     SKGServices::stringToDouble ( atts.at ( posvalue ) ) );

                                if ( err.isSucceeded() ) err=document->stepForward ( i+1 );
                            }
                        }

                        if ( err.isSucceeded() ) err=document->endTransaction ( true );
                        else  document->endTransaction ( false );

                        //Lines treated
                        if ( err.isSucceeded() ) err=document->stepForward ( 3 );
                    }
                }
            }
        }
        if ( err.isSucceeded() ) err=document->endTransaction ( true );
        else  document->endTransaction ( false );
    }

    return err;
}

SKGError SKGImportExportManager::importCSV()
{
    SKGError err;
    SKGTRACEINRC ( 2, "SKGImportExportManager::importCSV", err );
    SKGTRACEL ( 10 ) << "Input filename=" << fileName << endl;

    //Begin transaction
    if ( document )
    {
        err=document->beginTransaction ( "#INTERNAL#", 3 );
        if ( err.isSucceeded() )
        {
            //Create account if needed
            QDateTime now=QDateTime::currentDateTime();
            QString postFix=SKGServices::dateToSqlString ( now );

            //Default mapping
            if ( getCSVMapping().count() ==0 )
            {
                err= setCSVMapping ( NULL );
                if ( err.isSucceeded() ) err=document->sendMessage ( i18nc ( "An information message",  "Use automatic search of the columns" ) );
            }

            //Step 1 done
            if ( err.isSucceeded() ) err=document->stepForward ( 1 );

            //Open file
            if ( err.isSucceeded() )
            {
                QFile file ( fileName );
                if ( !file.open ( QIODevice::ReadOnly | QIODevice::Text ) )
                {
                    err.setReturnCode ( ERR_INVALIDARG );
                    err.setMessage ( i18nc ( "Error message",  "Open file '%1' failed",fileName ) );
                }
                else
                {
                    QTextStream stream ( &file );
                    if ( !codec.isEmpty() ) stream.setCodec ( codec.toAscii().constData() );

                    //Ignore useless lines
                    int headerIndex=getCSVHeaderIndex();
                    if (headerIndex==-1)
                    {
                        err.setReturnCode ( ERR_FAIL );
                        err.setMessage ( i18nc ( "Error message",  "Header not found in csv file" ) );
                    }

                    for ( int i=0; i<=headerIndex; ++i )
                        stream.readLine();

                    //Get data column
                    QStringList dates;
                    QStringList lines;
                    if (err.isSucceeded())
                    {
                        int posdate=csvMapping.indexOf ( "date" );
                        if ( posdate!=-1 )
                        {
                            while ( !stream.atEnd() )
                            {
                                //Read line
                                QString line=stream.readLine().trimmed();
                                if ( !line.isEmpty() )
                                {
                                    lines.push_back ( line );

                                    //Get date
                                    QStringList field=SKGServices::splitCSVLine ( line, csvSeparator );
                                    if ( posdate<field.count() ) dates.push_back ( field.at ( posdate ).trimmed() );
                                }
                            }
                        }
                    }

                    //close file
                    file.close();

                    //Select dateformat
                    QString dateFormat=SKGServices::getDateFormat ( dates );
                    if ( err.isSucceeded() && dateFormat.isEmpty() )
                    {
                        err.setReturnCode ( ERR_FAIL );
                        err.setMessage ( i18nc ( "Error message",  "Date format not supported" ) );
                    }
                    if ( err.isSucceeded() )
                        err=document->sendMessage ( i18nc ( "An information message",  "Import of '%1' with code '%2' and date format '%3'",fileName,codec,dateFormat ) );

                    //Step 2 done
                    if ( err.isSucceeded() ) err=document->stepForward ( 2 );

                    //Treat all lines
                    if ( err.isSucceeded() )
                    {
                        int nb=lines.size();
                        err=document->beginTransaction ( "#INTERNAL#", nb );

                        //Save last mapping used in a settings
                        QString mappingDesc;
                        int nbMap=csvMapping.count();
                        for ( int i=0; i<nbMap; ++i )
                        {
                            if ( i ) mappingDesc+='|';
                            mappingDesc+=csvMapping.at ( i );
                        }
                        if ( err.isSucceeded() ) err=document->setParameter ( "SKG_LAST_CSV_MAPPING_USED", mappingDesc );

                        SKGUnitObject defUnit;
                        SKGAccountObject defAccount;
                        int nbOperationsNotImported=0;
                        QMap<int, SKGOperationObject> mapGroup;
                        QMap<int, SKGOperationObject> mapOperation;
                        for ( int i = 0; err.isSucceeded() && i < nb; ++i )
                        {
                            SKGOperationObject currentOperation ( document );
                            SKGSubOperationObject currentSubOperation ( document );

                            //Valuate mandatory attribute with default value
                            if ( csvMapping.indexOf ( "unit" ) ==-1 )
                            {
                                err=getDefaultUnit ( defUnit );
                                if ( err.isSucceeded() ) err=currentOperation.setUnit ( defUnit );
                            }
                            if ( err.isSucceeded() && csvMapping.indexOf ( "account" ) ==-1 )
                            {
                                err=getDefaultAccount ( defAccount );
                                if ( err.isSucceeded() ) err=currentOperation.setParentAccount ( defAccount );
                            }

                            QString line=lines.at ( i );
                            QByteArray hash = QCryptographicHash::hash ( line.toUtf8(), QCryptographicHash::Md5 );

                            SKGObjectBase opWithThisHash;
                            if ( SKGObjectBase::getObject ( document, "operation", "t_imported IN ('Y','P') AND t_import_id='"+QString ( hash.toHex() ) +'\'', opWithThisHash ).isSucceeded() ) nbOperationsNotImported++;
                            else
                            {
                                QString skg_op_original_amount;
                                QStringList atts=SKGServices::splitCSVLine ( line, csvSeparator );
                                int nbcol=csvMapping.count();
                                if ( atts.count() <nbcol )
                                {
                                    err=SKGError ( ERR_INVALIDARG, i18nc ( "Error message", "Invalid number of columns in line %1. Expected %2. Found %3.",
                                                                           headerIndex+1+i+1, nbcol, atts.count() ) );
                                }
                                int initialBalance=false;
                                int idgroup=0;
                                int idtransaction=0;
                                QStringList propertiesAtt;
                                QStringList propertiesVal;
                                for ( int c=0; err.isSucceeded() && c<nbcol; ++c )
                                {
                                    QString col=csvMapping[c];
                                    if (!col.isEmpty())
                                    {
                                        QString val;
                                        if ( c>=0 && c<atts.count() ) val=atts.at ( c ).trimmed();
                                        if ( col=="date" )
                                        {
                                            err=currentOperation.setDate ( SKGServices::stringToTime ( SKGServices::dateToSqlString ( val, dateFormat ) ).date() );
                                            if ( val=="0000-00-00" ) initialBalance=true;
                                        }
                                        else if ( col=="number" )
                                        {
                                            if ( !val.isEmpty() ) err=currentOperation.setNumber ( SKGServices::stringToInt ( val ) );
                                        }
                                        else if ( col=="mode" )
                                        {
                                            err=currentOperation.setMode ( val );
                                        }
                                        else if ( col=="payee" )
                                        {
                                            SKGPayeeObject payeeObj;
                                            err=SKGPayeeObject::createPayee(document, val, payeeObj);
                                            if ( err.isSucceeded() ) err=currentOperation.setPayee ( payeeObj );
                                        }
                                        else if ( col=="comment" )
                                        {
                                            QString comment=currentOperation.getComment();
                                            if (!comment.isEmpty()) comment+=' ';
                                            comment+=val;
                                            err=currentOperation.setComment ( comment );
                                            if ( err.isSucceeded() ) err=currentSubOperation.setComment ( comment );
                                        }
                                        else if ( col=="status" )
                                        {
                                            err=currentOperation.setStatus ( val=="C" || val=="Y" ?  SKGOperationObject::CHECKED : val=="P" ?  SKGOperationObject::POINTED :SKGOperationObject::NONE );
                                        }
                                        else if ( col=="bookmarked" )
                                        {
                                            err=currentOperation.bookmark ( val=="Y" );
                                        }
                                        else if ( col=="idgroup" )
                                        {
                                            idgroup=SKGServices::stringToInt(val);
                                        }
                                        else if ( col=="idtransaction" )
                                        {
                                            idtransaction=SKGServices::stringToInt(val);
                                        }
                                        else if ( col=="amount" )
                                        {
                                            if ( csvMapping.contains ( "quantity" ) )
                                            {
                                                //209705 vvvv
                                                skg_op_original_amount=val;
                                                //209705 ^^^^
                                            }
                                            else
                                            {
                                                err=currentSubOperation.setQuantity ( SKGServices::stringToDouble ( val ) );
                                            }
                                        }
                                        else if ( col=="quantity" )
                                        {
                                            err=currentSubOperation.setQuantity ( SKGServices::stringToDouble ( val ) );
                                        }
                                        else if ( col=="sign" )
                                        {
                                            if ( val=="DEBIT" || val=="-" ) // krazy:exclude=doublequote_chars
                                            {
                                                double cval=currentSubOperation.getQuantity();
                                                if ( cval>0 ) err=currentSubOperation.setQuantity ( -cval );
                                            }
                                        }
                                        else if ( col=="unit" )
                                        {
                                            //Looking for unit
                                            SKGUnitObject unit ( document );
                                            if ( val!=defUnit.getName() ) //For performance
                                            {
                                                err=unit.setName ( val );
                                                if ( err.isSucceeded() ) err=unit.setSymbol ( val );
                                                if ( err.isSucceeded() && unit.load().isFailed() )  err=unit.save ( false ); //Save only

                                                //This unit is now the default one, it's better for performance
                                                defUnit=unit;
                                            }
                                            else
                                            {
                                                unit=defUnit;
                                            }

                                            SKGUnitValueObject unitval;
                                            if ( err.isSucceeded() ) err=unit.addUnitValue ( unitval );
                                            if ( err.isSucceeded() )
                                            {
                                                int posAmount=csvMapping.indexOf ( "amount" );
                                                int posQuantity=csvMapping.indexOf ( "quantity" );
                                                if ( posAmount!=-1 && posQuantity!=-1 )
                                                {
                                                    err=unitval.setQuantity ( SKGServices::stringToDouble ( atts.at ( posAmount ) ) /SKGServices::stringToDouble ( atts.at ( posQuantity ) ) );
                                                }
                                                else
                                                {
                                                    err=unitval.setQuantity ( 1 );
                                                }
                                            }
                                            if ( err.isSucceeded() ) err=unitval.setDate ( now.date() );
                                            if ( err.isSucceeded() ) err=unitval.save();
                                            if ( err.isSucceeded() ) err=currentOperation.setUnit ( unit );
                                        }
                                        else if ( col=="account" )
                                        {
                                            //Looking for account
                                            if ( val!=defAccount.getName() ) //For performance
                                            {
                                                SKGAccountObject account ( document );
                                                account.setName ( val );
                                                err=account.load();
                                                if ( err.isFailed() )
                                                {
                                                    //Not found, we have to create one
                                                    SKGBankObject bank ( document );
                                                    QString name=i18nc ( "Noun",  "Bank for import %1",postFix );
                                                    err=bank.setName ( name );
                                                    if ( err.isSucceeded() && bank.load().isFailed() )
                                                    {
                                                        err=bank.save ( false );  //Save only
                                                        if ( err.isSucceeded() ) err=document->sendMessage ( i18nc ( "An information message",  "Default bank '%1' created for import",name ) );
                                                    }
                                                    if ( err.isSucceeded() ) err=bank.addAccount ( account );
                                                    if ( err.isSucceeded() ) err=account.setName ( val );
                                                    if ( err.isSucceeded() && account.load().isFailed() )  err=account.save ( false );  //Save only
                                                }

                                                //This account is now the default one, it's better for performance
                                                defAccount=account;
                                            }
                                            if ( err.isSucceeded() ) err=currentOperation.setParentAccount ( defAccount );
                                        }
                                        else if ( col=="category" )
                                        {
                                            //Set Category
                                            if ( !val.isEmpty() )
                                            {
                                                SKGCategoryObject Category;
                                                val.replace ( '/', OBJECTSEPARATOR );
                                                val.replace ( ':', OBJECTSEPARATOR );
                                                val.replace ( ',', OBJECTSEPARATOR );
                                                val.replace ( ';', OBJECTSEPARATOR );
                                                err=SKGCategoryObject::createPathCategory ( document,val, Category );
                                                if ( err.isSucceeded() )  err=currentSubOperation.setCategory ( Category );
                                            }
                                        }
                                        else
                                        {
                                            //A property
                                            propertiesAtt.push_back(col);
                                            propertiesVal.push_back(val);
                                        }
                                    }
                                }

                                if ( err.isSucceeded() && initialBalance )
                                {
                                    //Specific values for initial balance
                                    err=currentOperation.setStatus ( SKGOperationObject::CHECKED );
                                    if ( err.isSucceeded() ) err=currentOperation.setAttribute ( "d_date", "0000-00-00" );
                                }
                                if ( err.isSucceeded() ) err=currentOperation.setImportID ( hash.toHex() );
                                if ( err.isSucceeded() )
                                {
                                    if ( idtransaction!=0 )
                                    {
                                        if (mapOperation.contains(idtransaction))
                                        {
                                            currentOperation=mapOperation[idtransaction];
                                            skg_op_original_amount="";
                                        }
                                        else
                                        {
                                            err=currentOperation.save ();
                                            mapOperation[idtransaction]=currentOperation;
                                        }
                                    }
                                    else
                                    {
                                        err=currentOperation.save ( false );  //Save only
                                    }
                                }
                                if ( err.isSucceeded() && idgroup!=0 )
                                {
                                    if (mapGroup.contains(idgroup))
                                    {
                                        err=currentOperation.setGroupOperation ( mapGroup[idgroup] );
                                        if ( err.isSucceeded() ) err=currentOperation.save ();
                                    }
                                    mapGroup[idgroup]=currentOperation;
                                }

                                if ( err.isSucceeded() ) err=currentSubOperation.setParentOperation ( currentOperation );
                                if ( err.isSucceeded() ) err=currentSubOperation.save ( false, false );  //Save only without reload

                                //209705 vvvv
                                if ( err.isSucceeded() && !skg_op_original_amount.isEmpty() )
                                {
                                    err=currentOperation.setProperty ( "SKG_OP_ORIGINAL_AMOUNT", skg_op_original_amount );
                                }
                                //209705 ^^^^

                                //Add properties
                                int nbp=propertiesAtt.count();
                                for (int p=0; err.isSucceeded() && p<nbp; ++p)
                                {
                                    err=currentOperation.setProperty (propertiesAtt.at(p) , propertiesVal.at(p) );
                                }
                            }
                            if ( err.isSucceeded() && i%20==0 ) err = SKGServices::executeSqliteOrder ( document, "ANALYZE" );
                            if ( err.isSucceeded() ) err=document->stepForward ( i+1 );
                        }

                        if ( err.isSucceeded() ) err=document->endTransaction ( true );
                        else  document->endTransaction ( false );

                        if ( err.isSucceeded() && nbOperationsNotImported )
                            err=document->sendMessage ( i18np ( "one operation not imported because it already exists","%1 operations not imported because they already exists",nbOperationsNotImported ) );

                        //Lines treated
                        if ( err.isSucceeded() ) err=document->stepForward ( 3 );
                    }
                }
            }
        }
        if ( err.isSucceeded() ) err=document->endTransaction ( true );
        else  document->endTransaction ( false );
    }

    return err;
}

bool SKGImportExportManager::isOFXSupported()
{
#ifdef SKG_ENABLE_OFX
    return true;
#else
    return false;
#endif
}

#ifdef SKG_ENABLE_OFX
SKGError SKGImportExportManager::importOFX()
{
    SKGError err;
    SKGTRACEINRC ( 2, "SKGDocumentBank::importOFX", err );
    SKGTRACEL ( 10 ) << "Input filename=" << fileName << endl;

    if ( document )
    {
        err=document->beginTransaction ( "#INTERNAL#" );
        if ( err.isSucceeded() )
        {
            SKGImportExportManager::nbOperationsNotImported=0;
            SKGImportExportManager::nbOperationsImported=0;
            SKGImportExportManager::ofxError=SKGError();
            SKGImportExportManager::ofxInitialBalance.clear();

            LibofxContextPtr ctx = libofx_get_new_context();

            ofx_set_account_cb ( ctx, ofxAccountCallback, this );
            ofx_set_statement_cb ( ctx, ofxStatementCallback, this );
            ofx_set_transaction_cb ( ctx, ofxTransactionCallback, this );
            //ofx_set_security_cb(ctx, ofxSecurityCallback, this);
            int rc=libofx_proc_file ( ctx, fileName.toUtf8(), AUTODETECT );
            if ( rc )  err=SKGError ( ERR_FAIL, i18nc ( "Error message",  "Import OFX file '%1' failed",fileName ) );
            if ( err.isSucceeded() ) err=SKGImportExportManager::ofxError;
            {
                //This is an option ==> no error management
                SKGError err;
                int nb=SKGImportExportManager::ofxInitialBalance.count();
                for ( int i=0; err.isSucceeded() && i<nb; ++i )
                {
                    OfxStatementData data=SKGImportExportManager::ofxInitialBalance.at ( i );

                    //Get account
                    SKGAccountObject act;
                    if ( err.isSucceeded() ) err=getAccount(data.account_ptr, document, act);

                    //Date of balance
                    QDate date= ( data.ledger_balance_date_valid==true ? QDateTime::fromTime_t ( data.ledger_balance_date ).date() : QDate::currentDate() );

                    //Get unit
                    SKGUnitObject unit;
                    if ( err.isSucceeded() ) err=getDefaultUnit ( unit );

                    //Current amount
                    double currentAmount=act.getAmount ( date );

                    //Update account
                    if ( err.isSucceeded() ) err=act.setInitialBalance ( data.ledger_balance-currentAmount, unit );
                    if ( err.isSucceeded() ) err=act.save();
                    if ( err.isSucceeded() ) err=document->sendMessage ( i18nc ( "An information message","The initial balance of '%1' has been set with OFX file content", act.getName()) );
                }
            }
            SKGImportExportManager::ofxInitialBalance.clear();

            if ( err.isSucceeded() ) err=document->endTransaction ( true );
            else  document->endTransaction ( false );

            if ( err.isSucceeded() && SKGImportExportManager::nbOperationsNotImported )
                err=document->sendMessage ( i18np ( "one operation not imported because it already exists","%1 operations not imported because they already exists",SKGImportExportManager::nbOperationsNotImported ) );
        }
    }
    return err;
}

SKGError SKGImportExportManager::getAccount(OfxAccountData* iAccountData, SKGDocumentBank* iDoc, SKGAccountObject& oAccount)
{
    SKGError err;
    QString accountNumber=iAccountData->account_id;
    QString bankNumber=iAccountData->bank_id;
    //Correction BUG 234771 vvvvv
    accountNumber=accountNumber.trimmed();
    bankNumber=bankNumber.trimmed();
    //Correction BUG 234771 ^^^^^
    if ( accountNumber.startsWith ( bankNumber+' ' ) )
    {
        accountNumber=accountNumber.right ( accountNumber.length()-bankNumber.length()-1 );
        QStringList splitNumbers = accountNumber.split ( ' ' );
        if ( splitNumbers.count() ==2 ) accountNumber=splitNumbers.at ( 1 );
    }

    //Check if account is already existing
    err= SKGObjectBase::getObject ( iDoc, "v_account", "t_number='"+accountNumber+'\'', oAccount );

    return err;
}

int SKGImportExportManager::ofxStatementCallback ( struct OfxStatementData data, void * pv )
{
    if ( SKGImportExportManager::ofxError.isFailed() ) return 0;
    SKGTRACEINRC ( 5, "SKGImportExportManager::ofxStatementCallback", SKGImportExportManager::ofxError );

    SKGImportExportManager* impotExporter= static_cast<SKGImportExportManager*>(pv);
    if ( !impotExporter ) return 0;
    SKGDocumentBank* doc = impotExporter->getDocument();
    if ( !doc ) return 0;

    ////Get data
    OfxAccountData* accountData=data.account_ptr;
    if ( accountData && data.ledger_balance_valid==true )
    {
        //Get account
        SKGAccountObject act;
        SKGImportExportManager::ofxError=getAccount(accountData, doc, act);
        if ( SKGImportExportManager::ofxError.isSucceeded() )
        {
            if ( act.getNbOperation()>1 )
            {
                SKGImportExportManager::ofxError=doc->sendMessage ( i18nc ( "An information message","The initial balance of '%1' has not been set because some operations are already existing",act.getName() ) );
            }
            else
            {
                ofxInitialBalance.push_back ( data );
            }
        }
    }

    return SKGImportExportManager::ofxError.getReturnCode();
}

int SKGImportExportManager::ofxAccountCallback ( struct OfxAccountData data, void * pv )
{
    if ( SKGImportExportManager::ofxError.isFailed() ) return 0;
    SKGTRACEINRC ( 5, "SKGImportExportManager::ofxAccountCallback", SKGImportExportManager::ofxError );

    SKGImportExportManager* impotExporter= static_cast<SKGImportExportManager*>(pv);
    if ( !impotExporter ) return 0;
    SKGDocumentBank* doc = impotExporter->getDocument();
    if ( !doc ) return 0;

    SKGObjectBase tmp;
    if ( data.account_id_valid==true )
    {
        QString agencyNumber=data.branch_id;
        QString accountNumber=data.account_id;
        QString bankNumber=data.bank_id;
        //Correction BUG 234771 vvvvv
        accountNumber=accountNumber.trimmed();
        bankNumber=bankNumber.trimmed();
        //Correction BUG 234771 ^^^^^
        if ( accountNumber.startsWith ( bankNumber+' ' ) )
        {
            accountNumber=accountNumber.right ( accountNumber.length()-bankNumber.length()-1 );
            QStringList splitNumbers = accountNumber.split ( ' ' );
            if ( splitNumbers.count() ==2 )
            {
                agencyNumber=splitNumbers.at ( 0 );
                accountNumber=splitNumbers.at ( 1 );
            }
        }

        //Check if account is already existing
        SKGAccountObject account;
        SKGImportExportManager::ofxError=getAccount(&data, doc, account);
        if ( SKGImportExportManager::ofxError.isSucceeded() )
        {
            //Already existing
            account=tmp;
            SKGImportExportManager::ofxError=impotExporter->setDefaultAccount ( &account );
        }
        else
        {
            //Not existing
            QString bankId= ( data.bank_id_valid == true ? data.bank_id : ( data.broker_id_valid == true ? data.broker_id : i18nc ( "Adjective, an unknown item","Unknown" ) ) );
            if ( !bankId.isEmpty() )
            {
                //Check if bank is already existing
                SKGBankObject bank;
                SKGImportExportManager::ofxError= SKGObjectBase::getObject ( doc, "v_bank", "t_bank_number='"+bankId+'\'', tmp );
                if ( SKGImportExportManager::ofxError.isSucceeded() )
                {
                    //Already existing
                    bank=tmp;
                }
                else
                {
                    //Create new bank
                    bank=SKGBankObject ( doc );
                    SKGImportExportManager::ofxError=bank.setName ( bankId );
                    if ( SKGImportExportManager::ofxError.isSucceeded() ) SKGImportExportManager::ofxError=bank.setNumber ( data.bank_id );
                    if ( SKGImportExportManager::ofxError.isSucceeded() ) SKGImportExportManager::ofxError=bank.save();
                }

                //Create new account
                QString name=data.account_name;
                if ( name.isEmpty() ) name=data.account_id;

                if ( SKGImportExportManager::ofxError.isSucceeded() ) SKGImportExportManager::ofxError=bank.addAccount ( account );
                if ( SKGImportExportManager::ofxError.isSucceeded() ) SKGImportExportManager::ofxError=account.setName ( name );
                if ( SKGImportExportManager::ofxError.isSucceeded() ) SKGImportExportManager::ofxError=account.setNumber ( accountNumber );
                if ( SKGImportExportManager::ofxError.isSucceeded() ) SKGImportExportManager::ofxError=account.setAgencyNumber ( agencyNumber );
                if ( SKGImportExportManager::ofxError.isSucceeded() ) SKGImportExportManager::ofxError=account.setComment ( data.account_name );
                SKGAccountObject::AccountType type=SKGAccountObject::CURRENT;
                if ( data.account_type_valid==true )
                {
                    switch ( data.account_type )
                    {
                    case OfxAccountData::OFX_CHECKING:
                    case OfxAccountData::OFX_SAVINGS:
                    case OfxAccountData::OFX_CREDITLINE:
                    case OfxAccountData::OFX_CMA:
                        type = SKGAccountObject::CURRENT;
                        break;
                    case OfxAccountData::OFX_MONEYMRKT:
                    case OfxAccountData::OFX_INVESTMENT:
                        type = SKGAccountObject::INVESTMENT;
                        break;
                    case OfxAccountData::OFX_CREDITCARD:
                        type = SKGAccountObject::CREDITCARD;
                        break;
                    default:
                        break;
                    }
                }

                if ( SKGImportExportManager::ofxError.isSucceeded() ) SKGImportExportManager::ofxError=account.setType ( type );
                if ( SKGImportExportManager::ofxError.isSucceeded() ) SKGImportExportManager::ofxError=account.save();

                if ( SKGImportExportManager::ofxError.isSucceeded() ) SKGImportExportManager::ofxError=impotExporter->setDefaultAccount ( &account );

                SKGTRACEL(10) << "Account [" << name << "] created" << endl;
            }
        }
    }
    if ( data.currency_valid==true )
    {
        //Check if unit is already existing
        SKGUnitObject unit ( doc );
        SKGImportExportManager::ofxError= SKGObjectBase::getObject ( doc, "v_unit", "t_name='"+QString ( data.currency ) +"' OR t_name like '%("+QString ( data.currency ) +")%'", tmp );
        if ( SKGImportExportManager::ofxError.isSucceeded() )
        {
            //Already existing
            unit=tmp;
            SKGImportExportManager::ofxError=impotExporter->setDefaultUnit ( &unit );
        }
        else
        {
            //Create new account
            SKGImportExportManager::ofxError=unit.setName ( data.currency );
            if ( SKGImportExportManager::ofxError.isSucceeded() ) SKGImportExportManager::ofxError=unit.setSymbol ( data.currency );
            if ( SKGImportExportManager::ofxError.isSucceeded() ) SKGImportExportManager::ofxError=unit.save();

            if ( SKGImportExportManager::ofxError.isSucceeded() ) SKGImportExportManager::ofxError=impotExporter->setDefaultUnit ( &unit );
        }
    }
    return SKGImportExportManager::ofxError.getReturnCode();
}

int SKGImportExportManager::ofxTransactionCallback ( struct OfxTransactionData data, void * pv )
{
    if ( SKGImportExportManager::ofxError.isFailed() ) return 0;
    SKGTRACEINRC ( 5, "SKGImportExportManager::ofxTransactionCallback", SKGImportExportManager::ofxError );

    SKGImportExportManager* impotExporter= static_cast<SKGImportExportManager*>(pv);
    if ( !impotExporter ) return 0;
    SKGDocumentBank* doc = impotExporter->getDocument();
    if ( !doc ) return 0;

    //Get account
    SKGAccountObject account;
    SKGImportExportManager::ofxError=getAccount(data.account_ptr, doc, account);
    if ( SKGImportExportManager::ofxError.isSucceeded() )
    {
        //Get operation date
        QDate date=QDateTime::fromTime_t ( data.date_posted_valid==true ? data.date_posted : data.date_initiated ).date();

        //Get unit
        SKGUnitObject unit;
        SKGImportExportManager::ofxError=impotExporter->getDefaultUnit ( unit );

        //Create id
        QString ID;
        if ( data.fi_id_valid==true )
        {
            if (QString(data.fi_id).count('0')!=QString(data.fi_id).count()) ID = QString ( "ID-" ) + data.fi_id;
        }
        else if ( data.reference_number_valid==true )
        {
            if (QString(data.reference_number).count('0')!=QString(data.reference_number).count()) ID = QString ( "REF-" ) + data.reference_number;
        }

        //Check if already imported
        SKGObjectBase opWithThisHash;
        if ( (!ID.isEmpty() && SKGObjectBase::getObject ( doc, "operation", "t_imported IN ('Y','P') AND t_import_id='"+QString ( ID ) +'\'', opWithThisHash ).isSucceeded())
                || ( data.invtransactiontype_valid==true && data.invtransactiontype==OFX_SPLIT ) )
        {
            //Yes
            SKGImportExportManager::nbOperationsNotImported++;
        }
        else
        {
            //No
            //Create operation
            SKGOperationObject ope;
            if ( SKGImportExportManager::ofxError.isSucceeded() ) SKGImportExportManager::ofxError=account.addOperation ( ope );
            if ( SKGImportExportManager::ofxError.isSucceeded() ) SKGImportExportManager::ofxError=ope.setDate ( date );
            if ( SKGImportExportManager::ofxError.isSucceeded() ) SKGImportExportManager::ofxError=ope.setUnit ( unit );
            if ( SKGImportExportManager::ofxError.isSucceeded() ) SKGImportExportManager::ofxError=ope.setAttribute ( "t_imported", "T" );
            if ( SKGImportExportManager::ofxError.isSucceeded() ) SKGImportExportManager::ofxError=ope.setImportID ( ID );
            if ( SKGImportExportManager::ofxError.isSucceeded() )
            {
                QString payeeName;
                if ( data.payee_id_valid==true ) payeeName=data.payee_id;
                else if ( data.name_valid==true ) payeeName=data.name;
                if (!payeeName.isEmpty())
                {
                    SKGPayeeObject payeeObj;
                    SKGImportExportManager::ofxError=SKGPayeeObject::createPayee(doc, payeeName, payeeObj);
                    if ( SKGImportExportManager::ofxError.isSucceeded() ) SKGImportExportManager::ofxError=ope.setPayee ( payeeObj );
                }
            }
            if ( SKGImportExportManager::ofxError.isSucceeded() && data.memo_valid==true ) SKGImportExportManager::ofxError=ope.setComment ( data.memo );
            if ( SKGImportExportManager::ofxError.isSucceeded() && data.check_number_valid==true ) SKGImportExportManager::ofxError=ope.setNumber ( SKGServices::stringToInt ( data.check_number ) );
            if ( SKGImportExportManager::ofxError.isSucceeded() && data.invtransactiontype_valid ) SKGImportExportManager::ofxError=ope.setMode ( i18nc ( "Noun, the title of an item","Title" ) );
            if ( SKGImportExportManager::ofxError.isSucceeded() ) SKGImportExportManager::ofxError=ope.save();

            //Create sub operation
            SKGSubOperationObject subop;
            if ( SKGImportExportManager::ofxError.isSucceeded() ) SKGImportExportManager::ofxError=ope.addSubOperation ( subop );
            if ( SKGImportExportManager::ofxError.isSucceeded() && data.amount_valid==true ) SKGImportExportManager::ofxError=subop.setQuantity ( data.amount+ ( data.commission_valid==true ? data.commission : 0 ) );
            if ( SKGImportExportManager::ofxError.isSucceeded() && data.memo_valid==true ) SKGImportExportManager::ofxError=subop.setComment ( data.memo );
            if ( SKGImportExportManager::ofxError.isSucceeded() ) SKGImportExportManager::ofxError=subop.save();

            //Commission
            if ( SKGImportExportManager::ofxError.isSucceeded() && data.commission_valid==true && data.amount_valid==true && data.commission>0 )
            {
                //Create splitter operation
                SKGSubOperationObject subop2;
                if ( SKGImportExportManager::ofxError.isSucceeded() ) SKGImportExportManager::ofxError=ope.addSubOperation ( subop2 );
                if ( SKGImportExportManager::ofxError.isSucceeded() ) SKGImportExportManager::ofxError=subop2.setComment ( i18nc ( "Noun, a quantity of money taken by a financial institution to perform an operation","Commission" ) );
                if ( SKGImportExportManager::ofxError.isSucceeded() ) SKGImportExportManager::ofxError=subop2.setQuantity ( -data.commission );
                if ( SKGImportExportManager::ofxError.isSucceeded() ) SKGImportExportManager::ofxError=subop2.save();
            }

            SKGImportExportManager::nbOperationsImported++;
        }
    }
    return SKGImportExportManager::ofxError.getReturnCode();
}
#endif

SKGError SKGImportExportManager::importQIF()
{
    SKGError err;
    SKGTRACEINRC ( 2, "SKGImportExportManager::importQIF", err );
    SKGTRACEL ( 10 ) << "Input filename=" << fileName << endl;

    //Info for QIF format:
    //http://mb-net.net/Debian/src/gnucash/gnucash-2.2.6/src/import-export/qif-import/file-format.txt
    //http://web.intuit.com/support/quicken/docs/d_qif.html

    //Begin transaction
    if ( document )
    {
        err=document->beginTransaction ( "#INTERNAL#", 3 );
        if ( err.isSucceeded() )
        {
            //Create account if needed
            QDateTime now=QDateTime::currentDateTime();
            QString postFix=SKGServices::dateToSqlString ( now );

            //Step 1 done
            if ( err.isSucceeded() ) err=document->stepForward ( 1 );

            //Open file
            QFile file ( fileName );
            if ( !file.open ( QIODevice::ReadOnly | QIODevice::Text ) )
            {
                err.setReturnCode ( ERR_INVALIDARG );
                err.setMessage ( i18nc ( "Error message",  "Open file '%1' failed",fileName ) );
            }
            else
            {
                QTextStream stream ( &file );
                if ( !codec.isEmpty() ) stream.setCodec ( codec.toAscii().constData() );

                //load file in memory
                QStringList lines;
                QStringList dates;
                bool inAccountSection=false;
                while ( !stream.atEnd() )
                {
                    //Read line
                    //Check line if line is empty or is a commented
                    QString line=stream.readLine().trimmed();
                    if ( line.length() >0 && line[0]!='#' )
                    {
                        lines.push_back ( line );
                        //Manage !Account section
                        if ( line=="^" && inAccountSection )
                        {
                            inAccountSection=false;
                        }
                        else if ( QString::compare ( line, "!account", Qt::CaseInsensitive ) ==0 )
                        {
                            inAccountSection=true;
                        }

                        //We try to find automatically the date format
                        else if ( !inAccountSection &&line[0]=='D' )
                        {
                            dates.push_back ( line.right ( line.length()-1 ) );
                        }
                    }
                }

                //close file
                file.close();

                //Select dateformat
                QString dateFormat=SKGServices::getDateFormat ( dates );
                if ( dateFormat.isEmpty() )
                {
                    err.setReturnCode ( ERR_FAIL );
                    err.setMessage ( i18nc ( "Error message",  "Date format not supported" ) );
                }
                if ( err.isSucceeded() )
                    err=document->sendMessage ( i18nc ( "An information message",  "Import of '%1' with code '%2' and date format '%3'",fileName,codec,dateFormat ) );

                //Step 2 done
                if ( err.isSucceeded() ) err=document->stepForward ( 2 );

                //Treat all lines
                if ( err.isSucceeded() )
                {
                    SKGAccountObject* account=NULL;
                    SKGOperationObject currentOperation;
                    SKGOperationObject payement;
                    SKGPayeeObject currentPayee;
                    SKGSubOperationObject currentSubOperation;
                    QDate currentOperationDate;
                    QString transferAccount;
                    QString stringForHash;
                    QString currentUnitForInvestment;
                    QChar inSection='B';
                    bool currentOperationInitialized=false;
                    bool latestSubCatMustBeRemoved=false;
                    bool investmentAccount=false;
                    bool div=false;
                    bool automaticAccount=true;
                    int quantityFactor=1;
                    int nbOperationsNotImported=0;
                    double currentUnitPrice=1;
                    double checkOperationAmount=0;
                    double checkSuboperationsAmount=0;

                    int nb=lines.size();
                    err=document->beginTransaction ( "#INTERNAL#", nb );
                    for ( int i = 0; err.isSucceeded() && i < nb; ++i )
                    {
                        QString line=lines.at ( i );
                        QString val;
                        QChar op=line[0];
                        if ( line.length() >1 ) val=line.right ( line.length()-1 ).trimmed();

                        //Manage !Account section
                        if ( QString::compare ( line, "!type:bank", Qt::CaseInsensitive ) ==0 ||
                                QString::compare ( line, "!type:cash", Qt::CaseInsensitive ) ==0 ||
                                QString::compare ( line, "!type:ccard", Qt::CaseInsensitive ) ==0 ||
                                QString::compare ( line, "!type:oth a", Qt::CaseInsensitive ) ==0 ||
                                QString::compare ( line, "!type:oth l", Qt::CaseInsensitive ) ==0 ||
                                QString::compare ( line, "!type:invst", Qt::CaseInsensitive ) ==0 )
                        {
                            inSection='B';
                            investmentAccount= ( QString::compare ( val, "type:invst", Qt::CaseInsensitive ) ==0 );

                            //Set type of account
                            if ( account==NULL )
                            {
                                SKGAccountObject defAccount;
                                err=getDefaultAccount ( defAccount );
                                if ( err.isSucceeded() ) err=defAccount.addOperation ( currentOperation );
                                if ( err.isSucceeded() ) account=new SKGAccountObject ( defAccount );
                            }

                            if ( err.isSucceeded() && account )
                            {
                                err=account->setType ( QString::compare ( line, "!type:bank", Qt::CaseInsensitive ) ==0 ? SKGAccountObject::CURRENT :
                                                       ( QString::compare ( line, "!type:ccard", Qt::CaseInsensitive ) ==0 ? SKGAccountObject::CREDITCARD :
                                                         ( QString::compare ( line, "!type:invst", Qt::CaseInsensitive ) ==0 ? SKGAccountObject::INVESTMENT :
                                                           ( QString::compare ( line, "!type:oth a", Qt::CaseInsensitive ) ==0 ? SKGAccountObject::ASSETS : SKGAccountObject::OTHER ) ) ) );
                                if ( err.isSucceeded() ) err=account->save();
                            }
                        }
                        else if ( QString::compare ( line, "!account", Qt::CaseInsensitive ) ==0 )
                        {
                            inSection='A';
                            automaticAccount=false;
                        }
                        else if ( QString::compare ( line, "!type:cat", Qt::CaseInsensitive ) ==0 )
                        {
                            inSection='C';
                            if ( err.isSucceeded() ) err=document->sendMessage ( i18nc ( "An information message",  "Categories found and imported" ) );
                        }
                        else if ( line.at ( 0 ) =='!' )
                        {
                            inSection='?';
                        }
                        else if ( inSection=='C' )
                        {
                            //Category creation
                            if ( op=='N' )
                            {
                                SKGCategoryObject Category;
                                val.replace ( '/', OBJECTSEPARATOR );
                                val.replace ( ':', OBJECTSEPARATOR );
                                val.replace ( ',', OBJECTSEPARATOR );
                                val.replace ( ';', OBJECTSEPARATOR );
                                err=SKGCategoryObject::createPathCategory ( document,val, Category );
                            }
                        }
                        else if ( inSection=='A' )
                        {
                            //Account creation
                            if ( op=='N' )
                            {
                                //Check if the account already exist
                                SKGAccountObject account2;
                                err=SKGNamedObject::getObjectByName ( document, "account", val, account2 );
                                if ( err.isFailed() )
                                {
                                    //Create account
                                    SKGBankObject bank ( document );
                                    err=bank.setName ( i18nc ( "Noun",  "Bank for import %1",postFix ) );
                                    if ( err.isSucceeded() && bank.load().isFailed() ) err=bank.save ( false );
                                    if ( err.isSucceeded() ) err=bank.addAccount ( account2 );
                                    if ( err.isSucceeded() ) err=account2.setName ( val );
                                    if ( err.isSucceeded() && account2.load().isFailed() ) err=account2.save ( false );  //Save only
                                }

                                if ( err.isSucceeded() )
                                {
                                    if ( account )
                                    {
                                        delete account;
                                        account=NULL;
                                    }
                                    account=new SKGAccountObject ( account2 );
                                }
                            }
                            else if ( op=='D' )
                            {
                                if ( account ) err=account->setNumber ( val );
                            }
                            else if ( op=='T' )
                            {
                                if ( account ) err=account->setType ( val=="Bank" ? SKGAccountObject::CURRENT : ( val=="CCard" ? SKGAccountObject::CREDITCARD : ( val=="Invst" ? SKGAccountObject::INVESTMENT : ( val=="Oth A" ? SKGAccountObject::ASSETS : SKGAccountObject::OTHER ) ) ) );
                            }
                            else if ( op=='^' )
                            {
                                //^ 	End of entry
                                //save
                                if ( account ) err=account->save();
                            }
                        }
                        else if ( inSection=='B' )
                        {
                            //Operation creation
                            /*
                            >>>> Items for Non-Investment Accounts <<<<
                            DONE	D      Date
                            DONE	T      Amount
                            	U      Transaction amount (higher possible value than T)
                            DONE	C      Cleared status
                            DONE	N      Number (check or reference number)
                            DONE	P      Payee/description
                            DONE	M      Memo
                            DONE	A      Address (up to 5 lines; 6th line is an optional message)
                            DONE	L      Category (category/class or transfer/class)
                            DONE	S      Category in split (category/class or transfer/class)
                            DONE	E      Memo in split
                            DONE	$      Dollar amount of split
                            	%      Percentage of split if percentages are used
                            	F      Reimbursable business expense flag
                            	X      Small Business extensions
                            DONE	^      End of entry

                            >>>> Items for Investment Accounts <<<<
                            DONE	D  	Date
                            	N 	Action
                            DONE	Y 	Security
                            DONE	I 	Price
                            DONE	Q 	Quantity (number of shares or split ratio)
                            DONE	T 	Transaction amount
                            DONE	C 	Cleared status
                            	P 	Text in the first line for transfers and reminders
                            DONE	M 	Memo
                            	O 	Commission
                            	L 	Account for the transfer
                            	$ 	Amount transferred
                            	^ 	End of entry
                            */
                            stringForHash+=line;
                            if ( op=='D' )
                            {
                                //D  	Date
                                /*
                                Dates in US QIF files are usually in the format MM/DD/YY, although
                                four-digit years are not uncommon.  Dates sometimes occur without the
                                slash separator, or using other separators in place of the slash,
                                commonly '-' and '.'.  US Quicken seems to be using the ' to indicate
                                post-2000 two-digit years (such as 01/01'00 for Jan 1 2000).  Some
                                banks appear to be using a completely undifferentiated numeric QString
                                formateed YYYYMMDD in downloaded QIF files.
                                */
                                //Operation creation
                                if ( err.isSucceeded() )
                                {
                                    if ( account!=NULL ) err=account->addOperation ( currentOperation );
                                    else
                                    {
                                        SKGAccountObject defAccount;
                                        err=getDefaultAccount ( defAccount );
                                        if ( err.isSucceeded() ) err=defAccount.addOperation ( currentOperation );
                                    }
                                    currentOperationInitialized=true;
                                }

                                //Set date
                                currentOperationDate=SKGServices::stringToTime ( SKGServices::dateToSqlString ( val, dateFormat ) ).date();
                                if ( err.isSucceeded() ) err=currentOperation.setDate ( currentOperationDate );

                                //Set unit
                                if ( err.isSucceeded() )
                                {
                                    //Create unit if needed
                                    SKGUnitObject unit;
                                    err=getDefaultUnit ( unit, &currentOperationDate );
                                    if ( err.isSucceeded() )  err=currentOperation.setUnit ( unit );
                                }

                                if ( err.isSucceeded() ) currentOperation.save();

                                //Create suboperation
                                if ( err.isSucceeded() ) err=currentOperation.addSubOperation ( currentSubOperation );
                            }
                            else if ( op=='Y' )
                            {
                                //Y 	Security
                                if ( !div )
                                {
                                    currentUnitForInvestment=val;

                                    SKGUnitObject unit ( document );
                                    err = unit.setName ( currentUnitForInvestment );
                                    if ( unit.load().isFailed() )
                                    {
                                        if ( err.isSucceeded() ) err = unit.setSymbol ( currentUnitForInvestment );
                                        if ( err.isSucceeded() ) err = unit.setType ( investmentAccount ? SKGUnitObject::SHARE : SKGUnitObject::CURRENCY );
                                        if ( err.isSucceeded() ) err = unit.save ( false );  //Save only
                                    }
                                    if ( err.isSucceeded() ) err=currentOperation.setUnit ( unit );
                                }
                                else
                                {
                                    //For dividend, if comment is empty, we set the security in comment
                                    if ( currentOperation.getComment().isEmpty() ) err=currentOperation.setComment ( val );
                                }
                            }
                            else if ( op=='O' )
                            {
                                //O 	Commission
                                //Get previous quantity
                                double quantity=SKGServices::stringToDouble ( val );
                                SKGObjectBase::SKGListSKGObjectBase subops;
                                payement.getSubOperations ( subops );
                                if ( subops.count() )
                                {
                                    SKGSubOperationObject subpayement=subops.at ( 0 );
                                    err=subpayement.setQuantity ( subpayement.getQuantity() +quantity );
                                    if ( err.isSucceeded() ) err=subpayement.save ();
                                }

                                SKGSubOperationObject subcommission;
                                if ( !payement.exist() )
                                {
                                    //We have to create a new operation
                                    if ( account!=NULL ) err=account->addOperation ( payement );
                                    else
                                    {
                                        SKGAccountObject defAccount;
                                        err=getDefaultAccount ( defAccount );
                                        if ( err.isSucceeded() ) err=defAccount.addOperation ( payement );
                                    }
                                    if ( err.isSucceeded() ) err=payement.setDate ( currentOperationDate );
                                    if ( err.isSucceeded() )
                                    {
                                        //Create unit if needed
                                        SKGUnitObject unit;
                                        err=getDefaultUnit ( unit, &currentOperationDate );
                                        if ( err.isSucceeded() )  err=payement.setUnit ( unit );
                                    }
                                    if ( err.isSucceeded() ) err=payement.save();
                                }
                                if ( err.isSucceeded() ) err=payement.addSubOperation ( subcommission );
                                if ( err.isSucceeded() ) err=subcommission.setQuantity ( -quantity );
                                if ( err.isSucceeded() ) err=subcommission.save ( false, false );  //Save only whitout reload
                            }
                            else if ( op=='I' )
                            {
                                //I 	Price
                                currentUnitPrice=SKGServices::stringToDouble ( val );
                                if ( currentUnitPrice && !currentUnitForInvestment.isEmpty() )
                                    err=document->addOrModifyUnitValue ( currentUnitForInvestment, currentOperationDate, currentUnitPrice );
                            }
                            else if ( op=='N' )
                            {
                                if ( investmentAccount )
                                {
                                    //N 	Action
                                    /*
                                    QIF N Line    Notes
                                    ============  =====
                                    Aktab         Same as ShrsOut.
                                    AktSplit      Same as StkSplit.
                                    Aktzu         Same as ShrsIn.
                                    Buy           Buy shares.
                                    BuyX          Buy shares. Used with an L line.
                                    Cash          Miscellaneous cash transaction. Used with an L line.
                                    CGMid         Mid-term capital gains.
                                    CGMidX        Mid-term capital gains. For use with an L line.
                                    CGLong        Long-term capital gains.
                                    CGLongX       Long-term capital gains. For use with an L line.
                                    CGShort       Short-term capital gains.
                                    CGShortX      Short-term capital gains. For use with an L line.
                                    ContribX      Same as XIn. Used for tax-advantaged accounts.
                                    CvrShrt       Buy shares to cover a short sale.
                                    CvrShrtX      Buy shares to cover a short sale. Used with an L line.
                                    Div           Dividend received.
                                    DivX          Dividend received. For use with an L line.
                                    Errinerg      Same as Reminder.
                                    Exercise      Exercise an option.
                                    ExercisX      Exercise an option. For use with an L line.
                                    Expire        Mark an option as expired. (Uses D, N, Y & M lines)
                                    Grant         Receive a grant of stock options.
                                    Int           Same as IntInc.
                                    IntX          Same as IntIncX.
                                    IntInc        Interest received.
                                    IntIncX       Interest received. For use with an L line.
                                    K.gewsp       Same as CGShort. (German)
                                    K.gewspX      Same as CGShortX. (German)2307068
                                    Kapgew        Same as CGLong. Kapitalgewinnsteuer.(German)
                                    KapgewX       Same as CGLongX. Kapitalgewinnsteuer. (German)
                                    Kauf          Same as Buy. (German)
                                    KaufX         Same as BuyX. (German)
                                    MargInt       Margin interest paid.
                                    MargIntX      Margin interest paid. For use with an L line.
                                    MiscExp       Miscellaneous expense.
                                    MiscExpX      Miscellaneous expense. For use with an L line.
                                    MiscInc       Miscellaneous income.
                                    MiscIncX      Miscellaneous income. For use with an L line.
                                    ReinvDiv      Reinvested dividend.
                                    ReinvInt      Reinvested interest.
                                    ReinvLG       Reinvested long-term capital gains.
                                    Reinvkur      Same as ReinvLG.
                                    Reinvksp      Same as ReinvSh.
                                    ReinvMd       Reinvested mid-term capital gains.
                                    ReinvSG       Same as ReinvSh.
                                    ReinvSh       Reinvested short-term capital gains.
                                    Reinvzin      Same as ReinvDiv.
                                    Reminder      Reminder. (Uses D, N, C & M lines)
                                    RtrnCap       Return of capital.
                                    RtrnCapX      Return of capital. For use with an L line.
                                    Sell          Sell shares.
                                    SellX         Sell shares. For use with an L line.
                                    ShtSell       Short sale.
                                    ShrsIn        Deposit shares.
                                    ShrsOut       Withdraw shares.
                                    StkSplit      Share split.
                                    Verkauf       Same as Sell. (German)
                                    VerkaufX      Same as SellX. (German)
                                    Vest          Mark options as vested. (Uses N, Y, Q, C & M lines)
                                    WithDrwX      Same as XOut. Used for tax-advantaged accounts.
                                    XIn           Transfer cash from another account.
                                    XOut          Transfer cash to another account.
                                    */
                                    val=val.toLower();
                                    if ( val.contains ( "div" ) && val!="reinvdiv" )
                                    {
                                        //TODO err=currentOperation.setProperty ( "SKG_OP_ORIGINAL_AMOUNT", "" );
                                        div=true;
                                    }
                                    else if ( val.contains ( "sell" ) || val.contains ( "verkauf" ) || val.contains ( "miscexp" ) ) quantityFactor=-1;
                                    //Correction 214851 vvvv
                                    //err=currentOperation.setComment ( val );
                                    //if ( err.isSucceeded() ) err=currentOperation.setMode ( i18nc ( "Noun, the title of an item","Title" ) );
                                    //Correction 214851 ^^^^
                                }
                                else
                                {
                                    //N 	Num (check or reference number)
                                    //Set number
                                    bool ok;
                                    int number=val.toInt ( &ok );
                                    if ( ok && number!=0 )
                                    {
                                        err=currentOperation.setNumber ( number );
                                    }
                                    else
                                    {
                                        err=currentOperation.setMode ( val );
                                    }
                                }
                            }
                            else if ( op=='Q' )
                            {
                                //Q 	Quantity (number of shares or split ratio)
                                //Set value
                                if ( !val.isEmpty() )
                                {
                                    double previousQuantity=currentSubOperation.getQuantity();
                                    if ( previousQuantity )
                                    {
                                        //We have to create a new operation
                                        if ( account!=NULL ) err=account->addOperation ( payement );
                                        else
                                        {
                                            SKGAccountObject defAccount;
                                            err=getDefaultAccount ( defAccount );
                                            if ( err.isSucceeded() ) err=defAccount.addOperation ( payement );
                                        }
                                        if ( err.isSucceeded() ) err=payement.setDate ( currentOperationDate );
                                        if ( err.isSucceeded() )
                                        {
                                            //Create unit if needed
                                            SKGUnitObject unit;
                                            err=getDefaultUnit ( unit, &currentOperationDate );
                                            if ( err.isSucceeded() )  err=payement.setUnit ( unit );
                                        }
                                        if ( err.isSucceeded() ) err=payement.save();  //Save only
                                        if ( err.isSucceeded() ) err=currentOperation.setGroupOperation ( payement );

                                        SKGSubOperationObject subpayement;
                                        if ( err.isSucceeded() ) err=payement.addSubOperation ( subpayement );
                                        if ( err.isSucceeded() ) err=subpayement.setQuantity ( -previousQuantity );
                                        if ( err.isSucceeded() ) err=subpayement.save();
                                    }

                                    if ( err.isSucceeded() ) err=currentSubOperation.setQuantity ( quantityFactor*SKGServices::stringToDouble ( val ) );
                                }
                            }
                            else if ( op=='T' )
                            {
                                //T 	Amount
                                //Set value
                                checkOperationAmount=SKGServices::stringToDouble ( val );
                                err=currentSubOperation.setQuantity ( checkOperationAmount/currentUnitPrice );
                                if ( err.isSucceeded() && investmentAccount )
                                {
                                    err=currentOperation.setProperty ( "SKG_OP_ORIGINAL_AMOUNT", val );
                                }
                            }
                            else if ( op=='$' )
                            {
                                //Dollar amount of split
                                //Set value
                                if ( !investmentAccount )
                                {
                                    double vald=SKGServices::stringToDouble ( val );
                                    checkSuboperationsAmount+=vald;
                                    err=currentSubOperation.setQuantity ( vald );

                                    //save
                                    if ( err.isSucceeded() ) err=currentSubOperation.save();

                                    //Create suboperation
                                    if ( err.isSucceeded() ) err=currentOperation.addSubOperation ( currentSubOperation );

                                    latestSubCatMustBeRemoved=true;
                                }
                            }
                            else if ( op=='P' )
                            {
                                //P Payee
                                //Set Payee
                                //Clean QIF coming from bankperfect
                                val.remove ( "[auto]" );

                                err=SKGPayeeObject::createPayee(document, val, currentPayee);
                                if ( err.isSucceeded() ) err=currentOperation.setPayee ( currentPayee );
                            }
                            else if ( op=='A' )
                            {
                                //A      Address (up to 5 lines; 6th line is an optional message)
                                QString add=currentPayee.getAddress();
                                if (!add.isEmpty()) add+=' ';
                                add+=val;
                                err=currentPayee.setAddress(add);
                                if ( err.isSucceeded() ) err=currentPayee.save();
                            }
                            else if ( op=='M' )
                            {
                                //M 	Memo
                                //Set Memo
                                err=currentOperation.setComment ( val );
                            }
                            else if ( op=='E' )
                            {
                                //E 	Memo in split
                                //Set Memo
                                err=currentSubOperation.setComment ( val );
                            }
                            else if ( op=='S' || op=='L' )
                            {
                                //S 	Category in split (Category/Transfer/Class)//L 	Category (Category/Subcategory/Transfer/Class)
                                //LCategory of transaction
                                //L[Transfer account]
                                //LCategory of transaction/Class of transaction
                                //L[Transfer account]/Class of transaction//Set Category
                                if ( !val.isEmpty() )
                                {
                                    //Correction bug 216520 vvvv
                                    //If the operation is splitter, it is not a transfer
                                    if ( op=='S' ) transferAccount="";
                                    //Correction bug 216520 vvvv

                                    if ( val[0]=='[' )
                                    {
                                        int pos=val.indexOf ( ']' );
                                        if ( pos!=-1 )
                                        {
                                            SKGPayeeObject payeeObj;
                                            currentOperation.getPayee(payeeObj);
                                            bool opening=(payeeObj.getName().compare ( OPENINGBALANCE, Qt::CaseInsensitive ) == 0);

                                            //If the very first Bank transaction in the file has a payee of "Opening Balance", the L line contains the name of the account that the file describes. This is not a transfer
                                            if ( op=='L' && automaticAccount && account && opening)
                                            {
                                                QString accountName=val.mid ( 1, pos-1 );

                                                SKGAccountObject newAccount ( document );
                                                err=newAccount.setName ( accountName );
                                                if ( err.isSucceeded() )
                                                {
                                                    if ( newAccount.exist() )
                                                    {
                                                        //Oups, the real account is existing and it is another one
                                                        err=newAccount.load();

                                                        //We move the operation in the right account
                                                        if ( err.isSucceeded() ) err=currentOperation.setParentAccount ( newAccount );
                                                        if ( err.isSucceeded() ) err=currentOperation.save();

                                                        //We delete the previous account if empty
                                                        if ( err.isSucceeded() )
                                                        {
                                                            if ( account->getNbOperation() ==0 ) err=account->remove();
                                                            delete account;
                                                            account=new SKGAccountObject ( newAccount );
                                                        }
                                                    }
                                                    else
                                                    {
                                                        err=account->setName ( accountName );
                                                        if ( err.isSucceeded() ) err=account->save();
                                                    }
                                                }
                                            }
//                                            if ( op=='L' && currentOperation.getPayee().compare ( "Opening Balance", Qt::CaseInsensitive ) !=0 && !investmentAccount)
                                            if ( op=='L' && !opening )
                                            {
                                                transferAccount=val.mid ( 1, pos-1 );
                                                if ( transferAccount==account->getName() ) transferAccount="";
                                            }
                                            val=val.mid ( pos+2 );
                                        }
                                    }
                                    if ( err.isSucceeded() && !val.isEmpty() )
                                    {
                                        SKGCategoryObject Category;
                                        val.replace ( '/', OBJECTSEPARATOR );
                                        val.replace ( ':', OBJECTSEPARATOR );
                                        val.replace ( ',', OBJECTSEPARATOR );
                                        val.replace ( ';', OBJECTSEPARATOR );
                                        err=SKGCategoryObject::createPathCategory ( document,val, Category );
                                        if ( err.isSucceeded() )  err=currentSubOperation.setCategory ( Category );
                                    }
                                }
                            }
                            else if ( op=='C' )
                            {
                                //C 	Cleared status
                                //Set status
                                err=currentOperation.setStatus ( ( val=="C" || val=="*" ? SKGOperationObject::POINTED : ( val=="R" || val=="X" ? SKGOperationObject::CHECKED : SKGOperationObject::NONE ) ) );
                            }
                            else if ( op=='^' )
                            {
                                //^ 	End of entry
                                //save

                                if ( currentOperationInitialized )
                                {
                                    QByteArray hash = QCryptographicHash::hash ( stringForHash.toUtf8(), QCryptographicHash::Md5 );
                                    SKGObjectBase opWithThisHash;
                                    if ( SKGObjectBase::getObject ( document, "operation", "t_imported IN ('Y','P') AND t_import_id='"+QString ( hash.toHex() ) +'\'', opWithThisHash ).isSucceeded() )
                                    {
                                        err=currentOperation.setStatus ( SKGOperationObject::NONE ); //To be sure that remove is possible
                                        if ( err.isSucceeded() ) err=currentOperation.save();
                                        if ( err.isSucceeded() ) err=currentOperation.remove();
                                        nbOperationsNotImported++;
                                    }
                                    else
                                    {
                                        SKGPayeeObject payeeObj;
                                        currentOperation.getPayee(payeeObj);
                                        bool opening=(payeeObj.getName().compare ( OPENINGBALANCE, Qt::CaseInsensitive ) == 0);
                                        if ( err.isSucceeded() && opening )
                                        {
                                            //Specific values for initial balance
                                            err=currentOperation.setStatus ( SKGOperationObject::CHECKED );
                                            if ( err.isSucceeded() ) err=currentOperation.setAttribute ( "d_date", "0000-00-00" );
                                        }

                                        if ( err.isSucceeded() ) err=currentOperation.setImportID ( hash.toHex() );
                                        if ( err.isSucceeded() ) err=currentOperation.save();
                                        if ( !latestSubCatMustBeRemoved && err.isSucceeded() ) err=currentSubOperation.save();

                                        //Create transfer is needed
                                        if ( err.isSucceeded() && !transferAccount.isEmpty() )
                                        {
                                            //Get origin op
                                            SKGOperationObject opOrigin ( document, currentOperation.getID() );
                                            SKGAccountObject accountOrigin;
                                            if ( err.isSucceeded() ) err=opOrigin.getParentAccount ( accountOrigin );
                                            if ( err.isSucceeded() )
                                            {
                                                //Is the created operation already existing as a transfer?
                                                QString wc="t_import_id='QIF TRANSFER' AND t_ACCOUNT='"+accountOrigin.getName() +
                                                           "' AND  f_QUANTITY="+opOrigin.getAttribute ( "f_QUANTITY" ) +
                                                           " ORDER BY ABS(julianday(d_date) - julianday('2008-01-01')) ASC";
                                                SKGObjectBase::SKGListSKGObjectBase obs;
                                                SKGObjectBase::getObjects ( document, "v_operation_display",wc, obs );
                                                if ( obs.count() )
                                                {
                                                    //We have to merge them and we do not need to create the transfer
                                                    SKGOperationObject firstOne=obs.at ( 0 );
                                                    err=opOrigin.setStatus ( SKGOperationObject::NONE ); //To be sure we can delete it
                                                    if ( err.isSucceeded() ) err=opOrigin.save();
                                                    if ( err.isSucceeded() ) err=firstOne.merge ( opOrigin );
                                                }
                                                else
                                                {
                                                    //Create target account if needed
                                                    SKGAccountObject accountTransfer ( document );
                                                    accountTransfer.setName ( transferAccount );
                                                    if ( !accountTransfer.exist() )
                                                    {
                                                        //The account is created in the same bank by default
                                                        SKGBankObject bankOrigin;
                                                        if ( err.isSucceeded() ) err=accountOrigin.getBank ( bankOrigin );
                                                        if ( err.isSucceeded() ) err=accountTransfer.setBank ( bankOrigin );
                                                        if ( err.isSucceeded() ) err=accountTransfer.save();
                                                    }
                                                    else err=accountTransfer.load();

                                                    //Create operation
                                                    SKGUnitObject unit;
                                                    opOrigin.getUnit ( unit );

                                                    SKGOperationObject opTransfer;
                                                    if ( err.isSucceeded() ) err=accountTransfer.addOperation ( opTransfer );
                                                    if ( err.isSucceeded() ) err=opTransfer.setDate ( opOrigin.getDate() );
                                                    if ( err.isSucceeded() ) err=opTransfer.setComment ( opOrigin.getComment() );
                                                    SKGPayeeObject payeeObj;
                                                    opTransfer.getPayee ( payeeObj );
                                                    if ( err.isSucceeded() ) err=opTransfer.setPayee ( payeeObj );
                                                    if ( err.isSucceeded() ) err=opTransfer.setStatus ( opOrigin.getStatus() );
                                                    if ( err.isSucceeded() ) err=opTransfer.setUnit ( unit );
                                                    if ( err.isSucceeded() ) err=opTransfer.setImportID ( "QIF TRANSFER" ); //To be able to identify it later
                                                    if ( err.isSucceeded() ) err=opTransfer.setGroupOperation ( opOrigin );
                                                    if ( err.isSucceeded() ) err=opTransfer.save();

                                                    SKGSubOperationObject subopTransfer;
                                                    if ( err.isSucceeded() ) err=opTransfer.addSubOperation ( subopTransfer );
                                                    if ( err.isSucceeded() ) err=subopTransfer.setQuantity ( -SKGServices::stringToDouble ( opOrigin.getAttribute ( "f_QUANTITY" ) ) );
                                                    if ( err.isSucceeded() ) err=subopTransfer.save();
                                                }
                                            }
                                        }
                                    }
                                }

                                //Check Sum($)=T for incident 214462
                                QString checkOperationAmountString=SKGServices::doubleToString ( checkOperationAmount );
                                QString checkSuboperationsAmountString=SKGServices::doubleToString ( checkSuboperationsAmount );
                                if ( checkOperationAmount!=0 && checkSuboperationsAmount!=0 && checkOperationAmountString!=checkSuboperationsAmountString )
                                {
                                    err=document->sendMessage ( i18nc ( "An information message",  "Warning: The total amount of the operation (%1) is different to the sum of the sub-operations (%2).",checkOperationAmountString, checkSuboperationsAmountString ) );
                                }

                                //Initialize variables
                                currentOperationInitialized=false;
                                latestSubCatMustBeRemoved=false;
                                currentUnitForInvestment="";
                                quantityFactor=1;
                                currentUnitPrice=1;
                                stringForHash="";
                                checkOperationAmount=0;
                                checkSuboperationsAmount=0;
                                transferAccount="";
                                payement=SKGOperationObject();
                            }
                            else
                            {
                                // A 	Address (up to five lines; the sixth line is an optional message)
                            }
                        }

                        if ( err.isSucceeded() && i%100==0 ) err = SKGServices::executeSqliteOrder ( document, "ANALYZE" );
                        if ( err.isSucceeded() ) err=document->stepForward ( i+1 );
                    }

                    if ( account )
                    {
                        delete account;
                        account=NULL;
                    }
                    if ( err.isSucceeded() ) err=document->endTransaction ( true );
                    else  document->endTransaction ( false );

                    if ( err.isSucceeded() && nbOperationsNotImported )
                        err=document->sendMessage ( i18np ( "one operation not imported because it already exists","%1 operations not imported because they already exists",nbOperationsNotImported ) );

                    //Lines treated
                    if ( err.isSucceeded() ) err=document->stepForward ( 3 );
                }
            }
        }
        if ( err.isSucceeded() ) err=document->endTransaction ( true );
        else  document->endTransaction ( false );
    }

    return err;
}

SKGError SKGImportExportManager::cleanBankImport()
{
    SKGError err;
    SKGTRACEINRC ( 2, "SKGImportExportManager::cleanBankImport", err );

    //Begin transaction
    if ( document )
    {
        err=document->beginTransaction ( "#INTERNAL#", 3 );
        if ( err.isSucceeded() )
        {
            //Step 1 Clean operations without mode and with comment with double space
            SKGObjectBase::SKGListSKGObjectBase operations;
            if ( err.isSucceeded() ) err=SKGObjectBase::getObjects ( document, "operation", "t_imported!='N' and t_mode='' and t_comment like '%  %'", operations );

            int nb=operations.count();
            for ( int i=0; err.isSucceeded() && i<nb ; ++i )
            {
                SKGOperationObject op=operations[i];

                //Comment is like this: <TYPE>  <INFO>
                //Example: RETRAIT DAB             20/01/08 11H44 013330 LCL GAILLAC 000497
                QRegExp rx ( "(.+) {2,}(.+)" );
                QString comment=op.getComment();
                if ( rx.indexIn ( comment ) !=-1 )
                {
                    //Get parameters
                    QString mode =rx.cap ( 1 );
                    QString info =rx.cap ( 2 );

                    //Modify
                    err=op.setComment ( info.trimmed() );
                    if ( err.isSucceeded() )  err=op.setMode ( mode.trimmed() );
                    if ( err.isSucceeded() )  err=op.save ( true, false ); //No reload
                }
            }

            //Step 1 done
            if ( err.isSucceeded() ) err=document->stepForward ( 1 );

            //Step 2 Clean operations without mode and with comment
            if ( err.isSucceeded() ) err=SKGObjectBase::getObjects ( document, "operation", "t_imported!='N' and t_mode='' and t_comment!=''", operations );

            nb=operations.count();
            for ( int i=0; err.isSucceeded() && i<nb ; ++i )
            {
                SKGOperationObject op=operations[i];

                //Comment is like this: <TYPE> <INFO>
                //Example: RETRAIT DAB 14-05-16607-482390
                QRegExp rx ( "(\\S+) +(.+)" );
                QString comment=op.getComment();
                if ( rx.indexIn ( comment ) !=-1 )
                {
                    //Get parameters
                    QString mode =rx.cap ( 1 );
                    QString info =rx.cap ( 2 );

                    //Modify
                    err=op.setComment ( info.trimmed() );
                    if ( err.isSucceeded() )  err=op.setMode ( mode.trimmed() );
                    if ( err.isSucceeded() )  err=op.save ( true, false ); //No reload
                }
            }

            //Step 2 done
            if ( err.isSucceeded() ) err=document->stepForward ( 2 );

            //Step 3 Clean cheque without number
            if ( err.isSucceeded() ) err=SKGObjectBase::getObjects ( document, "operation", "t_imported!='N' and i_number=0 and lower(t_mode)='cheque'", operations );

            nb=operations.count();
            for ( int i=0; err.isSucceeded() && i<nb ; ++i )
            {
                SKGOperationObject op=operations[i];

                //Comment is like this: <TYPE>  <INFO>
                //Example: RETRAIT DAB             20/01/08 11H44 013330 LCL GAILLAC 000497
                QRegExp rx ( "(\\d+)" );
                QString comment=op.getComment();
                if ( rx.indexIn ( comment ) !=-1 )
                {
                    //Get parameters
                    int number =SKGServices::stringToInt ( rx.cap ( 1 ) );

                    //Modify
                    err=op.setNumber ( number );
                    if ( err.isSucceeded() )  err=op.save ( true, false ); //No reload
                }
            }

            //Step 3 done
            if ( err.isSucceeded() ) err=document->stepForward ( 3 );
        }

        if ( err.isSucceeded() ) err=document->endTransaction ( true );
        else  document->endTransaction ( false );
    }

    return err;
}

SKGError SKGImportExportManager::anonymize ()
{
    SKGError err;
    SKGTRACEINRC ( 2, "SKGImportExportManager::anonymize", err );
    if (document)
    {
        if (document->isFileModified())
        {
            err=SKGError(ERR_ABORT, i18nc ( "An information message",  "The document must be saved to be anonymized." ) );
        }
        else
        {
            {
                SKGBEGINTRANSACTION ( *document, "##INTERNAL##", err );
                if ( err.isSucceeded() )
                {
                    QString sql="UPDATE bank SET t_bank_number='', t_name='bank_'||id;;"
                                "UPDATE account SET t_number='', t_agency_number='', t_agency_address='', t_comment='', t_name='account_'||id;;"
                                "UPDATE category SET t_name='category_'||id;;"
                                "UPDATE payee SET t_address='', t_name='payee_'||id;;"
                                "UPDATE refund SET t_comment='', t_name='tracker_'||id;;"
                                "UPDATE operation SET t_comment='';;"
                                "UPDATE suboperation SET t_comment='', f_value=f_value%1234.56;;"
                                ;
                    err=SKGServices::executeSqliteOrder ( document, sql);
                }
            }

            if (err.isSucceeded()) err=document->removeAllTransactions();
        }
    }
    return err;
}

SKGError SKGImportExportManager::findAndGroupTransfers ( int& oNbOperationsMerged )
{
    SKGError err;
    SKGTRACEINRC ( 2, "SKGImportExportManager::findAndGroupTransfers", err );

    oNbOperationsMerged=0;

    //Begin transaction
    if ( document )
    {
        err=document->beginTransaction ( "#INTERNAL#", 3 );
        if ( err.isSucceeded() )
        {
            //Look for operations with
            //  Same units
            //  Same dates
            //  Null i_group_id
            //  Different accounts
            //  Oposite amounts
            SKGStringListList listTmp;
            //+A.i_group_id=0 AND +B.i_group_id=0 is for avoiding to use bad index
            err=SKGServices::executeSelectSqliteOrder ( document,
                    "SELECT A.id, B.id FROM v_operation_tmp1 A, v_operation_tmp1 B WHERE A.id<=B.id AND A.rc_unit_id=B.rc_unit_id AND A.d_date=B.d_date AND A.rd_account_id!=B.rd_account_id AND A.f_QUANTITY=-B.f_QUANTITY AND +A.i_group_id=0 AND +B.i_group_id=0 AND A.f_QUANTITY!=0",
                    listTmp );
            //Step 1 done
            if ( err.isSucceeded() ) err=document->stepForward ( 1 );

            SKGStringListList listTmp2;
            //+A.i_group_id=0 AND +B.i_group_id=0 is for avoiding to use bad index
            if ( err.isSucceeded() ) err=SKGServices::executeSelectSqliteOrder ( document,
                                             "SELECT A.id, B.id FROM v_operation A, v_operation_tmp1 B, parameters P WHERE P.t_uuid_parent=B.id||'-operation' AND A.rc_unit_id!=B.rc_unit_id AND P.t_name='SKG_OP_ORIGINAL_AMOUNT' AND A.d_date=B.d_date AND A.rd_account_id!=B.rd_account_id AND A.f_CURRENTAMOUNT=-P.t_value AND +A.i_group_id=0 AND +B.i_group_id=0 AND A.f_CURRENTAMOUNT!=0",
                                             listTmp2 );

            //Step 2 done
            if ( err.isSucceeded() ) err=document->stepForward ( 2 );
            listTmp2.removeAt ( 0 ); //Remove header
            listTmp+=listTmp2;

            //Group
            {
                oNbOperationsMerged=listTmp.count();
                if ( err.isSucceeded() ) err=document->beginTransaction ( "#INTERNAL#", oNbOperationsMerged-1 );
                for ( int i=1; err.isSucceeded() && i<oNbOperationsMerged ; ++i ) //First line ignored because of it's header
                {
                    SKGOperationObject op1 ( document, SKGServices::stringToInt ( listTmp.at ( i ).at ( 0 ) ) );
                    SKGOperationObject op2 ( document, SKGServices::stringToInt ( listTmp.at ( i ).at ( 1 ) ) );

                    err=op2.setGroupOperation ( op1 );
                    if ( err.isSucceeded() ) err=op2.save ( true, false ); //No reload
                    if ( err.isSucceeded() ) err=document->stepForward ( i );
                }
                if ( err.isSucceeded() ) err=document->endTransaction ( true );
                else  document->endTransaction ( false );
            }
            oNbOperationsMerged= ( oNbOperationsMerged-1 ) *2;

            //Step 3 done
            if ( err.isSucceeded() ) err=document->stepForward ( 3 );

        }
        if ( err.isSucceeded() ) err=document->endTransaction ( true );
        else  document->endTransaction ( false );
    }

    return err;
}

#include "skgimportexportmanager.moc"
