#include <iostream>
#include <boost/bind.hpp>
#include "message.h"
#include "messagebasictypes.h"
#include "tcpmessageclient.h"
using namespace std;


TCPMessageClient::TCPMessageClient(asio::io_service& io_service, const char * host, const char * port):
  ioservice(io_service),
  timer(ioservice),
  resolver(ioservice),
  socket(ioservice),
  m_host(host),
  m_port(port)
{
  startResolver();
}

TCPMessageClient::~TCPMessageClient()
{
}


void TCPMessageClient::startResolver()
{
  asio::ip::tcp::resolver::query query(m_host, m_port);
  stopAfterOneReceivedMessage=false;
  sendQueueCurrentlySending=false;
  resolver.async_resolve(query,
                         boost::bind(&TCPMessageClient::handleResolve, this, asio::placeholders::error,asio::placeholders::iterator));
}


void TCPMessageClient::closeAndScheduleResolve()
{
  socket.close();

  boost::posix_time::time_duration expiry_time=boost::posix_time::seconds(3);
  timer.expires_from_now(expiry_time);
  timer.async_wait(boost::bind(&TCPMessageClient::startResolver, this));
}


void TCPMessageClient::handleResolve(const asio::error_code& err, asio::ip::tcp::resolver::iterator endpoint_iterator)
{
  if (!err)
  {
    // Attempt a connection to the first endpoint in the list. Each endpoint
    // will be tried until we successfully establish a connection.
    asio::ip::tcp::endpoint endpoint = *endpoint_iterator;
    socket.async_connect(endpoint,
                         boost::bind(&TCPMessageClient::handleConnect, this,
                                     asio::placeholders::error, ++endpoint_iterator));
  }
  else
  {
    std::cout << "TCPMessageClient::handleResolve error: " << err.message() << endl;
    closeAndScheduleResolve();
  }
}


void TCPMessageClient::handleConnect(const asio::error_code& err, asio::ip::tcp::resolver::iterator endpoint_iterator)
{
  if (!err)
  {
    // start handleReadMessageSize, handleReadMessage Event loop
    asio::async_read(socket,
                     asio::buffer(data, sizeof(uint32)),
                     asio::transfer_at_least(sizeof(uint32)),
                     boost::bind(&TCPMessageClient::handleReadMessageSize, this, asio::placeholders::error, asio::placeholders::bytes_transferred));

    connectionReadySignal();
  }
  else if (endpoint_iterator != asio::ip::tcp::resolver::iterator())
  {
    // The connection failed. Try the next endpoint in the list.
    socket.close();
    asio::ip::tcp::endpoint endpoint = *endpoint_iterator;
    socket.async_connect(endpoint,
                         boost::bind(&TCPMessageClient::handleConnect, this,
                                     asio::placeholders::error, ++endpoint_iterator));
  }
  else
  {
    std::cout << "TCPMessageClient::handleConnect error: " << err.message() << endl;
    connectionNotReadySignal();
    closeAndScheduleResolve();
  }
}


void TCPMessageClient::queueAndSendMessageSlot(Message & message)
{

  if (sendQueue.size()<maxSendQueueSize)
  {
    if (message.size()<=maxMessageIOSize)
    {
      sendQueue.push_back(message);
      Msg::pushFrontint32(sendQueue.back(),message.size());
    }
  }
  startNewTransmission();
}


void TCPMessageClient::startNewTransmission()
{
  if ((false==sendQueueCurrentlySending) && (sendQueue.size()>0))
  {
    Message & message=sendQueue.front();
    sendQueueCurrentlySending=true;
    asio::async_write(socket,asio::buffer(message.getDataPtr(), message.size()),
                      boost::bind(&TCPMessageClient::handleWriteMessage, this, asio::placeholders::error));
  }
}


void TCPMessageClient::handleWriteMessage(const asio::error_code& err)
{
  // cout << "UDPMessageClient::handleSendTo" << endl;
  // cout << "sendQueue.front().size()=" << sendQueue.front().size() << endl;

  if (!err)
  {
    sendQueue.pop_front();
    sendQueueCurrentlySending=false;
    startNewTransmission();
  }
  else
  {
    cout << "TCPMessageClient::handleWriteMessage error: " << err.message() << endl;
    connectionNotReadySignal();
    closeAndScheduleResolve();
  }
}


void TCPMessageClient::handleReadMessageSize(const asio::error_code& err, size_t length)
{
  if (!err)
  {
    //cout << "handleReadMessageSize length=" << length << endl;
    Message sizeMessage(length,data);
    uint32 messageSizeTmp;
    Msg::popFrontuint32(sizeMessage, messageSizeTmp);
    messageSize=messageSizeTmp;

    asio::async_read(socket,
                     asio::buffer(data, messageSize),
                     asio::transfer_at_least(messageSize),
                     boost::bind(&TCPMessageClient::handleReadMessage, this, asio::placeholders::error, asio::placeholders::bytes_transferred));
  }
  else
  {
    std::cout << "TCPMessageClient::handleReadMessageSize error: " << err << endl;
    connectionNotReadySignal();
    closeAndScheduleResolve();
  }
}


void TCPMessageClient::handleReadMessage(const asio::error_code& err, size_t length)
{
  if (!err)
  {
    // Write all of the data that has been read so far.
    Message returnMessage(length,data);
    receivedMessageSignal(returnMessage);

    if (false==stopAfterOneReceivedMessage)
    {
      // read next size
      asio::async_read(socket,
                       asio::buffer(data, sizeof(uint32)),
                       asio::transfer_at_least(sizeof(uint32)),
                       boost::bind(&TCPMessageClient::handleReadMessageSize, this, asio::placeholders::error, asio::placeholders::bytes_transferred));
    }
  }
  else if (err != asio::error::eof)
  {
    std::cout << "TCPMessageClient::handleReadMessage error: " << err << endl;
    connectionNotReadySignal();
    closeAndScheduleResolve();
  }
}

