/* $Id: dns.c,v 1.6 2003/05/30 00:42:40 cgd Exp $ */

/*
 * Copyright 2001, 2003
 * Broadcom Corporation. All rights reserved.
 *
 * This software is furnished under license and may be used and copied only
 * in accordance with the following terms and conditions.  Subject to these
 * conditions, you may download, copy, install, use, modify and distribute
 * modified or unmodified copies of this software in source and/or binary
 * form. No title or ownership is transferred hereby.
 *
 * 1) Any source code used, modified or distributed must reproduce and
 *    retain this copyright notice and list of conditions as they appear in
 *    the source file.
 *
 * 2) No right is granted to use any trade name, trademark, or logo of
 *    Broadcom Corporation.  The "Broadcom Corporation" name may not be
 *    used to endorse or promote products derived from this software
 *    without the prior written permission of Broadcom Corporation.
 *
 * 3) THIS SOFTWARE IS PROVIDED "AS-IS" AND ANY EXPRESS OR IMPLIED
 *    WARRANTIES, INCLUDING BUT NOT LIMITED TO, ANY IMPLIED WARRANTIES OF
 *    MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE, OR
 *    NON-INFRINGEMENT ARE DISCLAIMED. IN NO EVENT SHALL BROADCOM BE LIABLE
 *    FOR ANY DAMAGES WHATSOEVER, AND IN PARTICULAR, BROADCOM SHALL NOT BE
 *    LIABLE FOR DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 *    CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 *    SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR
 *    BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 *    WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE
 *    OR OTHERWISE), EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include "ip.h"
#include "eth.h"
#include "libc.h"
#include "dns.h"
#include "mbuf.h"
#include "udp.h"
#include "misc.h"
#include "global.h"


#define DNS_CLIENT_PORT       1234
#define DNS_SERVER_PORT         53
#define DNS_FLAGS_RD         0x100
#define DNS_TYPE_A         1
#define DNS_CLASS_INET     1


/* 
 * Data Structures
 */

typedef struct {
	uint8_t trans_id[2];
	uint8_t flags[2];
	uint8_t num_questions[2];
	uint8_t num_answer_rrs[2];
	uint8_t num_auth_rrs[2];
	uint8_t num_extra_rrs[2];
} dns_packet_header_t;

typedef struct {
	uint8_t query_type[2];
	uint8_t query_class[2];
} dns_question_tail_t;

typedef struct {
	uint8_t type[2];
	uint8_t class[2];
	uint8_t ttl[4];
	uint8_t data_length[2];
	uint8_t data[4];
} dns_rr_tail;

typedef struct {
	unsigned int *flag;
	uint32_t ip;
	uint16_t trans_id;
} dns_query_handler_data_t;



/*
 * Static data
 */

static uint16_t dns_tid;


/*
 * Static functions
 */

/*
 * Take a ptr into an mbuf buffer that ostensibly points
 * to the start of a DNS-style name.  Skip the pointer 
 * over the name.  
 * 
 * Return the new pointer position, or NULL if we do something
 * silly like, say, try to go beyond the end of the mbuf
 */

static uint8_t *dns_skip_name(mbuf_t *mbuf, uint8_t *ptr)
{
	if (!mbuf_range_ok(mbuf, ptr, 0)) {
		ptr = 0;
		goto out;
	}
	if (*ptr & 0xc0) {
		if ((*ptr & 0xc0) != 0xc0) {
			ptr = 0;
			goto out;
		}
		/* It's a compressed name.  Just skip
		   it */
		ptr += 2;
	} else {
		/* It's a normal name */
		while (*ptr) {
			ptr += (*ptr) + 1;
			if (!mbuf_range_ok(mbuf, ptr, 0)) {
				ptr = 0;
				goto out;
			}
		}
		ptr++;
	}
 out:
	return ptr;
}


/*
 * Handler for dns replies.
 *  
 * If the transaction ID matches, and everything looks good,
 * pull out the response IP. 
 * 
 * This isn't terribly conservative in terms of sanity checking.  We
 * assume if the server send us a packet with a response, it's
 * for the request we sent.  We don't actually double-check the
 * name to make sure it matches
 */
static int dns_handler(mbuf_t *mbuf, void *data_ptr, uint32_t ignore1, uint32_t ignore2, uint16_t ignore3, uint16_t ignore4)
{
	int retval = 0;
	dns_packet_header_t *header;
	dns_query_handler_data_t *data;
	uint8_t *ptr;
	dns_rr_tail *rr_tail;

	if (mbuf->size < sizeof(dns_packet_header_t)) {
		goto out;
	}
	
	header = (dns_packet_header_t *)mbuf->front_ptr;
	data = (dns_query_handler_data_t *)data_ptr;
	
	if ((get_uint16_field(header->trans_id) != data->trans_id)
	    || (get_uint16_field(header->num_questions) > 1)
	    || (get_uint16_field(header->num_answer_rrs) == 0)) {
		goto out;
	}
	/* Looks like it's for us */
	retval = 1;
	ptr = mbuf->front_ptr + sizeof(dns_packet_header_t);
	ptr = dns_skip_name(mbuf, ptr);
	if (!ptr) {
		goto out;
	}
	ptr += sizeof(dns_question_tail_t);
	if (!mbuf_range_ok(mbuf, ptr, 0)) {
		goto out;
	}
	ptr = dns_skip_name(mbuf, ptr);
	if (!ptr) {
		goto out;
	}
	rr_tail = (dns_rr_tail *)ptr;
	if (!mbuf_range_ok(mbuf, rr_tail, sizeof(dns_rr_tail))) {
		goto out;
	}
	if ((get_uint16_field(rr_tail->type) != DNS_TYPE_A)
	    || (get_uint16_field(rr_tail->class) != DNS_CLASS_INET)
	    || (get_uint16_field(rr_tail->data_length) != 4)) {
		goto out;
	}
	data->ip = get_uint32_field(rr_tail->data);
	(*(data->flag)) = 1;
 out:
	return retval;
}


/*
 * Take a string of the format "some.domain.com" and
 * turn it into dns packet format, e.g. "\004some\006domain\003com\000". 
 * Write this to dest.  return 1 on success, 0 on failure.
 * 
 * Note that the caller needs to make sure that there is room
 * for strlen(src)+2 characters in dest. 
 */
static int dnsify_string(uint8_t *dest, char *str)
{
	uint8_t *write_ptr, *src_ptr;
	int len;
	int count = -1;
	len = lib_strlen(str);
	
	write_ptr = dest + len + 1; 
	src_ptr = str + len ;
	
	while (write_ptr > dest) {
		if (*src_ptr == '.') {
			if (count > 63) {
				return 1;
			}
			*write_ptr = count;
			count = 0;
		} else {
			count++;
			*write_ptr = *src_ptr;
		}
		write_ptr--;
		src_ptr--;
	}
	if (count > 63) {
		return 1;
	}
	*write_ptr = count;
	return 0;
}
	
/*
 * Exported interface.  These are described in the header
 */	
uint32_t dns_resolve_name(char *name)
{
	mbuf_t *mbuf;
	unsigned int tries = 4;
	unsigned int flag = 0;
	dns_query_handler_data_t data;
	dns_packet_header_t *header;
	dns_question_tail_t *tail;
	char realname[128];

	lib_strcpy(realname,name);
	if (lib_strchr(realname,'.') == 0) {
	    if (net_cfg.ip_domain && net_cfg.ip_domain[0]) {
		lib_strcat(realname,".");
		lib_strcat(realname,net_cfg.ip_domain);
		}
	    }

	data.flag = &flag;
	data.ip = 0;

	mbuf = lib_malloc(sizeof(mbuf_t));
	if (!mbuf) {
		lib_die("Out of memory");
	}

	if (net_cfg.ip_ns1) {
		udp_add_handler(dns_handler, &data, net_cfg.ip_ns1, net_cfg.ip_addr, DNS_SERVER_PORT, DNS_CLIENT_PORT);
	}
	if (net_cfg.ip_ns2) {
		udp_add_handler(dns_handler, &data, net_cfg.ip_ns2, net_cfg.ip_addr, DNS_SERVER_PORT, DNS_CLIENT_PORT);
	}


	while (tries--) {
		uint32_t  nameserver;
	       
		if (tries & 1) {
			nameserver = net_cfg.ip_ns1;
		} else {
			nameserver = net_cfg.ip_ns2;
		}

		if (!nameserver) {
			continue;
		}

		mbuf_init(mbuf, 0);
		if (mbuf_append_space(mbuf, sizeof(dns_packet_header_t)) != MBUF_SUCCESS) {
			lib_die("Out of memory");
		}
		header = (dns_packet_header_t *)mbuf->front_ptr;
		
		++dns_tid;
		set_uint16_field(header->trans_id, dns_tid);
		data.trans_id = dns_tid;
		set_uint16_field(header->flags, DNS_FLAGS_RD);
		set_uint16_field(header->num_questions, 1);
		lib_bzero(header->num_answer_rrs, 6);
		if (mbuf_append_space(mbuf, lib_strlen(realname)+2) != MBUF_SUCCESS) {
			lib_die("Out of memory");
		}
		if (dnsify_string(mbuf->front_ptr + sizeof(dns_packet_header_t), realname)) {
			goto cleanup;
		}
		
		tail = (dns_question_tail_t *)mbuf->end_ptr;
		if (mbuf_append_space(mbuf, sizeof(dns_question_tail_t)) != MBUF_SUCCESS) {
			lib_die("Out of memory");
		}
		
		set_uint16_field(tail->query_type, DNS_TYPE_A);
		set_uint16_field(tail->query_class, DNS_CLASS_INET);
		
		udp_send_packet(mbuf, net_cfg.ip_addr, nameserver, IP_FLAGS_NO_FRAGMENT, DNS_CLIENT_PORT, DNS_SERVER_PORT);
		
		eth_recv_loop(500000, &flag);
		if (flag) {
			break;
		}
	}

 cleanup:
	if (net_cfg.ip_ns1) {
		udp_add_handler(dns_handler, &data, net_cfg.ip_ns1, net_cfg.ip_addr, DNS_SERVER_PORT, DNS_CLIENT_PORT);
	}
	if (net_cfg.ip_ns2) {
		udp_add_handler(dns_handler, &data, net_cfg.ip_ns2, net_cfg.ip_addr, DNS_SERVER_PORT, DNS_CLIENT_PORT);
	}
	lib_free(mbuf);
	return data.ip;
	
}
