//  Copyright (c) CNES  2008
//
//  This software is part of CelestLab, a CNES toolbox for Scilab
//
//  This software is governed by the CeCILL  license under French law and
//  abiding by the rules of distribution of free software.  You can  use,
//  modify and/ or redistribute the software under the terms of the CeCILL
//  license as circulated by CEA, CNRS and INRIA at the following URL
//  'http://www.cecill.info'.

function [delta_v,dv1,dv2,anv1,anv2] = CL_man_hohmann(ai,af,mu)
// Delta V for a Hohmann transfer
//
// Calling Sequence
// [delta_v,dv1,dv2,anv1,anv2] = CL_man_hohmann(ai,af [,mu])
//
// Description
// <itemizedlist><listitem>
// <p>Computes the maneuvers of a Hohmann transfer from an initial
// circular orbit with semi major axis <b>ai</b> to a final circular 
// orbit with semi major axis <b>af</b>.</p>
// <p>The output argument <b>delta_v</b> is the sum of the norms 
// of the velocity increments required (|<b>dv1</b>| + |<b>dv2</b>|).</p>
// <p>Velocity increments are expressed in spherical coordinates in the QSW frame: 
// [lambda; phi; dv], where lambda is the in-plane angle (+%pi: towards planet and +%pi/2: 
// ~along velocity), phi is the out-of-plane angle, positive towards the angular momentum vector 
// (the angular momentum vector is perpendicular to the orbit plane and oriented according to 
// right hand rule), dv is the norm of the velocity increment. </p>
// <para/><inlinemediaobject><imageobject><imagedata fileref="hohmann.gif"/></imageobject></inlinemediaobject></listitem>
// </itemizedlist>
//
// Parameters
// ai : Semi-major axis of initial circular orbit. [m] (1xN)
// af : Semi-major axis of final circular orbit. [m] (1xN)
// mu : (optional) Gravitational constant. [m^3/s^2] (default value is %CL_mu)
// delta_v : Total delta-v required = |dv1| + |dv2|. [m/s] (1xN)
// dv1: First velocity increment, in spherical coordinates in the QSW frame [lambda;phi;|dv|] [rad,rad,m/s]. (3xN)
// dv2: Second velocity increment, in spherical coordinates in the QSW frame [lambda;phi;|dv|]. [rad,rad,m/s] (3xN)
// anv1: True anomaly at the location of the first velocity increment (in the initial orbit): as the initial orbit is circular, anv1 is set to 0 arbitrarily. (1xN)
// anv2: True anomaly at the location of the second velocity increment (in the intermediate orbit): %pi if ai < af, 0 otherwise. [rad] (1xN)
//
// Authors
// CNES - DCT/SB
//
// See also
// CL_man_biElliptic
//
// Examples
// // 7200km to 7000km :
// ai = 7200.e3;
// af = 7000.e3;
// [delta_v,dv1,dv2,anv1,anv2] = CL_man_hohmann(ai,af);
// // Check results :
// kep = [ai ; 0 ; %pi/2 ; 0 ; 0 ; anv1];
// kep1 = CL_man_applyDv(kep,dv1);
// kep1(6) = anv2;
// kep2 = CL_man_applyDv(kep1,dv2)

// Declarations:
if(~exists('%CL_mu')) then global %CL_mu; end;

// Code:

Nai = size(ai,2);
Naf = size(af,2);
if (Nai~=Naf) & ~(Nai==1|Naf==1) then CL__error('bad input argument size'); end
N = max(Nai,Naf);

if ~exists('mu','local') then mu=%CL_mu; end

vitp1 = sqrt(mu./ai);  //velocity at periapsis of initial orbit (1)
vitaf = sqrt(mu./af);  //velocity at apoapsis of final orbit (2)
dgatr =(af+ai)./2;  //elliptic transfer orbit semi major axis
vitatr = sqrt(mu.*ai./(dgatr.*af));  //apoapsis velocity
vitptr = sqrt(mu.*af./(dgatr.*ai));  //periapsis velocity

//impulsions in spheric coordinates in qsw frame
dv1 = zeros(3,N);
dv2 = zeros(3,N);

dv1(3,:) = vitptr - vitp1; //modulus of first impulsion
i1 = find(dv1(3,:)>0);
i2 = find(~(dv1(3,:)>0));
dv1(1,i1) = %pi/2;
dv1(1,i2) = %pi/2 + %pi;
dv1(2,:) = 0;
dv1(3,:) = abs(dv1(3,:));

dv2(3,:) = vitaf - vitatr; //modulus of second impulsion
i3 = find(dv2(3,:)>0);
i4 = find(~(dv2(3,:)>0));
dv2(1,i3) = %pi/2;
dv2(1,i4) = %pi/2 + %pi;
dv2(2,:) = 0;
dv2(3,:) = abs(dv2(3,:));

delta_v = dv1(3,:)+dv2(3,:);  //total impulsion needed

anv1 = zeros(ai);  // true anomalies of impulsions
anv2 = zeros(ai);
ii = find(ai < af);
anv2(ii) = %pi;

endfunction
