//  Copyright (c) CNES  2008
//
//  This software is part of CelestLab, a CNES toolbox for Scilab
//
//  This software is governed by the CeCILL  license under French law and
//  abiding by the rules of distribution of free software.  You can  use,
//  modify and/ or redistribute the software under the terms of the CeCILL
//  license as circulated by CEA, CNRS and INRIA at the following URL
//  'http://www.cecill.info'.

function [arg_lat_1,arg_lat_2,inters] = CL_gm_intersectPlanes(inc1,raan1,inc2,raan2)
// Arguments of latitude at the intersection of 2 orbit planes
//
// Calling Sequence
// [arg_lat_1,arg_lat_2,inters] = CL_gm_intersectPlanes(inc1,raan1,inc2,raan2)
//
// Description
// <itemizedlist><listitem>
// This function computes the argument of latitude at the intersection of 2 orbit planes. The orbit planes are defined by inclination (<emphasis role="bold">inc</emphasis>) and right ascension of ascending node (<emphasis role="bold">raan</emphasis>). 
// </listitem>
// <listitem>
// <para>Notes: </para>
// <para> - Only one pair of solutions is computed. The arguments of latitude for the second solution are arg_lat_1+pi and arg_lat_2+pi respectively.</para>
// <para> - If the orbits are circular, then arg_lat_1 and arg_lat_2 define the positions in the orbits where the orbit paths intersect. </para>
// <para> - If the planes are identical, the number of intersections is infinite. The output argument (<emphasis role="bold">inters</emphasis>)is then set to 0, otherwise it is equal to 1.</para>
// </listitem>
// </itemizedlist>
// <para><emphasis role="bold">( Last updated: 2010-06-03 )</emphasis></para>
//
// Parameters
// inc1: Inclination of orbit 1 [rad] (1xN)
// raan1: Right ascension of ascending node of orbit 1 [rad] (1xN)
// inc2: Inclination of orbit 2 [rad] (1xN)
// raan2: Right ascension of ascending node of orbit 2 [rad] (1xN)
// arg_lat_1: Argument of latitude in orbit 1 where the 2 planes intersect [rad] (1xN)
// arg_lat_2: Argument of latitude in orbit 2 where the 2 planes intersect [rad] (1xN)
// inters: Flag indicating if the planes intersect (inters = 1) or is there exists an infinity of intersections (inters=0) (1xN)
//
// Authors
// CNES - DCT/SB
//
// Examples
// inc1 = CL_deg2rad(98);
// raan1 = CL_deg2rad(102);
// inc2 = CL_deg2rad(100);
// raan2 = CL_deg2rad(126);
// [arg_lat_1,arg_lat_2]=CL_gm_intersectPlanes(inc1,raan1,inc2,raan2)
//


// Coherence des entrees

// Declarations:


// Code:

Ninc1 = size(inc1,2);
Ninc2 = size(inc2,2);
Nraan1 = size(raan1,2);
Nraan2 = size(raan2,2);
N = max(Ninc1,Ninc2,Nraan1,Nraan2);
coherence = Ninc1==N & Ninc2==N & Nraan1==N & Nraan2==N;
if ~coherence then CL__error('bad input argument size'); end

inters = ones(1,N);

// Moments cinetiques :
cin1 = [sin(raan1) .* sin(inc1)
    -cos(raan1) .* sin(inc1)
    cos(inc1) ];
cin2 = [sin(raan2) .* sin(inc2)
    -cos(raan2) .* sin(inc2)
    cos(inc2) ];

// vect = produit vectoriel des moments cinetiques :
vect = CL_cross(cin1,cin2);
Nvect = CL_colNorm(vect);

// normalisation et changement de signe :
jj = find(Nvect > %eps);
signeVectZ = sign(vect(3,:));
ii = find(signeVectZ == 0 );
signeVectZ(ii) = 1;

vect(1,jj) = vect(1,jj) .* signeVectZ(jj) ./ Nvect(jj);
vect(2,jj) = vect(2,jj) .* signeVectZ(jj) ./ Nvect(jj);
vect(3,jj) = vect(3,jj) .* signeVectZ(jj) ./ Nvect(jj);

arg_lat_1 = zeros(1,N);
arg_lat_2 = zeros(1,N);
cos_arg_lat_1 = cos(raan1(jj)) .* vect(1,jj) + sin(raan1(jj)) .* vect(2,jj);
cos_arg_lat_2 = cos(raan2(jj)) .* vect(1,jj) + sin(raan2(jj)) .* vect(2,jj);

arg_lat_1(jj) = real(acos( cos_arg_lat_1 ));
arg_lat_2(jj) = real(acos( cos_arg_lat_2 ));

ii = find(Nvect <= %eps);
if(ii~=[])
  inters(ii) = 0;
end


endfunction
