// Copyright (C) 1999-2012
// Smithsonian Astrophysical Observatory, Cambridge, MA, USA
// For conditions of distribution and use, see copyright notice in "copyright"

#include "circle.h"
#include "fitsimage.h"

Circle::Circle(const Circle& a) : BaseEllipse(a) {}

Circle::Circle(Base* p, const Vector& ctr,
	       double r, 
	       const char* clr, int* dsh, 
	       int wth, const char* fnt, const char* txt, 
	       unsigned short prop, const char* cmt,
	       const List<Tag>& tg, const List<CallBack>& cb)
  : BaseEllipse(p, ctr, 0, clr, dsh, wth, fnt, txt, prop, cmt, tg, cb)
{
  numAnnuli_ = 1;
  annuli_ = new Vector[numAnnuli_];
  annuli_[0] = Vector(r,r);

  strcpy(type_, "circle");
  numHandle = 4;

  updateBBox();
}

void Circle::edit(const Vector& v, int h)
{
  Matrix mm = bckMatrix();

  // calc dist between edge of circle and handle
  double d = annuli_[0].length() - annuli_[0][0];
  double r = (v * mm).length() - d;
  annuli_[0] = Vector(r,r);
  
  updateBBox();
  doCallBack(CallBack::EDITCB);
}

void Circle::analysis(AnalysisMethod mm, int which)
{
  switch (mm) {
  case PLOT3D:
    if (!analysisPlot3d_ && which) {
      addCallBack(CallBack::MOVECB, analysisPlot3dCB_[0], 
		  parent->options->cmdName);
      addCallBack(CallBack::EDITCB, analysisPlot3dCB_[0], 
		  parent->options->cmdName);
      addCallBack(CallBack::DELETECB, analysisPlot3dCB_[1], 
		  parent->options->cmdName);
    }
    if (analysisPlot3d_ && !which) {
      deleteCallBack(CallBack::MOVECB, analysisPlot3dCB_[0]);
      deleteCallBack(CallBack::EDITCB, analysisPlot3dCB_[0]);
      deleteCallBack(CallBack::DELETECB, analysisPlot3dCB_[1]);
    }

    analysisPlot3d_ = which;
    break;
  }
}

void Circle::analysisPlot3d(char* xname, char* yname)
{
  double* x;
  double* y;
  Vector ll = -annuli_[0] * Translate(center);
  Vector ur =  annuli_[0] * Translate(center);
  BBox bb(ll,ur) ;
  int num = parent->markerAnalysisPlot3d(&x, &y, this, bb);
  analysisPlot3dResult(xname, yname, x, y, num);
}

// list

void Circle::list(ostream& str, Coord::CoordSystem sys, Coord::SkyFrame sky,
		  Coord::SkyFormat format, int conj, int strip)
{
  FitsImage* ptr = parent->findFits(sys,center);
  listPre(str, sys, sky, ptr, strip, 0);
  
  switch (sys) {
  case Coord::IMAGE:
  case Coord::PHYSICAL:
  case Coord::DETECTOR:
  case Coord::AMPLIFIER:
    {
      Vector v = ptr->mapFromRef(center,sys);
      double r = ptr->mapLenFromRef(annuli_[0][0],sys);
      str << type_ << '(' << setprecision(8) << v[0] << ',' << v[1] << ','
	  << r <<  ')';
    }
    break;
  default:
    if (ptr->hasWCS(sys)) {
      if (ptr->hasWCSCel(sys)) {
	switch (format) {
	case Coord::DEGREES:
	  {
	    Vector v = ptr->mapFromRef(center,sys,sky);
	    double r = ptr->mapLenFromRef(annuli_[0][0],sys,Coord::ARCSEC);
	    str << type_ << '(' << setprecision(8) << v[0] << ',' << v[1] <<','
		<< r << '"' <<  ')';
	  }
	  break;
	case Coord::SEXAGESIMAL:
	  {
	    char buf[64];
	    ptr->mapFromRef(center,sys,sky,format,buf,64);
	    char ra[16];
	    char dec[16];
	    string x(buf);
	    istringstream wcs(x);
	    wcs >> ra >> dec;
	    double r = ptr->mapLenFromRef(annuli_[0][0],sys,Coord::ARCSEC);
	    str << type_ << '(' << ra << ',' << dec << ',' << r << '"' <<  ')';
	  }
	  break;
	}
      }
      else {
	Vector v = ptr->mapFromRef(center,sys);
	double r = ptr->mapLenFromRef(annuli_[0][0],sys);
	str << type_ << '(' << setprecision(8) << v[0] << ',' << v[1] <<','
	    << r <<  ')';
      }
    }
  }

  listPost(str, conj, strip);
}

void Circle::listXML(ostream& str, Coord::CoordSystem sys, Coord::SkyFrame sky, 
		     Coord::SkyFormat format)
{
  FitsImage* ptr = parent->findFits(sys,center);

  XMLRowInit();
  XMLRow(XMLSHAPE,type_);

  XMLRowCenter(ptr,sys,sky,format);
  XMLRowRadiusX(ptr,sys,annuli_[0]);

  XMLRowProps(ptr,sys);
  XMLRowEnd(str);
}

void Circle::listCiao(ostream& str, Coord::CoordSystem sys, int strip)
{
  FitsImage* ptr = parent->findFits();
  listCiaoPre(str);

  switch (sys) {
  case Coord::IMAGE:
  case Coord::PHYSICAL:
  case Coord::DETECTOR:
  case Coord::AMPLIFIER:
    {
      Vector v = ptr->mapFromRef(center,Coord::PHYSICAL);
      double r = ptr->mapLenFromRef(annuli_[0][0],Coord::PHYSICAL);
      str << type_ << '(' << setprecision(8) << v[0] << ',' << v[1] << ',' 
	  << r << ')';
    }
    break;
  default:
    if (ptr->hasWCSCel(sys)) {
      double r = ptr->mapLenFromRef(annuli_[0][0],sys,Coord::ARCMIN);
      char buf[64];
      ptr->mapFromRef(center,sys,Coord::FK5,Coord::SEXAGESIMAL,buf,64);
      char ra[16];
      char dec[16];
      string x(buf);
      istringstream wcs(x);
      wcs >> ra >> dec;
      str << type_ << '(' << ra << ',' << dec << ',' << r << '\'' << ')';
    }
  }

  listCiaoPost(str, strip);
}

void Circle::listSAOtng(ostream& str, Coord::CoordSystem sys, Coord::SkyFrame sky,
		     Coord::SkyFormat format, int strip)
{
  FitsImage* ptr = parent->findFits();
  listSAOtngPre(str, strip);

  switch (sys) {
  case Coord::IMAGE:
  case Coord::PHYSICAL:
  case Coord::DETECTOR:
  case Coord::AMPLIFIER:
    {

      Vector v = ptr->mapFromRef(center,Coord::IMAGE);
      double r = ptr->mapLenFromRef(annuli_[0][0],Coord::IMAGE);
      str << type_ << '(' << setprecision(8) << v[0] << ',' << v[1] << ',' 
	  << r << ')';
    }
    break;
  default:
    if (ptr->hasWCSCel(sys)) {
      switch (format) {
      case Coord::DEGREES:
	{
	  Vector v = ptr->mapFromRef(center,sys,sky);
	  double r = ptr->mapLenFromRef(annuli_[0][0],Coord::IMAGE);
	  str << type_ << '(' << setprecision(8) << v[0] << ',' << v[1] << ','
	      << r << ')';
	}
	break;
      case Coord::SEXAGESIMAL:
	{
	  char buf[64];
	  ptr->mapFromRef(center,sys,sky,format,buf,64);
	  double r = ptr->mapLenFromRef(annuli_[0][0],Coord::IMAGE);
	  char ra[16];
	  char dec[16];
	  string x(buf);
	  istringstream wcs(x);
	  wcs >> ra >> dec;
	  str << type_ << '(' << ra << ',' << dec << ',' << r << ')';
	}
	break;
      }
    }
  }

  listSAOtngPost(str,strip);
}

void Circle::listPros(ostream& str, Coord::CoordSystem sys, Coord::SkyFrame sky,
		      Coord::SkyFormat format, int strip)
{
  FitsImage* ptr = parent->findFits();

  switch (sys) {
  case Coord::IMAGE:
  case Coord::DETECTOR:
  case Coord::AMPLIFIER:
    sys = Coord::IMAGE;
  case Coord::PHYSICAL:
    {
      Vector v = ptr->mapFromRef(center,sys);
      double r = ptr->mapLenFromRef(annuli_[0][0],Coord::IMAGE);
      coord.listProsCoordSystem(str,sys,sky);
      str << "; " << type_ << ' ' << setprecision(8) << v << r;
    }
    break;
  default:
    if (ptr->hasWCSCel(sys)) {
      switch (format) {
      case Coord::DEGREES:
	{
	  Vector v = ptr->mapFromRef(center,sys,sky);
	  double r = ptr->mapLenFromRef(annuli_[0][0],sys,Coord::ARCSEC);
	  coord.listProsCoordSystem(str,sys,sky);
	  str << "; " << type_ << ' ' << setprecision(8) 
	      << v[0] << "d " << v[1] << "d "
	      << r << "\" ";
	}
	break;
      case Coord::SEXAGESIMAL:
	{
	  char buf[64];
	  ptr->mapFromRef(center,sys,sky,format,buf,64);
	  char ra[16];
	  char decc[16];
	  char *dec = decc;
	  string x(buf);
	  istringstream wcs(x);
	  wcs >> ra >> dec;
	  if (dec[0]=='+')
	    dec++;

	  double r = ptr->mapLenFromRef(annuli_[0][0],sys,Coord::ARCSEC);
	  coord.listProsCoordSystem(str,sys,sky);
	  str << "; " << type_ << ' ' << ra << ' ' << dec << ' '
	      << r << "\" ";
	}
	break;
      }
    }
  }

  listProsPost(str, strip);
}

void Circle::listSAOimage(ostream& str, int strip)
{
  FitsImage* ptr = parent->findFits();
  listSAOimagePre(str);

  Vector v = ptr->mapFromRef(center,Coord::IMAGE);
  str << type_ << '(' << setprecision(8) << v[0] << ',' << v[1] << ',' 
      << annuli_[0][0] << ')';

  listSAOimagePost(str, strip);
}

// special composite funtionallity

void Circle::setComposite(const Matrix& mx, double aa)
{
  center *= mx;
  updateBBox();
}

