# Samizdat resource display and focus management
#
#   Copyright (c) 2002-2009  Dmitry Borodaenko <angdraug@debian.org>
#
#   This program is free software.
#   You can distribute/modify this program under the terms of
#   the GNU General Public License version 3 or later.
#
# vim: et sw=2 sts=2 ts=8 tw=0

class ResourceController < Controller
  def initialize(request, id=nil)
    super

    @resource = Resource.new(@request, @id)

    # special case: force untranslated title to go with full rendering of a message
    @title = ('Message' == @resource.type) ?
      escape_title(Message.cached(@id).content.title) :
      @resource.title

    @links.update(@resource.links)
  end

  # render resource
  #
  def index
    @content_for_layout =
      if @request.has_key?('page')
        @resource.messages.to_s
      elsif @request.has_key?('related_page')
        related
      else
        related + box(@title, @resource.page)
      end
  end

  # vote on focus rating
  #
  def vote
    @member.assert_allowed_to('vote')

    focus, focus_id, rating = @request.values_at %w[focus focus_id rating]
    if focus_id = Resource.validate_id(focus_id)
      # manual entry overrides selection
      focus = focus_id
    end
    focus = Focus.new(@request, focus, @id) if focus

    if focus.kind_of?(Focus) and rating and action_confirmed?   # commit vote
      # rating is validated by Focus#rating=
      db.transaction {|db| focus.rating = rating }
      @request.redirect(@id)
    else   # display vote form
      vote_form = secure_form( 'resource/' + @id.to_s + '/vote',
        *focus_fields(focus) +
        [ [:br], [:submit, nil, _('Submit')] ]
      )
      @title = _('Vote') + ': ' + @title
      @content_for_layout = box(_('Vote'), vote_form) + box(@title, @resource.full)
    end
  end

  # RSS feed of related resources
  #
  def rss
    render_rss('resource/' + @id.to_s) do |maker|
      maker.channel.title = maker.channel.description = config['site']['name'] +
        ' / ' + Focus.focus_title(@title)
      maker.channel.link = @request.base + @id.to_s

      related_dataset[0]
    end
  end

  private

  # messages that are related to this focus
  #
  def related_dataset
    RdfDataSet.new(%{
SELECT ?msg
WHERE (rdf::subject ?stmt ?msg)
      (rdf::predicate ?stmt dc::relation)
      (rdf::object ?stmt #{@id})
      (s::inReplyTo ?msg ?parent)
      (dct::isVersionOf ?msg ?current)
      (s::rating ?stmt ?rating)
      (dc::date ?msg ?date)
      (s::hidden ?msg ?hidden)
LITERAL ?rating > 0 AND ?parent IS NULL AND ?current IS NULL #{exclude_hidden}
ORDER BY ?date DESC})
  end

  # related resources
  #
  def related
    page = (@request['related_page'] or 1).to_i
    dataset = related_dataset

    return '' if dataset.empty?

    rss_link = %{resource/#{@id}/rss}
    @feeds = { Focus.focus_title(@title) => rss_link }
    box(
      Focus.focus_title(@title) + page_number(page),
      list(
        dataset[page - 1].collect {|r,| Resource.new(@request, r).short },
        nav_rss(rss_link) + nav(dataset, 'related_page')
      )
    )
  end
end
