/*
 * ipl_tools - Linux for System z reipl and shutdown tools
 *
 * Command: chshut
 *
 * Copyright IBM Corp. 2008, 2010
 * Author(s): Hans-Joachim Picht <hans@linux.vnet.ibm.com>
 *            Michael Holzheu <holzheu@linux.vnet.ibm.com>
 */

#include "ipl_tools.h"

enum shutdown_action {
	SA_IPL,
	SA_REIPL,
	SA_DUMP,
	SA_STOP,
	SA_VMCMD,
};

enum shutdown_trigger {
	ST_HALT,
	ST_POFF,
	ST_REBOOT,
	ST_PANIC,
};

static const char *const usage_chshut =
"Usage: %s TRIGGER ACTION [COMMAND] [OPTIONS]\n"
"\n"
"Change action to be performed after shutdown.\n"
"\n"
"TRIGGER specifies when the action is performed:\n"
"  halt      System has been shut down (e.g. shutdown -h -H now)\n"
"  poff      System has been shut down for power off (e.g. shutdown -h -P now)\n"
"  reboot    System has been shut down for reboot (e.g. shutdown -r)\n"
"  Note: Depending on the distribution, \"halt\" might be mapped to \"poff\".\n"
"\n"
"ACTION specifies the action to be performed:\n"
"  ipl       IPL with previous settings\n"
"  reipl     IPL with re-IPL settings (see chreipl)\n"
"  stop      Stop all CPUs\n"
"  vmcmd     Run z/VM CP command defined by COMMAND\n"
"\n"
"COMMAND defines the z/VM CP command to issue.\n"
"\n"
"OPTIONS:\n"
"  -h, --help        Print this help, then exit\n"
"  -v, --version     Print version information, then exit\n";

static void print_usage_chshut_exit(void)
{
	printf(usage_chshut, g.prog_name);
	exit(0);
}

static void parse_chshut_options(int argc, char *argv[])
{
	int opt, idx;
	const struct option long_opts[] = {
		{ "help",	 no_argument,		NULL, 'h' },
		{ "version",	 no_argument,		NULL, 'v' },
		{ NULL,		 0,			NULL,  0  }
	};
	while ((opt = getopt_long(argc, argv, "hv", long_opts, &idx)) != -1) {
		switch (opt) {
		case 'h':
			print_usage_chshut_exit();
		case 'v':
			print_version_exit();
		default:
			print_help_hint_exit();
		}
	}
	if (!is_root())
		ERR_EXIT("You must be root to perform this operation");
}

static enum shutdown_trigger shutdown_trigger_get(const char *trigger)
{
	if (strcmp(trigger, "halt") == 0)
		return ST_HALT;
	if (strcmp(trigger, "poff") == 0)
		return ST_POFF;
	if (strcmp(trigger, "reboot") == 0)
		return ST_REBOOT;
	if (strcmp(trigger, "panic") == 0)
		return ST_PANIC;
	ERR_EXIT("Unknown shutdown trigger \"%s\" specified", trigger);
}

static enum shutdown_action shutdown_action_get(const char *action)
{
	if (strcmp(action, "ipl") == 0)
		return SA_IPL;
	if (strcmp(action, "reipl") == 0)
		return SA_REIPL;
	if (strcmp(action, "dump") == 0)
		return SA_DUMP;
	if (strcmp(action, "stop") == 0)
		return SA_STOP;
	if (strcmp(action, "vmcmd") == 0)
		return SA_VMCMD;
	ERR_EXIT("Unknown shutdown action \"%s\" specified", action);
}

/*
 * Multiple CP commands can be specified via "vmcmd XY1 vmcmd XY2 ..."
 */
static void vmcmd_set(enum shutdown_trigger st, int argc, char *argv[])
{
	char vmcmd[1024];
	int first = 1, i;
	int vmcmd_length = 0;

	if (is_lpar())
		ERR_EXIT("vmcmd works only under z/VM");
	memset(vmcmd, 0, sizeof(vmcmd));
	for (i = 2; i < argc; i++) {
		if (strcmp(argv[i], "vmcmd") != 0)
			ERR_EXIT("Invalid vmcmd command specification");
		if (i == argc - 1)
			ERR_EXIT("vmcmd needs an additional argument");
		if (!first) {
			strcat(vmcmd, "\n");
			vmcmd_length++;
		} else {
			first = 0;
		}
		vmcmd_length += strlen(argv[i + 1]);
		if (vmcmd_length >= 127)
			ERR_EXIT("The vmcmd command must not exceed 127 "
				 "characters");

		strcat(vmcmd, argv[i + 1]);
		i++;
	}

	switch (st) {
	case ST_HALT:
		write_str(vmcmd, "vmcmd/on_halt");
		break;
	case ST_POFF:
		write_str(vmcmd, "vmcmd/on_poff");
		break;
	case ST_REBOOT:
		write_str(vmcmd, "vmcmd/on_reboot");
		break;
	case ST_PANIC:
		break;
	}
}

void cmd_chshut(int argc, char *argv[])
{
	enum shutdown_trigger st;
	enum shutdown_action sa;

	parse_chshut_options(argc, argv);

	if (argc < 2) {
		ERR("No trigger specified");
		print_help_hint_exit();
	}
	st = shutdown_trigger_get(argv[1]);
	if (st == ST_PANIC)
		ERR_EXIT("Please use \"service dumpconf\" for "
			 "configuring the panic action");
	if (argc < 3) {
		ERR("No action specified");
		print_help_hint_exit();
	}
	sa = shutdown_action_get(argv[2]);

	if (sa == SA_VMCMD) {
		vmcmd_set(st, argc, argv);
	} else if (argc != 3) {
		ERR("Too many parameters specified");
		print_help_hint_exit();
	}

	switch (st) {
	case ST_HALT:
		write_str(argv[2], "shutdown_actions/on_halt");
		break;
	case ST_POFF:
		write_str(argv[2], "shutdown_actions/on_poff");
		break;
	case ST_REBOOT:
		write_str(argv[2], "shutdown_actions/on_reboot");
		break;
	case ST_PANIC:
		break;
	}
}
