# ifndef _SKIT_CSR_ALGO_AMUX_H
# define _SKIT_CSR_ALGO_AMUX_H
///
/// This file is part of Rheolef.
///
/// Copyright (C) 2000-2009 Pierre Saramito <Pierre.Saramito@imag.fr>
///
/// Rheolef is free software; you can redistribute it and/or modify
/// it under the terms of the GNU General Public License as published by
/// the Free Software Foundation; either version 2 of the License, or
/// (at your option) any later version.
///
/// Rheolef is distributed in the hope that it will be useful,
/// but WITHOUT ANY WARRANTY; without even the implied warranty of
/// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
/// GNU General Public License for more details.
///
/// You should have received a copy of the GNU General Public License
/// along with Rheolef; if not, write to the Free Software
/// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
/// 
/// =========================================================================
//
// CSR: Compressed Sparse Row format
//
// algorithm-oriented generic library
// inspired from sparskit2 fortran library
//
// author: Pierre.Saramito@imag.fr
//
// date: 7 march 1997
//
# include "rheolef/config.h"
# include "rheolef/csr.h"
# include "rheolef/vec.h"
namespace rheolef { 

//@!\vfill\listofalgorithms

/*@! 
 \vfill \pagebreak \mbox{} \vfill \begin{algorithm}[h]
  \caption{{\tt amulx}: vec := csr*vec sparse matrix - dense vector product}
  \begin{algorithmic}
    \INPUT {the matrix in CSR format and the dense vector}
      ia(0:nrow), ja(0:nnz-1), a(0:nnz-1),
      x(0:ncol-1)
    \ENDINPUT
    \OUTPUT {the result as dense vector}
      y(0:nrow-1)
    \ENDOUTPUT
    \NOTE {the algorithm is not in-place}
    \ENDNOTE
    \BEGIN 
      \FORTO {i := 0} {nrow-1}
	sum := 0
        \FORTO {p := ia(i)} {ia(i+1)-1}
	    sum := sum + a(p) * x(ja(p))
        \ENDFOR
	y(i) := sum
      \ENDFOR
    \END
 \end{algorithmic} \end{algorithm}
 \vfill \pagebreak \mbox{} \vfill
*/

// y := a*x; sparse matrix * dense vector product
template <
    class InputIterator1,
    class InputIterator2,
    class InputIterator3,
    class RandomAcessIterator,
    class OutputIterator>
void
csr_amux (
    InputIterator1      ia,
    InputIterator1      last_ia,
    InputIterator2      ja,
    InputIterator3      first_a,
    RandomAcessIterator x,
    OutputIterator      y)
{
    typedef typename std::iterator_traits<OutputIterator>::value_type T;
    register InputIterator3 a = first_a + (*ia++);
    while (ia != last_ia) {
	register T sum = 0;
        register InputIterator3 last_a = first_a + (*ia++);
	while (a != last_a)
	    sum += (*a++) * x [*ja++];
	(*y++) = sum;
    }
}
// y += a*x;
template <
    class InputIterator1,
    class InputIterator2,
    class InputIterator3,
    class RandomAcessIterator,
    class OutputIterator>
void
csr_pamux (
    InputIterator1      ia,
    InputIterator1      last_ia,
    InputIterator2      ja,
    InputIterator3      first_a,
    RandomAcessIterator x,
    OutputIterator      y)
{
    typedef typename std::iterator_traits<OutputIterator>::value_type T;
    register InputIterator3 a = first_a + (*ia++);
    while (ia != last_ia) {
	register T sum = 0;
        register InputIterator3 last_a = first_a + (*ia++);
	while (a != last_a)
	    sum += (*a++) * x [*ja++];
	(*y++) += sum;
    }
}
//@!\vfill
}// namespace rheolef
#endif // _SKIT_CSR_ALGO_AMUX_H
