\name{kalcount}
\title{Repeated Measurements Models for Counts with Frailty or Serial
Dependence}
\alias{kalcount}
\usage{
kalcount(response=NULL, times=NULL, origin=0,
	intensity="exponential", depend="independence", update="Markov",
	mu=NULL, shape=NULL, density=FALSE, ccov=NULL, tvcov=NULL, preg=NULL,
	ptvc=NULL, pbirth=NULL, pintercept=NULL, pshape=NULL, pinitial=1,
	pdepend=NULL, pfamily=NULL, envir=parent.frame(), print.level=0,
	ndigit=10, gradtol=0.00001, steptol=0.00001, fscale=1, iterlim=100,
	typsiz=abs(p), stepmax=10*sqrt(p\%*\%p))
}
\description{
\code{kalcount} is designed to handle repeated measurements models
with time-varying covariates. The distributions have two extra
parameters as compared to the functions specified by \code{intensity}
and are generally longer tailed than those distributions. Dependence
among observations on a unit can be through gamma or power variance
family frailties (a type of random effect), with or without
autoregression, or serial dependence over time.

By default, a gamma mixture of the distribution specified in
\code{intensity} is used, as the conditional distribution in the
\code{serial} dependence models, and as a symmetric multivariate
(random effect) model for \code{frailty} dependence.

Unless specified otherwise, the time origin is taken to be zero. The
given times are the ends of the periods in which the counts occurred.

Here, the variance, with exponential intensity, is a quadratic
function of the mean, whereas, for \code{\link[repeated]{nbkal}}, it is
proportional to the mean function.

If the counts on a unit are clustered, not longitudinal, use the
failty dependence with the default exponential intensity, yielding a
multivariate negative binomial distribution.

Nonlinear regression models can be supplied as formulae where
parameters are unknowns in which case factor variables cannot be used and
parameters must be scalars. (See \code{\link[rmutil]{finterp}}.)

Marginal and individual profiles can be plotted using
\code{\link[rmutil]{mprofile}} and \code{\link[rmutil]{iprofile}} and
residuals with \code{\link[rmutil]{plot.residuals}}.
}
\arguments{
\item{response}{A list of two column matrices with counts and
corresponding times for each individual, one matrix or dataframe of
counts, or an object of class, \code{response} (created by
\code{\link[rmutil]{restovec}}) or \code{repeated} (created by
\code{\link[rmutil]{rmna}} or \code{\link[rmutil]{lvna}}). If the
\code{repeated} data object contains more than one response variable,
give that object in \code{envir} and give the name of the response
variable to be used here.}
\item{times}{When response is a matrix, a vector of possibly unequally
spaced times when they are the same for all individuals or a matrix of
times. Not necessary if equally spaced. Ignored if response has class,
\code{response} or \code{repeated}.}
\item{origin}{If the time origin is to be before the start of
observations, a positive constant to be added to all times.}
\item{intensity}{The form of function to be put in the Pareto distribution.
Choices are exponential, Weibull, gamma, log normal, log logistic,
log Cauchy, log Student, and gen(eralized) logistic.}
\item{depend}{Type of dependence. Choices are \code{independence},
\code{frailty}, and \code{serial}.}
\item{update}{Type of for serial dependence. Choices are
\code{Markov}, \code{serial}, \code{event}, \code{cumulated},
\code{count}, and \code{kalman}. With \code{frailty} dependence, weighting by
length of observation time may be specified by setting update to \code{time}.}
\item{mu}{A regression function for the location parameter or a
formula beginning with ~, specifying either a linear regression
function in the Wilkinson and Rogers notation (a log link is assumed)
or a general function with named unknown parameters. Give the initial
estimates in \code{preg} if there are no time-varying covariates and
in \code{ptvc} if there are.}
\item{shape}{A regression function for the shape parameter or a
formula beginning with ~, specifying either a linear regression
function in the Wilkinson and Rogers notation or a general function
with named unknown parameters. It must yield one value per observation.}
\item{density}{If TRUE, the density of the function specified in
\code{intensity} is used instead of the intensity.}
\item{ccov}{A vector or matrix containing time-constant baseline
covariates with one row per individual, a model formula using
vectors of the same size, or an object of class, \code{tccov} (created
by \code{\link[rmutil]{tcctomat}}). If response has class, \code{repeated},
the covariates must be supplied as a Wilkinson and Rogers formula
unless none are to be used or \code{mu} is given.}
\item{tvcov}{A list of matrices with time-varying covariate values,
observed in the time periods in \code{response}, for each individual
(one column per variable), one matrix or dataframe of such covariate
values, or an object of class, \code{tvcov} (created by
\code{\link[rmutil]{tvctomat}}). If response has class, \code{repeated}, the
covariates must be supplied as a Wilkinson and Rogers formula unless
none are to be used or \code{mu} is given.}
\item{preg}{Initial parameter estimates for the regression model:
intercept plus one for each covariate in \code{ccov}. If \code{mu} is
a formula or function, the parameter estimates must be given here only
if there are no time-varying covariates. If \code{mu} is a formula
with unknown parameters, their estimates must be supplied either in
their order of appearance in the expression or in a named list.}
\item{ptvc}{Initial parameter estimates for the coefficients of the
time-varying covariates, as many as in \code{tvcov}. If \code{mu} is a
formula or function, the parameter estimates must be given here if
there are time-varying covariates present.}
\item{pbirth}{If supplied, this is the initial estimate for the
coefficient of the birth model.}
\item{pintercept}{The initial estimate of the intercept for the
generalized logistic intensity.}
\item{pshape}{An initial estimate for the shape parameter of the
intensity function (except exponential intensity). If \code{shape} is
a function or formula, the corresponding initial estimates. If
\code{shape} is a formula with unknown parameters, their estimates
must be supplied either in their order of appearance in the expression
or in a named list.}
\item{pinitial}{An initial estimate for the initial parameter. With
\code{frailty} dependence, this is the frailty parameter.}
\item{pdepend}{An initial estimate for the serial dependence
parameter. For \code{frailty} dependence, if a value is given here, an
autoregression is fitted as well as the frailty.}
\item{pfamily}{An optional initial estimate for the second parameter
of a two-parameter power variance family mixture instead of the
default gamma mixture. This yields a gamma mixture as \code{family ->
0}, an inverse Gauss mixture for \code{family = 0.5}, and a compound
distribution of a Poisson-distributed number of gamma distributions
for \code{-1 < family < 0}.}
\item{envir}{Environment in which model formulae are to be
interpreted or a data object of class, \code{repeated}, \code{tccov},
or \code{tvcov}; the name of the response variable should be given in
\code{response}.
If \code{response} has class \code{repeated}, it is used as the
environment.}
\item{others}{Arguments controlling \code{\link{nlm}}.}
}
\value{
A list of classes \code{kalcount} and \code{recursive} is returned.
}
\author{J.K. Lindsey}
\seealso{
\code{\link[growth]{carma}}, \code{\link[growth]{elliptic}},
\code{\link[rmutil]{finterp}}, \code{\link[repeated]{gar}},
\code{\link[rmutil]{gettvc}}, \code{\link[repeated]{gnlmm}},
\code{\link[gnlm]{gnlr}}, \code{\link[rmutil]{iprofile}},
\code{\link[repeated]{kalseries}}, \code{\link[event]{kalsurv}},
\code{\link[rmutil]{mprofile}}, \code{\link[repeated]{nbkal}},
\code{\link[rmutil]{read.list}}, \code{\link[rmutil]{restovec}},
\code{\link[rmutil]{rmna}}, \code{\link[rmutil]{tcctomat}},
\code{\link[rmutil]{tvctomat}}.
}
\examples{
treat <- c(0,0,1,1)
tr <- tcctomat(treat)
dose <- # matrix(rpois(20,10),ncol=5)
	matrix(c(9,13,16,7,12,6,9,10,11,9,10,10,7,9,9,9,8,10,15,4),
		ncol=5,byrow=TRUE)
dd <- tvctomat(dose)
y <- # matrix(rpois(20,1+3*rep(treat,5)),ncol=5)
	restovec(matrix(c(1,1,1,1,0,1,0,1,0,5,3,3,4,1,4,4,2,3,2,5),
		ncol=5,byrow=TRUE))
reps <- rmna(y, ccov=tr, tvcov=dd)
#
# log normal intensity, independence model
kalcount(y, intensity="log normal", dep="independence", preg=1,
	pshape=0.1)
# random effect
kalcount(y, intensity="log normal", dep="frailty", pinitial=0.1,
	preg=1, psh=0.1)
# serial dependence
kalcount(y, intensity="log normal", dep="serial", pinitial=0.1,
	preg=1, pdep=0.01, psh=0.1)
# random effect and autoregression
kalcount(y, intensity="log normal", dep="frailty", pinitial=0.1,
	pdep=0.1, preg=1, psh=0.1)
# add time-constant variable
kalcount(y, intensity="log normal", pinitial=0.1, psh=0.1,
	preg=c(1,0), ccov=treat)
# or equivalently
kalcount(y, intensity="log normal", mu=~treat, pinitial=0.1,
	psh=0.1, preg=c(1,0))
# or
kalcount(y, intensity="log normal", mu=~b0+b1*treat,
	pinitial=0.1, psh=0.1, preg=c(1,0), envir=reps)
# add time-varying variable
kalcount(y, intensity="log normal", pinitial=0.1, psh=0.1,
	preg=c(1,0), ccov=treat, ptvc=0, tvc=dose)
# or equivalently, from the environment
dosev <- as.vector(t(dose))
kalcount(y, intensity="log normal", mu=~b0+b1*rep(treat,rep(5,4))+b2*dosev,
	pinitial=0.1, psh=0.1, ptvc=c(1,0,0))
# or from the reps data object
kalcount(y, intensity="log normal", mu=~b0+b1*treat+b2*dose,
	pinitial=0.1, psh=0.1, ptvc=c(1,0,0), envir=reps)
# try power variance family
kalcount(y, intensity="log normal", mu=~b0+b1*treat+b2*dose,
	pinitial=0.001, psh=14, ptvc=c(5,-1,0.1), envir=reps,
	pfamily=0.8)
}
\keyword{models}
