#' Summarise and mutate multiple columns.
#'
#' Apply one or more functions to one or more columns. Grouping variables
#' are always excluded from modification.
#'
#' @param tbl a tbl
#' @param funs List of function calls, generated by \code{\link{funs}}, or
#'   a character vector of function names.
#' @param vars,... Variables to include/exclude in mutate/summarise.
#'   You can use same specifications as in \code{\link{select}}. If missing,
#'   defaults to all non-grouping variables.
#'
#'   For standard evaluation versions (ending in \code{_}) these can
#'   be either a list of expressions or a character vector.
#' @examples
#' # One function
#' by_species <- iris %>% group_by(Species)
#' by_species %>% summarise_each(funs(length))
#' by_species %>% summarise_each(funs(mean))
#' by_species %>% summarise_each(funs(mean), Petal.Width)
#' by_species %>% summarise_each(funs(mean), matches("Width"))
#'
#' by_species %>% mutate_each(funs(half = . / 2))
#' by_species %>% mutate_each(funs(min_rank))
#'
#' # Two functions
#' by_species %>% summarise_each(funs(min, max))
#' by_species %>% summarise_each(funs(min, max), Petal.Width, Sepal.Width)
#' by_species %>% summarise_each(funs(min, max), matches("Width"))
#'
#' # Alternative function specification
#' iris %>% summarise_each(funs(ul = length(unique(.))))
#' by_species %>% summarise_each(funs(ul = length(unique(.))))
#'
#' by_species %>% summarise_each(c("min", "max"))
#'
#' # Alternative variable specification
#' summarise_each_(iris, funs(max), names(iris)[-5])
#' summarise_each_(iris, funs(max), list(quote(-Species)))
#' @aliases summarise_each_q mutate_each_q
#' @export
summarise_each <- function(tbl, funs, ...) {
  summarise_each_(tbl, funs, lazyeval::lazy_dots(...))
}

#' @export
#' @rdname summarise_each
summarise_each_ <- function(tbl, funs, vars) {
  vars <- colwise_(tbl, funs_(funs), vars)
  summarise_(tbl, .dots = vars)
}

#' @rdname summarise_each
#' @export
summarize_each <- summarise_each

#' @rdname summarise_each
#' @export
summarize_each_ <- summarise_each_

#' @export
summarise_each_q <- function(...) {
  .Deprecated("summarise_each_")
  summarise_each_(...)
}

#' @export
#' @rdname summarise_each
mutate_each <- function(tbl, funs, ...) {
  mutate_each_(tbl, funs, lazyeval::lazy_dots(...))
}

#' @export
#' @rdname summarise_each
mutate_each_ <- function(tbl, funs, vars) {
  vars <- colwise_(tbl, funs_(funs), vars)
  mutate_(tbl, .dots = vars)
}

#' @export
mutate_each_q <- function(...) {
  .Deprecated("mutate_each_")
  mutate_each_(...)
}


colwise_ <- function(tbl, calls, vars) {
  stopifnot(is.fun_list(calls))

  if (length(vars) == 0) {
    vars <- lazyeval::lazy_dots(everything())
  }
  vars <- select_vars_(tbl_vars(tbl), vars, exclude = as.character(groups(tbl)))

  out <- vector("list", length(vars) * length(calls))
  dim(out) <- c(length(vars), length(calls))

  vars <- enc2native(vars)
  for (i in seq_along(vars)) {
    for (j in seq_along(calls)) {
      out[[i, j]] <- lazyeval::interp(calls[[j]],
        .values = list(. = as.name(vars[i])))
    }
  }
  dim(out) <- NULL

  if (length(calls) == 1) {
    names(out) <- names(vars)
  } else if (length(vars) == 1) {
    names(out) <- names(calls)
  } else {
    grid <- expand.grid(var = names(vars), call = names(calls))
    names(out) <- paste(grid$var, grid$call, sep = "_")
  }

  out
}

