\name{VCF-class}
\docType{class}

% Class
\alias{class:VCF}
\alias{VCF-class}
\alias{CollapsedVCF}
\alias{class:CollapsedVCF}
\alias{CollapsedVCF-class}
\alias{ExpandedVCF}
\alias{class:ExpandedVCF}
\alias{ExpandedVCF-class}

% Constructors:
\alias{VCF}
\alias{updateObject,VCF-method}

% Coercion methods:

% Accessor methods:
\alias{ref}
\alias{ref,VCF-method}
\alias{ref<-}
\alias{ref<-,VCF,DNAStringSet-method}
\alias{alt}
\alias{alt,VCF-method}
\alias{alt<-}
\alias{alt<-,CollapsedVCF,CharacterList-method}
\alias{alt<-,ExpandedVCF,character-method}
\alias{alt<-,CollapsedVCF,DNAStringSetList-method}
\alias{alt<-,ExpandedVCF,DNAStringSet-method}
\alias{qual}
\alias{qual,VCF-method}
\alias{qual<-}
\alias{qual<-,VCF,numeric-method}
\alias{filt}
\alias{filt,VCF-method}
\alias{filt<-}
\alias{filt<-,VCF,character-method}
\alias{fixed}
\alias{fixed,VCF-method}
\alias{fixed<-}
\alias{fixed<-,VCF,DataFrame-method}
\alias{info}
\alias{info,VCF-method}
\alias{info<-}
\alias{info<-,VCF,DataFrame-method}
\alias{rowData,VCF-method}
\alias{rowData<-,VCF-method}
\alias{mcols<-,VCF,DataFrame-method}
\alias{dimnames<-,VCF,list-method}
\alias{geno}
\alias{geno,VCF-method}
\alias{geno,VCF,ANY-method}
\alias{geno,VCF,character-method}
\alias{geno,VCF,numeric-method}
\alias{geno,VCFHeader,ANY-method}
\alias{geno<-}
\alias{geno<-,VCF,character,matrix-method}
\alias{geno<-,VCF,numeric,matrix-method}
\alias{geno<-,VCF,missing,matrix-method}
\alias{geno<-,VCF,missing,SimpleList-method}
\alias{strand,VCF-method}
\alias{strand<-,VCF-method}
\alias{header,VCF-method}
\alias{header<-}
\alias{header<-,VCF,VCFHeader-method}

% Subset methods
\alias{[,VCF-method}
\alias{[,VCF,ANY,ANY-method}
\alias{[<-,VCF,ANY,ANY,VCF-method}

% Combine methods
\alias{cbind,VCF-method}
\alias{rbind,VCF-method}

% Other methods / functions
\alias{genome,VCF-method}
\alias{seqlevels,VCF-method}
\alias{expand,CollapsedVCF-method}
\alias{expand,ExpandedVCF-method}
\alias{restrictToSNV}

% show method:
\alias{show,VCF-method}
\alias{show,CollapsedVCF-method}
\alias{show,ExpandedVCF-method}

\title{VCF class objects}

\description{
  The VCF class is a virtual class extended from \code{SummarizedExperiment}. 
  The subclasses, \code{CompressedVCF} and \code{ExtendedVCF}, are containers 
  for holding data from Variant Call Format files.
}

\section{Constructors}{
  \describe{
    \item{}{
      \code{readVcf(rowData = GRanges(), colData = DataFrame(), 
                exptData = SimpleList(), fixed = DataFrame(), 
                info = DataFrame(), geno = SimpleList(), 
                ..., collapsed=TRUE, verbose = FALSE)}
      Creates a CollapsedVCF class from data in a Variant Call
      Format file on disk. 
    }
    \item{}{
      \code{VCF(rowData = GRanges(), colData = DataFrame(), 
                exptData = SimpleList(header = VCFHeader()), 
                fixed = DataFrame(), info = DataFrame(), 
                geno = SimpleList(), ..., collapsed=TRUE, 
                verbose = FALSE)}
      Creates CollapsedVCF when \code{collapsed = TRUE} and an
      ExpandedVCF when \code{collapsed = FALSE}.
    }
    \item{}{
      \code{expand(rowData = GRanges(), colData = DataFrame(), 
                exptData = SimpleList(), fixed = DataFrame(), 
                info = DataFrame(), geno = SimpleList(), 
                ..., collapsed=TRUE, verbose = FALSE)}
      Creates an ExpandedVCF from a CollapsedVCF.
    }
  }
}

\section{Accessors}{
  In the following code snippets \code{x} is a CollapsedVCF or ExpandedVCF 
  object.
  \describe{
    \item{}{
      \code{rowData(x)}, \code{rowData(x) <- value}:
      Gets or sets a \code{GRanges} constructed from the CHROM, POS and ID 
      fields of the VCF file with REF, ALT, QUAL and FILT as metadata columns. 
      The ID's serve as the \code{rownames}; if they are NULL, \code{rownames} 
      are constructed from  CHROM:POS. \code{value} must be a \code{GRanges} 
      with metadata columns.

      The \code{paramRangeID} metadata column groups ranges as
      specified with \code{ScanVcfParam}. If \code{ScanVcfParam} was
      not used with \code{readVcf} these values will be \code{NA}. 

      The metadata columns can be accessed with \code{mcols(x)$<variable>} or 
      the following accessors:
      \itemize{
        \item{\code{ref(x)}, \code{ref(x) <- value}:
          Gets or sets the reference allele (REF). \code{value} must 
          be a \code{DNAStringSet}. 
        }
        \item{\code{alt(x)}, \code{alt(x) <- value}:
          Gets or sets the alternate allele data (ALT). When \code{x} is 
          a CollapsedVCF, \code{value} must be a \code{DNAStringSetList}
          or \code{CompressedCharacterList}. For ExtendedVCF, \code{value}
          must be a \code{DNAStringSet} or \code{character}.
        }
        \item{\code{qual(x)}, \code{qual(x) <- value}:
          Returns or sets the quality scores (QUAL). \code{value} must 
          be an \code{numeric(1L)}.
        }
        \item{\code{filt(x)}, \code{filt(x) <- value}:
          Returns or sets the filter data (FILT). \code{value} must 
          be a \code{character(1L)}.
        }
      }
    }
    \item{}{
      \code{info(x)}, \code{info(x) <- value}:
       Gets or sets a DataFrame of INFO variables. 
    }
    \item{}{
      \code{geno(x)}, \code{geno(x) <- value}:
       Gets a SimpleList of genotype data.
       \code{value} is a SimpleList. To replace a single variable in
       the SimpleList use \code{geno(x)$variable <- value}; in this 
       case \code{value} must be a matrix or array. By default
       row names are returned; to override specify
       \code{geno(vcf, withDimnames=FALSE)}.
    }
    \item{}{
      \code{exptData(x)}:
       Gets a SimpleList of experiment-related data. By default this
       list includes the \sQuote{header} information from the VCF file. 
       See the use of \code{header()} for details in extracting
       header information. 
    }
    \item{}{
      \code{colData(x)}, \code{colData(x) <- value}:
      Gets or sets a \code{DataFrame} of sample-specific information. Each row 
      represents a sample in the VCF file. \code{value} must be a 
      \code{DataFrame} with rownames representing the samples in the VCF 
      file.
    }
    \item{}{
      \code{genome(x)}:
      Extract the \code{genome} information from the \code{GRanges} in 
      the \code{rowData} slot.
    }
    \item{}{
      \code{seqlevels(x)}:
      Extract the \code{seqlevels} from the \code{GRanges} in the 
      \code{rowData} slot.
    }
    \item{}{
      \code{strand(x)}:
      Extract the \code{strand} from the \code{GRanges} in the 
      \code{rowData} slot.
    }
    \item{}{
      \code{header(x)}, \code{header(x)<- value}:
      Get or set the VCF header information. Replacement value
      must be a \code{VCFHeader} object. To modify individual elements 
      use \code{info<-}, \code{geno<-} or \code{meta<-} on a 
      \sQuote{VCFHeader} object. See ?\code{VCFHeader} man page for
      details.
       \itemize{
         \item{\code{info(header(x))}}
         \item{\code{geno(header(x))}}
         \item{\code{meta(header(x))}}
         \item{\code{samples(header(x))}}
      }
    }
  }
}

\section{Subsetting and combining}{
  In the following code \code{x} is a VCF object, and \dots is a list
  of VCF objects.
  \describe{
    \item{}{
      \code{x[i, j]}, \code{x[i, j] <- value}:
      Gets or sets rows and columns. \code{i} and \code{j} can be integer
      or logical vectors. \code{value} is a replacement \code{VCF} object.
    }
    \item{}{
      \code{cbind(...)}, \code{rbind(...)}:
      \code{cbind} combines objects with identical ranges (\code{rowData}) 
      but different samples (columns in \code{assays}). The colnames in
      \code{colData} must match or an error is thrown. Columns with
      duplicate names in \code{fixed}, \code{info} and \code{mcols(rowData(VCF))} 
      must contain the same data.

      \code{rbind} combines objects with different ranges (\code{rowData}) 
      and the same subjects (columns in \code{assays}). Columns with
      duplicate names in \code{colData} must contain the same data. 
      The \sQuote{Samples} columns in \code{colData} (created by \code{readVcf}) 
      are renamed with a numeric extension ordered as they were input to \code{rbind}
      e.g., \dQuote{Samples.1, Samples.2, ...} etc. 

      \code{exptData} from all objects are combined into a
      \code{SimpleList} with no name checking.

      ## TODO: header information not yet combined
    }
  }
}

\section{expand}{
  In the following code snippets \code{x} is a CollapsedVCF object.
  \describe{
    \item{}{
      \code{expand(x)}:
      Expand (unlist) the ALT column of a CollapsedVCF object to one row 
      per ALT value. Variables with Number='A' have one value per alternate
      allele and are expanded accordingly. The 'AD' genotype field
      is expanded into REF/ALT pairs. For all other fields, the rows
      are replicated to match the elementLengths of ALT.

      The output is an ExpandedVCF object with ALT as a \code{DNAStringSet} 
      or \code{character} (structural variants). 
    }
  }
}

\section{Variant Type}{
  Functions to identify variant type include \link{isSNV}, 
  \link{isInsertion}, \link{isDeletion}, \link{isIndel}, 
  \link{isSubstitution} and \link{isTransition}. See the ?\code{isSNV} 
  man page for details.
}

\section{Arguments}{
  \describe{
    \item{geno}{A \code{list} or \code{SimpleList} of matrix elements,
      or a \code{matrix} containing the genotype information from a
      VCF file. If present, these data immediately follow the FORMAT
      field in the VCF. 

      Each element of the list must have the same dimensions, and dimension 
      names (if present) must be consistent across elements and with the row 
      names of \code{rowData}, \code{colData}. 
    }
    \item{info}{A \code{DataFrame} of data from the INFO field of a VCF file. 
      The number of rows must match that in the \code{rowData} object.
    }
    \item{fixed}{A \code{DataFrame} of REF, ALT, QUAL and FILTER 
      fields from a VCF file. The number of rows must match that
      of the \code{rowData} object.
    }
    \item{rowData}{A \code{GRanges} instance describing the ranges of interest. 
      Row names, if present, become the row names of the \code{VCF}. The length 
      of the \code{GRanges} must equal the number of rows of the matrices in 
      \code{geno}.
    }
    \item{colData}{A \code{DataFrame} describing the samples. Row names, if 
      present, become the column names of the \code{VCF}.
    }
    \item{exptData}{A \code{SimpleList} describing the header of the VCF file or 
      additional information for the overall experiment. 
    }
    \item{...}{For \code{cbind} and \code{rbind} a list of VCF objects.
      For all other methods \dots are additional arguments passed to methods. 
    }
    \item{collapsed}{A \code{logical(1)} indicating whether a 
      CollapsedVCF or ExpandedVCF should be created. The ALT in a
      CollapsedVCF is a \code{DNAStringSetList} while in a
      ExpandedVCF it is a \code{DNAStringSet}. 
    }
    \item{verbose}{A \code{logical(1)} indicating whether messages
      about data coercion during construction should be printed.
    }
  }
}

\details{
  The \code{VCF} class is a virtual class with two concrete subclasses,
  \code{CollapsedVCF} and \code{ExtendedVCF}. 

  Slots unique to \code{VCF} and subclasses,
  \itemize{
    \item \code{fixed}: A \link{DataFrame} containing information from the 
           REF, ALT, QUAL and FILTER fields from a VCF file.
    \item \code{info}: A \link{DataFrame} containing information from the INFO 
           fields from a VCF file.
  }

  Slots inherited from \code{SummarizedExperiment},
  \itemize{
    \item \code{exptData}: A \link{SimpleList}-class instance containing the 
          file header or other information about the overall experiment.
    \item \code{rowData}: A \link{GRanges}-class instance defining the
           variant ranges and associated metadata columns of REF, ALT, QUAL 
           and FILTER.
    \item \code{colData}: A \link{DataFrame}-class instance describing the 
          samples and associated metadata.
    \item \code{geno}: The \code{assays} slot from \code{SummarizedExperiment}
          has been renamed as \code{geno} for the VCF class. This slot contains
          the genotype information immediately following the FORMAT field in a 
          VCF file. Each element of the \code{list} or \code{SimpleList} is a 
          matrix or array. 
  }

  It is expected that users will not create instances of the VCF class
  but instead one of the concrete subclasses, CollapsedVCF or ExpandVCF.
  CollapsedVCF contains the ALT data as a \code{DNAStringSetList} allowing 
  for multiple alleles per variant. The ExpandedVCF ALT data is a 
  \code{DNAStringSet} where the ALT column has been expanded to create a 
  flat form of the data with one row per variant-allele combination. In 
  the case of strucutral variants, ALT will be a \code{CompressedCharacterList}
  or \code{character} in the collapsed or expanded forms.
}

\author{Valerie Obenchain <vobencha@fhcrc.org>}

\seealso{
  \link{GRanges},
  \link[IRanges]{DataFrame},
  \link[IRanges]{SimpleList},
  \link[GenomicRanges]{SummarizedExperiment},
  \code{\link{readVcf}},
  \code{\link{writeVcf}}
}

\examples{
fl <- system.file("extdata", "structural.vcf", package="VariantAnnotation")
vcf <- readVcf(fl, genome="hg19")

## ----------------------------------------------------------------
## Accessors 
## ----------------------------------------------------------------
## Variant locations are stored in a GRanges in the rowData slot.
rowData(vcf)

## Individual fields can be extracted with ref(), alt(), qual() 
## and filt().
qual(vcf)
ref(vcf)

## All 'info' fields can be extracted together along
## with the GRanges of locations.
head(info(vcf))

## All genotype fields can be seen with geno(). Individual
## fields are accessed with '$' or '[['.
geno(vcf)
identical(geno(vcf)$GQ, geno(vcf)[[2]])

## ----------------------------------------------------------------
## Renaming seqlevels and subsetting 
## ----------------------------------------------------------------
## Overlap and matching operations require that the objects
## being compared have the same seqlevels (chromosome names).
## It is often the case that the seqlevesls in on of the objects
## needs to be modified to match the other. In this VCF, the 
## seqlevels are numbers instead of preceded by "chr" or "ch". 

seqlevels(vcf)

## Rename the seqlevels to start with 'chr'.
vcf2 <- renameSeqlevels(vcf, paste0("chr", seqlevels(vcf))) 
seqlevels(vcf2)

## The VCF can also be subset by seqlevel using 'keepSeqlevels'
## or 'dropSeqlevels'. See ?keepSeqlevels for details. 
vcf3 <- keepSeqlevels(vcf2, "chr2")
seqlevels(vcf3)

## ----------------------------------------------------------------
## Header information 
## ----------------------------------------------------------------

## Header data can be modified in the 'meta', 'info' and 'geno'
## slots of the VCFHeader object. See ?VCFHeader for details.

## Current 'info' fields.
rownames(info(header(vcf)))

## Add a new field to the header.
newInfo <- DataFrame(Number=1, Type="Integer",
                     Description="Number of Samples With Data",
                     row.names="NS")
info(header(vcf)) <- rbind(info(header(vcf)), newInfo)
rownames(info(header(vcf)))

## ----------------------------------------------------------------
## CollapsedVCF 
## ----------------------------------------------------------------
## readVCF() produces a CollapsedVCF object.
fl <- system.file("extdata", "ex2.vcf", 
                  package="VariantAnnotation")
vcf <- readVcf(fl, genome="hg19")
vcf

## The ALT column is a DNAStringSetList to allow for more
## than one alternate allele per variant.
alt(vcf)

## For structural variants ALT is a CharacterList.
fl <- system.file("extdata", "structural.vcf", 
                  package="VariantAnnotation")
vcf <- readVcf(fl, genome="hg19")
alt(vcf)

## ----------------------------------------------------------------
## ExpandedVCF
## ----------------------------------------------------------------
## ExpandedVCF is the 'flattened' counterpart of CollapsedVCF.
## The ALT and all variables with Number='A' in the header are
## expanded to one row per alternate allele.
fl <- system.file("extdata", "ex2.vcf", package="VariantAnnotation")
vcf <- readVcf(fl, genome="hg19")
vcfLong <- expand(vcf)
alt(vcfLong)

## ----------------------------------------------------------------
## restrictToSNV() 
## ----------------------------------------------------------------
## restrictToSNV() returns a subset VCF containing SNVs only.
## The VCF can be a CollapsedVCF or ExpandedVCF. 'alt(x)' must
## contain nucleotides (i.e., no structural variants).

vcf1 <- VCF(rowData = GRanges("chr1", IRanges(1:4*3, width=c(1, 2, 1, 1))))
alt(vcf1) <- DNAStringSetList("A", c("TT"), c("G", "A"), c("TT", "C"))
ref(vcf1) <- DNAStringSet(c("G", c("AA"), "T", "G"))

## SNVs are present in rows 1 (single ALT value), 3 (both ALT values) 
## and 4 (1 of the 2 ALT values).
fixed(vcf1)[c("REF", "ALT")]

## CollapsedVcf
res1 <- restrictToSNV(vcf1)
rowData(res1)[ ,1:2]

## ExpandedVcf
vcf2 <- expand(vcf1)
res2 <- restrictToSNV(vcf2)
rowData(res2)[ ,1:2]
}
