// Copyright (c) 2007, Google Inc.
// All rights reserved.
//
// Author: Li Liu
//
// Redistribution and use in source and binary forms, with or without
// modification, are permitted provided that the following conditions are
// met:
//
//     * Redistributions of source code must retain the above copyright
// notice, this list of conditions and the following disclaimer.
//     * Redistributions in binary form must reproduce the above
// copyright notice, this list of conditions and the following disclaimer
// in the documentation and/or other materials provided with the
// distribution.
//     * Neither the name of Google Inc. nor the names of its
// contributors may be used to endorse or promote products derived from
// this software without specific prior written permission.
//
// THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
// "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
// LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
// A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
// OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
// SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
// LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
// DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
// THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
// (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
// OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

#ifndef CLIENT_LINUX_HANDLER_SIGCONTEXT_H__
#define CLIENT_LINUX_HANDLER_SIGCONTEXT_H__

#include <stdint.h>
#include <sys/ucontext.h>

#include "google_breakpad/common/minidump_format.h"
#include "client/linux/handler/cpu_context.h"

// Forward declaration of signal context.
struct sigcontext;

namespace google_breakpad {

//
// SignalContext
//
// Wrapper class around the raw signal context which depends on CPU type (i386
// and x86_64).
// It provides CPU independent way of accessing common fields in the signal
// context.
class SignalContext {
  public:
   SignalContext();

   // Set the raw signal context.
   void SetSignalContext(const struct sigcontext *sig_ctx);

   // Set raw signal context from ucontext_t.
   void SetUContext(const ucontext_t *ucontext);

   // Get the raw signal context.
   const struct sigcontext *GetRawContext() const;

   // Get the current instruction address.
   uintptr_t GetIP() const;

   // Get the frame pointer.
   uintptr_t GetFramePointer() const;

   // Get the stack pointer.
   uintptr_t GetStackPointer() const;

   // Check if current signal context is empty or not.
   bool IsEmpty() const;

   // Fill the context section of minidump file.
   bool CopyTo(RawContext *context) const;

  private:
   const struct sigcontext *sig_ctx_;
};

}  // namespace google_breakpad

#endif  // CLIENT_LINUX_HANDLER_SIGCONTEXT_H__
