/***************************************************************************
                          qgsnetworkaccessmanager.h  -  description
                             -------------------
    begin                : 2010-05-08
    copyright            : (C) 2010 by Juergen E. Fischer
    email                : jef at norbit dot de
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#ifndef QGSNETWORKACCESSMANAGER_H
#define QGSNETWORKACCESSMANAGER_H

#include <QList>
#include <QStringList>
#include <QNetworkAccessManager>
#include <QNetworkProxy>
#include <QNetworkRequest>

/*
 * \class QgsNetworkAccessManager
 * \brief network access manager for QGIS
 * \ingroup core
 * \since 1.5
 *
 * This class implements the QGIS network access manager.  It's a singleton
 * that can be used across QGIS.
 *
 * Plugins can insert proxy factories and thereby redirect requests to
 * individual proxies.
 *
 * If no proxy factories are there or none returns a proxy for an URL a
 * fallback proxy can be set.  There's also a exclude list that defines URLs
 * that the fallback proxy should not be used for, then no proxy will be used.
 *
 */
class CORE_EXPORT QgsNetworkAccessManager : public QNetworkAccessManager
{
    Q_OBJECT

  public:
    //! returns a pointer to the single instance
    // and creates that instance on the first call.
    static QgsNetworkAccessManager* instance();

    QgsNetworkAccessManager( QObject *parent = nullptr );

    //! destructor
    ~QgsNetworkAccessManager();

    //! insert a factory into the proxy factories list
    void insertProxyFactory( QNetworkProxyFactory *factory );

    //! remove a factory from the proxy factories list
    void removeProxyFactory( QNetworkProxyFactory *factory );

    //! retrieve proxy factory list
    const QList<QNetworkProxyFactory *> proxyFactories() const;

    //! retrieve fall back proxy (for urls that no factory returned proxies for)
    const QNetworkProxy &fallbackProxy() const;

    //! retrieve exclude list (urls shouldn't use the fallback proxy)
    const QStringList &excludeList() const;

    //! set fallback proxy and URL that shouldn't use it.
    void setFallbackProxyAndExcludes( const QNetworkProxy &proxy, const QStringList &excludes );

    //! Get name for QNetworkRequest::CacheLoadControl
    static QString cacheLoadControlName( QNetworkRequest::CacheLoadControl theControl );

    //! Get QNetworkRequest::CacheLoadControl from name
    static QNetworkRequest::CacheLoadControl cacheLoadControlFromName( const QString &theName );

    //! Setup the NAM according to the user's settings
    void setupDefaultProxyAndCache();

    //! return whether the system proxy should be used
    bool useSystemProxy() const { return mUseSystemProxy; }

  public slots:
    /** Send GET request, calls get().
     * Emits requestSent().
     * @param request request to be sent
     * @deprecated use get() directly
     */
    Q_DECL_DEPRECATED void sendGet( const QNetworkRequest & request );
    /** Abort and delete reply.
     * @param reply reply to be aborted.
     * @deprecated use abort() and deleteLayer() on the reply directly
     */
    Q_DECL_DEPRECATED void deleteReply( QNetworkReply * reply );

  signals:
    void requestAboutToBeCreated( QNetworkAccessManager::Operation, const QNetworkRequest &, QIODevice * );
    void requestCreated( QNetworkReply * );
    void requestTimedOut( QNetworkReply * );
    /** Emitted when request was sent by request()
     * @param reply request reply
     * @param sender the object which called request() slot.
     * @deprecated only emitted from deprecated sendGet
     */
    void requestSent( QNetworkReply * reply, QObject *sender );

  private slots:
    void abortRequest();

  protected:
    virtual QNetworkReply *createRequest( QNetworkAccessManager::Operation op, const QNetworkRequest &req, QIODevice *outgoingData = nullptr ) override;

  private:
    QList<QNetworkProxyFactory*> mProxyFactories;
    QNetworkProxy mFallbackProxy;
    QStringList mExcludedURLs;
    bool mUseSystemProxy;
    bool mInitialized;
    static QgsNetworkAccessManager *smMainNAM;
};

#endif // QGSNETWORKACCESSMANAGER_H

