/*
  Plee the Bear

  Copyright (C) 2005-2009 Julien Jorge, Sebastien Angibaud

  This program is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License as published by the
  Free Software Foundation; either version 2 of the License, or (at your
  option) any later version.

  This program is distributed in the hope that it will be useful, but WITHOUT
  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
  more details.

  You should have received a copy of the GNU General Public License along
  with this program; if not, write to the Free Software Foundation, Inc.,
  51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

  contact: plee-the-bear@gamned.org

  Please add the tag [PTB] in the subject of your mails.
*/
/**
 * \file honeypot.cpp
 * \brief Implementation of the ptb::honeypot class.
 * \author Sebastien Angibaud
 */
#include "ptb/item/honeypot.hpp"

#include "universe/derived_item_handle.hpp"
#include "ptb/item/plee/plee.hpp"
#include "ptb/level_variables.hpp"
#include "generic_items/decorative_item.hpp"

BASE_ITEM_EXPORT( honeypot, ptb )

/*----------------------------------------------------------------------------*/
/**
 * \brief Contructor.
 */
ptb::honeypot::honeypot()
  : m_given(false)
{
  set_phantom(true);
  set_can_move_items(false);
} // honeypot::honeypot()

/*----------------------------------------------------------------------------*/
/**
 * \brief Load the media required by this class.
 */
void ptb::honeypot::pre_cache()
{
  get_level_globals().load_animation("animation/shiny-honey-jar.canim");
} // honeypot::pre_cache()

/*----------------------------------------------------------------------------*/
/**
 * \brief Initialize the item.
 */
void ptb::honeypot::build()
{
  super::build();

  set_animation
    ( get_level_globals().get_animation("animation/shiny-honey-jar.canim") );

  set_size( 64, 64 );
} // honeypot::build()

/*----------------------------------------------------------------------------*/
/**
 * \brief Check if the collision is with a player.
 * \param that The other item of the collision.
 * \param info Some informations about the collision.
 */
void ptb::honeypot::collision_check_and_apply
( bear::engine::base_item& that, bear::universe::collision_info& info )
{
  plee* p = dynamic_cast<plee*>(&that);
  
  if ( p != NULL )
    {
      if ( ! m_given )
        {
          create_decoration();
          m_given = true;
          level_variables::set_honeypots_found
            ( get_level(), 
              level_variables::get_honeypots_found(get_level()) + 1 );
          
          if ( level_variables::get_honeypots_found(get_level()) % 5 == 0 )
            {
              give_life(1);
              give_life(2);
            }
          
          kill();
        }
      else
        default_collision(info);
    }
  else
    default_collision(info);
} // honeypot::collision_check_and_apply()

/*----------------------------------------------------------------------------*/
/**
 * \brief Call collision_check_and_apply().
 * \param that The other item of the collision.
 * \param info Some informations about the collision.
 */
void ptb::honeypot::collision
( bear::engine::base_item& that, bear::universe::collision_info& info )
{
  collision_check_and_apply(that, info);
} // honeypot::collision()
/*----------------------------------------------------------------------------*/
/**
 * \brief Give a life at each player.
 * \param index The index of the player that receives a life.
 */
void ptb::honeypot::give_life(unsigned int index) const
{
  bear::universe::derived_item_handle_maker<plee>::handle_type player;
  plee::get_instance_message msg;
  
  get_level_globals().send_message(plee::player_name(index),msg);
  player = msg.get_instance();

  if ( player.get() != NULL ) 
    {
      game_variables::set_lives_count
        ( index, 
          game_variables::get_lives_count(index) + 1);
    }  
} // honeypot::give_life()

/*----------------------------------------------------------------------------*/
/**
 * \brief Create a decoration.
 */
void ptb::honeypot::create_decoration()
{
  bear::decorative_item* item = new bear::decorative_item;
  
  item->set_animation
    (get_level_globals().get_animation
     ("animation/corrupting-bonus-disapearing.canim"));
  item->get_rendering_attributes().set_angle(get_visual_angle());
  item->set_kill_when_finished(true);
  item->set_z_position(get_z_position()-1);
  
  new_item( *item );
  
  item->set_center_of_mass(get_center_of_mass());
  
  CLAW_ASSERT( item->is_valid(),
               "The decoration of corrupting_bonus isn't correctly initialized" );
} // honeypot::create_decoration()
