/* ----------------------------------------------------------------------------
 * pbb_errno.c
 * libpbb funtions to handle error messages
 *
 * Copyright 2002 Matthias Grimm
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version
 * 2 of the License, or (at your option) any later version.
 * ----------------------------------------------------------------------------*/
#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <stdio.h>
#include <unistd.h>
#include <syslog.h>
#include <stdarg.h>
#include <string.h>

#if defined(HAVE_ICONV) && defined(HAVE_LANGIINFO_CODESET) && defined(ENABLE_NLS)
#  include <iconv.h>
#  include <langinfo.h>
#  include <libintl.h>
#endif

#include "pbb.h"
#include "gettext_macros.h"

extern struct libbase libdata;

/* This function is depreciated and should not be used in new developments or
 * changes of existing clients. Use replacement function print_msg() instead.
 */
void
print_error (char *msg, ...)
{
	va_list args;
	
	va_start (args, msg);
	print_message (PBB_ERR, msg, args);
	va_end (args);
}

void
print_msg (enum pbberrtype type, char *msg, ...)
{
	va_list args;
	
	va_start (args, msg);
	print_message (type, msg, args);
	va_end (args);
}

void
print_message (enum pbberrtype type, char *msg, va_list args)
{
	struct libbase *base = &libdata;
	va_list targs;
	char inbuffer[200], *inptr = inbuffer;
	int offset, level;

#if defined(HAVE_ICONV) && defined(HAVE_LANGIINFO_CODESET) && defined(ENABLE_NLS)
	iconv_t cvh;
	char outbuffer[200], *outptr = outbuffer;
	size_t count, inidx, outidx;
#endif
	
	va_copy (targs, args);
	switch (type) {
		case PBB_ERR:
			snprintf (inbuffer, sizeof(inbuffer) - 1, "%s: ", _("ERROR"));
			level = LOG_ERR;
			break;
		case PBB_WARN:
			snprintf (inbuffer, sizeof(inbuffer) - 1, "%s: ", _("WARNING"));
			level = LOG_WARNING;
			break;
		case PBB_INFO:
			snprintf (inbuffer, sizeof(inbuffer) - 1, "%s: ", _("INFO"));
		default:
			level = LOG_INFO;
	}	
	offset = strlen (inbuffer);
	vsnprintf (&inbuffer[offset], sizeof(inbuffer) - offset - 1, msg, targs);
	va_end (targs);
	
#if defined(HAVE_ICONV) && defined(HAVE_LANGIINFO_CODESET) && defined(ENABLE_NLS)
	inidx = strlen(inbuffer);
	outidx = sizeof(outbuffer);
	
	if ((cvh = iconv_open (nl_langinfo(CODESET),
				bind_textdomain_codeset(textdomain(NULL), NULL))) != (iconv_t) -1) {
		count = iconv (cvh, &inptr, &inidx, &outptr, &outidx);
		inptr = count == -1 ? inbuffer : outbuffer;
		*outptr = '\0';  /* iconv doesn't terminate strings with '\0' */
		iconv_close (cvh);
	}
#endif
	
/*                    PBB_ERR       PBB_WARN     PBB_INFO
 *                   (LOG_ERR)    (LOG_WARNING) (LOG_INFO)
 * PBBDS_NONE         stderr          stderr       stderr
 * PBBDS_PREPARE  stderr + syslog     syslog       syslog
 * PBBDS_PARENT   stderr + syslog     syslog       syslog
 * PBBDS_CHILD        syslog          syslog       syslog
 */
	if (base->daemon != PBBDS_NONE) {
		if (type == PBB_ERR && base->daemon != PBBDS_CHILD)
			fprintf(stderr, inptr);
		syslog(level, inptr);
	} else
		fprintf(stderr, inptr);
}

