#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <sys/types.h>
#include <sys/stat.h>
#include <string.h>

#include <stdio.h>

#ifndef WIN32
#include <unistd.h>
#endif

#include "mudclient.h"

#include <gdk/gdkkeysyms.h>

#include "Message.h"
#include "PapayaList.h"
#include "MUD.h"
#include "MUDSelector.h"
#include "Prefs.h"
#include "PreferenceHandler.h"

extern Prefs * globalPreferences;
extern PreferenceHandler * preferenceHandler;

extern MUDSelector * mud_selector;
static GtkWidget * mcw = NULL;
static GtkWidget * me = NULL;
static GtkListStore * mcw_list_store = NULL;
static PapayaList mlist;

void on_mcw_edit_clicked(GtkButton * button, gpointer data);
void on_mcw_connect_clicked(GtkButton * button, gpointer data);
void on_mcw_delete_clicked(GtkButton * button, gpointer data);
void on_mcw_prefs_clicked(GtkButton * button, gpointer data);

// Redraws the mud_selector->clist widget.

MUD * MUDSelector::findMUD(char * name) {

  for (PapayaListElement * tmp = mlist.firstElement; tmp; tmp = tmp->getNext()) {
    MUD * mud = (MUD *)tmp->getData();

    if (!strcmp(mud->getName(), name))
      return mud;
  }

  return NULL;
}

void mcw_redraw_list() {
  GtkTreeIter iter;
  
  gtk_list_store_clear(GTK_LIST_STORE(mcw_list_store));

  for (PapayaListElement * tmp = mlist.firstElement; tmp; tmp = tmp->getNext()) {
    MUD * mud = (MUD *)tmp->getData();
    if (!mud->getDeleted()) {
      
      gtk_list_store_append(mcw_list_store, &iter);

      gtk_list_store_set(mcw_list_store, &iter,
			 0, mud->getName(),
			 1, mud->getHostname(),
			 2, mud->getPort(),
			 3, mud->getLoginName(),
			 -1);
    }
  }
}

void on_me_cancel_clicked(GtkButton * b, gpointer data) {

  MUD * n = (MUD *)data;

  if (me) {
    gtk_widget_hide(me);
    gtk_widget_destroy(me);
    me = NULL;

    // Remove this element from the list - only if was created with a NEW rather
    // than an edit.
    if (strlen(n->getName()) == 0) {
      for (PapayaListElement * tmp = mlist.firstElement; tmp; tmp = tmp->getNext()) {
	if (tmp->getData() == n) {
	  mlist.deleteEntry(tmp);
	  return;
	}
      }
    }
  }
}

void on_mud_editor_delete_event(GtkWindow * window, GdkEvent * event, gpointer data) {
	on_me_cancel_clicked(NULL, data);
}

void checkPrefs(MUD * m) {
  char filename[1024];
  FILE * fp;
  sprintf(filename, "prefs/%s", m->getName());

  fp = openFile(filename, NULL, "r");
  if (!fp) {
    globalPreferences->save(filename);
    return;
  }
  fclose(fp);
}

void on_me_ok_clicked(GtkButton * b, gpointer data) {
  if (me) {

    // Retrieve the data from the widget's entry widgets.
    MUD * m = (MUD *)data;

    gchar * name = m->getName();
    gchar * new_name = (gchar *)gtk_entry_get_text(GTK_ENTRY(mud_selector->mcw_name));

    if (strcmp(name, new_name)
	&& mud_selector->findMUD(new_name)) {
      new Message ("Information", _("A MUD already exists with that name.  Please choose another."), true);
      return;
    }

    m->setName((gchar *)gtk_entry_get_text(GTK_ENTRY(mud_selector->mcw_name)));
    m->setHostname((gchar *)gtk_entry_get_text(GTK_ENTRY(mud_selector->mcw_host)));
    m->setPort(atoi((gchar *)gtk_entry_get_text(GTK_ENTRY(mud_selector->mcw_port))));
    m->setLoginName((gchar *)gtk_entry_get_text(GTK_ENTRY(mud_selector->mcw_login)));
    m->setLoginTrigger((gchar *)gtk_entry_get_text(GTK_ENTRY(GTK_COMBO(mud_selector->mcw_login_trigger)->entry)));
    m->setPassword((gchar *)gtk_entry_get_text(GTK_ENTRY(mud_selector->mcw_password)));
    m->setPasswordTrigger((gchar *)gtk_entry_get_text(GTK_ENTRY(GTK_COMBO(mud_selector->mcw_password_trigger)->entry)));

    if (!GTK_TOGGLE_BUTTON(mud_selector->ownprefs_check)->active) {
      if (!m->getPreferences())
	m->setPreferences(new Prefs(m));
      checkPrefs(m); // This saves the prefs.
    }

    gtk_widget_hide(me);
    gtk_widget_destroy(me);
    me = NULL;

    if (mcw)
      mcw_redraw_list();
  }

}

void on_connection_wizard_button_clicked(GtkButton * b, gpointer data, gint i) {

  if (!mcw) {
    mcw = mud_selector->create_mcw();
  } else {
    gtk_window_present((GtkWindow *)mcw);
  }

  mcw_redraw_list();
  gtk_widget_show(mcw);

}

void
on_mcw_new_clicked                     (GtkButton       *button,
                                        gpointer         user_data)
{

  if (!me) {
    MUD * mud = new MUD();
    mud->incrRefCount(true);
    PapayaListElement * tmp = mlist.newEntry(NULL, mud);
    me = mud_selector->create_me((MUD *)tmp->getData());
  }
  gtk_widget_show(me);

}

int mcw_row = -1;

void selection_made (GtkWidget * clist, gint row, gint column, GdkEventButton * event, gpointer data) {

  mcw_row = row;
  int disappear = 1; // Should the window disappear at the end?

  if (!event) {
    return;
  }

  if(event->type == GDK_2BUTTON_PRESS) {
    if (event->button == 3) {
      disappear = 0;
    }
    on_mcw_connect_clicked(NULL, &disappear);
  }
}

void selection_cancelled (GtkWidget * clist, gint row, gint column, GdkEventButton * event, gpointer data) {
  mcw_row = -1;
}

// From prefs_interface.cpp
extern GtkWidget * prefs_widget;
//GtkWidget * create_prefs_dialog(MUD *);

void prefs_function(GtkTreeModel * model, GtkTreePath * path, GtkTreeIter * iter, gpointer data) {

  char * mud_name;

  // Figure out which MUD this selected row is.
  gtk_tree_model_get(model, iter, 0, &mud_name, -1);

  // Find the corresponding MUD entry and delete it.
  for (PapayaListElement * tmp = mlist.firstElement; tmp; tmp=tmp->getNext()) {
    MUD * m = (MUD *)tmp->getData();

    if (!strcmp(m->getName(), mud_name)) {
      preferenceHandler->createWindow(m);
      g_free(mud_name);
      return;
    }
  }
  g_free(mud_name);
}

void on_mcw_prefs_clicked(GtkButton * button, gpointer data) {

  gtk_tree_selection_selected_foreach( gtk_tree_view_get_selection(GTK_TREE_VIEW(data)),
				       prefs_function,
				       data );

}

int initialise_network_mud(MUD *);

void
on_mcw_close_clicked                   (GtkButton       *button,
                                        gpointer         user_data)
{
  if (!mcw)
    return;

  gtk_widget_hide(mcw);
  gtk_widget_destroy(mcw);
  mcw = NULL;

  mud_selector->saveMUDS("mcw.dat");
}

bool
on_mcw_wm_close                        (GtkWidget       *widget,
                                        GdkEvent        *event,
                                        gpointer        user_data)
{
  on_mcw_close_clicked(NULL, NULL);
  return TRUE;
}

MUDSelector::MUDSelector() {
  loadMUDS("mcw.dat");
}

MUDSelector::~MUDSelector() {
  saveMUDS("mcw.dat");

  for (PapayaListElement * ele = mlist.firstElement; ele; ele = ele->getNext()) {
    if (ele->getData()) {
      delete (MUD *)ele->getData();
      ele->setData(NULL);
    }
  }
}

void MUDSelector::loadMUDS(char * file) {

  FILE * fp;
  char line[1024];
  
  fp = openFile(file, NULL, "r");
  if (!fp) {
    new Message("Information", _("Papaya was unable to load the MUD Connection Wizard data.  Papaya will continue to load.  You should create an entry in the MUD Connection Wizard for your favourite MUD."), true);
    return;
  }

  while (fgets(line, 1024, fp))
    addMUD(line);

  fclose(fp);
}

void MUDSelector::saveMUDS(char * file) {

  FILE * fp;

  fp = openFile(file, NULL, "w");
  if (!fp) {
    new Message("Error", _("Papaya is unable to save your MUD Connection Wizard data.  If you have made any changes this session you will lose them."), true);
    return;
  }

  for (PapayaListElement * tmp = mlist.firstElement; tmp; tmp = tmp->getNext()) {
    MUD * m = (MUD *)tmp->getData();

    fprintf(fp, "%s`%s`%d`",
	    strlen(m->getName()) ? m->getName() : "<Empty>",
	    strlen(m->getHostname()) ? m->getHostname() : "<Empty>",
	    m->getPort());

    if (strlen(m->getLoginName()) > 0)
      fprintf(fp, "%s`", m->getLoginName());
    else
      fprintf(fp, "<EMPTY>`");
    
    if (strlen(m->getLoginTrigger()) > 0)
      fprintf(fp, "%s`", m->getLoginTrigger());
    else
      fprintf(fp, "<EMPTY>`");

    if (strlen(m->getPassword()) > 0)
      fprintf(fp, "%s`", m->getPassword());
    else
      fprintf(fp, "<EMPTY>`");

    if (strlen(m->getPasswordTrigger()) > 0)
      fprintf(fp, "%s`", m->getPasswordTrigger());
    else
      fprintf(fp, "<EMPTY>`");

    fprintf(fp, "%d`", (bool)m->getPreferences());
    
    fprintf(fp, "\n");
  }

  fclose(fp);
}

void MUDSelector::addMUD(char * line) {

  char name[1024];
  char hostname[1024];
  int port;
  char login_name[1024];
  char login_name_trigger[1024];
  char password[1024];
  char password_trigger[1024];
  int res = 0;
  int ownprefs;

  // name, hostname, port, login name, login trigger string, password, password trigger string
  // High ascii separator -> 255

  if (strlen(line) == 0)
    return;

  res = sscanf(line, "%[^`]`%[^`]`%d`%[^`]`%[^`]`%[^`]`%[^`]`%d",
	       name, hostname, &port, 
	       login_name, login_name_trigger, password, password_trigger, &ownprefs);
  
  if (res < 7 || res > 8) {
    new Message("Error", _("Papaya has detected a malformed line in your MUD Connection Wizard save data.  The data on that line will not be loaded and Papaya will attempt to continue loading the rest of the data."), true);
    return;
  }

  if (res == 7)
    ownprefs = 0;

  MUD * m = new MUD(name, hostname, port, login_name, login_name_trigger, password, password_trigger);

  m->initPrefs((bool)ownprefs);
  m->incrRefCount(true);

  mlist.newEntry(NULL, m);
  return;
}

void on_mcw_row_double_clicked(GtkTreeView * view, void * arg1, GtkTreeViewColumn * column, gpointer data) {
  on_mcw_connect_clicked(NULL, data);
}

GtkWidget * MUDSelector::create_mcw() {

  GtkWidget *mud_connection_wizard;
  GtkWidget *hbox1;
  GtkWidget *scrolledwindow1;
  GtkWidget *vbuttonbox1;
  GtkWidget *mcw_new;
  GtkWidget *mcw_edit;
  GtkWidget *mcw_delete;
  GtkWidget *mcw_connect;
  GtkWidget *mcw_close;
  GtkWidget *mcw_prefs;

  mud_connection_wizard = gtk_window_new (GTK_WINDOW_TOPLEVEL);
  gtk_window_set_title (GTK_WINDOW (mud_connection_wizard), "MUD Connection Wizard");
  gtk_window_set_default_size (GTK_WINDOW (mud_connection_wizard), 550, 320);

  hbox1 = gtk_hbox_new (FALSE, 0);
  gtk_widget_ref (hbox1);
  gtk_widget_show (hbox1);
  gtk_container_add (GTK_CONTAINER (mud_connection_wizard), hbox1);

  scrolledwindow1 = gtk_scrolled_window_new (NULL, NULL);
  g_object_set_data_full(G_OBJECT(mud_connection_wizard), "scrolledwindow1",
			 scrolledwindow1, 
			 (GtkDestroyNotify)gtk_widget_unref);
  gtk_widget_ref (scrolledwindow1);
  gtk_widget_show (scrolledwindow1);
  gtk_box_pack_start (GTK_BOX (hbox1), scrolledwindow1, TRUE, TRUE, 0);

  mcw_list_store = gtk_list_store_new(4,
				      G_TYPE_STRING,
				      G_TYPE_STRING,
				      G_TYPE_INT,
				      G_TYPE_STRING);
  
  GtkWidget * mcw_view = gtk_tree_view_new_with_model(GTK_TREE_MODEL(mcw_list_store));
  
  gtk_tree_view_set_rules_hint(GTK_TREE_VIEW(mcw_view), true);

  GtkCellRenderer * cell_renderer = gtk_cell_renderer_text_new();

  GtkTreeViewColumn * column = gtk_tree_view_column_new_with_attributes(_("MUD Name"),
						    cell_renderer,
						    "text", 0,
						    NULL);
  gtk_tree_view_append_column(GTK_TREE_VIEW(mcw_view), column);

  column = gtk_tree_view_column_new_with_attributes(_("Hostname"),
						    cell_renderer,
						    "text", 1,
						    NULL);
  gtk_tree_view_append_column(GTK_TREE_VIEW(mcw_view), column);

  column = gtk_tree_view_column_new_with_attributes(_("Port"),
						    cell_renderer,
						    "text", 2,
						    NULL);
  gtk_tree_view_append_column(GTK_TREE_VIEW(mcw_view), column);

  column = gtk_tree_view_column_new_with_attributes(_("Character"),
						    cell_renderer,
						    "text", 3,
						    NULL);
  gtk_tree_view_append_column(GTK_TREE_VIEW(mcw_view), column);

  gtk_widget_show(mcw_view);

  gtk_container_add (GTK_CONTAINER (scrolledwindow1), mcw_view);

  vbuttonbox1 = gtk_vbutton_box_new ();
  g_object_set_data_full(G_OBJECT(mud_connection_wizard), "vbuttonbox1",
				  vbuttonbox1, 
				  (GtkDestroyNotify)gtk_widget_unref);

  gtk_widget_ref (vbuttonbox1);
  gtk_widget_show (vbuttonbox1);
  gtk_box_pack_start (GTK_BOX (hbox1), vbuttonbox1, FALSE, FALSE, 0);
  gtk_button_box_set_layout (GTK_BUTTON_BOX (vbuttonbox1), GTK_BUTTONBOX_START);

  gtk_box_set_spacing(GTK_BOX(vbuttonbox1), 10);

  mcw_new = gtk_button_new_with_label (_("New"));
  gtk_widget_ref (mcw_new);
  gtk_widget_show (mcw_new);
  gtk_container_add (GTK_CONTAINER (vbuttonbox1), mcw_new);
  GTK_WIDGET_SET_FLAGS (mcw_new, GTK_CAN_DEFAULT);

  mcw_edit = gtk_button_new_with_label (_("Edit"));
  gtk_widget_ref (mcw_edit);
  gtk_widget_show (mcw_edit);
  gtk_container_add (GTK_CONTAINER (vbuttonbox1), mcw_edit);
  GTK_WIDGET_SET_FLAGS (mcw_edit, GTK_CAN_DEFAULT);

  mcw_delete = gtk_button_new_with_label (_("Delete"));
  gtk_widget_ref (mcw_delete);
  gtk_widget_show (mcw_delete);
  gtk_container_add (GTK_CONTAINER (vbuttonbox1), mcw_delete);
  GTK_WIDGET_SET_FLAGS (mcw_delete, GTK_CAN_DEFAULT);

  mcw_connect = gtk_button_new_with_label (_("Connect"));
  gtk_widget_ref (mcw_connect);
  gtk_widget_show (mcw_connect);
  gtk_container_add (GTK_CONTAINER (vbuttonbox1), mcw_connect);
  GTK_WIDGET_SET_FLAGS (mcw_connect, GTK_CAN_DEFAULT);

  mcw_prefs = gtk_button_new_with_label (_("Edit Preferences"));
  gtk_widget_ref(mcw_prefs);
  gtk_widget_show(mcw_prefs);
  gtk_container_add(GTK_CONTAINER(vbuttonbox1), mcw_prefs);
  GTK_WIDGET_SET_FLAGS (mcw_prefs, GTK_CAN_DEFAULT);

  mcw_close = gtk_button_new_from_stock(GTK_STOCK_CLOSE);

  gtk_widget_ref (mcw_close);
  gtk_widget_show (mcw_close);
  gtk_container_add (GTK_CONTAINER (vbuttonbox1), mcw_close);
  GTK_WIDGET_SET_FLAGS (mcw_close, GTK_CAN_DEFAULT);

  g_signal_connect_data (mcw_new, "clicked",
			 GTK_SIGNAL_FUNC (on_mcw_new_clicked),
			 mcw_view, NULL, (GConnectFlags)0);
  g_signal_connect_data (mcw_edit, "clicked",
			 GTK_SIGNAL_FUNC (on_mcw_edit_clicked),
			 mcw_view, NULL, (GConnectFlags)0);
  g_signal_connect_data (mcw_delete, "clicked",
			 GTK_SIGNAL_FUNC (on_mcw_delete_clicked),
			 mcw_view, NULL, (GConnectFlags)0);
  g_signal_connect_data (mcw_connect, "clicked",
			 GTK_SIGNAL_FUNC (on_mcw_connect_clicked),
			 mcw_view, NULL, (GConnectFlags)0);
  g_signal_connect_data (mcw_prefs, "clicked",
			 GTK_SIGNAL_FUNC (on_mcw_prefs_clicked),
			 mcw_view, NULL, (GConnectFlags)0);
  g_signal_connect_data (mcw_close, "clicked",
			 GTK_SIGNAL_FUNC (on_mcw_close_clicked),
			 mcw_view, NULL, (GConnectFlags)0);
  g_signal_connect_data (mud_connection_wizard, "delete_event",
			 GTK_SIGNAL_FUNC(on_mcw_wm_close),
			 NULL, NULL, (GConnectFlags)0);
  g_signal_connect_data (mcw_view, "row-activated",
			 GTK_SIGNAL_FUNC(on_mcw_row_double_clicked),
			 mcw_view, NULL, (GConnectFlags)0);

  return mud_connection_wizard;
}

GtkWidget * MUDSelector::create_me(MUD * mud) {
  GtkWidget *mud_editor;
  GtkWidget *vbox1;
  GtkWidget *table1;
  GtkWidget *label6;
  GtkWidget *label5;
  GtkWidget *label7;
  GtkWidget *label4;
  GtkWidget *label3;
  GtkWidget *label2;
  GtkWidget *label1;
  GtkWidget *hbuttonbox1;
  GtkWidget *button1;
  GtkWidget *button2;

  mud_editor = gtk_window_new (GTK_WINDOW_TOPLEVEL);
  g_object_set_data(G_OBJECT(mud_editor), "mud_editor", mud_editor);
  gtk_window_set_title (GTK_WINDOW (mud_editor), "MUD Editor");
  gtk_window_set_default_size (GTK_WINDOW (mud_editor), 700, -1);
  //  gtk_window_set_resizable(GTK_WINDOW(mud_editor), false);
  g_signal_connect_data( G_OBJECT(mud_editor), "delete_event",
	  GTK_SIGNAL_FUNC(on_mud_editor_delete_event),
	  mud, NULL, (GConnectFlags)0);


  vbox1 = gtk_vbox_new (FALSE, 0);
  gtk_widget_ref (vbox1);
  g_object_set_data_full(G_OBJECT(mud_editor), "vbox1", vbox1,
			 (GtkDestroyNotify) gtk_widget_unref);
  gtk_widget_show (vbox1);
  gtk_container_add (GTK_CONTAINER (mud_editor), vbox1);

  table1 = gtk_table_new (7, 2, FALSE);
  gtk_widget_ref (table1);
  g_object_set_data_full (G_OBJECT(mud_editor), "table1", table1,
			  (GtkDestroyNotify) gtk_widget_unref);
  gtk_widget_show (table1);
  gtk_box_pack_start (GTK_BOX (vbox1), table1, FALSE, FALSE, 0);
  gtk_container_set_border_width (GTK_CONTAINER (table1), 3);
  gtk_table_set_col_spacings (GTK_TABLE (table1), 3);

  mcw_name = gtk_entry_new ();
  gtk_widget_ref (mcw_name);
  g_object_set_data_full (G_OBJECT(mud_editor), "mcw_name", mcw_name,
			  (GtkDestroyNotify)  gtk_widget_unref);
  gtk_widget_show (mcw_name);
  gtk_table_attach (GTK_TABLE (table1), mcw_name, 1, 2, 0, 1,
                    (GtkAttachOptions) (GTK_EXPAND | GTK_FILL),
                    (GtkAttachOptions) (0), 0, 0);

  mcw_host = gtk_entry_new ();
  gtk_widget_ref (mcw_host);
  g_object_set_data_full(G_OBJECT(mud_editor), "mcw_host", mcw_host,
			 (GtkDestroyNotify) gtk_widget_unref);
  gtk_widget_show (mcw_host);
  gtk_table_attach (GTK_TABLE (table1), mcw_host, 1, 2, 1, 2,
                    (GtkAttachOptions) (GTK_EXPAND | GTK_FILL),
                    (GtkAttachOptions) (0), 0, 0);

  mcw_port = gtk_entry_new ();
  gtk_widget_ref (mcw_port);
  g_object_set_data_full (G_OBJECT(mud_editor), "mcw_port", mcw_port,
			  (GtkDestroyNotify) gtk_widget_unref);
  gtk_widget_show (mcw_port);
  gtk_table_attach (GTK_TABLE (table1), mcw_port, 1, 2, 2, 3,
                    (GtkAttachOptions) (GTK_EXPAND | GTK_FILL),
                    (GtkAttachOptions) (0), 0, 0);

  mcw_login = gtk_entry_new ();
  gtk_widget_ref (mcw_login);
  g_object_set_data_full(G_OBJECT(mud_editor), "mcw_login", mcw_login,
			 (GtkDestroyNotify) gtk_widget_unref);
  gtk_widget_show (mcw_login);
  gtk_table_attach (GTK_TABLE (table1), mcw_login, 1, 2, 4, 5,
                    (GtkAttachOptions) (GTK_EXPAND | GTK_FILL),
                    (GtkAttachOptions) (0), 0, 0);

  GList * list = NULL;
  list = g_list_append(list, (void *)"By what name do you wish to be known?");
  list = g_list_append(list, (void *)"Login:");
  mcw_login_trigger = gtk_combo_new();
  gtk_widget_ref (mcw_login_trigger);
  gtk_combo_set_popdown_strings(GTK_COMBO(mcw_login_trigger), list);

  g_object_set_data_full (G_OBJECT (mud_editor), "mcw_login_trigger", mcw_login_trigger,
			  (GtkDestroyNotify) gtk_widget_unref);
  gtk_widget_show (mcw_login_trigger);
  gtk_table_attach (GTK_TABLE (table1), mcw_login_trigger, 1, 2, 3, 4,
                    (GtkAttachOptions) (GTK_EXPAND | GTK_FILL),
                    (GtkAttachOptions) (0), 0, 0);



  mcw_password = gtk_entry_new ();
  gtk_widget_ref (mcw_password);
  g_object_set_data_full (G_OBJECT (mud_editor), "mcw_password", mcw_password,
			  (GtkDestroyNotify) gtk_widget_unref);
  gtk_widget_show (mcw_password);
  gtk_table_attach (GTK_TABLE (table1), mcw_password, 1, 2, 6, 7,
                    (GtkAttachOptions) (GTK_EXPAND | GTK_FILL),
                    (GtkAttachOptions) (0), 0, 0);

  list = NULL;
  list = g_list_append(list, (void *)"Password:");
  mcw_password_trigger = gtk_combo_new ();
  gtk_widget_ref (mcw_password_trigger);
  gtk_combo_set_popdown_strings(GTK_COMBO(mcw_password_trigger), list);

  g_object_set_data_full (G_OBJECT (mud_editor), "mcw_password_trigger", mcw_password_trigger,
			  (GtkDestroyNotify) gtk_widget_unref);
  gtk_widget_show (mcw_password_trigger);
  gtk_table_attach (GTK_TABLE (table1), mcw_password_trigger, 1, 2, 5, 6,
                    (GtkAttachOptions) (GTK_EXPAND | GTK_FILL),
                    (GtkAttachOptions) (0), 0, 0);


  label6 = gtk_label_new (_("Password"));
  gtk_widget_ref (label6);
  g_object_set_data_full (G_OBJECT (mud_editor), "label6", label6,
			  (GtkDestroyNotify) gtk_widget_unref);
  gtk_widget_show (label6);
  gtk_table_attach (GTK_TABLE (table1), label6, 0, 1, 6, 7,
                    (GtkAttachOptions) (GTK_FILL),
                    (GtkAttachOptions) (0), 0, 0);
  gtk_misc_set_alignment (GTK_MISC (label6), 7.45058e-09, 0.5);

  label5 = gtk_label_new (_("Name Trigger String"));
  gtk_widget_ref (label5);
  g_object_set_data_full (G_OBJECT (mud_editor), "label5", label5,
			  (GtkDestroyNotify) gtk_widget_unref);
  gtk_widget_show (label5);
  gtk_table_attach (GTK_TABLE (table1), label5, 0, 1, 3, 4,
                    (GtkAttachOptions) (GTK_FILL),
                    (GtkAttachOptions) (0), 0, 0);
  gtk_misc_set_alignment (GTK_MISC (label5), 7.45058e-09, 0.5);

  label7 = gtk_label_new (_("Password Trigger String"));
  gtk_widget_ref (label7);
  g_object_set_data_full (G_OBJECT (mud_editor), "label7", label7,
                            (GtkDestroyNotify) gtk_widget_unref);
  gtk_widget_show (label7);
  gtk_table_attach (GTK_TABLE (table1), label7, 0, 1, 5, 6,
                    (GtkAttachOptions) (GTK_FILL),
                    (GtkAttachOptions) (0), 0, 0);
  gtk_label_set_justify (GTK_LABEL (label7), GTK_JUSTIFY_RIGHT);
  gtk_misc_set_alignment (GTK_MISC (label7), 7.45058e-09, 0.5);

  label4 = gtk_label_new (_("Character Name"));
  gtk_widget_ref (label4);
  g_object_set_data_full (G_OBJECT (mud_editor), "label4", label4,
			  (GtkDestroyNotify) gtk_widget_unref);
  gtk_widget_show (label4);
  gtk_table_attach (GTK_TABLE (table1), label4, 0, 1, 4, 5,
                    (GtkAttachOptions) (GTK_FILL),
                    (GtkAttachOptions) (0), 0, 0);
  gtk_misc_set_alignment (GTK_MISC (label4), 7.45058e-09, 0.5);

  label3 = gtk_label_new (_("Port"));
  gtk_widget_ref (label3);
  g_object_set_data_full (G_OBJECT (mud_editor), "label3", label3,
			  (GtkDestroyNotify) gtk_widget_unref);
  gtk_widget_show (label3);
  gtk_table_attach (GTK_TABLE (table1), label3, 0, 1, 2, 3,
                    (GtkAttachOptions) (GTK_FILL),
                    (GtkAttachOptions) (0), 0, 0);
  gtk_misc_set_alignment (GTK_MISC (label3), 7.45058e-09, 0.5);

  label2 = gtk_label_new (_("Host"));
  gtk_widget_ref (label2);
  g_object_set_data_full (G_OBJECT (mud_editor), "label2", label2,
			  (GtkDestroyNotify) gtk_widget_unref);
  gtk_widget_show (label2);
  gtk_table_attach (GTK_TABLE (table1), label2, 0, 1, 1, 2,
                    (GtkAttachOptions) (GTK_FILL),
                    (GtkAttachOptions) (0), 0, 0);
  gtk_misc_set_alignment (GTK_MISC (label2), 7.45058e-09, 0.5);

  label1 = gtk_label_new (_("MUD Name"));
  gtk_widget_ref (label1);
  g_object_set_data_full (G_OBJECT (mud_editor), "label1", label1,
			  (GtkDestroyNotify) gtk_widget_unref);
  gtk_widget_show (label1);
  gtk_table_attach (GTK_TABLE (table1), label1, 0, 1, 0, 1,
                    (GtkAttachOptions) (GTK_FILL),
                    (GtkAttachOptions) (0), 0, 0);
  gtk_misc_set_alignment (GTK_MISC (label1), 7.45058e-09, 0.5);

  // Own preferences check box

  ownprefs_check = gtk_check_button_new_with_label(_("Use global preferences"));
  gtk_widget_ref(ownprefs_check);
  g_object_set_data_full (G_OBJECT(mud_editor), "ownprefs_check", ownprefs_check,
			  (GtkDestroyNotify) gtk_widget_unref);
  gtk_widget_show(ownprefs_check);
  gtk_table_attach (GTK_TABLE (table1), ownprefs_check, 0, 2, 7, 8,
		    (GtkAttachOptions) (GTK_FILL),
		    (GtkAttachOptions) (0), 0, 0);
  
  // End of own preferences entry.

  hbuttonbox1 = gtk_hbutton_box_new ();
  gtk_widget_ref (hbuttonbox1);
  g_object_set_data_full (G_OBJECT (mud_editor), "hbuttonbox1", hbuttonbox1,
			  (GtkDestroyNotify) gtk_widget_unref);
  gtk_widget_show (hbuttonbox1);
  gtk_box_pack_end (GTK_BOX (vbox1), hbuttonbox1, FALSE, FALSE, 0);
  gtk_button_box_set_layout (GTK_BUTTON_BOX (hbuttonbox1), GTK_BUTTONBOX_SPREAD);

  button2 = gtk_button_new_from_stock(GTK_STOCK_CANCEL);
  gtk_widget_ref (button2);
  g_object_set_data_full (G_OBJECT (mud_editor), "button2", button2,
			  (GtkDestroyNotify) gtk_widget_unref);
  gtk_widget_show (button2);
  gtk_container_add (GTK_CONTAINER (hbuttonbox1), button2);
  GTK_WIDGET_SET_FLAGS (button2, GTK_CAN_DEFAULT);

  button1 = gtk_button_new_from_stock(GTK_STOCK_OK);
  gtk_widget_ref (button1);
  g_object_set_data_full (G_OBJECT (mud_editor), "button1", button1,
			  (GtkDestroyNotify) gtk_widget_unref);
  gtk_widget_show (button1);
  gtk_container_add (GTK_CONTAINER (hbuttonbox1), button1);
  GTK_WIDGET_SET_FLAGS (button1, GTK_CAN_DEFAULT);

  g_signal_connect_data(button1, "clicked",
			GTK_SIGNAL_FUNC(on_me_ok_clicked),
			mud,
			NULL,
			(GConnectFlags)0);
  g_signal_connect_data(button2, "clicked",
			GTK_SIGNAL_FUNC(on_me_cancel_clicked),
			mud,
			NULL,
			(GConnectFlags)0);
  return mud_editor;
}


/**
 * GTK 2 button functions.
 */

void edit_function(GtkTreeModel * model, GtkTreePath * path, GtkTreeIter * iter, gpointer data) {

  char * mud_name;

  // Figure out which MUD this selected row is.
  gtk_tree_model_get(model, iter, 0, &mud_name, -1);

  // Find the corresponding MUD entry and delete it.
  for (PapayaListElement * tmp = mlist.firstElement; tmp; tmp=tmp->getNext()) {
    MUD * mud = (MUD *)tmp->getData();

    if (!strcmp(mud->getName(), mud_name)) {
      me = mud_selector->create_me(mud);

      gtk_entry_set_text(GTK_ENTRY(mud_selector->mcw_name), mud->getName());
      gtk_entry_set_text(GTK_ENTRY(mud_selector->mcw_host), mud->getHostname());

      char buf[1024];
      sprintf(buf, "%d", mud->getPort());

      gtk_entry_set_text(GTK_ENTRY(mud_selector->mcw_port), buf);

      gtk_entry_set_text(GTK_ENTRY(mud_selector->mcw_login), mud->getLoginName());

      gtk_entry_set_text(GTK_ENTRY(GTK_COMBO(mud_selector->mcw_login_trigger)->entry), mud->getLoginTrigger());
      gtk_entry_set_text(GTK_ENTRY(mud_selector->mcw_password), mud->getPassword());
      gtk_entry_set_text(GTK_ENTRY(GTK_COMBO(mud_selector->mcw_password_trigger)->entry), mud->getPasswordTrigger());


      if (!mud->getPreferences())
        gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(mud_selector->ownprefs_check), TRUE);

      gtk_widget_show(me);
      g_free(mud_name);
      return;
    }
  }
  g_free(mud_name);
}

void on_mcw_edit_clicked(GtkButton * button, gpointer data) {

  gtk_tree_selection_selected_foreach( gtk_tree_view_get_selection(GTK_TREE_VIEW(data)),
				       edit_function,
				       data );


}


void connect_function(GtkTreeModel * model, GtkTreePath * path, GtkTreeIter * iter, gpointer data) {
  char * mud_name;

  // Figure out which MUD this selected row is.
  gtk_tree_model_get(model, iter, 0, &mud_name, -1);

  // Find the corresponding MUD entry and delete it.
  for (PapayaListElement * tmp = mlist.firstElement; tmp; tmp=tmp->getNext()) {
    MUD * mud = (MUD *)tmp->getData();

    if (!strcmp(mud->getName(), mud_name)) {

      initialise_network_mud(mud);
    }
  }

  g_free(mud_name);
}

void on_mcw_connect_clicked(GtkButton * button, gpointer data) {

  GtkTreeSelection * selection = gtk_tree_view_get_selection(GTK_TREE_VIEW(data));
  if (!selection) {
    debug ("GtkTreeSelection is NULL.");
    return;
  }

  gtk_tree_selection_selected_foreach(selection,
				      connect_function,
				      data);

  if (mcw) {
    gtk_widget_hide(mcw);
    gtk_widget_destroy(mcw);
    mcw = NULL;
  }

  mud_selector->saveMUDS("mcw.dat");

}

void delete_function(GtkTreeModel * model, GtkTreePath * path, GtkTreeIter * iter, gpointer data) {

  char * mud_name = NULL;

  // Figure out which MUD this selected row is.
  gtk_tree_model_get(model, iter, 0, &mud_name, -1);

  // Find the corresponding MUD entry and delete it.
  for (PapayaListElement * tmp = mlist.firstElement; tmp; tmp=tmp->getNext()) {
    MUD * m = (MUD *)tmp->getData();
    if (!strcmp(m->getName(), mud_name)) {
      m->decrRefCount(true);
      if (m->getRefCount() == 0) // Only delete the MUD if the ref count is 0.
	delete m;
      tmp->setData(NULL);
      mlist.deleteEntry(tmp);
      g_free(mud_name);
      return;
    }
  }
  g_free(mud_name);
}

void on_mcw_delete_clicked(GtkButton * button, gpointer data) {

  gtk_tree_selection_selected_foreach( gtk_tree_view_get_selection(GTK_TREE_VIEW(data)),
				       delete_function,
				       data );


  mcw_redraw_list();
}
