
/*
 * Copyright (C) 2004-2005 Maximilian Schwerin
 *
 * This file is part of oxine a free media player.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 * $Id: odk_osd.c 2658 2007-08-22 12:11:40Z mschwerin $
 *
 * This file is part of the odk system.
 *
 */

#include "config.h"

#include <assert.h>
#include <errno.h>
#include <math.h>
#include <stdio.h>
#include <unistd.h>

#ifdef HAVE_IMAGEMAGICK
#include <sys/types.h>
#include <magick/api.h>
#include <magick/ImageMagick.h>

/* These undefs are necessary, as ImageMagick has these defined in the
 * public headers thus colliding with our own defines of the same name. */
#undef PACKAGE_BUGREPORT
#undef PACKAGE_NAME
#undef PACKAGE_STRING
#undef PACKAGE_TARNAME
#undef PACKAGE_VERSION
#endif

#ifdef HAVE_GDK_PIXBUF
#define GDK_PIXBUF_DISABLE_DEPRECATED
#include <gdk-pixbuf/gdk-pixbuf.h>
#endif

#include "heap.h"
#include "i18n.h"
#include "logger.h"
#include "odk_private.h"
#include "utils.h"


#ifdef DEBUG
#define VALIDATE_POINT(x,y)    {        \
    if ((x < 0)                         \
        || (x >= odk->osd.width)) {     \
        fatal ("x-value is outside "    \
               "the valid drawing "     \
               "area: 0 <= %d <= %d",   \
                x, odk->osd.width);     \
        assert (x >= 0);                \
        assert (x < odk->osd.width);    \
    }                                   \
    if ((y < 0)                         \
        || (y >= odk->osd.height)) {    \
        fatal ("y-value is outside "    \
               "the valid drawing "     \
               "area: 0 <= %d <= %d",   \
               y, odk->osd.height);     \
        assert (y >= 0);                \
        assert (y < odk->osd.height);   \
    }                                   \
}
#else
#define VALIDATE_POINT(x,y)    {        \
    if ((x < 0)                         \
        || (x >= odk->osd.width)) {     \
        return;                         \
    }                                   \
    if ((y < 0)                         \
        || (y >= odk->osd.height)) {    \
        return;                         \
    }                                   \
}
#endif

#define CLAMP(x, low, high)  (((x) > (high)) ? (high) : (((x) < (low)) ? (low) : (x)))

void
odk_osd_set_resolution (odk_t * odk, int w, int h)
{
    odk->osd.base_width = w;
    odk->osd.base_height = h;
}


inline int
odk_osd_get_width (odk_t * odk)
{
    return odk->osd.base_width;
}


inline int
odk_osd_get_height (odk_t * odk)
{
    return odk->osd.base_height;
}


/// Get aspect ratio of OSD resolution.
static inline double
odk_osd_get_aspect (odk_t * odk)
{
    return ((double) odk->osd.base_width) / ((double) odk->osd.base_height);
}


bool
odk_osd_use_unscaled_osd (odk_t * odk)
{
    return odk->osd.is_unscaled_available && odk->osd.use_unscaled
        && odk_current_has_video (odk) && !odk_current_is_logo_mode (odk);
}


/**
 * Shows the passed OSD.
 *
 * @param osd                   The OSD to show.
 */
static void
odk_osd_show_osd (odk_t * odk, xine_osd_t * osd)
{
    if (odk_osd_use_unscaled_osd (odk)) {
        xine_osd_show_unscaled (osd, 0);
    }
    else {
        xine_osd_show (osd, 0);
    }
}


void
odk_osd_swap (odk_t * odk)
{
    if (odk->osd.x_osd == odk->osd.x_osd_0) {
        odk->osd.x_osd = odk->osd.x_osd_1;
    }
    else {
        odk->osd.x_osd = odk->osd.x_osd_0;
    }
}


void
odk_osd_show (odk_t * odk)
{
    if (odk->osd.x_osd == odk->osd.x_osd_0) {
        odk_osd_show_osd (odk, odk->osd.x_osd_0);
        xine_osd_hide (odk->osd.x_osd_1, 0);
        xine_osd_clear (odk->osd.x_osd_1);
    }
    else {
        odk_osd_show_osd (odk, odk->osd.x_osd_1);
        xine_osd_hide (odk->osd.x_osd_0, 0);
        xine_osd_clear (odk->osd.x_osd_0);
    }

    odk->osd.is_visible = true;
}


void
odk_osd_hide_images (odk_t * odk)
{
#ifdef HAVE_OSD_IMAGE
    int i = 0;
    for (; i < odk->image_cache.fill; i++) {
        odk_osd_image_t *cur = odk->image_cache.entries[i].image;
        if (cur->x_osd && cur->is_visible) {
            xine_osd_hide (cur->x_osd, 0);
            cur->is_visible = false;
        }
    }
#endif
}


void
odk_osd_hide (odk_t * odk)
{
    if (odk->osd.x_osd) {
        xine_osd_hide (odk->osd.x_osd, 0);
    }

#ifdef HAVE_OSD_IMAGE
    odk_osd_hide_images (odk);
#endif

    odk->osd.is_visible = false;
}


void
odk_osd_clear (odk_t * odk)
{
    xine_osd_clear (odk->osd.x_osd);

    odk->osd.is_visible = false;
}


/**
 * Converts RGB color to YUV color.
 */
static uint32_t
rgb2yuv (uint32_t rgb)
{
    uint8_t r = (rgb >> 16) & 0xff;
    uint8_t g = (rgb >> 8) & 0xff;
    uint8_t b = (rgb >> 0) & 0xff;

#if 0
    printf ("\nRGB 0x%06X r:%3X g:%3X b:%3X\n", rgb, r, g, b);
    printf ("RGB 0x%06X r:%3d g:%3d b:%3d\n", rgb, r, g, b);
#endif

    uint8_t y = ((66 * r + 129 * g + 25 * b + 128) >> 8) + 16;
    uint8_t u = ((-38 * r - 74 * g + 112 * b + 128) >> 8) + 128;
    uint8_t v = ((112 * r - 94 * g - 18 * b + 128) >> 8) + 128;

    uint32_t yuv = ((y << 16) | (v << 8) | u);

#if 0
    printf ("YUV 0x%06X y:%3X u:%3X v:%3X\n", yuv, y, u, v);
    printf ("YUV 0x%06X y:%3d u:%3d v:%3d\n", yuv, y, u, v);
#endif

    return yuv;
}


/**
 * Returns the index of the color in the palette. If the color is not in the
 * palette yet, it is added.
 *
 * @param palette               A color palette.
 * @param color                 A YUV color value.
 * @param transparency          A transparency value.
 */
static int
get_color (odk_palette_t * palette, uint32_t color, uint8_t transparency)
{
    /* The first entry of the palette should always be transparent */
    if (palette->num_colors == 0) {
        palette->colors[0] = 0x00;
        palette->transparency[0] = 0x00;
        palette->num_colors++;
    }

    /* We ignore all entries that have no color */
    if (!color) {
        return 0;
    }

    int i = 0;
    int best_index = 0;
    uint8_t best_error = 0xff;

    /* What we do here is look for the color already in the palette, that best
     * matches the color we wish to add. The color must match. What is allowed
     * to be off a bit is the transparency. */
    for (; i < palette->num_colors; i++) {
        if (palette->colors[i] == color) {
            uint8_t current_error =
                abs (palette->transparency[i] - transparency);

            if (current_error == 0)
                return i;
            if (current_error < best_error) {
                best_error = current_error;
                best_index = i;
            }
        }
    }

    /* If the error in transparency is small enough we return the index of the
     * color we found. */
    if (best_error < 10) {
        return best_index;
    }

    /* If the error is to large and the palette is already full we return
     * index 0 which is completely transparent so that this color will not
     * appear in the output. */
    if (palette->num_colors == NUM_COLORS) {
        return 0;
    }

    /* If the error is to large and the palette still has space in it, we add
     * the new color. */
    palette->colors[palette->num_colors] = color;
    palette->transparency[palette->num_colors] = transparency;
    palette->num_colors++;

    return (palette->num_colors - 1);
}


int
odk_osd_get_color (odk_t * odk, uint32_t color, uint8_t transparency)
{
    int color_index = get_color (&odk->osd.palette, rgb2yuv (color),
                                 transparency);

    xine_osd_set_palette (odk->osd.x_osd_0,
                          odk->osd.palette.colors,
                          odk->osd.palette.transparency);
    xine_osd_set_palette (odk->osd.x_osd_1,
                          odk->osd.palette.colors,
                          odk->osd.palette.transparency);

    return color_index;
}


/**
 * Generates a gradient on palette out of given values.
 *
 * @param num_colors            The number of colors in the gradient.
 * @param from_color            The first color as a YUV color value. 
 * @param from_transparency     The first transparency value.
 * @param to_color              The last color as a YUV color value. 
 * @param to_transparency       The last transparency value.
 */
static void
palette_transition (odk_palette_t * palette, int num_colors,
                    uint32_t from_color, uint8_t from_transparency,
                    uint32_t to_color, uint8_t to_transparency)
{
    double transparency_step = to_transparency - from_transparency;
    transparency_step /= num_colors;

    uint8_t cb_from = from_color & 0xff;
    uint8_t cr_from = (from_color >> 8) & 0xff;
    uint8_t y_from = (from_color >> 16) & 0xff;

    uint8_t cb_to = to_color & 0xff;
    uint8_t cr_to = (to_color >> 8) & 0xff;
    uint8_t y_to = (to_color >> 16) & 0xff;

    double cb_step = (double) (cb_to - cb_from) / num_colors;
    double cr_step = (double) (cr_to - cr_from) / num_colors;
    double y_step = (double) (y_to - y_from) / num_colors;

    int first_index = palette->num_colors;
    int last_index = first_index + num_colors;

    int i = first_index + 1;
    for (; i < last_index; i++) {
        uint8_t cb_cur = cb_from + (int8_t) (cb_step * (i - first_index));
        uint8_t cr_cur = cr_from + (int8_t) (cr_step * (i - first_index));
        uint8_t y_cur = y_from + (int8_t) (y_step * (i - first_index));

        palette->colors[i] = cb_cur + (cr_cur << 8) + (y_cur << 16);
        palette->transparency[i] = from_transparency;
        palette->transparency[i] += transparency_step * (i - first_index);
    }

    palette->colors[first_index] = from_color;
    palette->transparency[first_index] = from_transparency;

    palette->colors[last_index] = to_color;
    palette->transparency[last_index] = to_transparency;

    palette->num_colors += num_colors;
}


int
odk_osd_alloc_text_palette (odk_t * odk,
                            uint32_t fg_color, uint8_t fg_transparency,
                            uint32_t bg_color, uint8_t bg_transparency,
                            uint32_t bo_color, uint8_t bo_transparency)
{
    int first_color = odk->osd.palette.num_colors;

    /* A xine text palette always consists of 11 colors of which color 0 is
     * transparent, color 1 is the background and color 10 is the foreground. */
    odk->osd.palette.colors[first_color] = 0;
    odk->osd.palette.transparency[first_color] = 0;
    odk->osd.palette.num_colors += 1;

    palette_transition (&odk->osd.palette, 10,
                        rgb2yuv (bg_color), bg_transparency,
                        rgb2yuv (fg_color), fg_transparency);

    xine_osd_set_palette (odk->osd.x_osd_0,
                          odk->osd.palette.colors,
                          odk->osd.palette.transparency);
    xine_osd_set_palette (odk->osd.x_osd_1,
                          odk->osd.palette.colors,
                          odk->osd.palette.transparency);

    return first_color;
}

#ifdef HAVE_OSD_IMAGE

static odk_osd_image_t *
odk_osd_image_new (odk_t * odk, const char *mrl,
                   int x, int y, int w, int h,
                   bool border_draw, uint32_t border_color,
                   bool use_separate_osd, int alignment)
{
    odk_osd_image_t *image = ho_new (odk_osd_image_t);

    image->mrl = ho_strdup (mrl);

    image->x = x;
    image->y = y;
    image->w = w;
    image->h = h;

    image->alignment = alignment;

    image->border_draw = border_draw;
    image->border_color = border_color;

    image->is_visible = false;

    /* This is set when loading the image. */
    image->x_osd = NULL;
    image->data = NULL;

    return image;
}


static void
odk_osd_image_free (odk_osd_image_t * image)
{
    if (image) {
        if (image->x_osd) {
            xine_osd_free (image->x_osd);
        }
        ho_free (image->data);
        ho_free (image->mrl);
        ho_free (image);
    }
}


/**
 * Loads an OSD image.
 *
 * @param mrl                   The MRL of the image to load.
 * @param x                     The x-coordinate of the image.
 * @param y                     The y-coordinate of the image.
 * @param w                     The width of the image.
 * @param h                     The height of the image.
 * @param border_draw           If TRUE a border of one pixel will be drawn
 *                              around the image.
 * @param border_color          The color of the border as an RGB color value.
 * @param use_separate_osd      If TRUE the image is drawn into a separate
 *                              OSD object that is saved inside the image
 *                              object. If FALSE only the image data is
 *                              loaded. This can later be drawn onto another
 *                              OSD object.
 * @param alignment             The alignment of the image.
 */
static odk_osd_image_t *
odk_osd_load_image (odk_t * odk, const char *mrl,
                    int x, int y, int w, int h,
                    bool border_draw, uint32_t border_color,
                    bool use_separate_osd, int alignment)
{
    clock_t start = clock ();
    odk_palette_t *palette = NULL;
    odk_osd_image_t *osd_image = NULL;

    assert (w > 0);
    assert (h > 0);

    if (!(odk->osd.hscale && odk->osd.vscale)) {
        error (_("Failed to create image with no size!"));
        return NULL;
    }

    /* Create OSD image data structure. */
    osd_image = odk_osd_image_new (odk, mrl, x, y, w, h,
                                   border_draw, border_color,
                                   use_separate_osd, alignment);

    /* Scale width and height to the current output size. */
    w *= odk->osd.hscale;
    h *= odk->osd.vscale;

    /* Scale position */
    x *= odk->osd.hscale;
    y *= odk->osd.vscale;

#if defined (HAVE_GDK_PIXBUF)

    GError *error = NULL;

    GdkPixbuf *pixbuf = gdk_pixbuf_new_from_file_at_scale (mrl, w, h, true,
                                                           &error);
    if (!pixbuf) {
        error (_("Could not read image '%s'!"), mrl);
        ho_free (osd_image);
        return NULL;
    }
    else {
        g_object_ref (G_OBJECT (pixbuf));
    }

    w = gdk_pixbuf_get_width (pixbuf);
    h = gdk_pixbuf_get_height (pixbuf);

    bool has_alpha = gdk_pixbuf_get_has_alpha (pixbuf);
    gint n_channels = gdk_pixbuf_get_n_channels (pixbuf);
    gint rowstride = gdk_pixbuf_get_rowstride (pixbuf);
    guchar *pixels = gdk_pixbuf_get_pixels (pixbuf);

    uint8_t *image_data = ho_malloc (w * h);

    if (use_separate_osd) {
        palette = ho_new (odk_palette_t);
        palette->num_colors = 0;
    }
    else {
        palette = &odk->osd.palette;
    }

    bool reduce_colors = false;
    {
        int col;
        int row;
        odk_palette_t *test_palette = ho_new (odk_palette_t);
        test_palette->num_colors = 0;
        get_color (test_palette, rgb2yuv (border_color), 0xff);
        for (row = 0; row < h; row++) {
            guchar *cline = pixels + (row + 0) * rowstride;

            for (col = 0; col < w; col++) {
                guchar *p0 = cline + (col + 0) * n_channels;

                guchar r = p0[0];
                guchar g = p0[1];
                guchar b = p0[2];
                guchar t = has_alpha ? p0[3] : 0xff;

                uint32_t yuv = rgb2yuv ((r << 16) | (g << 8) | b);
                if ((t != 0) && (get_color (test_palette, yuv, t) == 0)) {
                    reduce_colors = true;
                    break;
                }
            }
        }
        ho_free (test_palette);
    }

    /* If there are too many colors (> 256) in the image, they must be
     * reduced. We use Floyd Steinberg Dithering to do this. */
    if (reduce_colors) {
        gfloat shades = 8;
        gfloat factor = (float) (shades - 1) / (float) 255;

        gfloat *err0 = ho_malloc (sizeof (gfloat) * rowstride);
        gfloat *err1 = ho_malloc (sizeof (gfloat) * rowstride);

        gfloat *cerror = err0;
        gfloat *nerror = err1;

        gint col;
        gint row;
        for (row = 0; row < h; row++) {
            guchar *cline = pixels + (row + 0) * rowstride;

            gint dir = 1;
            gint beg = 0;
            gint end = w;
            if (row % 2 == 1) {
                dir = -1;
                beg = w - 1;
                end = -1;
            }

            /* Swap the error arrays. */
            gfloat *tmp = cerror;
            cerror = nerror;
            nerror = tmp;

            /* And set the array for the next line to zero. */
            for (col = 0; col < w * n_channels; col++) {
                nerror[col] = 0;
            }

            /*
             * Direction: ----->
             *    +------+------+------+
             *    |      |curr. |  p1  |
             *    |      |pixel | 7/16 |
             *    |      |      |      |
             *    +------+------+------+
             *    |  p2  |  p3  |  p4  |
             *    | 3/16 | 5/16 | 1/16 |
             *    |      |      |      |
             *    +------+------+------+
             */
            for (col = beg; col != end; col += dir) {
                guchar *p0 = cline + (col + 0) * n_channels;

                gint c;
                for (c = 0; c < n_channels; c++) {
                    gint p0 = col * n_channels + c;
                    gint p1 = (col + dir) * n_channels + c;
                    gint p2 = (col - dir) * n_channels + c;
                    gint p3 = col * n_channels + c;
                    gint p4 = (col + dir) * n_channels + c;

                    gfloat n = cline[p0] + cerror[p0];
                    gfloat o = n;

                    n = n * factor;
                    n = floor (n + 0.5);
                    n = n / factor;
                    n = CLAMP (n, 0.0, 255.0);

                    cline[p0] = (guint) n + 0.5;

                    gfloat e = o - n;
                    if (col - dir >= 0 && col - dir < w) {
                        nerror[p2] += e * 3 / 16;
                    }
                    {
                        nerror[p3] += e * 5 / 16;
                    }
                    if (col + dir >= 0 && col + dir < w) {
                        cerror[p1] += e * 7 / 16;
                        nerror[p4] += e * 1 / 16;
                    }
                }

                guchar r = p0[0];
                guchar g = p0[1];
                guchar b = p0[2];
                guchar t = has_alpha ? p0[3] : 0xff;

                uint32_t yuv = rgb2yuv ((r << 16) | (g << 8) | b);
                image_data[row * w + col] = get_color (palette, yuv, t);
            }
        }

        ho_free (err0);
        ho_free (err1);
    }

    else {
        int col;
        int row;
        for (row = 0; row < h; row++) {
            guchar *cline = pixels + (row + 0) * rowstride;

            for (col = 0; col < w; col++) {
                guchar *p0 = cline + (col + 0) * n_channels;

                guchar r = p0[0];
                guchar g = p0[1];
                guchar b = p0[2];
                guchar t = has_alpha ? p0[3] : 0xff;

                uint32_t yuv = rgb2yuv ((r << 16) | (g << 8) | b);
                image_data[row * w + col] = get_color (palette, yuv, t);
            }
        }
    }

    g_object_unref (pixbuf);

#elif defined (HAVE_IMAGEMAGICK)

    /* Initialize ImageMagick and load the image. */
    ExceptionInfo exception;
    Image *image = NULL;
    ImageInfo *image_info = NULL;
    {
        InitializeMagick (NULL);
        GetExceptionInfo (&exception);

        image_info = CloneImageInfo ((ImageInfo *) NULL);
        if (!image_info) {
            error (_("Could not read image '%s'!"), mrl);

            /* Destroy ImageMagick stuff */
            DestroyExceptionInfo (&exception);
            DestroyMagick ();

            /* Free OSD image */
            odk_osd_image_free (osd_image);

            return NULL;
        }
        strncpy (image_info->filename, mrl, MaxTextExtent);

        image = ReadImage (image_info, &exception);
        CatchException (&exception);
        if (!image) {
            error (_("Could not read image '%s'!"), mrl);

            /* Destroy ImageMagick stuff */
            DestroyImageInfo (image_info);
            DestroyExceptionInfo (&exception);
            DestroyMagick ();

            /* Free OSD image */
            odk_osd_image_free (osd_image);

            return NULL;
        }
    }

    /* If the image does not have the inteded size (very probable) the image
     * has to be scaled. */
    {
        int iw = image->columns;
        int ih = image->rows;

        /* Calculate the size and width of the image preserving the aspect
         * ratio of the image. */
        double des_ratio = (double) h / (double) w;
        double img_ratio = (double) ih / (double) iw;

        if (des_ratio < img_ratio) {
            w = ((double) h) / ((double) img_ratio);
        }
        else {
            h = ((double) w) * ((double) img_ratio);
        }

        /* We have to use SampleImage here so we don't introduce new colors
         * to the image. */
        if ((iw != w) || (ih != h)) {
            Image *sampled_image = SampleImage (image, w, h, &exception);
            CatchException (&exception);

            if (!sampled_image) {
                error (_("Could not scale image '%s'!"), mrl);

                /* Destroy ImageMagick stuff */
                DestroyImage (image);
                DestroyImageInfo (image_info);
                DestroyExceptionInfo (&exception);
                DestroyMagick ();

                /* Free OSD image */
                odk_osd_image_free (osd_image);

                return NULL;
            }
            else {
                DestroyImage (image);
                image = sampled_image;
            }
        }
    }

    /* If the image has more colors than xine-lib can handle we reduce the
     * number of colors. */
    {
        /* We fetch the number of unique colors of the image. */
        int num_colors = GetNumberColors (image, NULL, &exception);
        CatchException (&exception);

        /* The OSD palette has NUM_COLORS colors. The first entry 
         * of our palette is always transparent. */
        int max_colors = NUM_COLORS - 1;
        /* If necessary we leave space for the border color. */
        if (border_draw) {
            max_colors -= 1;
        }

        /* If necessary, we quantize the image. */
        if (num_colors > max_colors) {
            QuantizeInfo quantize_info;
            GetQuantizeInfo (&quantize_info);
            quantize_info.number_colors = max_colors;
#ifdef USE_YUV
            quantize_info.colorspace = YUVColorspace;
#endif
            QuantizeImage (&quantize_info, image);
        }
    }

    /* Get image pixels. */
    const PixelPacket *pp = AcquireImagePixels (image, 0, 0, w, h,
                                                &exception);
    CatchException (&exception);
    if (!pp) {
        error (_("Could not get pixel data!"));

        /* Destroy ImageMagick stuff */
        DestroyImage (image);
        DestroyImageInfo (image_info);
        DestroyExceptionInfo (&exception);
        DestroyMagick ();

        /* Free OSD image */
        odk_osd_image_free (osd_image);

        return NULL;
    }

    uint8_t *image_data = ho_malloc (w * h);

    if (use_separate_osd) {
        palette = ho_new (odk_palette_t);
        palette->num_colors = 0;
    }
    else {
        palette = &odk->osd.palette;
    }

    int col;
    int row;
    for (row = 0; row < h; row++) {
        for (col = 0; col < w; col++) {
#ifdef USE_YUV
            uint8_t y = ScaleQuantumToChar (pp->red);
            uint8_t u = ScaleQuantumToChar (pp->green);
            uint8_t v = ScaleQuantumToChar (pp->blue);
            uint32_t yuv = ((y << 16) | (v << 8) | u);
#else
            uint8_t r = ScaleQuantumToChar (pp->red);
            uint8_t g = ScaleQuantumToChar (pp->green);
            uint8_t b = ScaleQuantumToChar (pp->blue);
            uint32_t yuv = rgb2yuv ((r << 16) | (g << 8) | b);
#endif
            uint8_t t = 0xff - ScaleQuantumToChar (pp->opacity);
            image_data[row * w + col] = get_color (palette, yuv, t);
            pp++;
        }
    }

    /* Destroy ImageMagick stuff */
    DestroyImage (image);
    DestroyImageInfo (image_info);
    DestroyExceptionInfo (&exception);
    DestroyMagick ();

#endif /* HAVE_GDK_PIXBUF or HAVE_IMAGEMAGICK */

    /* Draw a border around the image. */
    if (border_draw) {
        uint32_t yuv = rgb2yuv (border_color);
        uint32_t color = get_color (palette, yuv, 0xff);

        int col;
        for (col = 0; col < w; col++) {
            image_data[col] = color;
            image_data[col + (h - 1) * w] = color;
        }

        int row;
        for (row = 0; row < h; row++) {
            image_data[row * w] = color;
            image_data[(row + 1) * w - 1] = color;
        }
    }

    if (alignment & ODK_ALIGN_RIGHT) {
        x -= w;
    }
    if (alignment & ODK_ALIGN_CENTER) {
        x -= w / 2;
    }
    if (alignment & ODK_ALIGN_VCENTER) {
        y -= h / 2;
    }
    if (alignment & ODK_ALIGN_BOTTOM) {
        y -= h;
    }
    if (x < 0) {
        x += odk_osd_get_width (odk) * odk->osd.hscale - w;
    }
    if (y < 0) {
        y += odk_osd_get_height (odk) * odk->osd.vscale - h;
    }

    if (use_separate_osd) {
        /* Create a new OSD object. */
        osd_image->x_osd = xine_osd_new (odk->main_stream,
                                         odk->osd.x_off, odk->osd.y_off,
                                         odk->osd.width, odk->osd.height);
        /* Set the palette. */
        xine_osd_set_palette (osd_image->x_osd, palette->colors,
                              palette->transparency);
        /* Draw the image to that OSD object. */
        xine_osd_draw_bitmap (osd_image->x_osd, image_data, x, y, w, h, NULL);
        /* We don't need the image data and the palette any more. */
        ho_free (image_data);
        ho_free (palette);
    }

    else {
        /* We keep the image data inside our OSD image. */
        osd_image->data = image_data;
        /* Set the palette. */
        xine_osd_set_palette (odk->osd.x_osd_0,
                              odk->osd.palette.colors,
                              odk->osd.palette.transparency);
        xine_osd_set_palette (odk->osd.x_osd_1,
                              odk->osd.palette.colors,
                              odk->osd.palette.transparency);
    }

#if defined (HAVE_GDK_PIXBUF)
    debug ("Loaded '%s' in %d ms (using gdk-pixbuf)", mrl,
           (clock () - start) * 1000 / CLOCKS_PER_SEC);
#elif defined (HAVE_IMAGEMAGICK)
    debug ("Loaded '%s' in %d ms (using ImageMagick)", mrl,
           (clock () - start) * 1000 / CLOCKS_PER_SEC);
#endif

    return osd_image;
}


/**
 * Initializes the internal image cache.
 */
static void
odk_osd_image_cache_init (odk_t * odk)
{
    int i = 0;
    for (; i < NUM_IMAGE_CACHE_ENTRIES; i++) {
        odk->image_cache.entries[i].image = NULL;
        odk->image_cache.entries[i].timestamp = 0;
    }
    odk->image_cache.fill = 0;
}


/**
 * Frees the internal image cache.
 */
static void
odk_osd_image_cache_free (odk_t * odk)
{
    int i = 0;
    for (; i < odk->image_cache.fill; i++) {
        odk_osd_image_free (odk->image_cache.entries[i].image);
        odk->image_cache.entries[i].image = NULL;
        odk->image_cache.entries[i].timestamp = 0;
    }
    odk->image_cache.fill = 0;
}


/**
 * Adds an image to the internal image cache.
 */
static bool
odk_osd_image_cache_add (odk_t * odk, odk_osd_image_t * image)
{
    if (!image) {
        return false;
    }

    assert (odk);

    if (odk->image_cache.fill < NUM_IMAGE_CACHE_ENTRIES) {
#if 0
        debug ("image cache: ADD (index=%d)", odk->image_cache.fill);
#endif
        odk->image_cache.entries[odk->image_cache.fill].image = image;
        odk->image_cache.entries[odk->image_cache.fill].timestamp =
            time (NULL);
        odk->image_cache.fill += 1;
    }

    else {
        /* Find the cache entry that has been used the least recently. */
        int i = 0;
        int j = 0;
        time_t ts = time (NULL);
        for (; i < NUM_IMAGE_CACHE_ENTRIES; i++) {
            image_cache_entry_t entry = odk->image_cache.entries[i];
            if (!entry.image->is_visible && (entry.timestamp < ts)) {
                j = i;
                ts = entry.timestamp;
            }
        }

        /* Remove and free that entry and add the new image in that place. */
#if 0
        debug ("image cache: REPLACE (index=%d)", j);
#endif
        if (!odk->image_cache.entries[j].image->is_visible) {
            odk_osd_image_free (odk->image_cache.entries[j].image);
            odk->image_cache.entries[j].image = image;
            odk->image_cache.entries[j].timestamp = time (NULL);
        }
        else {
            error (_("Could not free a cache entry for '%s'."), image->mrl);
            return false;
        }
    }

    return true;
}


/**
 * Retrieves an image from the internal image cache. If the image is not in
 * the cache yet the image is loaded from disk and added to the cache.
 */
static odk_osd_image_t *
odk_osd_image_cache_get (odk_t * odk, const char *mrl,
                         int x, int y, int w, int h,
                         bool border_draw, uint32_t border_color,
                         bool use_separate_osd, int alignment)
{
    odk_osd_image_t *image = NULL;

    int i = 0;
    for (; i < odk->image_cache.fill; i++) {
        image_cache_entry_t entry = odk->image_cache.entries[i];
        if ((strcmp (mrl, entry.image->mrl) == 0)
            && (entry.image->x == x)
            && (entry.image->y == y)
            && (entry.image->w == w)
            && (entry.image->h == h)
            && (entry.image->alignment == alignment)
            && ((entry.image->border_draw == border_draw))
            && ((entry.image->border_color == border_color))
            && ((use_separate_osd && entry.image->x_osd)
                || (!use_separate_osd && entry.image->data))) {
            entry.timestamp = time (NULL);
#if 0
            debug ("image cache: HIT (index=%d)", i);
#endif
            image = entry.image;
        }
    }

    if (!image) {
#if 0
        debug ("image cache: MISS");
#endif
        image = odk_osd_load_image (odk, mrl, x, y, w, h,
                                    border_draw, border_color,
                                    use_separate_osd, alignment);
        odk_osd_image_cache_add (odk, image);
    }

    return image;
}


/**
 * Draws the image into the standard OSD.
 */
odk_osd_image_t *
odk_osd_draw_image (odk_t * odk, const char *mrl,
                    int x, int y, int w, int h,
                    bool border_draw, uint32_t border_color, int alignment)
{
    odk_osd_image_t *image = NULL;

    assert (odk);
    assert (mrl);

    if (!file_exists (mrl)) {
        error (_("Could not open '%s': %s!"), mrl, strerror (errno));
        return NULL;
    }

    image = odk_osd_image_cache_get (odk, mrl, x, y, w, h,
                                     border_draw, border_color,
                                     false, alignment);

    if (image) {
        /* We only have to scale the location and size. The image data has
         * already been scaled to the correct size either in
         * odk_osd_adapt_size or on loading. */
        int x = (int) ((double) image->x * odk->osd.hscale);
        int y = (int) ((double) image->y * odk->osd.vscale);
        int w = (int) ((double) image->w * odk->osd.hscale);
        int h = (int) ((double) image->h * odk->osd.vscale);

        xine_osd_draw_bitmap (odk->osd.x_osd, image->data, x, y, w, h, NULL);
    }

    return image;
}


odk_osd_image_t *
odk_osd_show_image (odk_t * odk, const char *mrl,
                    int x, int y, int w, int h,
                    bool border_draw, uint32_t border_color, int alignment)
{
    odk_osd_image_t *image = NULL;

    assert (odk);
    assert (mrl);

    if (!file_exists (mrl)) {
        error (_("Could not open '%s': %s!"), mrl, strerror (errno));
        return NULL;
    }

    image = odk_osd_image_cache_get (odk, mrl, x, y, w, h,
                                     border_draw, border_color,
                                     true, alignment);
    if (image && !image->is_visible) {
        odk_osd_show_osd (odk, image->x_osd);
        image->is_visible = true;
    }

    return image;
}


odk_osd_image_t *
odk_osd_preload_image (odk_t * odk, const char *mrl,
                       int x, int y, int w, int h,
                       bool border_draw, uint32_t border_color, int alignment)
{
    odk_osd_image_t *image = NULL;

    assert (odk);
    assert (mrl);

    if (!file_exists (mrl)) {
        error (_("Could not open '%s': %s!"), mrl, strerror (errno));
        return NULL;
    }

    image = odk_osd_image_cache_get (odk, mrl, x, y, w, h,
                                     border_draw, border_color,
                                     true, alignment);

    return image;
}


void
odk_osd_hide_image (odk_osd_image_t * image)
{
    if (image && image->x_osd && image->is_visible) {
        xine_osd_hide (image->x_osd, 0);
        image->is_visible = false;
    }
}


bool
odk_osd_is_image_visible (odk_osd_image_t * image)
{
    return image && image->x_osd && image->is_visible;
}
#endif


void
odk_draw_vector (odk_t * odk, int x, int y, int w, int h,
                 odk_osd_vector_type_t type, int color)
{
    if (!(odk->osd.hscale && odk->osd.vscale))
        return;

    switch (type) {
    case OSD_VECTOR_ARROW_UP_DOUBLE:
        odk_draw_vector (odk, x + (w / 4), y, (w / 2), (h / 2),
                         OSD_VECTOR_ARROW_UP_SIMPLE, color);
        odk_draw_vector (odk, x + (w / 4), y + (h / 2), (w / 2), (h / 2),
                         OSD_VECTOR_ARROW_UP_SIMPLE, color);
        break;
    case OSD_VECTOR_ARROW_DOWN_DOUBLE:
        odk_draw_vector (odk, x + (w / 4), y, (w / 2), (h / 2),
                         OSD_VECTOR_ARROW_DOWN_SIMPLE, color);
        odk_draw_vector (odk, x + (w / 4), y + (h / 2), (w / 2), (h / 2),
                         OSD_VECTOR_ARROW_DOWN_SIMPLE, color);
        break;
    case OSD_VECTOR_ARROW_UP_SIMPLE:
        {
            int x1 = x;
            int x2 = x + w;
            int x3 = x + (w / 2);
            int y1 = y + h;
            int y2 = y + h;
            int y3 = y;
            odk_draw_triangle (odk, x1, y1, x2, y2, x3, y3, color, true);
        }
        break;
    case OSD_VECTOR_ARROW_DOWN_SIMPLE:
        {
            int x1 = x;
            int x2 = x + w;
            int x3 = x + (w / 2);
            int y1 = y;
            int y2 = y;
            int y3 = y + h;
            odk_draw_triangle (odk, x1, y1, x2, y2, x3, y3, color, true);
        }
        break;
    case OSD_VECTOR_MINUS:
        {
            int x1 = x;
            int y1 = y + (h / 2) - (h / 10);
            int w1 = w;
            int h1 = 2 * (h / 10);
            odk_draw_rect (odk, x1, y1, w1, h1, 0, color, true);
        }
        break;
    case OSD_VECTOR_PLUS:
        {
            int x1 = x + (w / 2) - (w / 10);
            int y1 = y;
            int w1 = 2 * (w / 10);
            int h1 = h;
            odk_draw_rect (odk, x1, y1, w1, h1, 0, color, true);
        }
        {
            int x1 = x;
            int y1 = y + (h / 2) - (h / 10);
            int w1 = w;
            int h1 = 2 * (h / 10);
            odk_draw_rect (odk, x1, y1, w1, h1, 0, color, true);
        }
        break;
    case OSD_VECTOR_HOME:
        {
            int x1 = x + (w / 2);
            int y1 = y;
            int x2 = x;
            int y2 = y + (h / 2);
            int x3 = x + w;
            int y3 = y + (h / 2);
            odk_draw_triangle (odk, x1, y1, x2, y2, x3, y3, color, true);
        }
        {
            int x1 = x + 2 * (w / 10);
            int y1 = y + h - (h / 10);
            int w1 = w - 4 * (w / 10);
            int h1 = (h / 10);
            odk_draw_rect (odk, x1, y1, w1, h1, 0, color, true);
        }
        {
            int x1 = x + (w / 10);
            int y1 = y + (h / 2);
            int w1 = (w / 10);
            int h1 = (h / 2);
            odk_draw_rect (odk, x1, y1, w1, h1, 0, color, true);
        }
        {
            int x1 = x + w - 2 * (w / 10);
            int y1 = y + (h / 2);
            int w1 = (w / 10);
            int h1 = (h / 2);
            odk_draw_rect (odk, x1, y1, w1, h1, 0, color, true);
        }
        break;
    case OSD_VECTOR_ARROW_UP:
        {
            int x1 = x + (w / 2);
            int y1 = y;
            int x2 = x;
            int y2 = y + (h / 2);
            int x3 = x + w;
            int y3 = y + (h / 2);
            odk_draw_triangle (odk, x1, y1, x2, y2, x3, y3, color, true);
        }
        {
            int x1 = x + (w / 2) - 2 * (w / 10);
            int y1 = y + (h / 2);
            int w1 = 4 * (w / 10);
            int h1 = (h / 2);
            odk_draw_rect (odk, x1, y1, w1, h1, 0, color, true);
        }
        break;
    case OSD_VECTOR_ARROW_DOWN:
        {
            int x1 = x;
            int y1 = y;
            int x2 = x + w;
            int y2 = y;
            int x3 = x + (w / 2);
            int y3 = y + (h / 2);
            odk_draw_triangle (odk, x1, y1, x2, y2, x3, y3, color, true);
        }
        {
            int x1 = x + (w / 2) - 2 * (w / 10);
            int y1 = y;
            int w1 = 4 * (w / 10);
            int h1 = (h / 2);
            odk_draw_rect (odk, x1, y1, w1, h1, 0, color, true);
        }
        break;
    case OSD_VECTOR_ARROW_LEFT:
        {
            int x1 = x + (w / 2);
            int y1 = y;
            int x2 = x;
            int y2 = y + (h / 2);
            int x3 = x + (w / 2);
            int y3 = y + h;
            odk_draw_triangle (odk, x1, y1, x2, y2, x3, y3, color, true);
        }
        {
            int x1 = x + (w / 2);
            int y1 = y + (h / 2) - 2 * (h / 10);
            int w1 = (w / 2);
            int h1 = 4 * (h / 10);
            odk_draw_rect (odk, x1, y1, w1, h1, 0, color, true);
        }
        break;
    case OSD_VECTOR_ARROW_RIGHT:
        {
            int x1 = x + (w / 2);
            int y1 = y;
            int x2 = x + w;
            int y2 = y + (h / 2);
            int x3 = x + (w / 2);
            int y3 = y + h;
            odk_draw_triangle (odk, x1, y1, x2, y2, x3, y3, color, true);
        }
        {
            int x1 = x;
            int y1 = y + (h / 2) - 2 * (h / 10);
            int w1 = (w / 2);
            int h1 = 4 * (h / 10);
            odk_draw_rect (odk, x1, y1, w1, h1, 0, color, true);
        }
        break;
    case OSD_VECTOR_CHECK:
        {
            int x1 = x + (w / 10);
            int x2 = x + (w / 3);
            int x3 = x + w - (w / 10) - 2;
            int y1 = y + (h / 2);
            int y2 = y + h - (h / 10);
            int y3 = y + (h / 10);

            odk_draw_line (odk, x1, y1, x2, y2, 3, color);
            odk_draw_line (odk, x2, y2, x3, y3, 3, color);
        }
        break;
    case OSD_VECTOR_VOLUME_MUTE:
        {
            int x1 = x;
            int y1 = y + h;
            int x2 = x + w;
            int y2 = y;

            odk_draw_line (odk, x1, y1, x2, y2, 3, color);
        }
    case OSD_VECTOR_VOLUME:
        {
            int x1 = x + (w / 10);
            int y1 = y + (h / 3);
            int w1 = (w / 3);
            int h1 = (h / 3);
            int x2 = x1 + w1 - 1;
            int y2 = y1 + h1 - 1;

            odk_draw_rect (odk, x1, y1, w1, 1, 0, color, true);
            odk_draw_rect (odk, x1, y2, w1, 1, 0, color, true);

            odk_draw_rect (odk, x1, y1, 1, h1, 0, color, true);
            odk_draw_rect (odk, x2, y1, 1, h1, 0, color, true);
        }
        {
            int x1 = x - 1 + 1 * (w / 3) + (w / 10);
            int x2 = x - 1 + 2 * (w / 3) + (w / 10);
            int y1 = y + 1 * (h / 3);
            int y2 = y;
            int y3 = y + 2 * (h / 3);
            int y4 = y + h;

            odk_draw_line (odk, x1, y1, x2, y2, 3, color);
            odk_draw_line (odk, x1, y3, x2, y4, 3, color);
            odk_draw_line (odk, x2, y2, x2, y4, 3, color);
        }
        break;
    default:
        warn (_("Unknown vector image type: %d!"), type);
        break;
    }
}


void
odk_draw_line (odk_t * odk, int x1, int y1, int x2, int y2,
               int width, int color)
{
    if (!(odk->osd.hscale && odk->osd.vscale))
        return;

    int px1 = round ((double) x1 * odk->osd.hscale);
    int py1 = round ((double) y1 * odk->osd.vscale);
    int px2 = round ((double) x2 * odk->osd.hscale);
    int py2 = round ((double) y2 * odk->osd.vscale);

    int px3 = px1 + width;
    int py3 = py1;
    int px4 = px2 + width;
    int py4 = py2;

    VALIDATE_POINT (px1, py1);
    VALIDATE_POINT (px2, py2);
    VALIDATE_POINT (px3, py3);
    VALIDATE_POINT (px4, py4);

    while (px1 < px3) {
        xine_osd_draw_line (odk->osd.x_osd, px1, py1, px2, py2,
                            XINE_OSD_TEXT1 + color);
        xine_osd_draw_point (odk->osd.x_osd, px1, py1,
                             XINE_OSD_TEXT1 + color);
        xine_osd_draw_point (odk->osd.x_osd, px2, py2,
                             XINE_OSD_TEXT1 + color);
        px1 += 1;
        px2 += 1;
    }
}


typedef struct {
    double x;
    double y;
} point_t;

static void
swap_points (point_t * p1, point_t * p2)
{
    double x = p1->x;
    double y = p1->y;
    p1->x = p2->x;
    p1->y = p2->y;
    p2->x = x;
    p2->y = y;
}

static void
line_draw (odk_t * odk, int minx, int miny, int maxx, int maxy, point_t * p1,
           point_t * p2, int color)
{
    if (p1->x > maxx)
        p1->x = maxx;
    if (p1->x < minx)
        p1->x = minx;
    if (p2->x > maxx)
        p2->x = maxx;
    if (p2->x < minx)
        p2->x = minx;
    if (p1->y > maxy)
        p1->y = maxy;
    if (p1->y < miny)
        p1->y = miny;
    if (p2->y > maxy)
        p2->y = maxy;
    if (p2->y < miny)
        p2->y = miny;

    int x1 = round (p1->x);
    int x2 = round (p2->x);
    xine_osd_draw_line (odk->osd.x_osd, x1, p1->y, x2, p2->y,
                        XINE_OSD_TEXT1 + color);
}


void
odk_draw_triangle (odk_t * odk, int x1, int y1, int x2, int y2,
                   int x3, int y3, int color, bool filled)
{
    if (!(odk->osd.hscale && odk->osd.vscale))
        return;

    point_t p[3];

    p[0].x = round ((double) x1 * odk->osd.hscale);
    p[0].y = round ((double) y1 * odk->osd.vscale);
    p[1].x = round ((double) x2 * odk->osd.hscale);
    p[1].y = round ((double) y2 * odk->osd.vscale);
    p[2].x = round ((double) x3 * odk->osd.hscale);
    p[2].y = round ((double) y3 * odk->osd.vscale);

    int i = 0;
    for (i = 0; i < 3; i++) {
        int j = i;
        for (; j < 3; j++) {
            if (p[i].y > p[j].y) {
                swap_points (&p[i], &p[j]);
            }
            else if ((p[i].y == p[j].y)
                     && (p[i].x > p[j].x)) {
                swap_points (&p[i], &p[j]);
            }
        }
    }

    assert (p[0].y <= p[1].y);
    assert (p[1].y <= p[2].y);

    int minx = odk->osd.width;
    int maxx = 0;
    int miny = odk->osd.height;
    int maxy = 0;
    for (i = 0; i < 3; i++) {
        if (p[i].x < minx)
            minx = p[i].x;
        if (p[i].x > maxx)
            maxx = p[i].x;
        if (p[i].y < miny)
            miny = p[i].y;
        if (p[i].y > maxy)
            maxy = p[i].y;
    }

    if (filled) {
        point_t *A = &p[0];
        point_t *B = &p[1];
        point_t *C = &p[2];

        double dx1 = 0;
        if (B->y - A->y > 0)
            dx1 = (B->x - A->x) / (B->y - A->y);
        double dx2 = 0;
        if (C->y - A->y > 0)
            dx2 = (C->x - A->x) / (C->y - A->y);
        double dx3 = 0;
        if (C->y - B->y > 0)
            dx3 = (C->x - B->x) / (C->y - B->y);

        point_t s;
        point_t *S = &s;
        S->x = A->x;
        S->y = A->y;
        point_t e;
        point_t *E = &e;
        E->x = A->x;
        E->y = A->y;
        if (dx1 > dx2) {
            for (; S->y <= B->y; S->y++, E->y++, S->x += dx2, E->x += dx1)
                line_draw (odk, minx, miny, maxx, maxy, S, E, color);
            E->x = B->x;
            E->y = B->y;
            for (; S->y <= C->y; S->y++, E->y++, S->x += dx2, E->x += dx3)
                line_draw (odk, minx, miny, maxx, maxy, S, E, color);
        }
        else {
            for (; S->y <= B->y; S->y++, E->y++, S->x += dx1, E->x += dx2)
                line_draw (odk, minx, miny, maxx, maxy, S, E, color);
            S->x = B->x;
            S->y = B->y;
            for (; S->y <= C->y; S->y++, E->y++, S->x += dx3, E->x += dx2)
                line_draw (odk, minx, miny, maxx, maxy, S, E, color);
        }
    }

    xine_osd_draw_line (odk->osd.x_osd, p[0].x, p[0].y, p[1].x, p[1].y,
                        XINE_OSD_TEXT1 + color);
    xine_osd_draw_line (odk->osd.x_osd, p[0].x, p[0].y, p[2].x, p[2].y,
                        XINE_OSD_TEXT1 + color);
    xine_osd_draw_line (odk->osd.x_osd, p[1].x, p[1].y, p[2].x, p[2].y,
                        XINE_OSD_TEXT1 + color);

    xine_osd_draw_point (odk->osd.x_osd, p[0].x, p[0].y,
                         XINE_OSD_TEXT1 + color);
    xine_osd_draw_point (odk->osd.x_osd, p[1].x, p[1].y,
                         XINE_OSD_TEXT1 + color);
    xine_osd_draw_point (odk->osd.x_osd, p[2].x, p[2].y,
                         XINE_OSD_TEXT1 + color);
}


/**
 * Plots part of a circle.
 *
 * @param xc                    circle center
 * @param yc                    circle center
 * @param xl                    circle border (left)
 * @param xr                    circle border (right)
 * @param x1                    bounding rectangle (top-left)
 * @param y1                    bounding rectangle (top-left)
 * @param x2                    bounding rectangle (bot-right)
 * @param y2                    bounding rectangle (bot-right)
 */
static void
plot_circ (odk_t * odk, int xc, int yc, int xl, int xr, int y,
           int x1, int y1, int x2, int y2, int color, bool filled)
{
    xl += xc;
    xr += xc;
    y += yc;

    if (xl < x1)
        xl = x1;
    if (xr > x2)
        xr = x2;
    if (y < y1)
        y = y1;
    if (y > y2)
        y = y2;

    if (!filled) {
        xine_osd_draw_point (odk->osd.x_osd, xl, y, XINE_OSD_TEXT1 + color);
        xine_osd_draw_point (odk->osd.x_osd, xr, y, XINE_OSD_TEXT1 + color);
    }
    else {
        xine_osd_draw_line (odk->osd.x_osd, xl, y, xr, y, color);
    }
}


void
odk_draw_circ (odk_t * odk, int x, int y, int r, int color, bool filled)
{
    if (!(odk->osd.hscale && odk->osd.vscale))
        return;

    int px = round ((double) x * odk->osd.hscale);
    int py = round ((double) y * odk->osd.vscale);
    int pr = round ((double) r * odk->osd.hscale);

    int px1 = round ((double) (x - r) * odk->osd.hscale);
    int px2 = round ((double) (x + r) * odk->osd.hscale);
    int py1 = round ((double) (y - r) * odk->osd.hscale);
    int py2 = round ((double) (y + r) * odk->osd.hscale);

    VALIDATE_POINT (px1, py1);
    VALIDATE_POINT (px2, py2);

    int F = 1 - pr;
    int X = 0;
    int Y = pr;

    plot_circ (odk, px, py, -pr, +pr, 0, px1, py1, px2, py2, color, filled);
    while (X < Y) {
        X += 1;
        if (F < 0) {
            F += 2 * X - 1;
        }
        else {
            F += 2 * (X - Y);
            Y -= 1;
        }

        plot_circ (odk, px, py, -X, +X, +Y, px1, py1, px2, py2,
                   color, filled);
        plot_circ (odk, px, py, -Y, +Y, +X, px1, py1, px2, py2,
                   color, filled);
        plot_circ (odk, px, py, -X, +X, -Y, px1, py1, px2, py2,
                   color, filled);
        plot_circ (odk, px, py, -Y, +Y, -X, px1, py1, px2, py2,
                   color, filled);
    }
}


static void
draw_round_rect (odk_t * odk, int x, int y, int w, int h, int r,
                 int color, bool filled)
{
    if (!(odk->osd.hscale && odk->osd.vscale))
        return;

    int px1 = round ((double) x * odk->osd.hscale);
    int py1 = round ((double) y * odk->osd.vscale);
    int px2 = round ((double) (x + w) * odk->osd.hscale);
    int py2 = round ((double) (y + h) * odk->osd.vscale);

    VALIDATE_POINT (px1, py1);
    VALIDATE_POINT (px2, py2);

    int pr = round ((double) r * odk->osd.hscale);
    int pw = round ((double) w * odk->osd.hscale) / 2.0 - pr;
    int ph = round ((double) h * odk->osd.hscale) / 2.0 - pr;

    int pxc = round ((double) (x + w / 2) * odk->osd.hscale);
    int pyc = round ((double) (y + h / 2) * odk->osd.vscale);

    int F = 1 - pr;
    int X = 0;
    int Y = pr;

    plot_circ (odk, pxc, pyc, -pr - pw, +pr + pw, 0, px1, py1, px2, py2,
               color, filled);
    while (X < Y) {
        X += 1;
        if (F < 0) {
            F += 2 * X - 1;
        }
        else {
            F += 2 * (X - Y);
            Y -= 1;
        }

        plot_circ (odk, pxc, pyc, -X - pw, +X + pw, -Y - ph, px1, py1, px2,
                   py2, color, filled);
        plot_circ (odk, pxc, pyc, -Y - pw, +Y + pw, -X - ph, px1, py1, px2,
                   py2, color, filled);
        plot_circ (odk, pxc, pyc, -X - pw, +X + pw, +Y + ph, px1, py1, px2,
                   py2, color, filled);
        plot_circ (odk, pxc, pyc, -Y - pw, +Y + pw, +X + ph, px1, py1, px2,
                   py2, color, filled);
    }

    xine_osd_draw_line (odk->osd.x_osd, pxc - pw, py1, pxc + pw + 1, py1,
                        color);
    xine_osd_draw_line (odk->osd.x_osd, pxc - pw, py2, pxc + pw + 1, py2,
                        color);
    xine_osd_draw_line (odk->osd.x_osd, px1, pyc - ph, px1, pyc + ph + 1,
                        color);
    xine_osd_draw_line (odk->osd.x_osd, px2, pyc - ph, px2, pyc + ph + 1,
                        color);

    if (filled) {
        xine_osd_draw_rect (odk->osd.x_osd, px1, pyc - ph, px2, pyc + ph + 1,
                            color, true);
    }
}


static void
draw_normal_rect (odk_t * odk, int x, int y, int w, int h,
                  int color, bool filled)
{
    if (!(odk->osd.hscale && odk->osd.vscale))
        return;

    int px1 = round ((double) x * odk->osd.hscale);
    int py1 = round ((double) y * odk->osd.vscale);
    int px2 = round ((double) (x + w) * odk->osd.hscale);
    int py2 = round ((double) (y + h) * odk->osd.vscale);

    VALIDATE_POINT (px1, py1);
    VALIDATE_POINT (px2, py2);

    /**
     * @bug This is a hack! We draw a point to the bottom righthand corner.
     *      This is necessary, because the bottom line of a non-filled
     *      rectangle is not displayed, if nothing else has a y-value that is
     *      greater. This is a xine-lib bug (I think).
     */
    if (!filled) {
        xine_osd_draw_point (odk->osd.x_osd, odk->osd.width, odk->osd.height,
                             XINE_OSD_TEXT1 + color);
    }

    /**
     * @bug xine-lib does not draw the point at the lower right hand corner of
     *      an unfilled rectangle.
     */
    xine_osd_draw_point (odk->osd.x_osd, px2, py2, XINE_OSD_TEXT1 + color);
    xine_osd_draw_rect (odk->osd.x_osd, px1, py1, px2, py2,
                        XINE_OSD_TEXT1 + color, false);

    /**
     * @bug xine-lib does not draw the correct width of a filled rectangle.
     */
    xine_osd_draw_rect (odk->osd.x_osd, px1, py1, px2, py2,
                        XINE_OSD_TEXT1 + color, filled);
}


void
odk_draw_rect (odk_t * odk, int x, int y, int w, int h, int r,
               int color, bool filled)
{
    if (r != 0) {
        draw_round_rect (odk, x, y, w, h, r, color, filled);
    }
    else {
        draw_normal_rect (odk, x, y, w, h, color, filled);
    }
}


void
odk_draw_text (odk_t * odk, int x, int y, const char *text,
               int alignment, int color_base)
{
    if (!(odk->osd.hscale && odk->osd.vscale))
        return;

    int pw = 0;
    int pd = 0;
    int ph = 0;
    {
        int w;
        int h1;
        int h2;
        /* We always want the same baseline for text. It should not matter if
         * the text contains characters like 'g' or 'f'. */
        odk_get_text_size (odk, "DVB", &w, &h1);
        odk_get_text_size (odk, "DVg", &w, &h2);

        int h = h1 - (h2 - h1);
        /* This moves the text up a few pixels, so that the top is really at
         * the top of the charaters. */
        y -= (h2 - h1);

        if (odk->osd.is_current_font_freetype) {
            if (alignment & ODK_ALIGN_VCENTER) {
                y += h / 2;
            }
            else if (alignment & ODK_ALIGN_TOP) {
                y += h;
            }
        }
        else {
            if (alignment & ODK_ALIGN_VCENTER) {
                y -= h / 2;
            }
            else if (alignment & ODK_ALIGN_BOTTOM) {
                y -= h;
            }
        }

        pd = round ((double) (h2 - h1) * odk->osd.vscale);
        ph = round ((double) h * odk->osd.vscale);
    }
    {
        int w;
        int h;
        /* The width has to be calculated using the real text. */
        odk_get_text_size (odk, text, &w, &h);

        if (alignment & ODK_ALIGN_CENTER) {
            x -= w / 2;
        }
        else if (alignment & ODK_ALIGN_RIGHT) {
            x -= w;
        }

        pw = round ((double) w * odk->osd.hscale);
    }

    int px = round ((double) x * odk->osd.hscale);
    int py = round ((double) y * odk->osd.vscale);

    VALIDATE_POINT (px, py);
    VALIDATE_POINT (px + pw, py + ph);

    xine_osd_draw_text (odk->osd.x_osd, px, py, text, color_base);

#ifdef DEBUG_TEXT_DRAWING
    xine_osd_draw_line (odk->osd.x_osd, px, py + pd, px + pw, py + pd,
                        color_base + 10);
    xine_osd_draw_line (odk->osd.x_osd, px, py + pd + ph / 2, px + pw,
                        py + pd + ph / 2, color_base + 10);
    xine_osd_draw_line (odk->osd.x_osd, px, py + pd + ph, px + pw,
                        py + pd + ph, color_base + 10);
#endif
}


void
odk_get_text_size (odk_t * odk, const char *text, int *width, int *height)
{
    int w;
    int h;

    if (!(odk->osd.hscale && odk->osd.vscale)) {
        if (width) {
            *width = 0;
        }
        if (height) {
            *height = 0;
        }
        return;
    }

    xine_osd_get_text_size (odk->osd.x_osd, text, &w, &h);

    if (width) {
        *width = round ((double) w / odk->osd.hscale);
    }
    if (height) {
        *height = round ((double) h / odk->osd.vscale);
    }
}


void
odk_osd_set_font (odk_t * odk, const char *font, int font_size)
{
    int psize;

    if (odk->osd.hscale < odk->osd.vscale) {
        psize = (int) ((double) font_size * odk->osd.hscale);
    }
    else {
        psize = (int) ((double) font_size * odk->osd.vscale);
    }

    /* smallest text size possible */
    if (psize < 16) {
        psize = 16;
    }

    if (strchr (font, '.') || strchr (font, '/')) {
        odk->osd.is_current_font_freetype = 1;
    }
    else {
        odk->osd.is_current_font_freetype = 0;
    }

    xine_osd_set_font (odk->osd.x_osd, font, psize);
}


void
odk_get_frame_size (odk_t * odk, int *w, int *h)
{
    if (odk->novideo_post != NULL) {
        int r;
        int f;

        while (!xine_get_current_frame (odk->main_stream,
                                        w, h, &r, &f, NULL));
    }
    else if (xine_get_stream_info (odk->main_stream,
                                   XINE_STREAM_INFO_HAS_VIDEO)) {
        *w = xine_get_stream_info (odk->main_stream,
                                   XINE_STREAM_INFO_VIDEO_WIDTH);
        *h = xine_get_stream_info (odk->main_stream,
                                   XINE_STREAM_INFO_VIDEO_HEIGHT);
    }
    else if (xine_get_stream_info (odk->animation_stream,
                                   XINE_STREAM_INFO_HAS_VIDEO)) {
        *w = xine_get_stream_info (odk->animation_stream,
                                   XINE_STREAM_INFO_VIDEO_WIDTH);
        *h = xine_get_stream_info (odk->animation_stream,
                                   XINE_STREAM_INFO_VIDEO_HEIGHT);
    }
    else if (xine_get_stream_info (odk->background_stream,
                                   XINE_STREAM_INFO_HAS_VIDEO)) {
        *w = xine_get_stream_info (odk->background_stream,
                                   XINE_STREAM_INFO_VIDEO_WIDTH);
        *h = xine_get_stream_info (odk->background_stream,
                                   XINE_STREAM_INFO_VIDEO_HEIGHT);
    }
    else {
        debug (_("Could not determine size of video!"));
    }
}


/**
 * Adapt the OSD to the current output size.
 */
void
odk_osd_adapt_size (odk_t * odk, xine_format_change_data_t * fcd)
{
    int x_off = 0;
    int y_off = 0;
    int width = 0;
    int height = 0;

    /* If we're going to use unscaled OSD, we use the width and height of
     * the output window. */
    if (odk_osd_use_unscaled_osd (odk)) {
        width = odk->win->get_width (odk->win);
        height = odk->win->get_height (odk->win);
    }

    /* If we got XINE_EVENT_FRAME_FORMAT_CHANGE we can use the data provided
     * by this event. */
    else if (fcd) {
        width = fcd->width;
        height = fcd->height;
    }

    /* Else we've got to get the data all by ourselves. */
    else {
        odk_get_frame_size (odk, &width, &height);
    }

    /* Adjust the OSD size to the current zoom level. */
    int zoom = config_get_number ("video.zoom");
    if (zoom < 100) {
        int w = width;
        int h = height;
        width = (width * zoom) / 100;
        height = (height * zoom) / 100;
        x_off = (w - width) / 2;
        y_off = (h - height) / 2;
    }

    /* Correct height and width to the OSD aspect ratio. */
    double osd_aspect = odk_osd_get_aspect (odk);
    double out_aspect = (double) width / (double) height;
    if (osd_aspect > out_aspect) {
        int real_height = height;
        height = width * odk_osd_get_height (odk) / odk_osd_get_width (odk);
        y_off += (real_height - height) / 2;
    }
    else if (osd_aspect < out_aspect) {
        int real_width = width;
        width = height * odk_osd_get_width (odk) / odk_osd_get_height (odk);
        x_off += (real_width - width) / 2;
    }

    /* If the current OSD already has the correct size, we do not have to do
     * anything. */
    if ((odk->osd.x_osd != NULL)
        && (odk->osd.x_osd_0 != NULL)
        && (odk->osd.x_osd_1 != NULL)
        && (odk->osd.x_off == x_off)
        && (odk->osd.y_off == y_off)
        && (odk->osd.width == width)
        && (odk->osd.height == height)) {
        return;
    }

    /* We hide all images so they don't get redrawn in the old size before
     * we have resized them to the new OSD size. */
#ifdef HAVE_OSD_IMAGE
    int i;
    for (i = 0; i < odk->image_cache.fill; i++) {
        odk_osd_image_t *cur = odk->image_cache.entries[i].image;
        if (cur->x_osd && cur->is_visible) {
            xine_osd_hide (cur->x_osd, 0);
        }
    }
#endif

    odk->osd.x_off = x_off;
    odk->osd.y_off = y_off;
    odk->osd.width = width;
    odk->osd.height = height;

    odk->osd.vscale = 0;
    if (odk_osd_get_height (odk)) {
        odk->osd.vscale = (double) height / (double) odk_osd_get_height (odk);
    }
    odk->osd.hscale = 0;
    if (odk_osd_get_width (odk)) {
        odk->osd.hscale = (double) width / (double) odk_osd_get_width (odk);
    }

    /* Free the old OSD, and create a new one. */
    if (odk->osd.x_osd_0) {
        xine_osd_free (odk->osd.x_osd_0);
    }
    odk->osd.x_osd_0 = xine_osd_new (odk->main_stream,
                                     odk->osd.x_off, odk->osd.y_off,
                                     odk->osd.width, odk->osd.height);
    if (odk->osd.x_osd_1) {
        xine_osd_free (odk->osd.x_osd_1);
    }
    odk->osd.x_osd_1 = xine_osd_new (odk->main_stream,
                                     odk->osd.x_off, odk->osd.y_off,
                                     odk->osd.width, odk->osd.height);
    odk->osd.x_osd = odk->osd.x_osd_0;

    xine_osd_set_palette (odk->osd.x_osd_0,
                          odk->osd.palette.colors,
                          odk->osd.palette.transparency);
    xine_osd_set_palette (odk->osd.x_osd_1,
                          odk->osd.palette.colors,
                          odk->osd.palette.transparency);

    /* Adapt the sizes of the images in the cache and show any image, that
     * is marked as visible. */
#ifdef HAVE_OSD_IMAGE
    for (i = 0; i < odk->image_cache.fill; i++) {
        odk_osd_image_t *cur = odk->image_cache.entries[i].image;

        if (cur->x_osd) {
            odk_osd_image_t *tmp = odk_osd_load_image (odk, cur->mrl,
                                                       cur->x, cur->y,
                                                       cur->w, cur->h,
                                                       cur->border_draw,
                                                       cur->border_color,
                                                       true, cur->alignment);
            if (tmp) {
                xine_osd_free (cur->x_osd);
                cur->x_osd = tmp->x_osd;
                ho_free (tmp->mrl);
                ho_free (tmp);

                if (cur->is_visible) {
                    odk_osd_show_osd (odk, cur->x_osd);
                }
            }
        }

        else if (cur->data) {
            odk_osd_image_t *tmp = odk_osd_load_image (odk, cur->mrl,
                                                       cur->x, cur->y,
                                                       cur->w, cur->h,
                                                       cur->border_draw,
                                                       cur->border_color,
                                                       false, cur->alignment);
            if (tmp) {
                ho_free (cur->data);
                cur->data = tmp->data;
                ho_free (tmp->mrl);
                ho_free (tmp);
            }
        }
    }
#endif

    /* We send an event to notify any listeners that the format of the OSD has
     * changed. */
    oxine_event_t ev;
    ev.type = OXINE_EVENT_OSD_FORMAT_CHANGED;
    odk_oxine_event_send (odk, &ev);
}


bool
odk_is_osd_visible (odk_t * odk)
{
    return odk->osd.is_visible;
}


void
odk_osd_init (odk_t * odk)
{
#if defined (HAVE_GDK_PIXBUF)
    g_type_init ();
#endif

    odk->osd.is_visible = false;
    odk->osd.is_current_font_freetype = false;
    odk->osd.is_unscaled_available = false;

    odk->osd.x_osd = NULL;
    odk->osd.x_osd_0 = NULL;
    odk->osd.x_osd_1 = NULL;
    odk->osd.hscale = 0;
    odk->osd.vscale = 0;
    odk->osd.x_off = 0;
    odk->osd.y_off = 0;
    odk->osd.width = 0;
    odk->osd.height = 0;
    odk->osd.palette.num_colors = 0;

    /* Test support of unscaled OSD. */
    xine_osd_t *x_osd = xine_osd_new (odk->main_stream, 0, 0, 10, 10);
    int capabilities = xine_osd_get_capabilities (x_osd);
    xine_osd_free (x_osd);
    odk->osd.is_unscaled_available = (capabilities & XINE_OSD_CAP_UNSCALED);
    if (odk->osd.is_unscaled_available) {
        info (_("The current video driver supports unscaled OSD!"));
    }
    else {
        warn (_("The current video driver does not support unscaled OSD!"));
    }
    odk->osd.use_unscaled = config_get_bool ("video.osd.use_unscaled");

#ifdef HAVE_OSD_IMAGE
    /* Initialize the image cache. */
    odk_osd_image_cache_init (odk);
#endif

    /* Adapt size of OSD to current window/ output size. */
    odk_osd_adapt_size (odk, NULL);
}


void
odk_osd_free (odk_t * odk)
{
#ifdef HAVE_OSD_IMAGE
    /* Free the image cache. */
    odk_osd_image_cache_free (odk);
#endif

    /* Free the OSD drawing area. */
    if (odk->osd.x_osd_0) {
        xine_osd_free (odk->osd.x_osd_0);
    }
    if (odk->osd.x_osd_1) {
        xine_osd_free (odk->osd.x_osd_1);
    }
    odk->osd.x_osd = NULL;
    odk->osd.x_osd_0 = NULL;
    odk->osd.x_osd_1 = NULL;
}
