//                                               -*- C++ -*-
/**
 *  @file  GumbelFactory.cxx
 *  @brief Factory for Gumbel distribution
 *
 *  (C) Copyright 2005-2012 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: schueller $
 *  @date:   $LastChangedDate: 2012-02-17 19:35:43 +0100 (Fri, 17 Feb 2012) $
 *  Id:      $Id: GumbelFactory.cxx 2392 2012-02-17 18:35:43Z schueller $
 */
#include "GumbelFactory.hxx"
#include "SpecFunc.hxx"

BEGIN_NAMESPACE_OPENTURNS




/* Default constructor */
GumbelFactory::GumbelFactory():
  DistributionImplementationFactory()
{
  // Nothing to do
}

/* Virtual constructor */
GumbelFactory * GumbelFactory::clone() const
{
  return new GumbelFactory(*this);
}

/* Here is the interface that all derived class must implement */

Gumbel * GumbelFactory::build(const NumericalSample & sample) const
{
  if (sample.getSize() == 0) throw InvalidArgumentException(HERE) << "Error: cannot build a Gumbel distribution from an empty sample";
  if (sample.getDimension() != 1) throw InvalidArgumentException(HERE) << "Error: can build a Gumbel distribution only from a sample of dimension 1, here dimension=" << sample.getDimension();
  NumericalScalar mu(sample.computeMean()[0]);
  NumericalScalar sigma(sample.computeStandardDeviationPerComponent()[0]);
  if (sigma <= 0.0) throw InternalException(HERE) << "Error: can build a Gumbel distribution only if sigma >= 0, here sigma=" << sigma;
  return Gumbel(mu, sigma, Gumbel::MUSIGMA).clone();
}

Gumbel * GumbelFactory::build(const NumericalPointCollection & parameters) const
{
  try {
    Gumbel distribution;
    distribution.setParametersCollection(parameters);
    return distribution.clone();
  }
  catch (InvalidArgumentException & ex)
    {
      throw InternalException(HERE) << "Error: cannot build a Gumbel distribution from the given parameters";
    }
}

Gumbel * GumbelFactory::build() const
{
  return Gumbel().clone();
}

END_NAMESPACE_OPENTURNS
