//                                               -*- C++ -*-
/**
 *  @file  t_CorrelationAnalysis_sobol.cxx
 *  @brief Test file fo the Sobol' indices computation
 *
 *  (C) Copyright 2005-2010 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: lebrun $
 *  @date:   $LastChangedDate: 2008-04-15 16:24:44 +0200 (mar, 15 avr 2008) $
 *  Id:      $Id:  t_CorrelationAnalysis_sobol.cxx 789 2008-04-15 14:24:44Z lebrun $
 */


#include "OT.hxx"
#include "OTtestcode.hxx"
#include "OStream.hxx"
#include "NumericalMathFunction.hxx"
#include "Uniform.hxx"
#include "NumericalPoint.hxx"
#include "SymmetricMatrix.hxx"
#include "math.h"
#include "IndependentCopula.hxx"
#include "ComposedDistribution.hxx"
#include "CorrelationAnalysis.hxx"
#include "Matrix.hxx"
#include "LinearNumericalMathFunction.hxx"
#include "Normal.hxx"
#include "Log.hxx"

using namespace OT;
using namespace OT::Test;
using namespace OT::Base::Common;
using namespace OT::Base::Type;
using namespace OT::Base::Func;
using namespace OT::Base::Stat;
using namespace OT::Uncertainty::Model;
using namespace OT::Uncertainty::Distribution;


typedef Collection<Distribution>										DistributionCollection;
typedef CorrelationAnalysis::SobolIndiceResult			SobolIndiceResult;
typedef CorrelationAnalysis::SobolIndiceParameters	SobolIndiceParameters;


int main(int argc, char *argv[])
{
  TESTPREAMBLE;
  OStream fullprint(std::cout);

  try
    {
      UnsignedLong dimension(3);
      UnsignedLong sampleSize(1000);

      // we create an analytical function
      Description input(dimension);
      input[0] = "x0";
      input[1] = "x1";
      input[2] = "x2";
      fullprint << "input=" << input << std::endl;
	
      Description foutput(1);
      foutput[0] = "f0";
      fullprint << "output=" << foutput << std::endl;
	
      Description formulas(foutput.getSize());
      formulas[0] = "sin(x0)+7*sin(x1)^2+0.1*x2^4*sin(x0)";
	
      NumericalMathFunction analytical(input, foutput, formulas);
      fullprint << "analytical=" << analytical << std::endl;

      // we create a collection of uniform distributions over [-Pi; Pi[
      DistributionCollection aCollection;
      for(UnsignedLong i = 0; i < dimension; ++i)
	{
	  aCollection.add(Uniform(-M_PI, +M_PI));
	}

      // we create an independent copula
      IndependentCopula aCopula(aCollection.getSize());
      aCopula.setName("an independent copula");

      // we create one distribution object
      ComposedDistribution aDistribution(aCollection, Copula(aCopula));
      aDistribution.setName("a uniform distribution");

      // we create two input samples for the function
      NumericalSample firstInputSample(aDistribution.getNumericalSample(sampleSize));
      NumericalSample secondInputSample(aDistribution.getNumericalSample(sampleSize));

      // Choose which indices to compute
      CorrelationAnalysis::SobolIndiceParameters indiceParameters;
      // Choose to compute indices until order 3 (requires computation of inferior order indices)
      indiceParameters.setMaximumOrder(3);
      // Choose to compute total order indices
      indiceParameters.setTotalIndiceComputation(true);

      // Compute the Sobol' indices
      SobolIndiceResult myResult(CorrelationAnalysis::SobolIndice(indiceParameters, firstInputSample, secondInputSample, analytical));

      // Retrieve the indices from result according to the selected indices via indiceParameters
      // firstOrderIndice[i] is the first order indice of variable i
      NumericalPoint firstOrderIndice(myResult.getFirstOrderIndice());
      // secondOrderIndice(i, j) is the second order indice for both variables i and j (i not equal to j)
      SymmetricMatrix secondOrderIndice(myResult.getSecondOrderIndice());
      // thirdOrderIndice(i, j, k) is the indice for the subset of variables {i, j, k} (i, j and k are different)
      SymmetricTensor thirdOrderIndice(myResult.getThirdOrderIndice());
      // totalOrder[i] is the total indice for variable i
      NumericalPoint totalOrderIndice(myResult.getTotalOrderIndice());

      // stream out the indices
      fullprint << "first order indices=" << firstOrderIndice << std::endl;
      fullprint << "second order indices=" << secondOrderIndice << std::endl;
      fullprint << "third order indices=" << thirdOrderIndice << std::endl;
      fullprint << "total order indices=" << totalOrderIndice << std::endl;
    }
  catch (TestFailed & ex)
    {
      std::cerr << ex << std::endl;
      return ExitCode::Error;
    }

  return ExitCode::Success;
}
