//                                               -*- C++ -*-
/**
 *  @file  VisualTest.cxx
 *  @brief StatTest implements statistical tests
 *
 *  (C) Copyright 2005-2010 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: dutka $
 *  @date:   $LastChangedDate: 2010-02-04 16:44:49 +0100 (jeu. 04 févr. 2010) $
 *  Id:      $Id: VisualTest.cxx 1473 2010-02-04 15:44:49Z dutka $
 */
#include <cmath>
#include "VisualTest.hxx"
#include "Curve.hxx"
#include "Cloud.hxx"
#include "BarPlot.hxx"
#include "Staircase.hxx"
#include "NumericalPoint.hxx"


namespace OpenTURNS
{
  namespace Uncertainty
  {
    namespace StatTest
    {

      typedef Base::Graph::Curve         Curve;
      typedef Base::Graph::Cloud         Cloud;
      typedef Base::Graph::BarPlot       BarPlot;
      typedef Base::Graph::Staircase     Staircase;
      typedef Base::Type::NumericalPoint NumericalPoint;

      /* Default constructor, needed by SWIG */
      VisualTest::VisualTest()
      {
	// Nothing to do
      }

      /* Draw the empirical CDF of the Sample when its dimension is 1 */
      VisualTest::Graph VisualTest::DrawEmpiricalCDF(const NumericalSample & sample,
						     const NumericalScalar xMin,
						     const NumericalScalar xMax)
	/* throw(InvalidDimensionException, InvalidArgumentException) */
      {
	// Value :    0    1/5  2/5  3/5    4/5    1
	// Data  : ------+-----+---+------+----+---------
	// Case 1: ------------------------------[----]--
        // Case 2: ------------------[---]---------------
	//         -[--]---------------------------------
	// Case 3: ----------[---]-----------------------
        //         ---[-----------------------------]----
	//         -------[-----------------]------------
	if (sample.getDimension() != 1) throw InvalidDimensionException(HERE) << "Error: can draw a CDF only if dimension equals 1, here dimension=" << sample.getDimension();
	if (xMax <= xMin) throw InvalidArgumentException(HERE) << "Error: cannot draw a CDF with xMax >= xMin";
	// Create the graph that will store the staircase representing the empirical CDF
	OSS oss;
	oss << sample.getName() << " cdf";
	Graph graphCDF(oss, "x", "cdf", true, "topright");
	// Here, we know that the sample size equals 1
	const UnsignedLong size(sample.getSize());
	const NumericalSample sortedSample(sample.sort(0));
	// Find the elements of the sample that are between xMin and xMax
	// First, the lower bound. iMin will be the index of the first element of the sorted sample that is greater or equals to xMin
	UnsignedLong iMin(0);
	while ((iMin < size) && (sortedSample[iMin][0] < xMin)) ++iMin;
	// If iMin == size , all the elements are < xMin: we are in a region where the CDF is constant, equals to 1.
	if (iMin == size)
	  {
	    NumericalSample data(2, 2);
	    data[0][0] = xMin;
	    data[0][1] = 1.0;
	    data[1][0] = xMax;
	    data[1][1] = 1.0;
	    const Staircase empiricalCDF(data, "red", "solid", "s", oss);
            graphCDF.addDrawable(empiricalCDF);
            return graphCDF;
	  }
	// The other case: some elements are >= xMin. Find the index of the greatest element <= xMax. We start at size - 1 and stop when the element at index iMax is <= xMax or when iMax == iMin. In the latest case, it means either that all the elements that are >= xMin are also > xMax: the empirical CDF is constant over [xMin, xMax] and its value is iMin * step or that the element at index iMin is the only one in ]xMin, xMax]. It can be decided by comparing the value at index iMax and xMax.
	UnsignedLong iMax(size - 1);
	while ((iMax > iMin) && (sortedSample[iMax][0] > xMax)) --iMax;
	const NumericalScalar step(1.0 / size);
        // If iMax == iMin and the value at index iMax is > xMax, constant value
	if ((iMin == iMax) && (sortedSample[iMax][0] > xMax))
	  {
	    NumericalSample data(2, 2);
	    const NumericalScalar value(iMin * step);
	    data[0][0] = xMin;
	    data[0][1] = value;
	    data[1][0] = xMax;
	    data[1][1] = value;
	    const Staircase empiricalCDF(data, "red", "solid", "s", oss);
            graphCDF.addDrawable(empiricalCDF);
            return graphCDF;
	  }
	// Usual case: there is at least one point between xMin and xMax, in fact iMax-iMin+1 elements. We add 2 other points at xMin and xMax.
	NumericalSample data(iMax - iMin + 3, 2);
	data[0][0] = xMin;
	// The first point, value iMin * step
	data[0][1] = iMin * step;
	UnsignedLong iData(1);
	// All the other points, ending value (i + 1) * step
	for (UnsignedLong i = iMin; i <= iMax; ++i)
	  {
	    data[iData][0] = sortedSample[i][0];
	    data[iData][1] = (1.0 + i) * step;
	    ++iData;
	  }
	// Last point, value (iMax + 1) * step
	data[iData][0] = xMax;
	data[iData][1] = (iMax + 1) * step;
	const Staircase empiricalCDF(data, "red", "solid", "s", oss);
	graphCDF.addDrawable(empiricalCDF);
	return graphCDF;
      }


      /* Draw the Histogram of the Sample when its dimension is 1 */
      VisualTest::Graph VisualTest::DrawHistogram(const NumericalSample & sample,
						  const UnsignedLong BarNumber)
	/* throw(InvalidDimensionException, InvalidArgumentException) */
      {
      	if (sample.getDimension() != 1) throw InvalidDimensionException(HERE) << "Error: can draw a Histogram only if dimension equals 1, here dimension=" << sample.getDimension();
	if (BarNumber == 0) throw InvalidArgumentException(HERE) << "Error: cannot draw an Histogram with 0 bar";

	// Construct the histogram
	// It will extends from min to max, with BarNumber bars.
	NumericalScalar min(sample.getMin()[0]);
	// Small adjustment in order to have classes defined as [x_k, x_k+1[ intervalles
	NumericalScalar max(sample.getMax()[0]);
	if (max == min) throw InvalidArgumentException(HERE) << "Error: cannot draw an Histogram for a sample with constant realizations";
#define Epsilon 1e-10
	const NumericalScalar delta(Epsilon * (max - min));
	max += delta;
	min -= delta;
	NumericalSample data(BarNumber, 2);
	NumericalScalar h((max - min) / BarNumber);
	for(UnsignedLong i = 0; i < BarNumber; ++i)
	  {
	    data[i][0] = h;
	    data[i][1] = 0.0;
	  }
	const UnsignedLong size(sample.getSize());
	const NumericalScalar step(1.0 / h);
	for(UnsignedLong i = 0; i < size; ++i)
	  {
	    const UnsignedLong index((UnsignedLong)(floor((sample[i][0] - min) * step)));
	    ++data[index][1];
	  }
	const NumericalScalar inverseArea(1.0 / (h * sample.getSize()));
	for(UnsignedLong i = 0; i < BarNumber; ++i)
	  {
	    data[i][1] *= inverseArea;
	  }
	// Create an empty graph
	Graph graphHist("sample histogram", "realizations", "frequency", true, "topright");

	// Create the barplot
	OSS oss;
	oss << sample.getName() << " histogram";
	const BarPlot histHist(data, min, "purple", "shaded", "solid", oss);
	// Then, draw it
	graphHist.addDrawable(histHist);
	return graphHist;
      }

      /* Draw the Histogram of the Sample when its dimension is 1, Normal empirical rule for bin number */
      VisualTest::Graph VisualTest::DrawHistogram(const NumericalSample & sample)
	/* throw(InvalidDimensionException) */
      {
      	if (sample.getDimension() != 1) throw InvalidDimensionException(HERE) << "Error: can draw a Histogram only if dimension equals 1, here dimension=" << sample.getDimension();
	const NumericalScalar hOpt(sample.computeStandardDeviationPerComponent()[0] * pow(24.0 * sqrt(M_PI) / sample.getSize(), 1.0 / 3.0));
	return DrawHistogram(sample, (UnsignedLong)(ceil((sample.getMax()[0] - sample.getMin()[0]) / hOpt + 0.5)));
      }

      /* Draw the QQplot of the two Samples when its dimension is 1 */
      VisualTest::Graph VisualTest::DrawQQplot(const NumericalSample & sample1,
					       const NumericalSample & sample2,
					       const UnsignedLong pointNumber)
	/* throw(InvalidDimensionException, InvalidArgumentException) */
      {
	if (sample1.getDimension() != 1) throw InvalidDimensionException(HERE) << "Error: can draw a QQplot only if dimension equals 1, here dimension=" << sample1.getDimension();
	if (sample2.getDimension() != 1) throw InvalidDimensionException(HERE) << "Error: can draw a QQplot only if dimension equals 1, here dimension=" << sample2.getDimension();	
	NumericalSample data(pointNumber, 2);
	const NumericalScalar step(1.0 / pointNumber);
	for (UnsignedLong i = 0; i < pointNumber; ++i)
	  {
	    const NumericalScalar q((i + 0.5) * step);
	    data[i][0] = sample1.computeQuantilePerComponent(q)[0];
	    data[i][1] = sample2.computeQuantilePerComponent(q)[0];
	  }
	OSS oss;
	oss << sample1.getName() << " qqplot";
	const Curve curveQQplot(data, oss);
	Graph graphQQplot("two sample QQplot", "sample1", "sample2", true, "topright");
	// First, the bisectrice
	NumericalSample diagonal(2, 2);
	diagonal[0][0] = data[0][0];
	diagonal[0][1] = diagonal[0][0];
	diagonal[1][0] = data[pointNumber - 1][0];
	diagonal[1][1] = diagonal[1][0];
	Curve bisectrice(diagonal);
	bisectrice.setColor("grey");
	bisectrice.setLineStyle("dashed");
	graphQQplot.addDrawable(bisectrice);
	// Then the QQ plot
	graphQQplot.addDrawable(curveQQplot);
	return graphQQplot;
      }

      /* Draw the QQplot of one Sample and one Distribution when its dimension is 1 */
      VisualTest::Graph VisualTest::DrawQQplot(const NumericalSample & sample1,
					       const Distribution & dist,
					       const UnsignedLong pointNumber)
	/* throw(InvalidDimensionException, InvalidArgumentException) */
      {
	if (sample1.getDimension() != 1) throw InvalidDimensionException(HERE) << "Error: can draw a QQplot only if dimension equals 1, here dimension=" << sample1.getDimension();
	if (dist.getDimension() != 1) throw InvalidDimensionException(HERE) << "Error: can draw a QQplot only if dimension equals 1, here dimension=" << dist.getDimension();	
	NumericalSample data(pointNumber, 2);
	const NumericalScalar step(1.0 / pointNumber);
	for (UnsignedLong i = 0; i < pointNumber; ++i)
	  {
	    const NumericalScalar q((i + 0.5) * step);
	    data[i][0] = sample1.computeQuantilePerComponent(q)[0];
	    data[i][1] = dist.computeQuantile(q)[0];
	  }
	OSS oss;
	oss << sample1.getName() << " qqplot";
	const Curve curveQQplot(data, oss);
	Graph graphQQplot("sample versus model QQplot", "sample", "model", true, "topright");
	// First, the bisectrice
	NumericalSample diagonal(2, 2);
	diagonal[0][0] = data[0][0];
	diagonal[0][1] = diagonal[0][0];
	diagonal[1][0] = data[pointNumber - 1][0];
	diagonal[1][1] = diagonal[1][0];
	Curve bisectrice(diagonal);
	bisectrice.setColor("grey");
	bisectrice.setLineStyle("dashed");
	graphQQplot.addDrawable(bisectrice);
	// Then the QQ plot
	graphQQplot.addDrawable(curveQQplot);
	return graphQQplot;
      }


      /* Draw the Henry curve for one Sample when its dimension is 1 */
      VisualTest::Graph VisualTest::DrawHenryLine(const NumericalSample & sample)
	/* throw(InvalidDimensionException, InvalidArgumentException) */
      {
	if (sample.getDimension() != 1) throw InvalidDimensionException(HERE) << "Error: can draw a Henry line only if dimension equals 1, here dimension=" << sample.getDimension();

	const UnsignedLong size(sample.getSize());
	const Normal dist;
	
	const NumericalSample sortedSample(sample.sort(0));
	NumericalSample data(size, 2);
	const NumericalScalar step(1.0 / size);
	for (UnsignedLong i = 0; i < size; ++i)
	  {
	    data[i][0] = sortedSample[i][0];
	    data[i][1] = dist.computeQuantile((i + 0.5) * step)[0];
	  }
	OSS oss;
	oss << sample.getName() << " Henry Curve";
	const Curve curveHenry(data, oss);
	Graph graphHenry("Henry Curve", "sample", "model", true, "topright");
	// First, the moment-estimated line
	const NumericalScalar mean(sample.computeMean()[0]);
	const NumericalScalar std(sample.computeStandardDeviationPerComponent()[0]);
	NumericalSample diagonal(2, 2);
	diagonal[0][0] = data[0][0];
	diagonal[0][1] = (diagonal[0][0] - mean) / std;
	diagonal[1][0] = data[size - 1][0];
	diagonal[1][1] = (diagonal[1][0] - mean) / std;
	Curve bisectrice(diagonal);
	bisectrice.setColor("grey");
	bisectrice.setLineStyle("dashed");
	graphHenry.addDrawable(bisectrice);
	// Then, the Henry line
	graphHenry.addDrawable(curveHenry);
	return graphHenry;
      }


      /* Draw the clouds of one Sample and one model when its dimension is 2 */
      VisualTest::Graph VisualTest::DrawClouds(const NumericalSample & sample,
					       const Distribution & dist)
	/* throw(InvalidDimensionException, InvalidArgumentException) */
      {
	if (sample.getDimension() != 2) throw InvalidDimensionException(HERE) << "Error: can draw sample clouds only if dimension equals 2, here dimension=" << sample.getDimension();
	if (dist.getDimension() != 2) throw InvalidDimensionException(HERE) << "Error: can draw distribution clouds only if dimension equals 2, here dimension=" << dist.getDimension();	

	const NumericalSample distSample(dist.getNumericalSample(sample.getSize()));
	
	const Cloud sampleCloud(sample, "blue", "fsquare","Sample Cloud");	
	const Cloud distCloud(distSample, "red", "fsquare","Model Cloud");
	
	OSS oss;
	oss << sample.getName() << " 2D Clouds";

	Graph myGraph("two samples clouds", "x1", "x2", true,"topright");
	// Then, draw it
	myGraph.addDrawable(sampleCloud);
	myGraph.addDrawable(distCloud);

	return myGraph;
      }

      /* Draw the clouds of two Sample and one model when its dimension is 2 */
      VisualTest::Graph VisualTest::DrawClouds(const NumericalSample & sample1,
					       const NumericalSample & sample2)
	/* throw(InvalidDimensionException, InvalidArgumentException) */
      {
	if (sample1.getDimension() != 2) throw InvalidDimensionException(HERE) << "Error: can draw sample clouds only if dimension equals 2, here dimension=" << sample1.getDimension();
	if (sample2.getDimension() != 2) throw InvalidDimensionException(HERE) << "Error: can draw sample clouds only if dimension equals 2, here dimension=" << sample2.getDimension();	
	
	const Cloud sampleCloud1(sample1, "blue", "fsquare","Sample1 Cloud");	
	const Cloud sampleCloud2(sample2, "red", "fsquare","Sample2 Cloud");
	
	OSS oss;
	oss << sample1.getName() << " 2D Clouds";

	Graph myGraph("two samples clouds", "x1", "x2", true,"topright");
	// Then, draw it
	myGraph.addDrawable(sampleCloud1);
	myGraph.addDrawable(sampleCloud2);

	return myGraph;
      }

      /* Draw the visual test for the LM when its dimension is 1 */
      VisualTest::Graph VisualTest::DrawLMVisualTest(const NumericalSample & sample1,
						     const NumericalSample & sample2,
						     const LinearModel & lm)
	/* throw(InvalidDimensionException, InvalidArgumentException) */
      {
	if (sample1.getDimension() != 1) throw InvalidDimensionException(HERE) << "Error: can draw a LM visual test only if dimension equals 1, here dimension=" << sample1.getDimension();
	if (sample2.getDimension() != 1) throw InvalidDimensionException(HERE) << "Error: can draw a LM visual test only if dimension equals 1, here dimension=" << sample2.getDimension();
	if (sample1.getSize() != sample2.getSize()) throw InvalidArgumentException(HERE) << "Error: can draw a LM visual test only if sample 1 and sample 2 have the same size, here sample 1 size=" << sample1.getSize() << " and sample 2 size=" << sample2.getSize();

	const NumericalSample y(lm.getPredicted(sample1));
	const UnsignedLong size(sample1.getSize());

	NumericalSample data2(size, 2);
	for (UnsignedLong i = 0; i < size; ++i)
	  {
	    data2[i][0] = sample1[i][0];
	    data2[i][1] = sample2[i][0];
	  }
	NumericalSample data1(size, 2);
	for (UnsignedLong i = 0; i < size; ++i)
	  {
	    data1[i][0] = sample1[i][0];
	    data1[i][1] = y[i][0];
	  }
	OSS oss;
	oss << sample1.getName() << " LinearModel visualTest";
	const Curve curveLMTest(data1, oss);
	const Cloud cloudLMTest(data2, "red", "fsquare","Original Sample");

	Graph graphLMTest("original sample versus LM modeled one", "x", "y", true, "topright");
	graphLMTest.addDrawable(curveLMTest);
	graphLMTest.addDrawable(cloudLMTest);
	return graphLMTest;
      }

      /* Draw the visual test for the LM residual */
      VisualTest::Graph VisualTest::DrawLMResidualTest(const NumericalSample & sample1,
						       const NumericalSample & sample2,
						       const LinearModel & lm)
	/* throw(InvalidDimensionException, InvalidArgumentException) */
      {
	if (sample1.getDimension() != 1) throw InvalidDimensionException(HERE) << "Error: can draw a LM residual visual test only if dimension equals 1, here dimension=" << sample1.getDimension();
	if (sample2.getDimension() != 1) throw InvalidDimensionException(HERE) << "Error: can draw a LM residual visual test only if dimension equals 1, here dimension=" << sample2.getDimension();
	if (sample1.getSize() != sample2.getSize()) throw InvalidArgumentException(HERE) << "Error: can draw a LM residual visual test only if sample 1 and sample 2 have the same size, here sample 1 size=" << sample1.getSize() << " and sample 2 size=" << sample2.getSize();

	const NumericalSample y(lm.getResidual(sample1, sample2));
	const UnsignedLong size(sample1.getSize());
	NumericalSample data(size - 1, 2);
	for (UnsignedLong i = 0; i < size - 1; ++i)
	  {
	    data[i][0] = y[i][0];
	    data[i][1] = y[i+1][0];
	  }

	OSS oss;
	oss << sample1.getName() << " LinearModel residual Test";
	const Cloud cloudLMRTest(data, "red", "fsquare", oss);
	
	Graph graphLMRTest("residual(i) versus residual(i-1)", "redidual(i-1)", "residual(i)", true, "topright");
	graphLMRTest.addDrawable(cloudLMRTest);
	return graphLMRTest;
      }

    } // namespace StatTest
  } // namespace Uncertainty
} // namespace OpenTURNS
