//                                               -*- C++ -*-
/**
 *  @file  RootStrategyImplementation.cxx
 *  @brief 
 *
 *  (C) Copyright 2005-2010 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: dutka $
 *  @date:   $LastChangedDate: 2010-02-04 16:44:49 +0100 (jeu. 04 févr. 2010) $
 *  Id:      $Id: RootStrategyImplementation.cxx 1473 2010-02-04 15:44:49Z dutka $
 */
#include <cstdlib>

#include "RootStrategyImplementation.hxx"
#include "Brent.hxx"
#include "Exception.hxx"
#include "ResourceMap.hxx"

namespace OpenTURNS
{

  namespace Uncertainty 
  {

    namespace Algorithm
    {

      /**
       * @class RootStrategyImplementation
       */

      CLASSNAMEINIT(RootStrategyImplementation);

      typedef Base::Common::NotYetImplementedException NotYetImplementedException;
      typedef Base::Common::ResourceMap                ResourceMap;
      typedef Base::Solver::Brent                      Brent;

      /* Distance from the origin at which the radial density is negligible */
      const NumericalScalar RootStrategyImplementation::DefaultMaximumDistance = 8.0;//strtod(ResourceMap::GetInstance().get("RootStrategyImplementation-DefaultMaximumDistance").c_str(), NULL);
      /* Step-size for the detection of the sign changes along a direction */
      const NumericalScalar RootStrategyImplementation::DefaultStepSize = 1.0;//strtod(ResourceMap::GetInstance().get("RootStrategyImplementation-DefaultStepSize").c_str(), NULL);

      /* Default constructor */
      RootStrategyImplementation::RootStrategyImplementation()
	: Base::Common::PersistentObject(),
	  solver_(new Brent()),
	  maximumDistance_(DefaultMaximumDistance),
	  stepSize_(DefaultStepSize),
	  isAlreadyComputedOrignValue_(false),
	  originValue_(0.0)
      {
	// Nothing to do
      }

      /* Parameters constructor */
      RootStrategyImplementation::RootStrategyImplementation(const Solver & solver)
	: Base::Common::PersistentObject(),
	  solver_(solver),
	  maximumDistance_(DefaultMaximumDistance),
	  stepSize_(DefaultStepSize),
	  isAlreadyComputedOrignValue_(false),
	  originValue_(0.0)
      {
	// Nothing to do
      }

      /* Parameters constructor */
      RootStrategyImplementation::RootStrategyImplementation(const Solver & solver,
							     const NumericalScalar maximumDistance,
							     const NumericalScalar stepSize)
	: Base::Common::PersistentObject(),
	  solver_(solver),
	  maximumDistance_(maximumDistance),
	  stepSize_(stepSize),
	  isAlreadyComputedOrignValue_(false),
	  originValue_(0.0)
      {
	// Nothing to do
      }

      /* Virtual constructor */
      RootStrategyImplementation * RootStrategyImplementation::clone() const
      {
	return new RootStrategyImplementation(*this);
      }

      /* Solve gives all the roots found applying the root strategy */
      RootStrategyImplementation::NumericalScalarCollection RootStrategyImplementation::solve(const NumericalMathFunction & function,
											      const NumericalScalar value)
      {
	throw NotYetImplementedException(HERE);
      }

      /* Solver accessor */
      void RootStrategyImplementation::setSolver(const Solver & solver)
      {
	solver_ = solver;
      }

      RootStrategyImplementation::Solver RootStrategyImplementation::getSolver() const
      {
	return solver_;
      }

      /* Maximum distance accessor */
      void RootStrategyImplementation::setMaximumDistance(const NumericalScalar maximumDistance)
      {
	maximumDistance_ = maximumDistance;
      }

      NumericalScalar RootStrategyImplementation::getMaximumDistance() const
      {
	return maximumDistance_;
      }

      /* Step size accessor */
      void RootStrategyImplementation::setStepSize(const NumericalScalar stepSize)
      {
	stepSize_ = stepSize;
      }

      NumericalScalar RootStrategyImplementation::getStepSize() const
      {
	return stepSize_;
      }

      /* Value of the performance function at the origin accessor */
      void RootStrategyImplementation::setOriginValue(const NumericalScalar originValue)
      {
	originValue_ = originValue;
	isAlreadyComputedOrignValue_ = true;
      }

      /* We have to store the value of the performance function at the origin for two reasons:
         + to save computations, as this value will be reused for each direction
	 + to check if the origin is in the failure space or not, wich change the meaning of the roots
      */
      NumericalScalar RootStrategyImplementation::getOriginValue() const
      {
	if (!isAlreadyComputedOrignValue_) throw NotDefinedException(HERE);
	return originValue_;
      }

      /* String converter */
      String RootStrategyImplementation::__repr__() const
      {
	OSS oss;
	oss << "class=" << RootStrategyImplementation::GetClassName()
	    << " solver=" << solver_
	    << " maximumDistance=" << maximumDistance_
	    << " stepSize=" << stepSize_;
	return oss;
      }

    } /* namespace Algorithm */
  } /* namespace Uncertainty */
} /* namespace OpenTURNS */
