//                                               -*- C++ -*-
/**
 * @file  CorrelationAnalysis.hxx
 * @brief CorrelationAnalysis implements computation of correlation coefficients
 *
 *  (C) Copyright 2005-2010 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: dutka $
 *  @date:   $LastChangedDate: 2010-02-04 16:44:49 +0100 (jeu. 04 févr. 2010) $
 *  Id:      $Id: CorrelationAnalysis.hxx 1473 2010-02-04 15:44:49Z dutka $
 */

#ifndef OPENTURNS_CORRELATIONANALYSIS_HXX
#define OPENTURNS_CORRELATIONANALYSIS_HXX

#include "OTprivate.hxx"
#include "NumericalSample.hxx"
#include "NumericalPoint.hxx"
#include "NumericalMathFunction.hxx"
#include "SymmetricTensor.hxx"

namespace OpenTURNS
{
  namespace Base
  {
    namespace Stat
    {

      /**
       * @class CorrelationAnalysis
       *
       */

      class CorrelationAnalysis
      {
      public:

        typedef Type::NumericalPoint        NumericalPoint;
        typedef Func::NumericalMathFunction NumericalMathFunction;


#ifndef SWIG
        /**
         * @class SobolIndiceParameters
         *      Used to select which indices to compute
         */
        class SobolIndiceParameters
          : public Common::PersistentObject
        {
          static const UnsignedLong DefaultMaximumOrder;
          static const Bool DefaultTotalIndiceComputation;

          CLASSNAME;
        public :
          /** default constructor */
          SobolIndiceParameters();

          /** Parameters constructor */
          SobolIndiceParameters(const UnsignedLong maximumOrder,
                                const Bool computeTotalIndice);

          /** Virtual constructor */
          virtual SobolIndiceParameters * clone() const;

          /** Maximum Order accessor */
          UnsignedLong getMaximumOrder(void) const;
          void setMaximumOrder(const UnsignedLong maximumOrder);

          /** Total Indice accessor */
          Bool getTotalIndiceComputation(void) const;
          void setTotalIndiceComputation(const Bool totalIndiceComputation);

          /** String converter */
          String __repr__() const;

        protected :
          UnsignedLong maximumOrder_;
          Bool totalIndiceComputation_;

        } ; // class SobolIndiceParameters
#else
        typedef OT::Base::Stat::CorrelationAnalysisSobolIndiceParameters SobolIndiceParameters;
#endif


#ifndef SWIG
        /**
         * @class SobolIndiceResult
         *      Stores every Sobol' indices
         */
        class SobolIndiceResult
          : public Common::PersistentObject
        {
          CLASSNAME;

        public :
          typedef Type::NumericalPoint  NumericalPoint;
          typedef Type::SymmetricMatrix SymmetricMatrix;
          typedef Type::SymmetricTensor SymmetricTensor;

          /** default constructor */
          SobolIndiceResult();

          /** Virtual constructor */
          virtual SobolIndiceResult * clone() const;

          /** first order Sobol' indices accessor */
          NumericalPoint getFirstOrderIndice() const;
          void setFirstOrderIndice(NumericalPoint firstOrderIndice);

          /** second order Sobol' indices accessor */
          SymmetricMatrix getSecondOrderIndice() const;
          void setSecondOrderIndice(const SymmetricMatrix secondOrderIndice);

          /** third order Sobol' indices accessor */
          SymmetricTensor getThirdOrderIndice() const;
          void setThirdOrderIndice(const SymmetricTensor thirdOrderIndice);

          /** total order (order = dimension - 1) Sobol' indices accessor */
          NumericalPoint getTotalOrderIndice() const;
          void setTotalOrderIndice(const NumericalPoint totalOrderIndice);

          /** String converter */
          String __repr__() const;

        protected :
          NumericalPoint firstOrderIndice_;
          SymmetricMatrix secondOrderIndice_;
          SymmetricTensor thirdOrderIndice_;
          NumericalPoint totalOrderIndice_;

        } ; // class SobolIndiceResult
#else
        typedef OT::Base::Stat::CorrelationAnalysisSobolIndiceResult SobolIndiceResult;
#endif


        /** Default constructor */
        CorrelationAnalysis();

        /** Compute the Pearson correlation coefficient between the component number index of the input sample and the 1D outputSample */
        static NumericalScalar PearsonCorrelation(const NumericalSample & inputSample,
                                                  const NumericalSample & outputSample,
                                                  const UnsignedLong index = 0);

        /** Compute the Spearman correlation coefficient between the component number index of the input sample and the 1D outputSample */
        static NumericalScalar SpearmanCorrelation(const NumericalSample & inputSample,
                                                   const NumericalSample & outputSample,
                                                   const UnsignedLong index = 0);

        /** Compute the Standard Regression Coefficients (SRC) between the input sample and the output sample */
        static NumericalPoint SRC(const Base::Stat::NumericalSample & inputSample,
                                  const Base::Stat::NumericalSample & outputSample);

        /** Compute the Partial Correlation Coefficients (PCC) between the input sample and the output sample */
        static NumericalPoint PCC(const NumericalSample & inputSample,
                                  const NumericalSample & outputSample);

        /** Compute the Standard Rank Regression Coefficients (SRRC) between the input sample and the output sample */
        static NumericalPoint SRRC(const NumericalSample & inputSample,
                                   const NumericalSample & outputSample);

        /** Compute the Partial Rank Correlation Coefficients (PRCC) between the input sample and the output sample */
        static NumericalPoint PRCC(const NumericalSample & inputSample,
                                   const NumericalSample & outputSample);

        /** compute the Sobol' indices given two input samples and a 1d function */
        static SobolIndiceResult SobolIndice(const SobolIndiceParameters & sobolParameter,
                                             const NumericalSample & firstInputSample,
                                             const NumericalSample & secondInputSample,
                                             const NumericalMathFunction & function);

      }; /* class CorrelationAnalysis */

      typedef OpenTURNS::Base::Stat::CorrelationAnalysis::SobolIndiceParameters CorrelationAnalysisSobolIndiceParameters;
      typedef OpenTURNS::Base::Stat::CorrelationAnalysis::SobolIndiceResult CorrelationAnalysisSobolIndiceResult;

    } /* namespace Stat */
  } /* namespace Base */
} /* namespace OpenTURNS */

#endif /* OPENTURNS_CORRELATIONANALYSIS_HXX */
