//                                               -*- C++ -*-
/**
 * @file  ProductNumericalMathEvaluationImplementation.cxx
 * @brief The class that implements the composition between numerical
 *        math functions implementations
 *
 * (C) Copyright 2005-2010 EADS
 *
 * Permission to copy, use, modify, sell and distribute this software
 * is granted provided this copyright notice appears in all copies.
 * This software is provided "as is" without express or implied
 * warranty, and with no claim as to its suitability for any purpose.
 *
 *
 * \author $LastChangedBy: lebrun $
 * \date   $LastChangedDate: 2008-07-03 08:31:34 +0200 (jeu, 03 jui 2008) $
 */

#include "ProductNumericalMathEvaluationImplementation.hxx"
#include "PersistentObjectFactory.hxx"

namespace OpenTURNS {

  namespace Base {

    namespace Func {

      CLASSNAMEINIT(ProductNumericalMathEvaluationImplementation);

      static Common::Factory<ProductNumericalMathEvaluationImplementation> RegisteredFactory("ProductNumericalMathEvaluationImplementation");

      /* Default constructor */
      ProductNumericalMathEvaluationImplementation::ProductNumericalMathEvaluationImplementation(const EvaluationImplementation & p_leftEvaluation,
												 const EvaluationImplementation & p_rightEvaluation) /* throw(InvalidArgumentException) */
	: NumericalMathEvaluationImplementation(),
	  p_leftEvaluation_(p_leftEvaluation),
	  p_rightEvaluation_(p_rightEvaluation)
      {
	// Check if the dimensions of the left and right functions are compatible
	if ((p_leftEvaluation->getOutputDimension() != 1) || (p_leftEvaluation->getOutputDimension() != 1)) throw InvalidArgumentException(HERE) << "Error: the output dimension of the two functions must be 1 in a product.";
	if (p_leftEvaluation_->getInputDimension() != p_rightEvaluation_->getInputDimension()) throw InvalidArgumentException(HERE) << "Error: the two functions must have the same input dimension.";
	setDescription(p_leftEvaluation->getDescription());
      }

      /* Virtual constructor */
      ProductNumericalMathEvaluationImplementation * ProductNumericalMathEvaluationImplementation::clone() const
      {
	return new ProductNumericalMathEvaluationImplementation(*this);
      }

      /* Comparison operator */
      Bool ProductNumericalMathEvaluationImplementation::operator ==(const ProductNumericalMathEvaluationImplementation & other) const
      {
	return true;
      }
  
      /* String converter */
      String ProductNumericalMathEvaluationImplementation::__repr__() const {
	OSS oss;
	oss << "class=" << ProductNumericalMathEvaluationImplementation::GetClassName()
	    << " name=" << getName()
	    << " leftEvaluation=" << p_leftEvaluation_->__repr__()
	    << " rightEvaluation=" << p_rightEvaluation_->__repr__();
	return oss;
      }
  
      /* Operator () */
      ProductNumericalMathEvaluationImplementation::NumericalPoint ProductNumericalMathEvaluationImplementation::operator() (const NumericalPoint & in) const
	/* throw(InvalidArgumentException,InternalException) */
      {
	callsNumber_++;
	return p_leftEvaluation_->operator()(in)[0] * p_rightEvaluation_->operator()(in);
      }

      /* Operator () */
      ProductNumericalMathEvaluationImplementation::NumericalSample ProductNumericalMathEvaluationImplementation::operator() (const NumericalSample & inSample) const
	/* throw(InvalidArgumentException,InternalException) */
      {
	const UnsignedLong size(inSample.getSize());
	callsNumber_ += size;
	NumericalSample leftSample(p_leftEvaluation_->operator()(inSample));
	NumericalSample rightSample(p_rightEvaluation_->operator()(inSample));
	for (UnsignedLong i = 0; i < size; ++i)
	  {
	    leftSample[i][0] *= rightSample[i][0];
	  }
	return leftSample;
      }

      /* Parameters value and description accessor */
      ProductNumericalMathEvaluationImplementation::NumericalPointWithDescription ProductNumericalMathEvaluationImplementation::getParameters() const
      {
	NumericalPointWithDescription rightParameters(p_rightEvaluation_->getParameters());
	UnsignedLong rightDimension(rightParameters.getDimension());
	Description rightDescription(rightParameters.getDescription());
	NumericalPointWithDescription leftParameters(p_leftEvaluation_->getParameters());
	UnsignedLong leftDimension(leftParameters.getDimension());
	Description leftDescription(leftParameters.getDescription());
	UnsignedLong dimension(rightDimension + leftDimension);
	NumericalPointWithDescription parameters(dimension);
	Description description(dimension);
	UnsignedLong index(0);
	for (UnsignedLong i = 0; i < leftDimension; i++)
	  {
	    parameters[index] = leftParameters[i];
	    description[index] = leftDescription[i];
	    index++;
	  }
	for (UnsignedLong i = 0; i < rightDimension; i++)
	  {
	    parameters[index] = rightParameters[i];
	    description[index] = rightDescription[i];
	    index++;
	  }
	parameters.setDescription(description);
	return parameters;
      }

      void ProductNumericalMathEvaluationImplementation::setParameters(const NumericalPointWithDescription & parameters)
      {
	NumericalPointWithDescription rightParameters(p_rightEvaluation_->getParameters());
	UnsignedLong rightDimension(rightParameters.getDimension());
	NumericalPointWithDescription leftParameters(p_leftEvaluation_->getParameters());
	UnsignedLong leftDimension(leftParameters.getDimension());
	Description description(parameters.getDescription());
	Description rightDescription(rightDimension);
	Description leftDescription(leftDimension);
	UnsignedLong index(0);
	for (UnsignedLong i = 0; i < leftDimension; ++i)
	  {
	    leftParameters[i] = parameters[index];
	    leftDescription[i] = description[index];
	    index++;
	  }
	leftParameters.setDescription(leftDescription);
	p_leftEvaluation_->setParameters(leftParameters);
	for (UnsignedLong i = 0; i < rightDimension; ++i)
	  {
	    rightParameters[i] = parameters[index];
	    rightDescription[i] = description[index];
	    index++;
	  }
	rightParameters.setDescription(rightDescription);
	p_rightEvaluation_->setParameters(rightParameters);
      }

      /* Accessor for input point dimension */
      UnsignedLong ProductNumericalMathEvaluationImplementation::getInputDimension() const
	/* throw(InternalException) */
      {
	return p_leftEvaluation_->getInputDimension();
      }
      
      /* Accessor for output point dimension */
      UnsignedLong ProductNumericalMathEvaluationImplementation::getOutputDimension() const
	/* throw(InternalException) */
      {
	return 1;

      }
     
      /* Method save() stores the object through the StorageManager */
      void ProductNumericalMathEvaluationImplementation::save(StorageManager::Advocate & adv) const
      {
	NumericalMathEvaluationImplementation::save(adv);
	adv.saveAttribute( "leftEvaluation_", *p_leftEvaluation_ );
	adv.saveAttribute( "rightEvaluation_", *p_rightEvaluation_ );
      }

      /* Method load() reloads the object from the StorageManager */
      void ProductNumericalMathEvaluationImplementation::load(StorageManager::Advocate & adv)
      {
	NumericalMathEvaluationImplementation::load(adv);
	adv.saveAttribute( "leftEvaluation_", *p_leftEvaluation_ );
	adv.saveAttribute( "rightEvaluation_", *p_rightEvaluation_ );
      }

    } /* namespace Func */
  } /* namespace Base */
} /* namespace OpenTURNS */
