/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 *
 * Copyright 2010 Oracle and/or its affiliates. All rights reserved.
 *
 * Oracle and Java are registered trademarks of Oracle and/or its affiliates.
 * Other names may be trademarks of their respective owners.
 *
 * The contents of this file are subject to the terms of either the GNU
 * General Public License Version 2 only ("GPL") or the Common
 * Development and Distribution License("CDDL") (collectively, the
 * "License"). You may not use this file except in compliance with the
 * License. You can obtain a copy of the License at
 * http://www.netbeans.org/cddl-gplv2.html
 * or nbbuild/licenses/CDDL-GPL-2-CP. See the License for the
 * specific language governing permissions and limitations under the
 * License.  When distributing the software, include this License Header
 * Notice in each file and include the License file at
 * nbbuild/licenses/CDDL-GPL-2-CP.  Oracle designates this
 * particular file as subject to the "Classpath" exception as provided
 * by Oracle in the GPL Version 2 section of the License file that
 * accompanied this code. If applicable, add the following below the
 * License Header, with the fields enclosed by brackets [] replaced by
 * your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * If you wish your version of this file to be governed by only the CDDL
 * or only the GPL Version 2, indicate your decision by adding
 * "[Contributor] elects to include this software in this distribution
 * under the [CDDL or GPL Version 2] license." If you do not indicate a
 * single choice of license, a recipient has the option to distribute
 * your version of this file under either the CDDL, the GPL Version 2 or
 * to extend the choice of license to its licensees as provided above.
 * However, if you add GPL Version 2 code and therefore, elected the GPL
 * Version 2 license, then the option applies only if the new code is
 * made subject to such option by the copyright holder.
 *
 * Contributor(s):
 *
 * Portions Copyrighted 2010 Sun Microsystems, Inc.
 */

package org.netbeans.modules.j2ee.weblogic9.registration;

import java.io.File;
import java.io.IOException;
import java.util.Properties;
import java.util.logging.Level;
import java.util.logging.Logger;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import org.netbeans.modules.j2ee.deployment.common.api.Version;
import org.netbeans.modules.j2ee.deployment.plugins.api.InstanceProperties;
import org.netbeans.modules.j2ee.weblogic9.WLDeploymentFactory;
import org.netbeans.modules.j2ee.weblogic9.WLPluginProperties;
import org.netbeans.modules.j2ee.weblogic9.ui.wizard.WLInstantiatingIterator;
import org.openide.filesystems.FileObject;
import org.openide.filesystems.FileUtil;
import org.openide.util.NbBundle;
import org.openide.util.Utilities;

/**
 * Registers a WebLogic instance by creating instance file in cluster config
 * directory. Designed to be called from installer.
 * <p>
 * Sample command line<br>
 * java -cp platform/core/core.jar:platform/lib/boot.jar:platform/lib/org-openide-modules.jar:platform/core/org-openide-filesystems.jar:platform/lib/org-openide-util.jar:platform/lib/org-openide-util-lookup.jar:enterprise/modules/org-netbeans-modules-j2eeapis.jar:enterprise/modules/org-netbeans-modules-j2eeserver.jar:enterprise/modules/org-netbeans-modules-j2ee-weblogic9.jar org.netbeans.modules.j2ee.weblogic9.registration.AutomaticRegistration %lt;clusterDir&gt; &lt;serverDir&gt; &lt;domainDir&gt; &lt;username&gt; &lt;password&gt;
 *
 * @author Petr Hejl
 * @see #main(args)
 */
public class AutomaticRegistration {

    private static final Logger LOGGER = Logger.getLogger(AutomaticRegistration.class.getName());

    /**
     * Performs registration/uregistration of server instance. May also list
     * existing weblogic instances.
     *
     * Exit codes:<p>
     * <ul>
     *   <li> 1: could not hadle cluster folder
     *   <li> 2: could not find/create config/J2EE/InstalledServers folder
     *   <li> 3: could not find server dir
     *   <li> 4: could not find domain dir
     *   <li> 5: unsupported version of Weblogic
     *   <li> 6: could not recognize the domain
     *   <li> 7: empty domain name
     *   <li> 8: the domain is in production mode
     *   <li> 9: the domain version does not match the server version
     *   <li> 10: could not write registration FileObject
     * </ul>
     * @param args command line arguments
     * <ul>
     *  <li>--add cluster_path server_dir domain_dir username password
     *  <li>--remove cluster_path server_dir domain_dir
     *  <li>--list cluster_path
     * </ul>
     */
    public static void main(String[] args) {
        if (args.length <= 0) {
            printHelpAndExit();
        }

        if ("--add".equals(args[0])) {
            if (args.length < 6) {
                printHelpAndExit();
            }
            int status = registerWebLogicInstance(args[1], args[2], args[3],
                    args[4], args[5]);
            System.exit(status);
        } else if ("--remove".equals(args[0])) {
            if (args.length < 4) {
                printHelpAndExit();
            }
            int status = unregisterWebLogicInstance(args[1], args[2], args[3]);
            System.exit(status);
        } else if ("--list".equals(args[0])) {
            if (args.length < 2) {
                printHelpAndExit();
            }
            list(args[1]);
        } else {
            printHelpAndExit();
        }
    }

    private static void printHelpAndExit() {
        System.out.println("Available actions:");
        System.out.println("\t--add <clusterDir> <serverDir> <domainDir> <username> <password>");
        System.out.println("\t--remove <clusterDir> <serverDir> <domainDir>");
        System.out.println("\t--list <clusterDir>");
        System.exit(-1);
    }

    private static int registerWebLogicInstance(String clusterDirValue, String serverDirValue,
            String domainDirValue, String username, String password) {
        // tell the infrastructure that the userdir is cluster dir
        System.setProperty("netbeans.user", clusterDirValue); // NOI18N

        FileObject serverInstanceDir = FileUtil.getConfigFile("J2EE/InstalledServers"); // NOI18N

        if (serverInstanceDir == null) {
            LOGGER.log(Level.INFO, "Cannot register the default WebLogic server. The config/J2EE/InstalledServers folder cannot be created."); // NOI18N
            return 2;
        }

        File serverDir = FileUtil.normalizeFile(new File(serverDirValue));
        if (!serverDir.exists()) {
            LOGGER.log(Level.INFO, "Cannot register the default WebLogic server. "
                    + "The server directory " + serverDirValue // NOI18N
                    + " does not exist."); // NOI18N
            return 3;
        }

        File domainDir = FileUtil.normalizeFile(new File(domainDirValue));
        if (!domainDir.exists()) {
            LOGGER.log(Level.INFO, "Cannot register the default WebLogic server. "
                    + "The domain directory " + domainDirValue // NOI18N
                    + " does not exist."); // NOI18N
            return 4;
        }

        Version version = WLPluginProperties.getServerVersion(serverDir);
        if (!WLPluginProperties.isSupportedVersion(version)) {
            LOGGER.log(Level.INFO, "Cannot register the default WebLogic server. "
                    + " The version " + version + " is not supported."); // NOI18N
            return 5;
        }

        Properties properties = WLPluginProperties.getDomainProperties(domainDir.getAbsolutePath());
        if (properties.isEmpty()) {
            LOGGER.log(Level.INFO, "Cannot register the default WebLogic server. "
                    + " The domain " + domainDirValue + " is not valid domain."); // NOI18N
            return 6;
        }

        String name = properties.getProperty(WLPluginProperties.ADMIN_SERVER_NAME);
        String port = properties.getProperty(WLPluginProperties.PORT_ATTR);
        String host = properties.getProperty(WLPluginProperties.HOST_ATTR);
        String domainName = properties.getProperty(WLPluginProperties.DOMAIN_NAME);
        String versionString = properties.getProperty(WLPluginProperties.DOMAIN_VERSION);
        Version domainVersion = versionString != null ? Version.fromJsr277OrDottedNotationWithFallback(versionString) : null;
        Boolean isProductionMode = (Boolean) properties.get(WLPluginProperties.PRODUCTION_MODE);
        if ((name != null) && (!name.equals(""))) { // NOI18N
            // address and port have minOccurs=0 and are missing in 90
            // examples server
            port = (port == null || port.equals("")) // NOI18N
            ? Integer.toString(WLDeploymentFactory.DEFAULT_PORT)
                    : port;
            host = (host == null || host.equals("")) ? "localhost" // NOI18N
                    : host;
        } else {
            LOGGER.log(Level.INFO, "Cannot register the default WebLogic server. "
                    + " The domain name is empty."); // NOI18N
            return 7;
        }

        if (isProductionMode != null && isProductionMode.booleanValue()) {
            LOGGER.log(Level.INFO, "Cannot register the default WebLogic server. "
                    + " The domain is in production mode."); // NOI18N
            return 8;
        }

        if (domainVersion != null
                && version != null
                && !version.equals(domainVersion)) {
            LOGGER.log(Level.INFO, "Cannot register the default WebLogic server. "
                    + " The domain version does not match the server version."); // NOI18N
            return 9;
        }

        // build URL
        StringBuilder urlTmp = new StringBuilder();
        urlTmp.append(WLDeploymentFactory.URI_PREFIX);
        urlTmp.append(host).append(":"); // NOI18N
        urlTmp.append(port).append(":"); // NOI18N
        urlTmp.append(serverDir.getAbsolutePath()).append(":"); // NOI18N
        urlTmp.append(domainDir.getAbsolutePath());

        final String url = urlTmp.toString();

        // make sure the server is not registered yet
        for (FileObject fo : serverInstanceDir.getChildren()) {
            if (url.equals(fo.getAttribute(InstanceProperties.URL_ATTR))) {
                // the server is already registered, do nothing
                return 0;
            }
        }

        String displayName = generateUniqueDisplayName(serverInstanceDir, version);
        boolean ok = registerServerInstanceFO(serverInstanceDir, url, displayName,
                serverDir.getAbsolutePath(), domainDir.getAbsolutePath(), domainName,
                port, username, password);
        if (ok) {
            return 0;
        } else {
            return 10;
        }
    }

    private static int unregisterWebLogicInstance(String clusterDirValue,
            String serverDirValue, String domainDirValue) {
        // tell the infrastructure that the userdir is cluster dir
        System.setProperty("netbeans.user", clusterDirValue); // NOI18N

        // we could do this via registry, but the classspath would explode
        FileObject serverInstanceDir = FileUtil.getConfigFile("J2EE/InstalledServers"); // NOI18N

        if (serverInstanceDir == null) {
            LOGGER.log(Level.INFO, "The config/J2EE/InstalledServers folder does not exist."); // NOI18N
            return 2;
        }

        Pattern pattern = Pattern.compile(
                "^" + Pattern.quote(WLDeploymentFactory.URI_PREFIX) // NOI18N
                + "(.+):(.+):" // NOI18N
                + Pattern.quote(serverDirValue)
                + ":" + Pattern.quote(domainDirValue) + "$"); // NOI18N

        try {
            for (FileObject f : serverInstanceDir.getChildren()) {
                String url = f.getAttribute(InstanceProperties.URL_ATTR).toString();
                if (url != null) {
                    if (pattern.matcher(url).matches()) {
                        f.delete();
                        return 0;
                    }
                }
            }
        } catch (IOException ex) {
            LOGGER.log(Level.INFO, "Cannot unregister the default WebLogic server."); // NOI18N
            LOGGER.log(Level.INFO, null, ex);
            return 6;
        }
        return 0;
    }

    private static int list(String clusterDirValue) {
        // tell the infrastructure that the userdir is cluster dir
        System.setProperty("netbeans.user", clusterDirValue); // NOI18N

        // we could do this via registry, but the classspath would explode
        FileObject serverInstanceDir = FileUtil.getConfigFile("J2EE/InstalledServers"); // NOI18N

        if (serverInstanceDir == null) {
            LOGGER.log(Level.INFO, "The config/J2EE/InstalledServers folder does not exist."); // NOI18N
            return 2;
        }

        Pattern pattern = Pattern.compile(
                "^" + Pattern.quote(WLDeploymentFactory.URI_PREFIX) // NOI18N
                + "(.+):(.+):(.+):(.+)$"); // NOI18N

        for (FileObject f : serverInstanceDir.getChildren()) {
            String url = f.getAttribute(InstanceProperties.URL_ATTR).toString();
            if (url != null) {
                Matcher matcher = pattern.matcher(url);
                if (matcher.matches()) {
                    System.out.println(matcher.group(3) + " " + matcher.group(4));
                }
            }
        }
        return 0;
    }

    /**
     * Generates a unique display name for the specified version of WebLogic
     *
     * @param serverInstanceDir /J2EE/InstalledServers folder
     * @param version WebLogic version
     *
     * @return a unique display name for the specified version of WebLogic
     */
    private static String generateUniqueDisplayName(FileObject serverInstanceDir, Version version) {
        // find a unique display name
        String displayName = NbBundle.getMessage(WLDeploymentFactory.class, "LBL_OracleWebLogic", version);
        boolean unique = true;
        int i = 1;
        while (true) {
            for (FileObject fo : serverInstanceDir.getChildren()) {
                if (displayName.equals(fo.getAttribute(InstanceProperties.DISPLAY_NAME_ATTR))) {
                    // there is already some server of the same name
                    unique = false;
                    break;
                }
            }
            if (unique) {
                break;
            }
            displayName = NbBundle.getMessage(WLDeploymentFactory.class, "LBL_OracleWebLogicAlt", version, i++);
            unique = true;
        };
        return displayName;
    }

    /**
     * Registers the server instance file object and set the default properties.
     *
     * @param serverInstanceDir /J2EE/InstalledServers folder
     * @param url server instance url/ID
     * @param displayName display name
     */
    private static boolean registerServerInstanceFO(FileObject serverInstanceDir, String url,
            String displayName, String serverRoot, String domainRoot, String domainName,
            String port, String username, String password) {

        String name = FileUtil.findFreeFileName(serverInstanceDir, "weblogic_autoregistered_instance", null); // NOI18N
        FileObject instanceFO;
        try {
            instanceFO = serverInstanceDir.createData(name);
            instanceFO.setAttribute(InstanceProperties.URL_ATTR, url);
            instanceFO.setAttribute(InstanceProperties.USERNAME_ATTR, username);
            instanceFO.setAttribute(InstanceProperties.PASSWORD_ATTR, password);
            instanceFO.setAttribute(InstanceProperties.DISPLAY_NAME_ATTR, displayName);
            instanceFO.setAttribute(InstanceProperties.HTTP_PORT_NUMBER, port);
            instanceFO.setAttribute(WLPluginProperties.SERVER_ROOT_ATTR, serverRoot);
            instanceFO.setAttribute(WLPluginProperties.DOMAIN_ROOT_ATTR, domainRoot);
            instanceFO.setAttribute(WLPluginProperties.DEBUGGER_PORT_ATTR,
                    WLInstantiatingIterator.DEFAULT_DEBUGGER_PORT);
            instanceFO.setAttribute(WLPluginProperties.DOMAIN_NAME, domainName);
            instanceFO.setAttribute(WLPluginProperties.PORT_ATTR, port);
            if (Utilities.isMac()) {
                instanceFO.setAttribute(WLPluginProperties.MEM_OPTS,
                        WLInstantiatingIterator.DEFAULT_MAC_MEM_OPTS);
            }
            return true;
        } catch (IOException e) {
            LOGGER.log(Level.INFO, "Cannot register the default WebLogic server."); // NOI18N
            LOGGER.log(Level.INFO, null, e);
        }
        return false;
    }
}
