/*
 * NASPRO - The NASPRO Architecture for Sound Processing
 * Core library
 *
 * Copyright (C) 2007-2012 NASPRO core development team
 *
 * See the COPYING file for license conditions.
 *
 * Part of the nacore_fs_dir_open() function is based on the dirent_buf_size()
 * function under:
 *
 * Copyright (C) 2005 Ben Hutchings
 *
 * Its license (see http://womble.decadent.org.uk/readdir_r-advisory.html)
 * allows to distribute the code under our license conditions.
 */

#define _POSIX_C_SOURCE	200809L
#define _BSD_SOURCE

#include "internal.h"

#include <assert.h>

#include <sys/types.h>
#include <unistd.h>
#include <dirent.h>

struct _nacore_fs_dir
  {
	DIR	*dir;
	size_t	 name_max;
  };

_NACORE_DEF const char * const nacore_fs_dir_sep = "/";

_NACORE_DEF nacore_fs_dir
nacore_fs_dir_open(const char *name)
{
	nacore_fs_dir ret;
	int fd;
	int errsv;
	long name_max;

	ret = malloc(sizeof(struct _nacore_fs_dir));
	if (ret == NULL)
	  {
		errno = ENOMEM;
		return NULL;
	  }

	ret->dir = (DIR *)opendir(name);
	if (ret->dir == NULL)
		goto opendir_err;

	fd = dirfd(ret->dir);
	assert(fd != -1);

	errsv = errno;
	errno = 0;
	name_max = fpathconf(fd, _PC_NAME_MAX);
	if (name_max == -1)
	  {
		assert(errno == 0);
#ifdef NAME_MAX
		name_max = (NAME_MAX > 255) ? NAME_MAX : 255;	
#else
		errno = ENOMEM;
		goto fpathconf_err;
#endif
	  }
	errno = errsv;

	ret->name_max = offsetof(struct dirent, d_name) + name_max + 1;
	ret->name_max = (ret->name_max > sizeof(struct dirent))
			? ret->name_max : sizeof(struct dirent);

	return ret;

#ifndef NAME_MAX
fpathconf_err:
	closedir(ret->dir);
#endif
opendir_err:
	errsv = errno;
	free(ret);
	errno = errsv;
	return NULL;
}

_NACORE_DEF nacore_fs_dir_entry
nacore_fs_dir_get_next_entry(nacore_fs_dir dir)
{
	nacore_fs_dir_entry ret;
	struct dirent *entry;

	ret = malloc(dir->name_max);
	if (ret == NULL)
	  {
		errno = ENOMEM;
		return NULL;
	  }

	do
	  {
		readdir_r(dir->dir, (struct dirent *)ret, &entry);
		if (entry == NULL)
		  {
			free(ret);
			return NULL;
		  }
	  }
	while ((strcmp(entry->d_name, ".") == 0)
	       || (strcmp(entry->d_name, "..") == 0));

	return ret;
}

_NACORE_DEF int
nacore_fs_dir_close(nacore_fs_dir dir)
{
	if (closedir(dir->dir) != 0)
		return EINTR;

	free(dir);

	return 0;
}

_NACORE_DEF const char *
nacore_fs_dir_entry_get_name(nacore_fs_dir_entry entry)
{
	return ((struct dirent *)entry)->d_name;
}

_NACORE_DEF void
nacore_fs_dir_entry_free(nacore_fs_dir_entry entry)
{
	free(entry);
}
