# -------------------------------------------------------------------------
#     This file is part of mMass - the spectrum analysis tool for MS.
#     Copyright (C) 2005-07 Martin Strohalm <mmass@biographics.cz>

#     This program is free software; you can redistribute it and/or modify
#     it under the terms of the GNU General Public License as published by
#     the Free Software Foundation; either version 2 of the License, or
#     (at your option) any later version.

#     This program is distributed in the hope that it will be useful,
#     but WITHOUT ANY WARRANTY; without even the implied warranty of
#     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#     GNU General Public License for more details.

#     Complete text of GNU GPL can be found in the file LICENSE in the
#     main directory of the program
# -------------------------------------------------------------------------

# Function: Mascot - Common dialog definition

# load libs
import wx
import string
import webbrowser
import tempfile
import os.path

# load modules
from nucleus import mwx


class dlgMascotSearch(wx.Dialog):
    """ Mascot - Common dialog definitions. """

    # ----
    def __init__(self, parent, search, title, peaklist, parameters, defaults):

        self.searchType = search
        self.title = title
        self.peaklist = peaklist
        self.parameters = parameters
        self.defaults = defaults

        # make title
        dialogTitle = 'Mascot Search - '
        if self.searchType == 'pmf':
            dialogTitle += 'Peptide Mass Fingerprint'
        elif self.searchType == 'seq':
            dialogTitle += 'Sequence Query'
        elif self.searchType == 'msms':
            dialogTitle += 'MS/MS Ion Search'

        # init dialog
        wx.Dialog.__init__(self, parent, -1, dialogTitle, style=wx.DEFAULT_DIALOG_STYLE)

        # make and rise dialog
        self.makeDialog()
    # ----


    # ----
    def makeDialog(self):
        """ Make search dialog."""

        # pack items
        mainSizer = wx.BoxSizer(wx.VERTICAL)

        if wx.Platform == '__WXMAC__':
            grid = wx.GridBagSizer(8,20)
            
            grid.Add(self.makeSearchInfoBox(), (0,0), (1,2), flag=wx.EXPAND)
            grid.Add(self.makeSequenceBox(), (1,0), (1,2), flag=wx.EXPAND)
            grid.Add(self.makeFixedModBox(), (2,0), flag=wx.EXPAND)
            grid.Add(self.makeVariableModBox(), (2,1), flag=wx.EXPAND)
            grid.Add(self.makeMassBox(), (3,0), flag=wx.EXPAND)
            grid.Add(self.makeQueryBox(), (3,1), flag=wx.EXPAND)
            grid.Add(self.makeResultsBox(), (4,0), (1,2), flag=wx.EXPAND)
            grid.AddGrowableCol(1)
            grid.AddGrowableCol(2)
            
            mainSizer.Add(grid, 0, wx.EXPAND|wx.TOP|wx.LEFT|wx.RIGHT, 20)
            mainSizer.Add(self.makeButtonBox(), 0, wx.ALL|wx.ALIGN_CENTER, 20)
        else:          
            grid = wx.GridBagSizer(5,5)

            grid.Add(self.makeSearchInfoBox(), (0,0), (1,2), flag=wx.EXPAND)
            grid.Add(self.makeSequenceBox(), (1,0), (1,2), flag=wx.EXPAND)
            grid.Add(self.makeFixedModBox(), (2,0), flag=wx.EXPAND)
            grid.Add(self.makeVariableModBox(), (2,1), flag=wx.EXPAND)
            grid.Add(self.makeMassBox(), (3,0), flag=wx.EXPAND)
            grid.Add(self.makeQueryBox(), (3,1), flag=wx.EXPAND)
            grid.Add(self.makeResultsBox(), (4,0), (1,2), flag=wx.EXPAND)
            grid.AddGrowableCol(1)
            grid.AddGrowableCol(2)

            mainSizer.Add(grid, 0, wx.EXPAND|wx.TOP|wx.LEFT|wx.RIGHT, 3)
            mainSizer.Add(self.makeButtonBox(), 0, wx.ALL|wx.ALIGN_CENTER, 5)

        # fit layout
        mainSizer.Fit(self)
        self.SetSizer(mainSizer)
        self.Centre()
    # ----


    # ----
    def makeSearchInfoBox(self):
        """ Make box for search title and user info. """

        # make items
        mainBox = wx.StaticBoxSizer(wx.StaticBox(self, -1, "Search title and user information"), wx.VERTICAL)
        grid = mwx.GridBagSizer()

        title_label = wx.StaticText(self, -1, "Search title: ")
        self.title_value = wx.TextCtrl(self, -1, self.title, size=(100, -1))

        username_label = wx.StaticText(self, -1, "Your name: ")
        self.username_value = wx.TextCtrl(self, -1, self.defaults['username'], size=(170, -1))

        email_label = wx.StaticText(self, -1, " E-mail: ")
        self.email_value = wx.TextCtrl(self, -1, self.defaults['email'], size=(175, -1))

        # pack items
        grid.Add(title_label, (0, 0), flag=wx.ALIGN_CENTER_VERTICAL|wx.ALIGN_RIGHT)
        grid.Add(self.title_value, (0, 1), (1, 3), flag=wx.EXPAND|wx.ALIGN_CENTER_VERTICAL)

        grid.Add(username_label, (1, 0), flag=wx.ALIGN_CENTER_VERTICAL|wx.ALIGN_RIGHT)
        grid.Add(self.username_value, (1, 1), flag=wx.EXPAND|wx.ALIGN_CENTER_VERTICAL)

        grid.Add(email_label, (1, 2), flag=wx.ALIGN_CENTER_VERTICAL|wx.ALIGN_RIGHT)
        grid.Add(self.email_value, (1, 3), flag=wx.EXPAND|wx.ALIGN_CENTER_VERTICAL)

        grid.AddGrowableCol(1)
        grid.AddGrowableCol(3)

        if wx.Platform == '__WXMAC__':
            mainBox.Add(grid, 1, wx.EXPAND, 0)
        else:
            mainBox.Add(grid, 1, wx.EXPAND|wx.ALL, 5)

        return mainBox
    # ----


    # ----
    def makeSequenceBox(self):
        """ Make box for sequence parameters. """

        # make items
        mainBox = wx.StaticBoxSizer(wx.StaticBox(self, -1, "Sequence parameters"), wx.VERTICAL)
        grid = mwx.GridBagSizer()

        taxonomy_label = wx.StaticText(self, -1, "Taxonomy: ")
        self.taxonomy_combo = wx.ComboBox(self, -1, size=(120, -1), choices=self.parameters['taxonomy'], style=wx.CB_READONLY)

        database_label = wx.StaticText(self, -1, "Database: ")
        self.database_combo = wx.ComboBox(self, -1, size=(120, -1), choices=self.parameters['database'], style=wx.CB_READONLY)

        enzyme_label = wx.StaticText(self, -1, " Enzyme: ")
        self.enzyme_combo = wx.ComboBox(self, -1, size=(120, -1), choices=self.parameters['enzyme'], style=wx.CB_READONLY)

        partials_label = wx.StaticText(self, -1, " Partials: ")
        self.partials_combo = wx.ComboBox(self, -1, size=(55, -1), choices=self.parameters['partials'], style=wx.CB_READONLY)

        # pack items
        grid.Add(taxonomy_label, (0, 0), flag=wx.ALIGN_CENTER_VERTICAL|wx.ALIGN_RIGHT)
        grid.Add(self.taxonomy_combo, (0, 1), (1, 5), flag=wx.EXPAND|wx.ALIGN_CENTER_VERTICAL)

        grid.Add(database_label, (1, 0), flag=wx.ALIGN_CENTER_VERTICAL|wx.ALIGN_RIGHT)
        grid.Add(self.database_combo, (1, 1), flag=wx.EXPAND|wx.ALIGN_CENTER_VERTICAL)

        grid.Add(enzyme_label, (1, 2), flag=wx.ALIGN_CENTER_VERTICAL|wx.ALIGN_RIGHT)
        grid.Add(self.enzyme_combo, (1, 3), flag=wx.EXPAND|wx.ALIGN_CENTER_VERTICAL)

        grid.Add(partials_label, (1, 4), flag=wx.ALIGN_CENTER_VERTICAL|wx.ALIGN_RIGHT)
        grid.Add(self.partials_combo, (1, 5), flag=wx.ALIGN_CENTER_VERTICAL)
        
        grid.AddGrowableCol(1)
        grid.AddGrowableCol(3)

        if wx.Platform == '__WXMAC__':
            mainBox.Add(grid, 1, wx.EXPAND, 0)
        else:
            mainBox.Add(grid, 1, wx.EXPAND|wx.ALL, 5)

        # set defaults
        if self.defaults['taxonomy'] in self.parameters['taxonomy']:
            self.taxonomy_combo.Select(self.parameters['taxonomy'].index(self.defaults['taxonomy']))
        else:
            self.taxonomy_combo.Select(0)

        if self.defaults['database'] in self.parameters['database']:
            self.database_combo.Select(self.parameters['database'].index(self.defaults['database']))
        else:
            self.database_combo.Select(0)

        if self.defaults['enzyme'] in self.parameters['enzyme']:
            self.enzyme_combo.Select(self.parameters['enzyme'].index(self.defaults['enzyme']))
        else:
            self.enzyme_combo.Select(0)

        if str(self.defaults['partials']) in self.parameters['partials']:
            self.partials_combo.Select(self.parameters['partials'].index(str(self.defaults['partials'])))
        else:
            self.partials_combo.Select(0)

        return mainBox
    # ----


    # ----
    def makeFixedModBox(self):
        """ Make box for fixed modifications. """

        # make items
        mainBox = wx.StaticBoxSizer(wx.StaticBox(self, -1, "Fixed modifications"), wx.VERTICAL)
        self.fixedMod_listbox = wx.ListBox(self, -1, size=(170, 80), choices=self.parameters['mods'], style=wx.LB_EXTENDED)

        # pack items
        if wx.Platform == '__WXMAC__':
            mainBox.Add(self.fixedMod_listbox, 1, wx.EXPAND, 0)
        else:
            mainBox.Add(self.fixedMod_listbox, 1, wx.EXPAND|wx.ALL, 5)

        # set defaults
        for mod in self.defaults['fixedmod'].split(';'):
            if mod in self.parameters['mods']:
                self.fixedMod_listbox.Select(self.parameters['mods'].index(mod))

        return mainBox
    # ----


    # ----
    def makeVariableModBox(self):
        """ Make box for variable modifications. """

        # make items
        mainBox = wx.StaticBoxSizer(wx.StaticBox(self, -1, "Variable modifications"), wx.VERTICAL)
        self.variableMod_listbox = wx.ListBox(self, -1, size=(170, 80), choices=self.parameters['mods'], style=wx.LB_EXTENDED)

        # pack items
        if wx.Platform == '__WXMAC__':
            mainBox.Add(self.variableMod_listbox, 1, wx.EXPAND, 0)
        else:
            mainBox.Add(self.variableMod_listbox, 1, wx.EXPAND|wx.ALL, 5)

        # set defaults
        for mod in self.defaults['variablemod'].split(';'):
            if mod in self.parameters['mods']:
                self.variableMod_listbox.Select(self.parameters['mods'].index(mod))

        return mainBox
    # ----


    # ----
    def makeMassBox(self):
        """ Make box for masstype and tolerance. """

        # make items
        mainBox = wx.StaticBoxSizer(wx.StaticBox(self, -1, "Mass parameters"), wx.VERTICAL)
        grid = mwx.GridBagSizer()

        proteinMass_label = wx.StaticText(self, -1, "Protein mass: ")
        self.proteinMass_value = wx.TextCtrl(self, -1, size=(90, -1), validator=mwx.txtValidator('float'))
        proteinMass_units = wx.StaticText(self, -1, " kDa")

        if self.searchType == 'msms':

            pepMass_label = wx.StaticText(self, -1, "Precursor mass: ")
            self.pepMass_value = wx.TextCtrl(self, -1, size=(90, -1), validator=mwx.txtValidator('float'))
            pepMass_units = wx.StaticText(self, -1, " m/z")

        massType_label = wx.StaticText(self, -1, "Mass type: ")
        self.massType_combo = wx.ComboBox(self, -1, size=(90, -1), choices=self.parameters['masstype'], style=wx.CB_READONLY)

        charge_label = wx.StaticText(self, -1, "Charge: ")
        if self.searchType == 'pmf':
            self.charge_combo = wx.ComboBox(self, -1, size=(90, -1), choices=self.parameters['pepcharge'], style=wx.CB_READONLY)
        elif self.searchType in ('seq', 'msms'):
            self.charge_combo = wx.ComboBox(self, -1, size=(90, -1), choices=self.parameters['msmscharge'], style=wx.CB_READONLY)

        pepTol_label = wx.StaticText(self, -1, "Peptide tol.: ")
        self.pepTol_value = wx.TextCtrl(self, -1, str(self.defaults['peptol']), size=(90, -1), validator=mwx.txtValidator('float'))
        self.pepTolUnits_combo = wx.ComboBox(self, -1, size=(60, -1), choices=self.parameters['peptolunits'], style=wx.CB_READONLY)

        if self.searchType in ('seq', 'msms'):

            msmsTol_label = wx.StaticText(self, -1, "MS/MS tol.: ")
            self.msmsTol_value = wx.TextCtrl(self, -1, str(self.defaults['msmstol']), size=(90, -1), validator=mwx.txtValidator('float'))
            self.msmsTolUnits_combo = wx.ComboBox(self, -1, size=(60, -1), choices=self.parameters['msmstolunits'], style=wx.CB_READONLY)

            instrument_label = wx.StaticText(self, -1, "Instrument: ")
            self.instrument_combo = wx.ComboBox(self, -1, size=(90, -1), choices=self.parameters['instrument'], style=wx.CB_READONLY)

            icat_label = wx.StaticText(self, -1, "ICAT: ")
            self.icat_checkbox = wx.CheckBox(self, -1, "", style=wx.ALIGN_RIGHT)

        # pack items
        row = 0

        grid.Add(proteinMass_label, (row, 0), flag=wx.ALIGN_CENTER_VERTICAL|wx.ALIGN_RIGHT)
        grid.Add(self.proteinMass_value, (row, 1), flag=wx.ALIGN_CENTER_VERTICAL)
        grid.Add(proteinMass_units, (row, 2), flag=wx.ALIGN_CENTER_VERTICAL)
        row += 1

        if self.searchType == 'msms':

            grid.Add(pepMass_label, (row, 0), flag=wx.ALIGN_CENTER_VERTICAL|wx.ALIGN_RIGHT)
            grid.Add(self.pepMass_value, (row, 1), flag=wx.ALIGN_CENTER_VERTICAL)
            grid.Add(pepMass_units, (row, 2), flag=wx.ALIGN_CENTER_VERTICAL)
            row += 1

        grid.Add(massType_label, (row, 0), flag=wx.ALIGN_CENTER_VERTICAL|wx.ALIGN_RIGHT)
        grid.Add(self.massType_combo, (row, 1), flag=wx.ALIGN_CENTER_VERTICAL)
        row += 1

        grid.Add(charge_label, (row, 0), flag=wx.ALIGN_CENTER_VERTICAL|wx.ALIGN_RIGHT)
        grid.Add(self.charge_combo, (row, 1), flag=wx.ALIGN_CENTER_VERTICAL)
        row += 1

        grid.Add(pepTol_label, (row, 0), flag=wx.ALIGN_CENTER_VERTICAL|wx.ALIGN_RIGHT)
        grid.Add(self.pepTol_value, (row, 1), flag=wx.ALIGN_CENTER_VERTICAL)
        grid.Add(self.pepTolUnits_combo, (row, 2), flag=wx.ALIGN_CENTER_VERTICAL)
        row += 1

        if self.searchType in ('seq', 'msms'):

            grid.Add(msmsTol_label, (row, 0), flag=wx.ALIGN_CENTER_VERTICAL|wx.ALIGN_RIGHT)
            grid.Add(self.msmsTol_value, (row, 1), flag=wx.ALIGN_CENTER_VERTICAL)
            grid.Add(self.msmsTolUnits_combo, (row, 2), flag=wx.ALIGN_CENTER_VERTICAL)
            row += 1

            grid.Add(instrument_label, (row, 0), flag=wx.ALIGN_CENTER_VERTICAL|wx.ALIGN_RIGHT)
            grid.Add(self.instrument_combo, (row, 1), (1, 2), flag=wx.EXPAND|wx.ALIGN_CENTER_VERTICAL)
            row += 1

            grid.Add(icat_label, (row, 0), flag=wx.ALIGN_CENTER_VERTICAL|wx.ALIGN_RIGHT)
            grid.Add(self.icat_checkbox, (row, 1), flag=wx.ALIGN_CENTER_VERTICAL)
            row += 1

        if wx.Platform == '__WXMAC__':
            mainBox.Add(grid, 1, wx.EXPAND, 0)
        else:
            mainBox.Add(grid, 1, wx.EXPAND|wx.ALL, 5)

        # set defaults
        if self.defaults['masstype'] in self.parameters['masstype']:
            self.massType_combo.Select(self.parameters['masstype'].index(self.defaults['masstype']))
        else:
            self.massType_combo.Select(0)

        if self.defaults['pepcharge'] in self.parameters['pepcharge']:
            self.charge_combo.Select(self.parameters['pepcharge'].index(self.defaults['pepcharge']))
        else:
            self.charge_combo.Select(0)

        if self.defaults['peptolunits'] in self.parameters['peptolunits']:
            self.pepTolUnits_combo.Select(self.parameters['peptolunits'].index(self.defaults['peptolunits']))
        else:
            self.pepTolUnits_combo.Select(0)

        if self.searchType in ('seq', 'msms'):

            if self.defaults['msmscharge'] in self.parameters['msmscharge']:
                self.charge_combo.Select(self.parameters['msmscharge'].index(self.defaults['msmscharge']))
            else:
                self.charge_combo.Select(0)

            if self.defaults['msmstolunits'] in self.parameters['msmstolunits']:
                self.msmsTolUnits_combo.Select(self.parameters['msmstolunits'].index(self.defaults['msmstolunits']))
            else:
                self.msmsTolUnits_combo.Select(0)

            if self.defaults['instrument'] in self.parameters['instrument']:
                self.instrument_combo.Select(self.parameters['instrument'].index(self.defaults['instrument']))
            else:
                self.instrument_combo.Select(0)

            self.icat_checkbox.SetValue(self.defaults['icat'])

        return mainBox
    # ----


    # ----
    def makeQueryBox(self):
        """ Make box for peaklist/query. """

        # make items
        mainBox = wx.StaticBoxSizer(wx.StaticBox(self, -1, "Peaklist / Query"), wx.VERTICAL)
        self.query_value = wx.TextCtrl(self, -1, self.peaklist, size=(170, 100), style=wx.TE_MULTILINE|wx.TE_DONTWRAP)

        # pack items
        if wx.Platform == '__WXMAC__':
            mainBox.Add(self.query_value, 1, wx.EXPAND, 0)
        else:
            mainBox.Add(self.query_value, 1, wx.EXPAND|wx.ALL, 5)

        return mainBox
    # ----


    # ----
    def makeResultsBox(self):
        """ Make box for results options. """

        # make items
        mainBox = wx.StaticBoxSizer(wx.StaticBox(self, -1, "Results options"), wx.HORIZONTAL)

        hits_label = wx.StaticText(self, -1, "Number of hits to show:")
        self.hits_combo = wx.ComboBox(self, -1, size=(80, -1), choices=self.parameters['showhits'], style=wx.CB_READONLY)

        self.overview_checkbox = wx.CheckBox(self, -1, "Overview")

        # pack items
        if wx.Platform == '__WXMAC__':
            mainBox.Add(hits_label, 0, 0)
            mainBox.Add(self.hits_combo, 0, wx.LEFT, 5)
            mainBox.Add(wx.StaticText(self, -1, ""), 1)
            mainBox.Add(self.overview_checkbox, 0, 0)
        else:
            mainBox.Add(hits_label, 0, wx.ALL, 5)
            mainBox.Add(self.hits_combo, 0, wx.BOTTOM, 5)
            mainBox.Add(wx.StaticText(self, -1, ""), 1)
            mainBox.Add(self.overview_checkbox, 0, wx.ALL, 5)

        # set defaults
        if str(self.defaults['showhits']) in self.parameters['showhits']:
            self.hits_combo.Select(self.parameters['showhits'].index(str(self.defaults['showhits'])))
        else:
            self.hits_combo.Select(0)

        self.overview_checkbox.SetValue(self.defaults['overview'])

        return mainBox
    # ----


    # ----
    def makeButtonBox(self):
        """ Make box for main buttons. """

        # make items
        Save_button = wx.Button(self, -1, "Save Defaults")
        Search_button = wx.Button(self, -1, "Search")
        Close_button = wx.Button(self, wx.ID_CANCEL, "Close")

        # pack items
        buttonBox = wx.BoxSizer(wx.HORIZONTAL)
        buttonBox.Add(Save_button, 0, wx.ALL, 5)
        buttonBox.Add(Search_button, 0, wx.ALL, 5)
        buttonBox.Add(Close_button, 0, wx.ALL, 5)

        # set events
        Save_button.Bind(wx.EVT_BUTTON, self.onSave)
        Search_button.Bind(wx.EVT_BUTTON, self.onSearch)
        Close_button.Bind(wx.EVT_BUTTON, self.onClose)

        return buttonBox
    # ----


    # ----
    def onSave(self, evt):
        """ Save current values as default. """

        # get and validate form params
        ctrlData = self.getData()

        # store params to config
        if ctrlData:
            self.defaults['database'] = ctrlData['database']
            self.defaults['email'] = ctrlData['email']
            self.defaults['enzyme'] = ctrlData['enzyme']
            self.defaults['fixedmod'] = string.join(ctrlData['fixedmod'], ';')
            self.defaults['masstype'] = ctrlData['masstype']
            self.defaults['overview'] = ctrlData['overview']
            self.defaults['partials'] = ctrlData['partials']
            self.defaults['peptol'] = ctrlData['peptol']
            self.defaults['peptolunits'] = ctrlData['peptolunits']
            self.defaults['showhits'] = ctrlData['showhits']
            self.defaults['taxonomy'] = ctrlData['taxonomy']
            self.defaults['username'] = ctrlData['username']
            self.defaults['variablemod'] = string.join(ctrlData['variablemod'], ';')

            if self.searchType == 'pmf':
                self.defaults['pepcharge'] = ctrlData['charge']
            elif self.searchType in ('seq', 'msms'):
                self.defaults['msmscharge'] = ctrlData['charge']
                self.defaults['msmstol'] = ctrlData['msmstol']
                self.defaults['msmstolunits'] = ctrlData['msmstolunits']
                self.defaults['instrument'] = ctrlData['instrument']
                self.defaults['icat'] = ctrlData['icat']
    # ----


    # ----
    def onSearch(self, evt):
        """ Send data to mascot server. """

        # get and validate form params
        ctrlData = self.getData()
        if not ctrlData:
            return

        # format data to html
        htmlData = self.formatData(ctrlData)

        # make temporary file
        try:
            path = os.path.join(tempfile.gettempdir(), 'mmass_mascot_search.html')
            htmlFile = file(path, 'w')
            htmlFile.write(htmlData.encode("utf-8"))
            htmlFile.close()
        except IOError:
            dlg = wx.MessageDialog(self, "Unable to create temporary search file.", "File Error", wx.OK|wx.ICON_ERROR)
            dlg.ShowModal()
            dlg.Destroy()
            return

        # open file
        try:
            webbrowser.open('file://'+path, autoraise=1)
        except:
            dlg = wx.MessageDialog(self, "Unable to open webbrowser.", "Browser Error", wx.OK|wx.ICON_ERROR)
            dlg.ShowModal()
            dlg.Destroy()
            return

        # info dialog
        dlg = wx.MessageDialog(self, "Data were send to Mascot server.", "Mascot Search", wx.OK|wx.ICON_INFORMATION)
        dlg.ShowModal()
        dlg.Destroy()
    # ----


    # ----
    def onClose(self, evt):
        """ Delete temp file and close dialog. """

        # delete temporary file
        path = os.path.join(tempfile.gettempdir(), 'mmass_mascot_search.html')
        try:
            os.unlink(path)
        except:
            pass

        # close dialog
        self.EndModal(wx.ID_CANCEL)
    # ----


    # ----
    def getData(self):
        """ Get form data. """

        ctrlData = {}
        errors = []

        # get title
        ctrlData['title'] = self.title_value.GetValue()

        # get username
        ctrlData['username'] = self.username_value.GetValue()
        if not ctrlData['username']:
          errors.append('Your name must be specified.')

        # get e-mail
        ctrlData['email'] = self.email_value.GetValue()
        if not ctrlData['email']:
          errors.append('E-mail must be specified.')

        # get sequence params
        ctrlData['taxonomy'] = self.taxonomy_combo.GetValue()
        ctrlData['database'] = self.database_combo.GetValue()
        ctrlData['enzyme'] = self.enzyme_combo.GetValue()
        ctrlData['partials'] = self.partials_combo.GetValue()

        # get fixed modifications
        ctrlData['fixedmod'] = []
        for mod in self.fixedMod_listbox.GetSelections():
            if mod:
                ctrlData['fixedmod'].append(self.parameters['mods'][mod])

        # get fixed modifications
        ctrlData['variablemod'] = []
        for mod in self.variableMod_listbox.GetSelections():
            if mod:
                ctrlData['variablemod'].append(self.parameters['mods'][mod])

        # get mass type
        ctrlData['masstype'] = self.massType_combo.GetValue()

        # get charge
        ctrlData['charge'] = self.charge_combo.GetValue()

        # get tolerance units
        ctrlData['peptolunits'] = self.pepTolUnits_combo.GetValue()
        if self.searchType in ('seq', 'msms'):
            ctrlData['msmstolunits'] = self.msmsTolUnits_combo.GetValue()

        # get protein mass
        protmass = self.proteinMass_value.GetValue()
        protmass = protmass.replace(',', '.')
        if not protmass:
            ctrlData['protmass'] = ''
        else:
            try:
                ctrlData['protmass'] = float(protmass)
                if ctrlData['protmass'] < 0:
                    errors.append('Protein mass cannot be negative.')
                elif ctrlData['protmass'] > 1000:
                    errors.append('Upper limit on protein mass cannot exceed 1000 kDa.')
            except ValueError:
                errors.append('Protein mass is not a number.')

        # get precursor mass
        if self.searchType == 'msms':
            pepmass = self.pepMass_value.GetValue()
            pepmass = pepmass.replace(',', '.')
            if not pepmass:
                errors.append('Precursor mass is not specified.')
            else:
                try:
                    ctrlData['pepmass'] = float(pepmass)
                    if ctrlData['pepmass'] < 100:
                        errors.append('Precursor mass must be at least 100 Da.')
                    elif ctrlData['pepmass'] > 16000:
                        errors.append('Precursor mass must be less than 16,000 Da.')
                except ValueError:
                    errors.append('Precursor mass is not a number.')

        # get peptide tolerance
        peptol = self.pepTol_value.GetValue()
        peptol = peptol.replace(',', '.')
        if not peptol:
            errors.append('Peptide tolerance is not specified.')
        else:
            try:
                ctrlData['peptol'] = float(peptol)
                if ctrlData['peptol'] < 0:
                    errors.append('Peptide tolerance cannot be negative.')
                elif ctrlData['peptol'] > 10 and ctrlData['peptolunits'] == 'Da':
                    errors.append('Peptide tolerance must be less than 10 Da.')
                elif ctrlData['peptol'] > 10000 and ctrlData['peptolunits'] == 'mmu':
                    errors.append('Peptide tolerance must be less than 10 000 mmu.')
                elif ctrlData['peptol'] > 10000 and ctrlData['peptolunits'] == 'ppm':
                    errors.append('Peptide tolerance must be less than 10 000 ppm.')
                elif ctrlData['peptol'] > 1 and ctrlData['peptolunits'] == '%':
                    errors.append('Peptide tolerance must be less than 1%.')
            except ValueError:
                errors.append('Peptide tolerance is not a number.\n')

        # get msms tolerance
        if self.searchType in ('seq', 'msms'):
            msmstol = self.msmsTol_value.GetValue()
            msmstol = msmstol.replace(',', '.')
            if not msmstol:
                errors.append('MS/MS tolerance is not specified.')
            else:
                try:
                    ctrlData['msmstol'] = float(msmstol)
                    if ctrlData['msmstol'] < 0:
                        errors.append('MS/MS tolerance cannot be negative.')
                    elif ctrlData['msmstol'] > 10 and ctrlData['msmstolunits'] == 'Da':
                        errors.append('MS/MS tolerance must be less than 10 Da.')
                    elif ctrlData['msmstol'] > 10000 and ctrlData['msmstolunits'] == 'mmu':
                        errors.append('MS/MS tolerance must be less than 10 000 mmu.')
                    elif ctrlData['msmstol'] > 10000 and ctrlData['msmstolunits'] == 'ppm':
                        errors.append('MS/MS tolerance must be less than 10 000 ppm.')
                    elif ctrlData['msmstol'] > 1 and ctrlData['msmstolunits'] == '%':
                        errors.append('MS/MS tolerance must be less than 1%.')
                except ValueError:
                    errors.append('MS/MS tolerance is not a number.\n')

        # get instrument
        if self.searchType in ('seq', 'msms'):
            ctrlData['instrument'] = self.instrument_combo.GetValue()

        # get ICAT checkbox
        if self.searchType in ('seq', 'msms'):
            if self.icat_checkbox.IsChecked():
                ctrlData['icat'] = 1
            else:
                ctrlData['icat'] = 0

        # get peaklist/query
        ctrlData['query'] = self.query_value.GetValue()
        if not ctrlData['query']:
            errors.append('Peaklist/query is not specified.\n')

        # get results number
        ctrlData['showhits'] = self.hits_combo.GetValue()

        # get overview checkbox
        if self.overview_checkbox.IsChecked():
            ctrlData['overview'] = 1
        else:
            ctrlData['overview'] = 0

        # check errors
        if errors:
            message = ''
            for error in errors:
                message += '%s\n' % (error)
            errorDlg = wx.MessageDialog(self, message, "Query Error", wx.OK|wx.ICON_ERROR)
            errorDlg.ShowModal()
            errorDlg.Destroy()
            return False
        else:
            return ctrlData
    # ----


    # ----
    def formatData(self, ctrlData):
        """ Format data to mascot html. """

        # set basic form params
        if self.searchType == 'pmf':
            search = 'PMF'
            title = 'MASCOT Peptide Mass Fingerprint'
        elif self.searchType == 'seq':
            search = 'SQ'
            title = 'MASCOT Sequence Query'
        elif self.searchType == 'msms':
            search = 'MIS'
            title = 'MASCOT MS/MS Ion Search'

        # page header
        buff = '<?xml version="1.0" encoding="utf-8"?>\n'
        buff += '<!DOCTYPE html PUBLIC "-//W3C//DTD XHTML 1.0 Strict//EN" "http://www.w3.org/TR/xhtml1/DTD/xhtml1-strict.dtd">\n'
        buff += '<html xmlns="http://www.w3.org/1999/xhtml" xml:lang="en" lang="en">\n'
        buff += '<head>\n'
        buff += '  <title>Matrix Science - Mascot - MS/MS Ion Search</title>\n'
        buff += '<style type="text/css"><!--\n'
        buff += 'body{text-align: center; margin: auto; font-size: .8em; font-family: Arial, Verdana, Geneva, Helvetica, sans-serif;}\n'
        buff += '#data{display:none;}\n'
        buff += '#info{width: 300px; margin: auto; margin-top: 2em; border: 1px solid #e3e3e3; background-color: #f2f2f2;}\n'
        buff += '#info p.sending{font-weight: bold;}\n'
        buff += '#info p.note{font-size: .85em;}\n'
        buff += 'h1{font-size: 1.3em; color: #fff; background-color: #4dac27; border-bottom: 1px solid #e3e3e3;margin: 0; padding: .2em;}\n'
        buff += '--></style>\n'
        buff += '</head>\n\n'

        buff += '<body onload="document.forms[0].submit()">\n'
        buff += '<form action="%s" id="mainSearch" enctype="multipart/form-data" method="post">\n' % (self.defaults['urlsearch'])
        buff += '<div style="display:none;">\n\n'

        # hidden params
        buff += '<input type="hidden" name="INTERMEDIATE" value="" />\n'
        buff += '<input type="hidden" name="FORMVER" value="1.01" />\n'
        buff += '<input type="hidden" name="SEARCH" value="%s" />\n' % (search)
        buff += '<input type="hidden" name="IATOL" value="0" />\n'
        buff += '<input type="hidden" name="IASTOL" value="0" />\n'
        buff += '<input type="hidden" name="IA2TOL" value="0" />\n'
        buff += '<input type="hidden" name="IBTOL" value="1" />\n'
        buff += '<input type="hidden" name="IBSTOL" value="0" />\n'
        buff += '<input type="hidden" name="IB2TOL" value="1" />\n'
        buff += '<input type="hidden" name="IYTOL" value="1" />\n'
        buff += '<input type="hidden" name="IYSTOL" value="0" />\n'
        buff += '<input type="hidden" name="IY2TOL" value="1" />\n'
        buff += '<input type="hidden" name="PEAK" value="AUTO" />\n'
        buff += '<input type="hidden" name="LTOL" value="" />\n'
        buff += '<input type="hidden" name="REPTYPE" value="peptide" />\n'
        buff += '<input type="hidden" name="ERRORTOLERANT" value="0" />\n'
        buff += '<input type="hidden" name="SHOWALLMODS" value="" />\n\n'

        # main params
        buff += '<input type="text" name="COM" value="%s" />\n' % (ctrlData['title'])
        buff += '<input type="text" name="USERNAME" value="%s" />\n' % (ctrlData['username'])
        buff += '<input type="text" name="USEREMAIL" value="%s" />\n' % (ctrlData['email'])
        buff += '<input type="text" name="TAXONOMY" value="%s" />\n' % (ctrlData['taxonomy'])
        buff += '<input type="text" name="DB" value="%s" />\n' % (ctrlData['database'])
        buff += '<input type="text" name="CLE" value="%s" />\n' % (ctrlData['enzyme'])
        buff += '<input type="text" name="PFA" value="%s" />\n' % (ctrlData['partials'])
        buff += '<input type="text" name="SEG" value="%s" />\n' % (ctrlData['protmass'])
        buff += '<input type="text" name="MASS" value="%s" />\n' % (ctrlData['masstype'])
        buff += '<input type="text" name="CHARGE" value="%s" />\n' % (ctrlData['charge'])
        buff += '<input type="text" name="TOL" value="%s" />\n' % (ctrlData['peptol'])
        buff += '<input type="text" name="TOLU" value="%s" />\n' % (ctrlData['peptolunits'])
        buff += '<input type="text" name="REPORT" value="%s" />\n' % (ctrlData['showhits'])

        # overview
        if ctrlData['overview']:
            buff += '<input type="text" name="OVERVIEW" value="ON" />\n\n'
        else:
            buff += '<input type="text" name="OVERVIEW" value="" />\n\n'

        # fixed modifications
        buff += '<select name="MODS" multiple="multiple">\n'
        for mod in ctrlData['fixedmod']:
            buff += '  <option value="%s" selected="selected">%s</option>\n' % (mod, mod)
        buff += '</select>\n\n'

        # variable modifications
        buff += '<select name="IT_MODS" multiple="multiple">\n'
        for mod in ctrlData['variablemod']:
            buff += '  <option value="%s" selected="selected">%s</option>\n' % (mod, mod)
        buff += '</select>\n\n'

        # msms and seq only
        if self.searchType in ('seq', 'msms'):
            buff += '<input type="text" name="ITOL" value="%s" />\n' % (ctrlData['msmstol'])
            buff += '<input type="text" name="ITOLU" value="%s" />\n' % (ctrlData['msmstolunits'])
            buff += '<input type="text" name="INSTRUMENT" value="%s" />\n' % (ctrlData['instrument'])

            if ctrlData['icat']:
                buff += '<input type="text" name="ICAT" value="ON" />\n\n'
            else:
                buff += '<input type="text" name="ICAT" value="" />\n\n'

        # query
        if self.searchType == 'msms':
            ctrlData['query'] = '\nBEGIN IONS\nPEPMASS=%s\n\n%s\nEND IONS\n' % (ctrlData['pepmass'], ctrlData['query'])
        buff += '<textarea name="QUE" rows="6" cols="30">%s</textarea>\n\n' % (ctrlData['query'])

        # end hidden part of the form
        buff += '</div>\n\n'

        # show mMass info
        buff += '<div id="info">\n'
        buff += '  <h1>mMass - %s</h1>\n' % (title)
        buff += '  <p class="sending">Sending data to MASCOT Server...<br />Please wait...</p>\n'
        buff += '  <p class="note">(Press Search button if data was not send automatically.)</p>\n'
        buff += '  <p><input type="submit" value="Search" /></p>\n'
        buff += '</div>\n\n'

        # page foot
        buff += '</form>\n'
        buff += '</body>\n'
        buff += '</html>\n'

        return buff
    # ----
