// Copyright (c) 2009, Jens Peter Secher <jpsecher@gmail.com>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.

class MercurialPristineTar
{
	public static function main()
	{
		var usage = "Usage: " + Constants.pristineTar() + " [-V|--version]"
		+ " [-v|--verbose] [version]"
		;
		var options = new GetPot( neko.Sys.args() );
		Util.maybeReportVersion( options, usage );
		// Collect verbosity options.
		verbosity = 0;
		while( options.got( ["--verbose","-v"] ) ) ++verbosity;
		// We cannot get to other branches if there are pending changes.
		Mercurial.exitIfUncommittedChanges();
		// Extract package information from the changelog file.
		Mercurial.switchToBranch( Constants.mainBranch() );
		var changeLog = DebianChangeLog.parseCurrentChangeLog();
		if( changeLog.version.debianRevision == null )
		{
			Util.die( "Cannot generate pristine tarball for a native package." );
		}
		var fileNameBase = changeLog.source + "_";
		// Use a specified version or the most recent from the changelog.
		Util.exitOnExtraneousOptions( options, usage );
		var version = options.unprocessed();
		if( version == null )
		{
			version = changeLog.version.upstream;
		}
		Util.exitOnExtraneousArguments( options, usage );
		fileNameBase += version;		
		// Find the matching tarball xdeltas, and map them to the component
		// name.
		var deltas = new Hash<Pristine>();
		Mercurial.switchToBranch( Constants.pristineBranch() );
		var entries = neko.FileSystem.readDirectory( "." );
		for( entry in entries )
		{
			var versionRegExp = new EReg
			(
				"^(" + RegExpUtil.escape( fileNameBase ) + "\\.orig" +
				"(-([A-Za-z0-9][-A-Za-z0-9]*))?" + "\\.tar\\.)([a-z0-9]+)\\." +
				Constants.pristineExtension() + "$",
				""
			);
			if( versionRegExp.match( entry ) )
			{
				// Read the branch revision.
				var revision = neko.io.File.getContent
				(
					versionRegExp.matched( 1 ) + Constants.revisionExtension()
				);
				var pristineInfo = new Pristine
				(
					versionRegExp.matched( 4 ),
					entry,
					versionRegExp.matched( 1 ) + Constants.xdeltaExtension(),
					revision
				);
				var component = versionRegExp.matched( 3 );
				// Main tarball wil have a null component.
				if( component == null ) component = ".";
				deltas.set( component, pristineInfo );
				info( 3, component + " => " + pristineInfo );
			}
		}
		if( Lambda.count( deltas ) == 0 )
		{
			die( "No xdeltas found for version " + version );
		}
		// Extract each component from its dedicated branch and recreate the
		// pristine tarball from the right revision by using the xdelta in the
		// pristine branch.
		for( component in deltas.keys() )
		{
			var pristineInfo = deltas.get( component );
			var tarball = pristineInfo.tarballDelta.substr
			(
				0, pristineInfo.tarballDelta.lastIndexOf(".")
			);
			info( 2, "Switching to revision " + pristineInfo.revision );
			infos( 3, Mercurial.update( pristineInfo.revision ) );
			// Recreate a tarball from the component branch contents by
			// transforming the file names so they are rooted.
			var recreated = "recreated-" + tarball;
			var tarArgs = [ "tar", "cf", recreated ];
			if( component == "." )
			{
				tarArgs = tarArgs.concat
				([
					"--transform", "s%^%" + changeLog.source +
					"-" + version + "/%"
				]);
			}
			for( entry in neko.FileSystem.readDirectory( "." ) )
			{
				if( ! Constants.precious().match( entry ) )
				{
					tarArgs.push( entry );
				}
			}
			info( 3, "Starting " + tarArgs.join(" ") );
			var pack = new Process( "sh", ["-c", tarArgs.join(" ")] );
			if( pack.process.exitCode() != 0 )
			{
				infos( 1, pack.stdout() );
				Util.writeErrors( pack.stderr() );
				die( "Could not recreate " + tarball );
			}
			// tarball.
			Mercurial.switchToBranch( Constants.pristineBranch() );
			var args =
			[
				"patch", "-p", pristineInfo.tarballDelta, recreated, tarball
			];
			info( 2, "Starting xdelta " + args.join(" ") );
			var xdelta = Process.runButExitOnError( "xdelta", args );
			info( 2, "Created " + tarball );
			// Use one of the pristine info to recreate the pristine compressed
			// tarball.
			var pristine = Util.pristineCompress
			(
				tarball,
				pristineInfo.compressionDelta,
				pristineInfo.compression
			);
			if( ! pristine )
			{
				die( "Could not recreate pristine tarball for " + component );
			}
			var compressedTarball = tarball + "." + pristineInfo.compression;
			neko.io.File.copy( compressedTarball, "../" + compressedTarball );
			info( 1, "Created ../" + compressedTarball );
			// Cleanup.
			FileUtil.unlink( tarball );
			FileUtil.unlink( recreated );
			FileUtil.unlink( compressedTarball );
			FileUtil.unlink( "bin" );
		}
		Mercurial.switchToBranch( Constants.mainBranch() );
		neko.Sys.exit( 0 );
	}

	//
	// Higher values means more output.
	//
	static var verbosity : Int;

	//
	// Always write a line of info to the log, but only Write to stdout if the
	// verbosity level is high enough.
	//
	static function info( level : Int, line : String )
	{
		if( verbosity >= level ) Util.writeInfo( line );
	}

	//
	// Always write lines of info to the log, but only Write to stdout if the
	// verbosity level is high enough.
	//
	static function infos( level : Int, lines : Iterator<String> )
	{
		for( line in lines )
		{
			if( verbosity >= level ) Util.writeInfo( line );
		}
	}

	//
	// Exit with an error message but swith back to main branch first.
	//
	static function die( msg : String )
	{
		Mercurial.switchToBranch( Constants.mainBranch() );
		Util.die( msg );
	}
}
