// Copyright (C) 1994 The New York Group Theory Cooperative
// See magnus/doc/COPYRIGHT for the full notice.

// Contents: Definition of SGofFreeGroup and SGofFreeGroupRep classes.
//
// Principal Authors: Sergei Lyutikov, Roger Needham
//
// Status: Draft
//
// Description:
//
//   A subgroup stores a shared-reference `copy' of its parent group.
//   Enhancements to the parent group are thus available to its
//   subgroups. The subgroup is generated by (formal) words interpretable
//   in the generators of the parent group.
//
// Special Notes:
//
//   Since SubgroupGraph works ONLY with freely reduced word all words
//   should be freely reduced here before passing them to methods of
//   SubgroupGraph.
//
// Revision History:
//
// * First implemented: april 12, 1995
//
// * 5/8/95 rn added members:
//     SGofFreeGroup SGofFreeGroup::MHallCompletion( )
//     SGofFreeGroupRep* SGofFreeGroupRep::MHallCompletion( )
//     SGofFreeGroupRep::SGofFreeGroupRep( const FreeGroup& parent,
//                                         SubgroupGraph SGG )
//
// * 10/16/96 @dp added members:
//     bool SGofFreeGroup::isMalnormal( Word& conjugator )
//     bool SGofFreeGroupRep::isMalnormal( Word& conjugator )
//
// * 03/03/97 @dp added:
//     void SGofFreeGroup::printWord( ostream& ostr, const Word& w ) const;
//

#ifndef _SGOFFREEGROUP_H_
#define _SGOFFREEGROUP_H_

#include "ObjectOf.h"
#include "RefCounter.h"
#include "FreeGroup.h"
#include "SubgroupGraph.h"

class SGofFreeGroupRep : public GenericRep {

public:

  // Constructors:

  // Copy constructor supplied by compiler

  SGofFreeGroupRep( const FreeGroup& parent, const VectorOf<Word>& gens );

  SGofFreeGroupRep( const FreeGroup& parent, SubgroupGraph SGG ) :
    computedNielsenBasis(false),
    theSubgroupGraph(SGG),
    builtSubgroupGraph(true),
    theGenerators(SGG.nielsenBasis()),
    theParentGroup(parent)
  { }


  // Representation methods:

  PureRep* clone( ) const { return new SGofFreeGroupRep(*this); }

  static const Type theSGofFreeGroupType;

  static Type type( ) { return theSGofFreeGroupType; }

  virtual Type actualType( ) const { return type(); }

  int hash() const;

  // Operators:

  // Methods dealing with group structure:
 
  int order( ) { return rank() == 0; }
  Bool isTrivial( ) { return rank() == 0; }
  Bool isFinite( ) { return rank() == 0; }
  Bool isInfinite( ) { return !isFinite(); }
  Bool isAbelian( ) { return rank() < 2; } 
  bool isMalnormal( Word& conjugator );

  SGofFreeGroupRep* join(SGofFreeGroupRep& SGR);
  // Returns the join of this subgroup and the argument.

  SGofFreeGroupRep* intersection(SGofFreeGroupRep& SGR);
  // Returns the intersection of this subgroup and the argument.

  Bool isNormal( );

  VectorOf<Word> normalizer( );

  VectorOf<Word> nielsenBasis( );
  // Returns a Nielsen basis for the subgroup.

  Word nielsenWord(int i);
  // Returns an  i^th element of the Nielsen basis.

  Word inNielsenWords(const Word& w);
  // Returns the word `w' written in elements of the Nielsen basis.

  int rank();
  // Returns the rank of this subgroup as a free group.

  // Methods dealing with group elements:

  // Trichotomy areEqual(const Elt&, const Elt&) const { return DONTKNOW; }
  //@rn can do

  // Elt multiply(const Elt&, const Elt&) const
  // result is as actual type of args: if formal words in subgroup
  // generators, then such; if elt in parent group, then such
  // if mixed, then nonsense

  // Elt inverseOf(const Elt&) const
  // @rn?
	 
  // Elt raiseToPower(const Elt&, int) const
  // @rn?

  // Elt conjugateBy(const Elt&, const Elt&) const

  // Elt commutator(const Elt&, const Elt&) const
  // @rn?

  Elt eval( const Word& w ) const;
  // `w' is in the subgroup's generators. Returns the word `w' written
  // in the free group's generators.

  Bool wordProblem( const Word& w ) const;
  // `w' is a word in the subgroup's generators. Returns TRUE iff
  // `w' is trivial in the free group.
  
  Bool conjugacyProblem( const Word& u, const Word& v ) const;
  // `u' and `v' are words in the subgroup's generators. Returns TRUE
  // iff `u' and `v' are conjugate in the free group.

  Bool contains(const Word& w);
  // Returns TRUE iff this subgroup contains `w'.

  Bool contains(const SetOf<Word>& S);
  // Returns TRUE iff this subgroup contains the subgroup generated by `S'.

  Bool contains(const VectorOf<Word>& V);
  // Returns TRUE iff this subgroup contains the subgroup generated by `V'.

  Bool contains(const SGofFreeGroupRep& SGR);
  // Returns TRUE iff this subgroup contains `SG'.

  Bool equalTo(const SetOf<Word>& S);
  // Returns TRUE iff this subgroup and the subgroup generated by `S' are equal.

  Bool conjugateInSubgroup(const Word& w, Word& conjugator);
  // Returns TRUE iff some conjugate of `w' is in the subgroup.
  // If TRUE, `conjugator' is set to the conjugator.

  Bool conjugateInSubgroup(const SetOf<Word>& S, Word& conjugator);
  // Returns TRUE iff some conjugate of the subgroup generated by `S' is
  // in the subgroup. If TRUE, `conjugator' is set to the conjugator.

  bool conjugateTo(const SetOf<Word>& S);
  // Returns true iff this subgroup and the argument are conjugate.

  long powerInSubgroup( const Word& w );
  // returns `the minimal power' or 0 if there are no powers of the
  // element `aWord' in H.

  int findIndex();
  // Returns the index of the subgroup or 0 if infinite.

  VectorOf<Word> findWhiteheadBasis();
  // Finds the subgroup of the free group authomorphic to this with
  // smallest sum of lengths of generators.
  // Returns a vector of generators.

  Bool isAFreeFactor();
  // Returns TRUE iff this subgroup is a free factor.

  Bool generatesTheFreeGroup();

  Word rightSchreierRepresentative(const Word& w);

  SGofFreeGroupRep* MHallCompletion( );

  void makeSubgroupGraph( );

  // I/O:
 
  void printOn(ostream&) const;

  SGofFreeGroupRep* readFrom(istream&, Chars&) const { return 0; }

  void printGenerator( ostream& ostr, int n ) const;
  void printGenerators( ostream& ostr ) const;
  void printWord( ostream& ostr, const Word& w ) const;
  // w is a word in the subgroup's generators.
  

  // Data members:

  //@rn Provisional flags:
  bool computedNielsenBasis;
  bool builtSubgroupGraph;

  VectorOf<Word> theGenerators;
  VectorOf<Word> NielsenBasis;

  SubgroupGraph theSubgroupGraph;

  //@rn VectorOf<Chars> theNamesOfGenerators;
  //@rn Someday the end user might name generators.

  FreeGroup theParentGroup;

private:

  SGofFreeGroupRep& operator = ( const SGofFreeGroupRep& );
  // disable assignment

};


//------------------------- SGofFreeGroup ---------------------------//


class SGofFreeGroup : public GenericObject {

public:

  ///////////////////////////////////////////////////////
  //                                                   //
  //  Constructors                                     //
  //                                                   //
  ///////////////////////////////////////////////////////

  SGofFreeGroup( const FreeGroup& parent, const VectorOf<Word>& gens )
  : GenericObject( new SGofFreeGroupRep(parent,gens) )
  { }
  // to make a finitely generated SGofFreeGroup with a vector of unnamed
  // generators


  ///////////////////////////////////////////////////////
  //                                                   //
  //  Accessors                                        //
  //                                                   //
  ///////////////////////////////////////////////////////

  int hash() const { return look()->hash(); }

  bool operator == ( const SGofFreeGroup& g) const;

  const FreeGroup& parentGroup( ) const { return look()->theParentGroup; }

  const VectorOf<Word>& generators( ) const { return look()->theGenerators; }

  static Type type( ) { return SGofFreeGroupRep::type(); }

  Type actualType( ) const { return look()->actualType(); }

  ///////////////////////////////////////////////////////
  //                                                   //
  //  Group structure methods                          //
  //                                                   //
  ///////////////////////////////////////////////////////
 
  int order( ) { return enhance()->order( ); }
  Bool isTrivial( ) { return enhance()->isTrivial( ); }
  Bool isFinite( ) { return enhance()->isFinite( ); }
  Bool isInfinite( ) { return enhance()->isInfinite( ); }
  Bool isAbelian( ) { return enhance()->isAbelian( ); }

  bool isMalnormal( Word& conjugator ) { return enhance()->isMalnormal(conjugator); }
  // A subgroup is malnormal iff the intersection of the subgroup and
  // conjugation of the one by any element from the difference 
  // of the parent group and the subgroup is not trivial.
  // The function returns `false' and a word to conjugate the 
  // if the subgroup is not malnormal. If the subgroup is malnormal
  // then it returns only `true' and the parameter `conjugator' is meanless.

  SGofFreeGroup join(SGofFreeGroup& SG) {
    return SGofFreeGroup(enhance()->join(*SG.enhance()));
  }
  // Returns the join of this subgroup and the argument.

  SGofFreeGroup intersection(SGofFreeGroup& SG) {
    return SGofFreeGroup(enhance()->intersection(*SG.enhance()));
  }
  // Returns the intersection of this subgroup and the argument.

  Bool isNormal( ) { return enhance()->isNormal(); }

  VectorOf<Word> normalizer( ) { return enhance()->normalizer(); }

  VectorOf<Word> nielsenBasis( ) { return enhance()->nielsenBasis(); }
  // Returns a Nielsen basis for this subgroup.

  Word nielsenWord(int i) { return enhance()->nielsenWord(i); }
  // Returns an  i^th element of the Nielsen basis.

  Word inNielsenWords(const Word& w) { return enhance()->inNielsenWords(w); }
  // Returns the word `w' written in elements of the Nielsen basis.

  int rank() { return enhance()->rank(); }
  // Returns the rank of this subgroup as a free group

  SGofFreeGroup MHallCompletion( ) {
	 return SGofFreeGroup( enhance()->MHallCompletion() );
  }


  ///////////////////////////////////////////////////////
  //                                                   //
  //  Methods which deal with subgroup elements        //
  //                                                   //
  ///////////////////////////////////////////////////////

  // Trichotomy areEqual(const Elt& e1, const Elt& e2) const; 
  // Elt firstElt( ) const;                                   
  // Elt nextElt(const Elt& e) const;                         
  // Elt multiply(const Elt& e1, const Elt& e2) const;        
  // Elt inverseOf(const Elt& e) const;                       
  // Elt raiseToPower(const Elt& e, int n) const;             
  // Elt conjugateBy(const Elt& e1, const Elt& e2) const;     
  // Elt commutator(const Elt& e1, const Elt& e2) const;      

  Elt eval( const Word& w ) const { return look()->eval(w); }
  // `w' is in the subgroup's generators. Returns the word `w' written
  // in the free group's generators.

  Bool wordProblem( const Word& w ) const {
    return look()->wordProblem(w);
  }
  // `w' is a word in the subgroup's generators. Returns TRUE iff
  // `w' is trivial in the free group.

  Bool conjugacyProblem( const Word& u, const Word& v ) const {
    return look()->conjugacyProblem(u,v);
  }

  Bool contains(const Word& w) { return enhance()->contains(w); }
  // Returns TRUE iff this subgroup contains `w'.

  Bool contains(const SetOf<Word>& S) { return enhance()->contains(S); }
  // Returns TRUE iff this subgroup contains the subgroup generated by `S'.

  Bool contains(const VectorOf<Word>& V) { return enhance()->contains(V); }
  // Returns TRUE iff this subgroup contains the subgroup generated by `V'.

  Bool contains(const SGofFreeGroup& SG) { 
    return enhance()->contains(*SG.look());
  }
  // Returns TRUE iff this subgroup contains `SG'.

  Bool equalTo(const SetOf<Word>& S) { return enhance()->equalTo(S); }
  // Returns TRUE iff this subgroup and the subgroup generated by `S' are equal.

  Bool conjugateInSubgroup(const Word& w, Word& conjugator) {
         return enhance()->conjugateInSubgroup(w,conjugator);
  }
  // Returns TRUE iff some conjugate of `w' is in the subgroup.
  // If TRUE, `conjugator' is set to the conjugator.

  Bool conjugateInSubgroup(const SetOf<Word>& S, Word& conjugator) {
         return enhance()->conjugateInSubgroup(S,conjugator);
  }
  // Returns TRUE iff some conjugate of the subgroup generated by `S' is
  // in the subgroup. If TRUE, `conjugator' is set to the conjugator.

  bool conjugateTo(const SetOf<Word>& S) {
    return enhance()->conjugateTo(S);
  }
  // Returns true iff this subgroup and the argument are conjugate.

  long powerInSubgroup( const Word& w ) {
         return enhance()->powerInSubgroup(w);
  }
  // returns `the minimal power' or 0 if there are no powers of the
  // element `aWord' in H.

  int findIndex() { return enhance()->findIndex(); }
  // Returns the index of the subgroup or 0 if infinite.

  VectorOf<Word> findWhiteheadBasis() {
    return enhance()->findWhiteheadBasis();
  }
  // Finds the subgroup of the free group authomorphic to this with
  // smallest sum of lengths of generators.
  // Returns a vector of generators.

  Bool isAFreeFactor() { return enhance()->isAFreeFactor(); }
  // Returns TRUE iff this subgroup is a free factor.

  Bool generatesTheFreeGroup() {
         return enhance()->generatesTheFreeGroup();
  }

  Word rightSchreierRepresentative(const Word& w) {
         return enhance()->rightSchreierRepresentative(w);
  }

  ///////////////////////////////////////////////////////
  //                                                   //
  //  Methods which deal with sets of group elements   //
  //                                                   //
  ///////////////////////////////////////////////////////
 
  // @rn Want these?
  // SetOf<Elt> setMultiply(const SetOf<Elt>& S1, const SetOf<Elt>& S2) const;
  // SetOf<Elt> setMultiply(const Elt& e, const SetOf<Elt>& S) const;
  // SetOf<Elt> setMultiply(const SetOf<Elt>& S, const Elt& e) const;
  // SetOf<Elt> conjugateBy(const SetOf<Elt>& S1, const SetOf<Elt>& S2) const;
  // SetOf<Elt> conjugateBy(const Elt& e, const SetOf<Elt>& S) const;
  // SetOf<Elt> conjugateBy(const SetOf<Elt>& S, const Elt& e) const;
  // void closeUnderInverses(SetOf<Elt>& S) const;
  // void closeUnderCyclicPermutations(SetOf<Word>& S) const;

  ///////////////////////////////////////////////////////
  //                                                   //
  //  I/O                                              //
  //                                                   //
  ///////////////////////////////////////////////////////
 
  void printWord( ostream& ostr, const Word& w ) const {
    look()->printWord( ostr, w );
  } 
  // w is a word in the subgroup's generators.

 
  ///////////////////////////////////////////////////////
  //                                                   //
  //  Representation access methods                    //
  //                                                   //
  ///////////////////////////////////////////////////////

protected:


  // Shadow representation accessors to get representations of the
  // right type in the members of this class:

  const SGofFreeGroupRep* look( ) const {
	 return (const SGofFreeGroupRep*)GenericObject::look();
  }
  SGofFreeGroupRep* enhance( ) const {
	 return (SGofFreeGroupRep*)GenericObject::enhance();
  }
  SGofFreeGroupRep* change( ) {
	 return (SGofFreeGroupRep*)GenericObject::change();
  }

  // Special wrapping constructor to wrap new representations (returned
  // by eg. delegated methods) and for GenericObject initialisation by derived
  // classes:

  SGofFreeGroup( SGofFreeGroupRep* newrep ) : GenericObject(newrep) { }

};




Word expressWordInSubgroupGenerators( const SGofFreeGroup& H, const Word& w);
// Returns the word `w' written in the given generators of the subgroup `H'.

#endif
