/******************************** LICENSE ********************************


 Copyright 2007 European Centre for Medium-Range Weather Forecasts (ECMWF)
 
 Licensed under the Apache License, Version 2.0 (the "License"); 
 you may not use this file except in compliance with the License. 
 You may obtain a copy of the License at 
 
 	http://www.apache.org/licenses/LICENSE-2.0
 
 Unless required by applicable law or agreed to in writing, software 
 distributed under the License is distributed on an "AS IS" BASIS, 
 WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. 
 See the License for the specific language governing permissions and 
 limitations under the License.


 ******************************** LICENSE ********************************/

/*!
    \file GDDriver.h
    \brief Definition of GDDriver.
    \author Graphics Section, ECMWF

    Started: Mon Oct 29 16:05:47 2007
*/

#ifndef MPP_GDDriver_H
#define MPP_GDDriver_H

#include <BaseDriver.h>
#include <GDDriverAttributes.h>
#include <XmlNode.h>

#include <gd.h>

namespace magics
{

/*! \class GDDriver
    \brief This driver produces output in various raster image formats.
    \ingroup drivers

    The raster image format driver produces output in various raster image formats
    thanks to GD ( http://www.libgd.org ).
*/
class GDDriver: public BaseDriver, public GDDriverAttributes
{
public:
	GDDriver();
	~GDDriver();
	void open();
	void close();

	void setJPG(bool b = true) const {jpg_ = b;}
	void setGIF(bool b = true) const {gif_ = b;}
	void setPNG(bool b = true) const {png_ = b;}
	void setAnimation(bool b = true) const {animated_ = b; if(b) gif_ = true;}

	/*!
	  \brief sets a new XML node
	*/
	void set(const XmlNode& node)
	{
		if ( magCompare(node.name(), "gif") ||
		     magCompare(node.name(), "gif_animation") ||
		     magCompare(node.name(), "png") ||
		     magCompare(node.name(), "jpeg") )
		{
			XmlNode basic = node;
			basic.name("driver");
			BaseDriver::set(basic);
			basic.name("gd");
			GDDriverAttributes::set(basic);
		}
	}

	/*!
	  \brief sets a new map
	*/
	void set(const map<string, string>& map)
	{
		BaseDriver::set(map);
		GDDriverAttributes::set(map);
	}

private:
	MAGICS_NO_EXPORT void startPage() const;
	MAGICS_NO_EXPORT void endPage() const;
	MAGICS_NO_EXPORT void project(const Layout& lay) const;
	MAGICS_NO_EXPORT void unproject() const;

	MAGICS_NO_EXPORT void setNewColour(const Colour &col) const;
	MAGICS_NO_EXPORT int setLineParameters(const LineStyle style, const float w) const;

	MAGICS_NO_EXPORT void renderPolyline(const int, float *, float *) const;
	MAGICS_NO_EXPORT void renderPolyline2(const int n, float *x, float *y) const;
	MAGICS_NO_EXPORT void renderSimplePolygon(const int, float *, float *) const;
	MAGICS_NO_EXPORT void renderText(const Text& text) const;
	MAGICS_NO_EXPORT void circle(const float x, const float y, const float r, const int) const;
	MAGICS_NO_EXPORT bool renderPixmap(float,float,float,float,int,int,unsigned char*,int, bool) const;
	MAGICS_NO_EXPORT bool renderCellArray(const Image& obj) const;

	MAGICS_NO_EXPORT float setY(const float y) const {return -y;}
	MAGICS_NO_EXPORT float setAngleY(const float y) const {return -y;}
	MAGICS_NO_EXPORT void renderLogo(float x, float y) const;
	MAGICS_NO_EXPORT void renderSymbols(const Symbol& symbol) const;
	MAGICS_NO_EXPORT float projectX(const float x) const {return offsetX_+(x*coordRatioX_);}
	MAGICS_NO_EXPORT float projectY(const float y) const {return offsetY_+(y*coordRatioY_);}

	mutable ofstream	xmlFile_;
	mutable FILE*		outFile_;
	mutable string	filename_;
	mutable string	currentFont_;
	mutable int		currentFontSize_;
	mutable int		currentColourIndex_;
	mutable int		currentLineStyleIndex_;
	mutable int		currentPageCount_;
	mutable int		lastPage_;    //!< for layers 
	mutable float		scaleFactor_; //!< needed because GD uses different sizes as CM
	mutable Layout*		currentLayout_;

	mutable vector<gdImagePtr> Images_;
	mutable gdImagePtr	currentImage_;
	mutable gdImagePtr	Brush_;
	mutable bool		animated_;
	mutable bool		jpg_;
	mutable bool		gif_;
	mutable bool		png_;
	mutable bool		clipping_;

	mutable int dimensionXglobal_;
	mutable int dimensionYglobal_;
	mutable float offsetX_;
	mutable float offsetY_;
	mutable stack<float> offsetsX_;
	mutable stack<float> offsetsY_;
	mutable stack<float> boxoffsetsX_;
	mutable stack<float> boxoffsetsY_;


	//! Method to print string about this class on to a stream of type ostream (virtual).
	void print(ostream&) const;
	MAGICS_NO_EXPORT void debugOutput(const string &s) const;

	//! Copy constructor - No copy allowed
	GDDriver(const GDDriver&);
	//! Overloaded << operator to copy - No copy allowed
	GDDriver& operator=(const GDDriver&);

	// -- Friends
	//! Overloaded << operator to call print().
	friend ostream& operator<<(ostream& s,const GDDriver& p)
		{ p.print(s); return s; }
};

} // namespace magics
#endif
