/*! \file TagHandler.cc
    \brief Implementation of the Template class TagHandler.
    
    Magics Team - ECMWF 2008
    
    Started: Mon 27-Oct-2008
    
    Changes:
    
*/



#include "TagHandler.h"
#include "XmlReader.h"
using namespace magics;

TagHandler::TagHandler() 
{
}


TagHandler::~TagHandler() 
{
}

/*!
 Class information are given to the output-stream.
*/		
void TagHandler::print(ostream& out)  const
{
	out << "TagHandler[";
	for (map<string, DefList >::const_iterator fam = definitions_.begin(); fam != definitions_.end(); ++fam) {
		out << " Famille-> " << fam->first;
		
		DefList defs = fam->second;
		for ( DefList::const_iterator def = defs.begin(); def != defs.end(); ++def) {
			string sep;
			string value;
			for ( std::set<string>::const_iterator v = def->second.begin(); v != def->second.end(); ++v) {
				if ((*v).empty()) continue;
				value += sep + *v; 
				sep ="/";
			}
			out << " (" << def->first << "=" << value <<  ")" << endl;
		}
	}
	out << "]";
}

void TagHandler::update(const string& family, const string& definition, const string& value)
{
	map<string, DefList >::iterator fam = definitions_.find(family);
	if ( fam == definitions_.end()) {
		definitions_.insert(make_pair(family, DefList()) );
		fam = definitions_.find(family);
	}
	DefList::iterator def = fam->second.find(definition);
	
	if ( def == fam->second.end() ) {
		fam->second.insert(make_pair(definition, std::set<string>()));
		def = fam->second.find(definition);
	}

	def->second.insert(value);
}


string TagHandler::get(const string& family, const string& definition)
{
	map<string, DefList >::iterator fam = definitions_.find(family);

	if ( fam == definitions_.end() ) return "";

	DefList::iterator def = fam->second.find(definition);

	if ( def == fam->second.end() ) return "";

	string sep = "";
	string value;
	for ( std::set<string>::const_iterator v = def->second.begin(); v != def->second.end(); ++v) {
		if ((*v).empty()) continue;
		value += sep + *v; 
		sep ="/";
	}
	return value;
}

TagConverter::TagConverter(TagHandler& owner) : owner_(owner), text_(0) {
		map_["font"] =  &TagConverter::font;
		map_["u"] =  &TagConverter::underline;
		map_["b"] =  &TagConverter::bold;
		map_["sup"] =  &TagConverter::superscript;
		map_["sub"] =  &TagConverter::subscript;
		map_["root"] =  &TagConverter::ignore;
		map_["xml"] =  &TagConverter::text;
		map_["data"] =  &TagConverter::data;
		map_["grib_info"] =  &TagConverter::grib;
		map_["magics_title"] =  &TagConverter::magics;
		map_["text"] =  &TagConverter::pass;
		map_["box"] =  &TagConverter::pass;
		map_["br"] =  &TagConverter::pass;
	}
	
TagConverter::~TagConverter() {}

void TagConverter::visit(const XmlNode& node)
{		
	map<string, ConvertFunction>::iterator function = map_.find(node.name());		
	if ( function != map_.end())  (this->*function->second)(node);			
	else 
		Log::error() << "Html Text handler: unknown tag [" << node.name() << "]" << endl;	
	// continue the visit...
		
}
void TagConverter::ignore(const XmlNode&) {}
void TagConverter::encoding(const string& encoding) { encoding_ = encoding; }
void TagConverter::text(const XmlNode& node) { 
	push();
	node.visit(*this);
	//check(node.data()); 
}

void TagConverter::grib(const XmlNode& node) { 
	//Log::dev()<< "checking grib!" << endl;
	push();
	const map<string, string>& attributes = node.attributes();
	string key =  ( attributes.find("id") != attributes.end() ) ? attributes.find("id")->second : ""; 
	if (attributes.find("key") != attributes.end() ) {			
				check(owner_.get("grib"+key, attributes.find("key")->second)); 
	}
	if (attributes.find("definition") != attributes.end() ) {			
		check(owner_.get("grib"+key, attributes.find("definition")->second)); 
	}
	node.visit(*this);
	pop();	
	
	font_ = top().font();
	elevation_ = top().elevation();
}

void TagConverter::pass(const XmlNode& node) { 
	
	
	node.visit(*this);
	check(node.data()); 

	
}

void TagConverter::magics(const XmlNode& node) { 
	//Log::dev()<< "checking magics title!" << endl;
	push();		
    
    check(owner_.get("grib", "magics")); 
 
	node.visit(*this);
	pop();	
	
	font_ = top().font();
	elevation_ = top().elevation();
}

void TagConverter::data(const XmlNode& node) { 
	
	
	push();
	const map<string, string>& attributes = node.attributes();
	if (attributes.find("data") != attributes.end() )
		check(attributes.find("data")->second); 
	node.visit(*this);
	pop();	
	
	font_ = top().font();
	elevation_ = top().elevation();
	
}
void TagConverter::font(const XmlNode& node) {
	const map<string, string>& attributes = node.attributes();
	
	if ( attributes.find("colour") != attributes.end() ) {
		Log::debug() << "set colour" << attributes.find("colour")->second << endl;
		font_.colour(Colour(attributes.find("colour")->second));
	}
		
	if ( attributes.find("size") != attributes.end() ) {		
		font_.size(tonumber(attributes.find("size")->second));	
		Log::debug() << "set size" << attributes.find("size")->second << endl;
	}
	
	if ( attributes.find("style") != attributes.end() ) {		
		font_.style(attributes.find("style")->second);	
		Log::debug() << "set style" << attributes.find("style")->second << endl;
	}
	
	//push();
	node.visit(*this);
	
	//check(node.data());
	
	
}

void TagConverter::underline(const XmlNode& node) {
	Log::debug() << "uderline" << endl;
	font_.style("underline");
	//push();
	node.visit(*this);
	//check(node.data());		
}
void TagConverter::push()
{
	NiceText nice = top();
	nice.font(font_);
	nice.elevation(elevation_);
	stack<NiceText>::push(nice);
}
void TagConverter::bold(const XmlNode& node) {
	Log::debug() << "bold" << endl;
	font_.style("bold");		
	//push();
	node.visit(*this);
	//check(node.data());			
}
void TagConverter::superscript(const XmlNode& node) {
	Log::debug() << "superscript" << endl;
	elevation_ = SUPERSCRIPT;
	//push();
	node.visit(*this);
	//check(node.data());		
}
void TagConverter::subscript(const XmlNode& node) {
	Log::debug() << "subcript" << endl;
	elevation_ = SUBSCRIPT;
	//push();
	node.visit(*this);
	//check(node.data());		
}
void TagConverter::check(const string& text) {
	
	//Log::dev()<< "Nice text to add ... " << text << x << endl;
	if (!text.empty() ) {				
		top().text(text);		
		label_ += text;
		text_->addNiceText(top());
	}
	

	
}

void TagConverter::entities(ostream& out) 
{
	string path = getEnvVariable("MAGPLUS_HOME") + MAGPLUS_PATH_TO_SHARE_ + "entities.dtd";

	try {
		ifstream in(path.c_str());		
		char c;
		while(in.get(c))
		{
			out << c;
		}
		in.close();
	}
	catch (...) { 
		Log::warning() << "Cannot find entities declaration : " << path << " [Ignore them]" << endl;
	}
}

void TagConverter::decode(const string& line, Text* text)
{
	text_ = text;
	elevation_ = NORMAL;
	
	NiceText nice;
	nice.font(font_);
	nice.elevation(elevation_);
	stack<NiceText>::push(nice);
	XmlReader parser(true);
	XmlTree tree;	

	ostringstream xml;
	xml << "<?xml version='1.0' ?> \n";
	entities(xml);
	xml << "<xml> \n";
	xml << line << "\n";
	xml << "\n</xml>";
	

	try {
		parser.decode(xml.str(), &tree);		
		tree.visit(*this);
	}
	catch (MagicsException& e) {
		Log::debug() << e.what() << endl;
		label_ += line;
		text_->addText(line, font_);
	}	

	} 


