/*
TerraLib - a library for developing GIS applications.
Copyright  2001, 2002, 2003 INPE and Tecgraf/PUC-Rio.

This code is part of the TerraLib library.
This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.

You should have received a copy of the GNU Lesser General Public
License along with this library.

The authors reassure the license terms regarding the warranties.
They specifically disclaim any warranties, including, but not limited to,
the implied warranties of merchantability and fitness for a particular
purpose. The library provided hereunder is on an "as is" basis, and the
authors have no obligation to provide maintenance, support, updates,
enhancements, or modifications.
In no event shall INPE be held liable to any party
for direct, indirect, special, incidental, or consequential damages arising
out of the use of this library and its documentation.
*/

#ifndef TEPDIEUCLIDIANBLENDSTRATEGY_HPP
  #define TEPDIEUCLIDIANBLENDSTRATEGY_HPP

  #include <TeSharedPtr.h>
  #include "TePDIBlendingStrategy.hpp"
  #include "TePDIParameters.hpp"
  
  #include <string>

  /**
   * @brief This is the class for Euclidian blending strategy.
   * @author Emiliano F. Castejon <castejon@dpi.inpe.br>
   * @ingroup PDIBlendingStrategies
   *
   * @note The required parameters are:
   * @param input_raster1 ( TePDITypes::TePDIRasterPtrType ) - 
   *  Input raster 1.
   * @param input_raster2 ( TePDITypes::TePDIRasterPtrType ) - 
   *  Input raster 2.
   * @param output_raster ( TePDITypes::TePDIRasterPtrType ) - Output raster.
   * @param channels1 ( std::vector< int > ) - The channels to process from 
   * input_raster1.
   * @param channels2 ( std::vector< int > ) - The channels to process from 
   * input_raster2.
   * Raster 1 polygon pointer ( related to
   * input_raster1 world reference - Just 1 linear ring allowed ).
   * @param raster2_pol_ptr ( TePDITypes::TePDIPolygonPtrType ) - 
   * Raster 2 polygon pointer ( related to 
   * input_raster2 world reference - Just 1 linear ring allowed ).
   * @param raster2_pol_offset_x ( double ) - Raster 2 polygon box X offset 
   * related to raster1 polygon box in input_raster1 matricial reference
   * ( offset_x = pol2_x - pol1_x ).
   * @param raster2_pol_offset_y ( double ) - Raster 2 polygon box Y offset 
   * related to raster1 polygon box in input_raster1 matricial reference
   * ( offset_y = pol2_y - pol1_y ). 
   *
   * @note The optional parameters are:
   *
   * @param dummy_value (double) - A dummy pixel value for use 
   * in pixels where no data is available ( input pixels with
   * dummy values will be ignored ).  
   * @param draw_blend_lines (double) - Draw the blending lines over
   * the image using the supplied pixel value.
   *
   * @note The output_raster parameters will take the input_raster1 as
   * reference.
   * @note All input rasters must have the same dimensions.
   *
   */
  class PDI_DLL TePDIEuclidianBlendStrategy : public TePDIBlendingStrategy
  {
    friend class TePDIEuclidianBlendSF;
  
    public :
      
      /**
       * @brief Default Destructor
       */
      ~TePDIEuclidianBlendStrategy();
      
      /**
       * @brief Checks if the supplied parameters fits the requirements of each
       * PDI strategy.
       *
       * @note Error log messages must be generated. No exceptions generated.
       *
       * @param parameters The parameters to be checked.
       * @return true if the parameters are OK. false if not.
       */
      bool CheckParameters( 
        const TePDIParameters& parameters ) const;       
      
    protected :
      
      /**
       * @brief Default constructor
       *
       * @param factoryName Factory name.
       */
      TePDIEuclidianBlendStrategy();
      
      /**
       * @brief Runs the algorithm implementation.
       *
       * @return true if OK. false on errors.
       */
      bool Implementation( const TePDIParameters& params );

      /**
       * @brief Transfers the raster1 non-intersected ared to output raster.
       *
       * @return true if OK. false on errors.
       */            
      bool transferRaster1();
      
      /**
       * @brief Generates the two blending lines equations ( a . X + b . y + c = 0 ).
       *
       * @note All geometries must be in the same reference system.
       * @note If ( line1_a == 0 ) <--> X axis parallels (horizontal lines).
       * @note If ( line1_b == 0 ) <--> Y axis parallels (vertical lines).
       *
       * @param polygon1 Raster1 polygon.
       * @param polygon2 Raster2 polygon.
       * @param inter_pol Intersection polygon.
       * @param line1_a Blending line 1 ( related to raster1 polygon ) a 
       * coefficient.
       * @param line1_b Blending line 1 ( related to raster1 polygon ) b
       * coefficient.
       * @param line1_c Blending line 1 ( related to raster1 polygon ) c
       * coefficient.
       * @param line2_a Blending line 2 ( related to raster2 polygon ) a 
       * coefficient.
       * @param line2_b Blending line 2 ( related to raster2 polygon ) b
       * coefficient.
       * @param line2_c Blending line 2 ( related to raster2 polygon ) c
       * coefficient.
       */            
      void generateBlendingLines( const TePolygon& polygon1,  
        const TePolygon& polygon2, const TePolygon& inter_pol, 
        double& line1_a, double& line1_b, double& line1_c,
        double& line2_a, double& line2_b, double& line2_c );  
     
      /**
       * @brief Draws a line over the supplied raster.
       *
       * @note If ( line_a == 0 ) <--> X axis parallels (horizontal lines).
       * @note If ( line_b == 0 ) <--> Y axis parallels (vertical lines).
       *
       * @param raster Raster pointer.
       * @param channel Raster channel.
       * @param value The pixel value to draw.
       * @param line_a Line 1 "a" coefficient.
       * @param line_b Line 1 "b" coefficient.
       * @param line_c Line 1 "c" coefficient.
       */          
      void drawLine( const TePDITypes::TePDIRasterPtrType& raster, 
        unsigned int channel, double value, double line_a, double line_b, 
        double line_c );          
  };
  
#endif



