#include "TePDIArithmetic.hpp"

#include <TeAgnostic.h>
#include "TePDIUtils.hpp"


TePDIArithmetic::TePDIArithmetic()
{
}


TePDIArithmetic::~TePDIArithmetic()
{
}


void TePDIArithmetic::ResetState( const TePDIParameters& params )
{
  TePDILevelRemap::ResetState( params );
}


bool TePDIArithmetic::CheckParameters( 
  const TePDIParameters& parameters ) const
{
  /* Checking for the correct allowed algorithm types */

  TePDIArithmetic::TePDIArithmeticType arithmetic_type;
  if( ! parameters.GetParameter( "arithmetic_type", arithmetic_type ) ) {
    TEAGN_LOGERR( "Missing parameter: arithmetic_type" );
    return false;
  }
  if( ( arithmetic_type != TePDIAType1 ) &&
      ( arithmetic_type != TePDIAType2 ) &&
      ( arithmetic_type != TePDIAType3 ) &&
      ( arithmetic_type != TePDIAType4 ) &&
      ( arithmetic_type != TePDIAType5 ) ) {

    TEAGN_LOGERR( "Invalid parameter: arithmetic_type" );
    return false;
  }

  /* Checking input_image1 */

  TePDITypes::TePDIRasterPtrType inRaster1;
  if( ! parameters.GetParameter( "input_image1", inRaster1 ) ) {

    TEAGN_LOGERR( "Missing parameter: input_image1" );
    return false;
  }
  if( ! inRaster1.isActive() ) {

    TEAGN_LOGERR( "Invalid parameter: input_image1 inactive" );
    return false;
  }
  if( inRaster1->params().status_ == TeRasterParams::TeNotReady ) {

    TEAGN_LOGERR( "Invalid parameter: input_image1 not ready" );
    return false;
  }
  
  /* Checking output_image */

  TePDITypes::TePDIRasterPtrType outRaster;
  if( ! parameters.GetParameter( "output_image", outRaster ) ) {

    TEAGN_LOGERR( "Missing parameter: output_image" );
    return false;
  }
  if( ! outRaster.isActive() ) {

    TEAGN_LOGERR( "Invalid parameter: output_image inactive" );
    return false;
  }
  if( outRaster->params().status_ == TeRasterParams::TeNotReady ) {

    TEAGN_LOGERR( "Invalid parameter: output_image not ready" );
    return false;
  }
  
  /* Checking input_image2 */

  TePDITypes::TePDIRasterPtrType inRaster2;
  if( arithmetic_type != TePDIAType1 ) {
    if( ! parameters.GetParameter( "input_image2", inRaster2 ) ) {

      TEAGN_LOGERR( "Missing parameter: input_image2" );
      return false;
    }
    if( ! inRaster2.isActive() ) {

      TEAGN_LOGERR( "Invalid parameter: input_image2 inactive" );
      return false;
    }
    if( inRaster2->params().status_ == TeRasterParams::TeNotReady ) {

      TEAGN_LOGERR( "Invalid parameter: input_image2 not ready" );
      return false;
    }
  }

  if( arithmetic_type != TePDIAType1 ) {
    TeRasterParams params1 = inRaster1->params();
    TeRasterParams params2 = inRaster2->params();

    TEAGN_TRUE_OR_RETURN( params1.ncols_ == params2.ncols_,
      "Image1 and Imaga2 have different columns number" );
    TEAGN_TRUE_OR_RETURN( params1.nlines_ == params2.nlines_,
      "Image1 and Imaga2 have different lines number" );
  }

  /* channel parameters checking */

  int img1_chan;
  TEAGN_TRUE_OR_RETURN( parameters.GetParameter( "img1_chan", img1_chan ),
    "Missing parameter: img1_chan" );
  TEAGN_TRUE_OR_RETURN( ( img1_chan < inRaster1->nBands() ) &&
    ( img1_chan >= 0 ), "Invalid parameter: img1_chan" );

  int img2_chan;
  if( arithmetic_type != TePDIAType1 ) {
    TEAGN_TRUE_OR_RETURN( parameters.GetParameter( "img2_chan", img2_chan ),
        "Missing parameter: img2_chan" );
    TEAGN_TRUE_OR_RETURN( ( img2_chan < inRaster2->nBands() ) &&
      ( img2_chan >= 0 ), "Invalid parameter: img2_chan" );
  }

  /* Checking for gain and offset */

  TEAGN_TRUE_OR_RETURN(
    parameters.CheckParameter< double >( "gain" ),
    "Missing parameter: gain" );

  TEAGN_TRUE_OR_RETURN(
    parameters.CheckParameter< double >( "offset" ),
    "Missing parameter: offset" );
    
  /* Checking photometric interpretation */
  
  TEAGN_TRUE_OR_RETURN( ( 
    ( inRaster1->params().photometric_[ img1_chan ] == 
      TeRasterParams::TeRGB ) ||
    ( inRaster1->params().photometric_[ img1_chan ] == 
      TeRasterParams::TeMultiBand ) ),
    "Invalid paramter - input_image1 (invalid photometric interpretation)" );
    
  if( inRaster2.isActive() ) {
    TEAGN_TRUE_OR_RETURN( ( 
      ( inRaster2->params().photometric_[ img2_chan ] == 
        TeRasterParams::TeRGB ) ||
      ( inRaster2->params().photometric_[ img2_chan ] == 
        TeRasterParams::TeMultiBand ) ),
      "Invalid parameter - input_image2 (invalid photometric interpretation)" );    
  }

  return true;
}


bool TePDIArithmetic::RunImplementation()
{
  /* Parameters aquirement */

  TePDIArithmeticType arithmetic_type;
  params_.GetParameter( "arithmetic_type", arithmetic_type );

  TePDITypes::TePDIRasterPtrType inRaster1;
  params_.GetParameter( "input_image1", inRaster1 );

  TePDITypes::TePDIRasterPtrType outRaster;
  params_.GetParameter( "output_image", outRaster );

  TePDITypes::TePDIRasterPtrType inRaster2;
  if( arithmetic_type != TePDIAType1 ) {
    params_.GetParameter( "input_image2", inRaster2 );
  }

  int img1_chan;
  params_.GetParameter( "img1_chan", img1_chan );

  int img2_chan;
  if( arithmetic_type != TePDIAType1 ) {
   params_.GetParameter( "img2_chan", img2_chan );
  }

  double gain;
  params_.GetParameter( "gain", gain );

  double offset;
  params_.GetParameter( "offset", offset );
  
  bool normalize_output_flag = false;
  if( params_.CheckParameter< int >( "normalize_output" ) ) {
    
    int temp_int = 0;
    params_.GetParameter( "normalize_output", temp_int );
    if( temp_int == 1 ) {
      normalize_output_flag = true;
    }
  }

  /* Setting the output raster */
  
  TeRasterParams outRaster_params = outRaster->params();
  
  outRaster_params.nBands( 1 );
  if( inRaster1->projection() != 0 ) {
    TeSharedPtr< TeProjection > proj( TeProjectionFactory::make( 
      inRaster1->projection()->params() ) );
    outRaster_params.projection( proj.nakedPointer() );
  }
  
  const TeBox bbox = inRaster1->params().boundingBox();
  
  outRaster_params.boundingBoxLinesColumns( bbox.x1(), bbox.y1(),
    bbox.x2(), bbox.y2(), inRaster1->params().nlines_,
    inRaster1->params().ncols_, TeBox::TeUPPERLEFT );
  outRaster_params.setPhotometric( TeRasterParams::TeMultiBand, -1 );

  TEAGN_TRUE_OR_RETURN( outRaster->init( outRaster_params ),
    "Output raster reset error" );

  /* Switching the arithmetic function pointer */

  remap_func_3_ptr_type remap_func_3 = 0;
  remap_func_4_ptr_type remap_func_4 = 0;

  switch( arithmetic_type ) {
    case TePDIAType1 :
    {
      remap_func_3 = &type1_arith;
      break;
    }
    case TePDIAType2 :
    {
      remap_func_4 = &type2_arith;
      break;
    }
    case TePDIAType3 :
    {
      remap_func_4 = &type3_arith;
      break;
    }
    case TePDIAType4 :
    {
      remap_func_4 = &type4_arith;
      break;
    }
    case TePDIAType5 :
    {
      remap_func_4 = &type5_arith;
      break;
    }
    default :
    {
      TEAGN_LOG_AND_RETURN( "Invalid arithmetic type" );
    }
  }

  /* Running remapping function */

  if( remap_func_3 != 0 ) {
    TEAGN_TRUE_OR_RETURN( RemapLevels( inRaster1, remap_func_3, img1_chan, 0, 
      gain, offset, 
      normalize_output_flag, outRaster ), "Level remapping error" );
  } else if( remap_func_4 != 0 ) {
    TEAGN_TRUE_OR_RETURN( RemapLevels( inRaster1, inRaster2, remap_func_4, 
      img1_chan, img2_chan, 0,
      gain, offset, normalize_output_flag, outRaster ), 
      "Level remapping error" );
  } else {
    TEAGN_LOG_AND_RETURN( "Invalid remap function" );
  }

  return true;
}




