/* MouseOverlay.java
 * =========================================================================
 * This file is part of the SWIRL Library - http://swirl-lib.sourceforge.net
 * 
 * Copyright (C) 2005-2008 Universiteit Gent
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * A copy of the GNU General Public License can be found in the file
 * LICENSE.txt provided with the source distribution of this program (see
 * the META-INF directory in the source jar). This license can also be
 * found on the GNU website at http://www.gnu.org/licenses/gpl.html.
 * 
 * If you did not receive a copy of the GNU General Public License along
 * with this program, contact the lead developer, or write to the Free
 * Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 * 
 */

package be.ugent.caagt.swirl.mouse;

import be.ugent.caagt.swirl.DefaultGenericSelectionModel;
import be.ugent.caagt.swirl.GenericSelectionModel;
import java.awt.Container;
import java.awt.Graphics;
import java.awt.event.MouseEvent;
import java.awt.event.MouseListener;
import java.awt.event.MouseMotionListener;
import javax.swing.JComponent;


/**
 * Invisible component which overlays another component in order to
 * handle mouse operations for it.<p>
 * The mouse operations are delegated to a {@link
 * MouseTool} which in turn is managed by a {@link GenericSelectionModel}.
 */
public class MouseOverlay extends JComponent
implements MouseListener, MouseMotionListener {
    
    //
    private GenericSelectionModel<MouseTool> mouseToolModel;

    /**
     * Return the underlying model which keeps track of what mouse tool
     * is currently active.
     */
    public GenericSelectionModel<MouseTool> getMouseToolModel() {
        return mouseToolModel;
    }

    /**
     * Change the tool to be used by this overlay.<p>
     * @link throws IllegalArgumentException when this tool cannot handle
     * the parent of this panel.
     */
    public void setTool(MouseTool tool) {
	if (tool == null)
	    mouseToolModel.setSelection(null);
	else {
	    if (! tool.canHandle((JComponent)getParent()))
		throw new IllegalArgumentException("Incorrect mouse tool for this panel");
	    mouseToolModel.setSelection(tool);
	}
    }
    
    /**
     * Creates a new instance of this class.
     * @param parent Parent component for this overlay
     * @param mouseToolModel Underlying model which keeps track of
     * the current mouse tool to be used
     */
    public MouseOverlay(JComponent parent, GenericSelectionModel<MouseTool> mouseToolModel) {
        this.mouseToolModel = mouseToolModel;
	
	addMouseListener(this);
	addMouseMotionListener(this);
	setFocusable(false);
	
	parent.add(this);
    }
    
    /**
     * Creates a new instance of this class with a default model.
     * @param parent Parent component for this overlay
     */
    public MouseOverlay(JComponent parent) {
	this (parent, new DefaultGenericSelectionModel<MouseTool>());
    }
    
    /**
     * Overridden to make this component always the same size as its
     * parent.
     */
    @Override public void doLayout() {
	Container parent = getParent();
	if (parent != null) {
	    setSize(parent.getWidth(),parent.getHeight());
	    setLocation(0, 0);
	}
    }
    
    /* ============================================================
     * MOUSE PRESSED / CLICKED / POPUP
     * ============================================================ */
    
    //
    private boolean wasPopup;
    
    //
    private boolean wasDragged;
    
    //
    private MouseEvent pressedEvent;
    
    //
    private MouseEvent savedEvent;
    
    //
    private MouseEvent previousEvent;
    
    //
    private MouseHandler activeHandler;
    
    /**
     * If the event is a popup trigger, dispatch it as a popup event to
     * the first handler that takes it. Otherwise remember state so it can
     * be dispatched during the first drag or release.
     */
    public void mousePressed(MouseEvent mouseEvent) {
	activeHandler = null;
	wasDragged = false;
	wasPopup = mouseEvent.isPopupTrigger();
	JComponent parent = (JComponent)getParent();
	if (wasPopup) {
            MouseTool tool = mouseToolModel.getSelection();
	    if (tool != null) {
		for (MouseHandler handler : tool) {
		    handler.doPopup(parent, mouseEvent);
		    if (mouseEvent.isConsumed())
			return;
		}
	    }
	    wasPopup = false; // popups not handled
	}
	pressedEvent = mouseEvent;
    }
    
    /**
     * Dispatch the dragged event to the currently active handler.
     * <p>If this is the first drag, first dispatch the cached pressed event to
     * the first handler that will take it and mark that handler as active.
     *
     * This implementation first calls {@link #handleDelayedPressed} when needed
     * and then {@link #handleDragged} if a handler was marked as active.
     */
    public void mouseDragged(MouseEvent mouseEvent) {
	if (wasPopup)
	    return; // a popup has already been processed
	
	if (! wasDragged) {
	    wasDragged = true;
	    handleDelayedPressed(pressedEvent);
	}
	if (activeHandler != null)
	    handleDragged(mouseEvent);
    }
    
    /**
     * Handles delayed pressed event. Dispatches the mouse event to
     * the first handler that will take it and mark that handler as active.
     */
    protected void handleDelayedPressed(MouseEvent mouseEvent) {
	JComponent parent = (JComponent)getParent();
            MouseTool tool = mouseToolModel.getSelection();
	if (tool != null) {
	    for (MouseHandler handler : tool) {
		handler.doMousePressed(parent, mouseEvent);
		previousEvent = mouseEvent;
		if (mouseEvent.isConsumed()) {
		    activeHandler = handler;
		    break;
		}
	    }
	}
    }
    
    /**
     * Handles dragged event. Dispatches the event to the active handler.
     */
    protected void handleDragged(MouseEvent mouseEvent) {
	JComponent parent = (JComponent)getParent();
	activeHandler.doMouseDragged(parent, mouseEvent, previousEvent, pressedEvent);
	savedEvent = previousEvent; // saved for paintDragging
	previousEvent = mouseEvent;
    }
    
    /**
     * If a drag event preceeds this event, send it to the active handler.
     * Otherwise, check whether this is a popup trigger.
     * If true, hand it as a popup event to the first handler that will take it.
     * Otherwise send it as a clicked event to the first handler that will take it.
     *
     * Dispatch the released event to the currently active handler.
     */
    public void mouseReleased(MouseEvent mouseEvent) {
	if (wasPopup)
	    return; // a popup has already been processed
	
	JComponent parent = (JComponent)getParent();
            MouseTool tool = mouseToolModel.getSelection();
	if (wasDragged) {
	    if (activeHandler != null)
		activeHandler.doMouseReleased(parent, mouseEvent, pressedEvent);
	} else if (mouseEvent.isPopupTrigger()) {
	    if (tool != null) {
		for (MouseHandler handler : tool) {
		    handler.doPopup(parent, mouseEvent);
		    if (mouseEvent.isConsumed())
			break;
		}
	    }
	} else {
	    if (tool != null) {
		for (MouseHandler handler : tool) {
		    handler.doMouseClicked(parent, mouseEvent);
		    if (mouseEvent.isConsumed())
			break;
		}
	    }
	}
	wasDragged = false;
	activeHandler = null;
    }
    
    /* ============================================================
     * OTHER MOUSE EVENTS
     * ============================================================ */
    
    // implements MouseListener
    public void mouseClicked(MouseEvent mouseEvent) {
	// no action, clicks are processed by mouseReleased
    }
    
    // implements MouseListener
    public void mouseEntered(MouseEvent mouseEvent) {
	// no action
    }
    
    // implements MouseListener
    public void mouseExited(MouseEvent mouseEvent) {
	// no action
    }
    
    // implements MouseMotionListener
    public void mouseMoved(MouseEvent mouseEvent) {
	// no action
    }
    
    /* ============================================================
     * VISUAL FEEDBACK WHILE DRAGGING
     * ============================================================ */
    
    /**
     * Let the active handler provide mouse feedback during a mouse gesture.
     */
    @Override protected void paintComponent(Graphics g) {
	if (wasDragged && activeHandler != null)
	    activeHandler.paintDragging((JComponent)getParent(), g, previousEvent, savedEvent, pressedEvent);
    }
    
}
