//
// $Id: SpectrumList_ChargeStateCalculatorTest.cpp 4922 2013-09-05 22:33:08Z pcbrefugee $
//
//
// Original author: Matt Chambers <matt.chambers <a.t> vanderbilt.edu>
//
// Copyright 2008 Vanderbilt University - Nashville, TN 37232
//
// Licensed under the Apache License, Version 2.0 (the "License"); 
// you may not use this file except in compliance with the License. 
// You may obtain a copy of the License at 
//
// http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software 
// distributed under the License is distributed on an "AS IS" BASIS, 
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. 
// See the License for the specific language governing permissions and 
// limitations under the License.
//


#include "SpectrumList_ChargeStateCalculator.hpp"
#include "pwiz/utility/misc/unit.hpp"
#include "pwiz/data/msdata/examples.hpp"
#include "pwiz/data/msdata/TextWriter.hpp"
#include "pwiz/data/common/CVTranslator.hpp"
#include "pwiz/utility/misc/Std.hpp"

using namespace pwiz::util;
using namespace pwiz::msdata;
using namespace pwiz::analysis;

ostream* os_ = 0;

ostream& operator<< (ostream& os, const vector<double>& v)
{
    os << "(";
    for (size_t i=0; i < v.size(); ++i)
        os << " " << v[i];
    os << " )";
    return os;
}

struct TestChargeStateCalculator
{
    // space-delimited doubles
    const char* inputMZArray;
    const char* inputIntensityArray;
    const char* inputChargeStateArray;
    const char* inputActivationTypeArray;
    double inputPrecursorMZ;

    bool overrideExistingChargeState;
    int minMultipleCharge;
    int maxMultipleCharge;
    double singlyChargedFraction;
    const char* chargeStateArray;
    bool useMakeMS2Behavior;

};

TestChargeStateCalculator testChargeStateCalculators[] =
{
    { "1 2 3 4 5", "10 20 30 40 50", "1", "CID", 5,
      true, 2, 3, 0.9, "1", false },

    { "1 2 3 4 5", "10 20 30 40 50", "1 2 3", "CID", 5,
      true, 2, 3, 0.9, "1", false },

    { "1 2 3 4 5", "10 20 30 40 50", "2 3", "CID", 5,
      true, 2, 3, 0.9, "1" },

    { "1 2 3 4 5", "10 20 30 40 50", "", "CID", 2.5,
      true, 2, 3, 0.9, "2 3", false },

    // input charge of 0 should be treated like no charge
    { "1 2 3 4 5", "10 20 30 40 50", "0", "CID", 2.5,
      true, 2, 3, 0.9, "2 3", false },

    { "1 2 3 4 5", "10 20 30 40 50", "2", "CID", 2.5,
      true, 3, 4, 0.9, "3 4 5", false },

    { "1 2 3 4 5", "10 20 30 40 50", "3 4 5", "CID", 2.5,
      true, 3, 4, 0.9, "3 4 5", false },

    { "1 2 3 4 5", "10 20 30 40 50", "3", "CID", 2.5,
      true, 2, 2, 0.9, "2", false },

    { "1 2 3 4 5", "10 20 30 40 50", "", "CID", 5,
      false, 2, 3, 0.9, "1", false },

    { "1 2 3 4 5", "10 20 30 40 50", "", "CID", 2.5,
      false, 2, 3, 0.9, "2 3", false },

    // input charge of 0 should be treated like no charge
    { "1 2 3 4 5", "10 20 30 40 50", "0", "CID", 2.5,
      false, 2, 3, 0.9, "2 3", false },

    { "1 2 3 4 5", "10 20 30 40 50", "1", "CID", 2.5,
      false, 2, 3, 0.9, "1", false },

    { "1 2 3 4 5", "10 20 30 40 50", "2 3", "CID", 5,
      false, 2, 3, 0.9, "1 2 3" },

    { "1 2 3 4 5", "10 20 30 40 50", "2 3", "CID", 2.5,
      false, 2, 4, 0.9, "2 3 4", false },

    // 20090326_Yest_LysC_1x6ETD_longGrad_01.05864.05864.7.dta
    { "100.2316 102.1654 114.1007 142.3967 147.1805 148.0554 158.1545 159.1372 204.4804 211.9715 216.0852 217.2059 218.3450 219.1636 230.0421 238.6137 245.4243 246.2292 246.8875 259.0931 260.1801 267.7267 272.4216 273.2703 274.2005 287.1424 288.3069 294.8814 295.7421 296.4242 299.0081 303.5564 305.5432 316.3794 317.2255 326.3209 329.1799 339.1173 341.1064 342.1323 344.3056 345.2133 347.3302 355.1378 356.2534 367.5323 370.0612 371.1212 372.3159 380.9781 383.1261 384.1235 385.7302 386.3345 388.3026 389.2090 390.2366 390.9072 400.0575 404.4477 411.1859 416.4116 419.3402 426.5632 427.2871 428.3036 429.2469 431.1707 432.6296 437.1998 438.0903 445.0817 446.1855 456.3277 457.2303 459.3148 460.3862 461.1172 463.4081 467.3364 472.6896 474.4414 475.2522 476.0383 477.0409 479.3283 480.0349 484.2828 485.2685 487.2802 490.2423 491.2538 492.0312 502.1874 503.1729 504.3073 511.1371 512.5167 513.3364 513.9659 528.2398 529.0281  530.1065 531.5367 533.0511 533.9404 535.1735 536.2697 537.0353 539.3831 542.0954 546.0742 547.6926 552.2240 553.3661 556.2790 557.2021 557.9160 559.4593 562.8941 565.2065 568.3622 569.3952 573.2255 574.2047 575.4551 576.5182 581.0674 583.7980 585.7489 586.8701 587.5425 589.6053 590.2707 591.2530 592.2204 593.2232 597.5678 598.3864 599.1964 600.0155 604.6537 606.2562 607.7433 610.8060 612.3967 613.2997 614.2175 615.4516 617.8012 622.2905 624.6938 626.3412 631.5930 632.6418 633.3755 636.5936 639.9253 640.9366 643.3186 644.1926 645.7126 647.3211 648.4583 649.2852 650.5535 651.3624 654.3332 657.2921 658.3475 659.8166 662.4810 664.3082 665.6356 668.1566 669.6567 670.2629 671.4838 672.3119 675.8893 677.0720 678.0397 680.3560 682.3088 683.4887 684.3668 689.4347 692.5178 693.4395 694.9922 696.2354 698.6488 699.7452 700.6685 701.5429 702.9158 704.6758 705.5378 706.4117 707.9384 708.7421 710.2140 711.4921 713.7796 714.5510 716.2544 717.9063 718.5756 719.7270 723.4227 725.6956 726.4857 727.2152 728.2474 729.0570 729.9482 730.6264 732.6265 733.2515 734.4771 738.3369 739.5363 740.2814 741.5637 749.6926 752.5176 753.2125 754.4744 755.3438 757.0353 758.0003 759.4985 760.3941 761.7000 762.8120 764.7708 765.6356 767.5706 768.6167 769.8208 771.1314 771.8177 773.3740 774.5911 775.1982 775.9609 776.8069 777.4185 778.2676 779.3928 781.2325 783.3708 784.3259 785.1341 786.0793 787.1502 787.9260 789.0554 790.2109 791.4124 792.5274 793.5887 794.5103 795.2405 796.2149 796.9224 799.5650 801.3645 803.8350 804.4755 805.2210 806.4171 807.4406 808.6633 809.6253 811.9946 814.5346 817.0883 818.0493 819.0435 820.9210 821.8782 824.1953 825.4860 826.4271 829.5283 830.4576 832.2156 832.8347 834.4541 836.0142 836.9937 838.1580 839.6205 841.3862 843.0901 843.8099 845.3442 846.6658 847.6355 850.0297 852.0378 853.2718 854.4008 856.4588 857.4885 859.5439 861.4915 862.1750 863.2495 864.8409 867.3325 868.4129 869.3459 870.4559 871.7208 872.5671 873.2695 874.4902 875.4397 877.8188 880.2540 882.8533 883.6810 884.6408 887.3281 889.8179 893.5707 894.5350 898.4689 899.2938 900.3057 901.4645 902.7376 903.6931 904.5297 910.3437 911.3130 912.6046 913.4197 914.2205 916.8642 918.2337 920.0554 921.9939 922.9043 924.7449 925.5725 926.3651 927.3116 928.2539 929.2107 930.2380 932.0637 933.0691 933.6707 935.9348 937.2978 939.5454 940.9717 941.9307 943.8423 944.8279 949.0307 952.2651 953.4227 954.1648 955.2634 956.2859 957.3259 957.9824 959.4771 960.2833 962.5200 963.5037 965.2302 967.8181 968.6046 969.2939 970.3937 971.6548 972.6874 976.8770 978.5676 980.8109 981.4683 982.4684 983.2844 983.9832 986.5053 987.1211 989.2344 993.3611 994.6191 996.0191 997.5510 999.1077 1000.4880 1002.7420 1005.3079 1006.2520 1007.1972 1008.3851 1009.3384 1010.3657 1012.2654 1013.2116 1014.0049 1015.4166 1017.0412 1017.6729 1019.2534 1021.5500 1023.4800 1024.4152 1025.5474 1026.6501 1027.8084 1028.9393 1030.0382 1031.1877 1031.8257 1033.3679 1034.3296 1035.4045 1036.1647 1037.3960 1038.4543 1039.1770 1040.1565 1042.1265 1043.2186 1044.4426 1045.5302 1046.7881 1047.5444 1049.1858 1050.2570 1051.4323 1053.2034 1054.4518 1055.5977 1056.4724 1057.2979 1058.4558 1059.4180 1060.2227 1060.9368 1062.2380 1063.0264 1063.7651 1065.1206 1066.2021 1067.3129 1068.5062 1069.2571 1070.4927 1071.5564 1072.6907 1073.7375 1074.5313 1075.2959 1076.3043 1077.2311 1078.0673 1079.6613 1081.9634 1083.8835 1084.7516 1087.9388 1088.6055 1090.0548 1090.7660 1091.6577 1092.5103 1093.2368 1093.8492 1094.5607 1095.9825 1096.6309 1098.0442 1098.8674 1100.5309 1101.6241 1103.2767 1104.0603 1106.0070 1106.9993 1109.4095 1112.0243 1114.4689 1115.4701 1116.4965 1118.4481 1120.3474 1122.0121 1123.3585 1124.0900 1126.5304 1127.5829 1129.7465 1131.1656 1133.5940 1134.8131 1136.3689 1137.7687 1138.9125 1139.9618 1140.6614 1142.9773 1143.7170 1144.4061 1145.0295 1148.4452 1150.0396 1151.6088 1153.8888 1155.4089 1157.7570 1159.0519 1160.5687 1163.3629 1164.2449 1165.6302 1167.2775 1170.1827 1171.9503 1173.6943 1178.1985 1178.9193 1180.1576 1181.7102 1183.7064 1187.6965 1189.9983 1191.4650 1192.9443 1193.8871 1194.8962 1195.7772 1196.4039 1197.9064 1198.9985 1199.7401 1201.5292 1206.3419 1207.0554 1207.7147 1208.9205 1210.1613 1211.1132 1211.7198 1212.8514 1213.9382 1215.0614 1215.8712 1216.8856 1218.7352 1220.2765 1221.1106 1221.7494 1222.8069 1224.0594 1224.9103 1225.7242 1226.5060 1227.3594 1229.0797 1229.9269 1230.7487 1231.5803 1232.2225 1232.8799 1235.1190 1236.3748 1236.9987 1237.8800 1240.5161 1242.5454 1245.7117 1248.2661 1249.0128 1250.0117 1253.4581 1254.8519 1258.7023 1261.1508 1262.4562 1263.8649 1266.9794 1268.4827 1271.5897 1272.6104 1276.4093 1278.5415 1279.4795 1280.1755 1283.2010 1284.1530 1285.4436 1286.6320 1287.8259 1288.8115 1292.2874 1294.6595 1295.3514 1296.9506 1297.6798 1299.2562 1300.0573 1301.4045 1303.1136 1303.7776 1304.8346 1305.4403 1306.5542 1307.6301 1308.7888 1310.1801 1311.2026 1312.3358 1313.4727 1314.5222 1315.2391 1316.4742 1317.2528 1317.9081 1319.5140 1322.6118 1323.4581 1324.4717 1325.4580 1326.8948 1328.9554 1329.6587 1330.7810 1331.5580 1332.9672 1338.9171 1340.0092 1341.0057 1342.1337 1343.0525 1343.9711 1344.8038 1346.7933 1348.2371 1351.8527 1352.8333 1353.9034 1354.5902 1357.1407 1359.8356 1368.5875 1370.5602 1371.7577 1372.9083 1375.4344 1382.9457 1383.8612 1385.3168 1387.3273 1390.1798 1391.5629 1392.4547 1393.5638 1394.3420 1395.7352 1397.0741 1398.1329 1398.8969 1400.7195 1402.5508 1403.6631 1407.2289 1410.7687 1411.7717 1412.4025 1413.7531 1415.5807 1416.5509 1417.6246 1418.7306 1419.4539 1420.1559 1421.1342 1421.9913 1422.8665 1423.7020 1425.9493 1427.1515 1428.4490 1429.0841 1430.0243 1431.8527 1434.8483 1436.1760 1436.9138 1437.8707 1439.5657 1442.8479 1450.1022 1451.2184 1453.8065 1456.0328 1456.6864 1458.0870 1459.1053 1461.9120 1462.6119 1463.7512 1469.5785 1473.6619 1475.8344 1478.1866 1479.9750 1480.7377 1482.8094 1484.3290 1485.4197 1487.3420 1493.9686 1496.9908 1498.8871 1499.9709 1501.5465 1503.8168 1505.7375 1506.5231 1507.2571 1507.9613 1509.2472 1510.8177 1511.9337 1512.8951 1516.2877 1516.9982 1517.9640 1519.2213 1521.8727 1523.5814 1524.6912 1525.5077 1526.3093 1528.0574 1528.8844 1529.6288 1530.7858 1531.8386 1533.4905 1534.8173 1536.1672 1537.7214 1539.0460 1539.8561 1540.9690 1542.5066 1544.0828 1544.8269 1545.9760 1546.8141 1547.5980 1548.7726 1549.4471 1550.2883 1551.1202 1552.0406 1552.9171 1554.0577 1555.2815 1556.1970 1556.9130 1557.9539 1559.1941 1560.8824 1562.7125 1563.6962 1565.8517 1566.8927 1567.9371 1568.8998 1569.9613 1571.1643 1572.1046 1572.8573 1573.7242 1574.4230 1577.9452 1578.9706 1579.7214 1580.5770 1581.3318 1582.5018 1583.2159 1584.3450 1585.7789 1586.8810 1588.1760 1589.1420 1590.6589 1591.7211 1593.4823 1594.6254 1595.4083 1596.3741 1597.6932 1598.7032",
      "  3.2418   2.2150  11.0869   2.2778  50.9633   2.2093  19.3287  12.4510   8.9364   3.3966  73.3816  35.4844  17.8207  12.0066   6.4694  15.7834   8.2228   6.0483   8.6328   2.2224  12.6325  17.8730  12.9276   3.4544  11.7616 142.6698  69.7587  11.4609 216.5886   1.8377   6.5047  48.8038   4.3386  46.1475   4.4074  18.6191   4.6289  20.0546   2.9302  11.5797  21.3002  15.2543   7.4326   6.0572   1.9788   6.9874 155.7842  10.7804  11.7532  24.5477  24.2512   9.8384  31.9294  10.7230  80.0561  21.8453   4.2641   7.5264   3.4045   1.9816   6.0726   5.6346  25.9069  18.8623  18.5661  45.0142  16.8058   2.2163   1.7374  25.7332  30.4077   8.3505  26.0232  27.2097  12.4612   2.6889   4.3828   3.8749  25.7056   4.8666   7.9613  26.5360  72.1348  46.1052   6.2061  38.5774  32.5843 121.5198  36.7320   9.0994   9.5700  13.7784   8.6465   3.4076  29.5869  10.2263  38.0570 222.2102  28.3271  36.8888 527.9606 559.1112 1000.0000  42.5538  29.9167  16.2195   9.4391 135.0116  10.7759   7.4709   4.8641   9.6174   8.8924  11.7112   9.1210   2.2504  14.3277   8.8646   2.4484   2.6109  10.5663  31.9595  14.9439  51.2389   9.4340   9.1117   3.9247   2.2127  11.9327  50.3476  40.0407  21.7258  28.2477 195.3547  56.9475 101.0778  18.8569  20.2152   9.4084  12.7033  15.9751  26.1827  25.2666  11.3724  12.1166 228.3339  51.6904  13.8419  19.1409  67.5237   7.0505  20.1022  11.5966  12.5294  29.2311  12.6644  89.0887  42.5493  22.1867 284.1091  75.3782   9.5436  17.6771  10.8152   7.7030  16.7860  15.3062   3.8434   8.9797  33.1646  20.8195  31.6203  42.3290  15.6282  32.7879  13.7429   5.5781 159.8926  49.8017  19.5154  10.0519   8.1359  49.1926  27.8555  12.2163  23.6995   9.8478   6.5212  27.3637  44.8395  12.2819  73.2868  79.3788  47.1691  41.8323  17.7027  47.3262 100.6576  30.6365  89.0225 111.0906  85.2993  47.1381  12.1153  16.9150  76.4411 126.5558 132.3647  17.5699  20.0839  49.8344  41.3531  89.9600  40.7703   4.1909  30.8658  17.8780  14.6302   3.6403  13.5900  26.9555  25.8994  55.4275   7.4319   8.8972  25.1711  10.7609  24.7685  43.4959  95.7611  12.8040  29.0010  32.5526  91.6369  78.5863 153.4670  99.6896  24.0892  46.4224  23.2663  59.5909  20.8479  44.3557   6.2475  21.2062  52.1075  36.3742  22.1843   4.5882  75.2949   8.3783  15.3799  46.0072  22.8688 113.2306  32.0690  18.1297  10.4917 115.5912 111.7756 401.1757 122.0876 279.1254  39.4676  62.1709   7.5180   4.9301  23.2444  25.8239   6.1557  76.0793  62.5210  49.6156  95.2091  36.9798  12.7502  16.5192  39.1498  56.7536   5.8498  34.9025  36.0609  13.3717  10.4189  19.5955  24.5496  11.9503  11.0001  25.1156  33.0185  26.1695   8.0101  25.6687  46.1801   8.8585  79.3093  60.3437  16.0549 128.9062  15.0835   7.2337  43.6132   3.9066  14.3446  15.2733  13.3906  15.3547  13.4817   4.1137  17.9837  37.4059 124.3921  82.3209  49.4677  37.0553  16.6742  10.0291  67.2750 132.8354  59.9224  39.5181  30.8779  21.0157  22.9907  13.9648   7.5204  30.9982  59.8321  82.4379  11.7083   5.0482  69.8900  17.8834  27.8587  27.7243  37.2471   9.1249 133.9667  81.9941   7.9074  10.7635  33.3638  51.3460  28.5408 120.0791  64.7641 130.6045  57.2454  43.0353  76.0876  47.7058  35.8746  93.3957  11.5280  86.3272  48.0687  17.4323  17.9122  32.9358  64.8450  65.4608 237.2285  63.2972  14.5888  12.9763  22.4495  26.9346  20.1092  22.2112   5.2457  26.2077  31.5904  26.9079   3.1633   7.7255  27.5440  55.5476  12.5317  12.2895  29.7237   7.8901  40.7258  29.4631  59.8513  42.8277  33.9833  11.4439  14.0811  27.8080  24.0819  20.2655   5.8329  95.8489  10.7876  27.9724  11.7674  65.8042   17.1769   75.2262  108.2203   54.5236  115.9897   53.0610   64.9889   56.1649   36.3425   43.3307   41.8708   12.8255   39.2026   10.7519    7.9070   28.0647   96.8431  272.0151   40.1899   88.3676  125.8584   48.9294   62.9378   38.2105   32.8664  173.3573   35.0060   12.7923   47.6801   24.0867  316.5758   58.3904  126.7407   50.1308   94.4073   33.2618   30.6582   56.5687    8.3901  227.6603  114.3906   32.3215   35.7905   86.9787  335.9061  253.4367  274.9273  308.5602  370.9917  115.7313   18.8021  110.6610   71.3167   72.6248   31.7910    2.7195   71.2045   11.4435    9.5659   60.2547  146.5376   49.1726   20.5347   26.9450    8.8278   17.1559   50.6427   17.5154   13.8876   35.0394   36.4334    5.2007   58.4728   91.9885   14.8054   51.3077   43.9949   22.6586   16.0143   38.2009   22.9566   16.7180   18.8799   42.6049   27.6320   22.4117   19.5303   15.1561   21.4431   95.0123   64.2030    3.9374  108.4517   37.7694    4.5797   31.8731   10.5401   12.9173    4.3468  100.2101    9.2353   59.5649   18.6624   14.4319   18.9445   41.4509  182.4317   54.1046   68.3520   48.8382   34.0989   36.4116    8.2339   23.0928   40.2105  105.4458   18.0327   14.3393   24.3977   19.6919   26.4091   38.8509   44.9871    2.7207   45.5520   49.5328    4.0999   36.4504   23.1417   65.7151    9.3434   82.1442   30.5353   17.5419   75.2710   13.3796   16.3657   28.0236   38.3099  136.1610   38.2127   13.3162   24.8838   32.5043   37.0730   11.8400    2.3494  104.0438   72.1625   65.9864   50.1828  145.5539  142.2270   95.9832   34.1003   74.0660   37.7689  262.6055  227.1480   37.2455   72.0068  133.5652   70.6917   58.9968   33.5739  136.1320  122.0077   34.4275   15.2904   52.0315  217.4032  294.8655   46.4175   18.8492   18.5775    5.8225  302.8139   33.9352   59.0499   23.7904   30.2314    5.1719    4.8413   27.1069   13.8646   34.0044   28.0126   28.6527   36.1438   16.9803   13.1914    2.2518   11.0453   13.7889   32.1950   18.0757   67.8071   59.1083   74.2689   60.3698   15.0569    9.5843   28.0170   59.0015   96.2570   44.1337   26.6404  115.1966   13.2806   46.2623   28.0777   20.1977    9.5533   14.0957   55.0649   28.3657    6.3383   17.4758    2.2125   66.3089  108.6709   88.2742   15.5212   29.2015    6.0010   31.4612   41.5906   92.0749   48.1478    3.2631   35.9574   17.0619    4.5782   16.4670   10.9796   19.0521   55.7497   18.4859   17.1694   15.7372   65.8272   16.5908  126.2662  340.1746   85.5386   32.7716    7.4511   25.9296    6.2617   18.8992   51.5927   35.4239   14.3629    8.7435    8.3693   14.0242    7.9631   18.5842   57.3299   38.7590   15.9937   21.4731    9.8567   20.5013   40.2602   41.4855   20.7565    8.4145   14.3733    7.0817    6.1302   13.6144   10.1045   12.4565    7.9210   13.6308   17.4817    8.1616   19.0192   26.1957   24.0622   40.4821  244.4002   75.6395   34.6522   69.4252   72.4496   23.6674   27.5617   17.3844   13.2189   11.9606   45.7776   13.5406   13.6217    5.9266   15.7507   23.6746   90.7586   21.9682   16.7471   43.0270   24.2355   12.6997   73.6207   18.9294    6.5011   94.5874    6.4875   18.8324   10.5245   18.8643   11.9761    9.0707   30.2404   14.7942   27.3304   11.0095    5.1887   36.7607   15.8796   35.9323   36.2018   24.5081   61.2506   10.0052    9.1100   27.4192   12.7560   34.0523   21.3068   10.2965   14.9880   29.3413   31.2773   22.6135  150.2002   52.4790   36.1888   23.6843   24.6079   29.6641   40.7802   32.2881   32.5135   23.1299   39.1300   89.2834  101.0283   50.9225   67.8559   13.3802    6.9135   16.0007   84.8277   90.0348   58.0272   54.7518   56.2134  133.9728   46.4745   13.6455   42.5110   82.2220   25.6897  102.5927   83.6057    6.7646  113.5944   43.7605   16.0641   15.1910   46.8437   34.5942  129.2386  301.2840   52.0523    2.2900   23.7619   33.4012   17.4250   31.0673   56.7097   20.9884   18.9671   52.8099   17.3569   19.3719   11.3217   23.9565   22.9341   34.2522   21.1709    4.2498   37.1278   36.2991    7.7404   36.1655  103.5929   51.3723  157.4559  111.6063   28.3911    4.4272    6.7148   55.8945   16.8600   37.5254   25.4733   30.2943",
      "7", "ETD", 529.7,
      true, 2, 4, 0.9, "7", false },

    // 20090326_Yest_LysC_1x6ETD_longGrad_01.09832.09832.6.dta
    { "113.9901 131.2311 132.1521 143.1788 146.3371 146.9454 152.4344 159.1886 175.0511 189.9796 202.1302 204.0196 217.2641 218.2109 219.1491 220.0296 229.4176 231.9133 243.0516 245.0983 246.2436 247.2485 259.4018 260.4774 261.3199 262.3853 271.1878 275.3297 284.0590 286.2119 287.1346 302.1248 303.4427 331.2704 334.0960 344.3830 345.0301 346.0869 348.1975 354.3492 357.2001 357.8897 359.1330 362.2360 367.1863 373.1064 374.0351 375.1785 375.9212 382.2288 382.9473 385.1367 388.4153 389.1718 390.3647 400.3410 402.7518 408.0059 414.3023 414.9655 416.0516 417.4536 418.4369 428.2923 429.4773 432.2854 439.2054 444.1362 445.3222 446.1862 447.4139 454.2898 455.2044 457.3239 461.2656 466.1697 470.2955 472.8702 474.5312 478.3126 482.5112 483.7782 487.1635 488.1677 489.2876 490.1572 491.2320 492.2506 495.2077 502.8879 504.5410 517.4409 518.2657 519.0779 520.2747 522.1584 528.1399 529.3173 531.1453 533.0059 537.3658 542.7713 543.5209 546.1447 547.0486 549.0401 553.2217 555.8703 557.9484 561.1995 562.0273 568.2609 574.1810 575.9733 577.3002 578.4282 579.5886 583.4700 584.7160 586.2695 587.2089 588.1901 589.3297 590.8264 596.1005 598.8209 599.4614 600.3657 601.3851 602.3392 604.3453 605.5561 612.6790 614.3278 615.7708 616.4172 618.3073 619.5785 625.9096 632.2582 634.2405 634.9560 636.0547 644.7774 647.3933 650.0336 655.0643 656.0058 658.1935 659.4125 660.4209 661.3862 662.3001 668.3742 669.5141 671.0125 672.2191 672.8754 677.6085 678.4711 684.4069 686.7532 690.8365 691.5204 692.7213 693.5538  694.6512 695.5095 696.4465 697.3856 698.2626 700.1693 701.5884 702.5576 704.4716 705.5855 706.3614 707.2552 708.9446 714.9379 719.9528 720.6530 721.4449 722.5185 723.3188 725.1766 730.1107 732.4985 741.5456 743.4431 745.9914 746.6793 748.4077 749.5251 755.5034 756.4111 757.1909 758.3714 760.3879 761.4011 763.5425 764.4570 766.6517 769.2108 770.2787 770.9116 775.0515 776.1511 777.3389 778.3056 779.3911 781.2295 783.2359 784.0864 785.2942 786.5158 788.1550 788.8935 791.2961 792.2492 793.4121 794.2834 794.8853 799.2510 800.1431 800.8760 802.1324 804.9783 807.6463 811.0676 813.3052 814.1748 818.6032 819.8773 820.5369 821.2690 823.3438 824.1729 824.8098 826.0527 826.7680 830.6498 831.5505 833.9725 834.7603 835.4443 837.2088 838.8663 840.3897 841.1557 843.3222 844.3497 845.4462 846.4551 847.4792 848.3829 849.9952 850.9881 852.7861 856.1987 856.9511 857.6238 859.9268 861.1257 862.1689 863.2673 864.1008 865.5912 867.3397 868.2449 870.1732 872.8954 874.0253 875.4199 876.3032 877.7094 878.4007 879.1054 880.8305 882.1107 884.2063 886.2969 887.2461 888.5373 889.4424 890.5363 891.4711 895.3788 896.5696 899.8397 900.5723 902.4795 903.3190 904.0875 904.8262 906.5564 907.4622 908.2554 909.3530 910.1611 911.2778 912.7902 913.5569 915.2399 917.2781 918.4448 920.1915 921.6672 925.0764 925.8985 927.8044 928.6380 929.7339 930.6306 932.0803 933.5059 934.3492 935.4163 936.2333 938.2280 939.6257 942.8901 943.7336 944.4680 945.0829 946.3312 947.3308 948.4017 949.3831 950.5120 951.1990 953.3817 955.2790 956.4444 957.4495 959.2953 960.3568 961.3559 962.2510 963.8233 964.6780 966.1072 967.2949 967.9041 969.3903 970.0180 972.5438 973.4310 974.6245 975.4291 976.2036 977.8686 978.8047 979.5064 980.8099 981.6885 982.8694 983.9150 986.1586 986.8849 988.6154 989.8579 991.2148 992.3150 993.6907 994.8007 995.8435 997.5283 998.2278 999.3420 1000.8774 1004.1041 1005.1081 1006.3886 1008.0444 1008.7605 1009.6301 1010.4102 1011.1902 1012.3957 1013.0112 1013.6276 1014.3865 1015.3806 1016.8649 1017.8294 1018.9973 1020.2168 1021.0508 1022.8062 1023.8928 1024.6741 1025.5602 1027.0774 1027.9182 1028.6938 1029.6450 1030.4016 1031.1428 1031.8799 1032.6040 1033.9194 1035.0521 1035.8300 1036.9064 1038.4028 1039.5544 1040.5033 1041.5558 1042.2976 1043.0061 1043.8560 1044.4673 1045.5693 1046.6289 1048.0798 1049.1487 1049.8953 1051.1707 1051.8075 1052.7578 1054.0664 1056.2206 1057.3115 1058.6471 1059.3623 1060.1843 1061.1813 1063.2637 1063.8937 1064.6082 1067.6022 1068.5436 1070.4143 1073.8324 1074.4854 1075.6296 1077.5215 1078.7728 1079.6797 1080.5966 1081.3116 1082.8440 1083.4913 1084.4801 1085.5917 1086.3186 1087.5177 1088.4213 1089.5858 1091.9376 1093.6687 1096.1257 1097.7383 1098.5665 1099.8446 1101.7408 1104.1158 1105.0193 1106.6691 1109.3883 1110.0417 1110.8109 1111.6754 1112.5588 1114.2137 1117.1703 1118.7562 1119.6354 1121.5552 1122.4783 1125.8068 1126.5062 1127.3027 1128.4645 1129.3394 1130.4718 1131.7727 1132.3931 1133.1520 1134.8618 1135.8544 1136.7268 1137.8665 1138.9940 1140.4271 1141.1722 1142.1083 1143.3627 1144.6188 1145.2980 1147.1053 1149.1876 1150.1573 1150.9237 1152.8646 1153.9529 1154.8275 1155.5189 1156.6376 1157.8301 1158.8705 1159.8807 1160.6486 1161.4875 1163.3159 1165.5519 1167.3486 1168.8508 1169.6659 1171.2729 1172.3824 1173.1112 1176.1285 1177.2781 1178.1449 1179.4564 1180.4569 1181.7239 1182.6207 1183.7595 1185.7343 1188.1429 1189.0431 1189.6774 1191.4640 1192.5618 1193.5658 1194.8416 1195.5509 1196.5480 1197.6661 1198.5800 1199.6740 1200.7408 1202.4299 1204.0262 1206.1571 1207.0011 1207.6847 1210.7443 1212.2711 1213.1403 1213.8700 1214.5084 1215.4905 1216.3856 1217.6212 1218.7476 1219.5159 1220.4584 1221.2958 1222.0941 1223.2582 1224.4431 1225.1934 1226.9341 1227.5494 1228.2169 1229.2250 1230.3502 1231.1119 1233.0883 1234.5504 1235.8893 1238.8624 1239.7577 1240.6467 1241.8114 1242.6564 1243.3212 1244.4535 1245.9132 1247.2084 1248.5177 1249.9370 1251.2473 1252.1356 1253.4420 1255.5792 1256.3678 1257.2438 1258.5654 1259.6080 1261.2924 1262.0133 1262.7606 1263.5397 1264.5056 1265.4819 1266.2118 1267.3071 1268.6116 1269.6219 1270.8040 1271.9053 1272.9305 1274.3118 1275.1403 1276.1952 1277.0780 1278.8470 1280.0765 1281.6877 1283.1969 1285.1021 1286.0947 1287.3153 1288.3663 1289.0408 1289.7804 1290.4738 1291.1433 1292.2286 1292.9154 1293.9307 1295.2352 1296.5833 1297.4636 1298.8580 1299.5538 1301.5502 1302.1819 1302.7853 1303.5138 1305.8978 1307.0115 1308.0680 1309.0383 1310.2512 1310.9983 1312.3962 1313.3129 1314.1918 1314.9108 1315.9288 1316.5973 1317.6083 1318.3568 1319.9128 1320.6820 1322.8466 1323.8477 1324.4873 1326.1058 1326.9562 1328.3691 1329.3285 1330.3256 1331.5922 1332.5153 1333.7073 1335.3253 1336.4666 1337.6293 1338.5098 1339.8680 1341.0546 1342.1763 1343.5287 1344.5222 1345.3486 1345.9896 1346.7664 1347.7000 1349.1268 1350.1544 1351.1793 1351.8549 1352.5951 1354.2925 1354.9467 1355.7598 1356.6874 1357.5726 1358.4680 1359.3387 1359.9834 1360.7842 1361.4207 1362.2504 1363.0944 1364.5067 1365.4633 1366.7472 1367.4696 1368.8267 1369.9672 1370.8593 1371.4626 1372.5177 1373.4644 1374.5326 1375.4686 1376.3918 1377.1687 1378.0614 1379.2006 1380.0397 1381.3709 1382.3823 1383.1343 1383.8593 1384.5813 1385.5964 1386.4283 1387.5092 1388.4943 1389.4509 1390.3361 1391.2609 1391.9382 1392.6139 1393.4957 1394.2208 1394.8849 1395.6302 1397.4410 1398.9955 1399.7090 1400.3776 1401.0082 1402.6064 1403.8135 1404.8600 1408.1729 1408.9955 1412.5879 1413.8361 1414.5884 1417.4901 1418.3378 1419.2512 1419.8925 1423.4171 1425.0929 1426.5471 1428.1871 1430.0979 1431.2125 1432.5609 1434.8376 1437.0071 1438.4773 1440.1837 1441.5282 1442.5403 1443.3225 1444.4987 1445.5729 1446.7213 1447.8928 1448.9817 1449.7942 1450.6229 1451.3451 1452.4407 1453.6246 1454.2584 1455.4415 1456.7585 1458.9193 1463.1229 1463.8900 1465.2791 1467.1940 1468.4724 1470.5989 1471.4646 1472.3035 1473.1034 1474.8070 1477.1874 1478.5344 1479.5206 1480.4601 1481.4905 1482.5090 1483.6610 1484.8904 1485.6355 1486.6951 1487.4030 1489.5216 1491.4435 1492.4117 1493.5546 1495.1490 1496.0404 1496.6921 1497.5963 1500.2513 1501.1467 1502.2924 1504.7384 1506.7328 1508.5890 1509.5657 1510.6417 1511.5657 1512.7714 1513.4022 1514.2220 1515.1892 1515.9622 1516.6791 1519.0306 1519.9967 1521.2291 1522.1139 1523.2526 1524.3158 1525.3829 1526.5032 1528.7216 1529.4207 1530.5813 1532.9574 1534.4117 1535.6248 1537.3634 1538.3651 1539.1365 1540.1622 1540.8871 1541.8827 1542.6510 1546.9823 1548.3737 1549.7220 1550.7257 1551.4734 1552.3499 1554.8102 1555.8829 1556.8192 1559.1213 1560.6205 1561.2299 1562.5834 1563.7611 1564.9724 1566.1536 1566.7748 1567.6414 1569.7770 1571.3156 1572.6013 1573.5187 1574.6320 1575.4623 1576.7740 1578.4156 1579.3646 1580.9421 1582.0262 1582.9362 1583.6995 1584.7473 1585.4773 1587.3556 1588.6100 1589.2535 1589.9210 1590.9388 1592.6239 1593.4801 1594.8108 1595.6331 1596.6759 1597.5315 1598.7552 1599.6293 1600.8079 1601.4324 1603.4369 1604.5647 1606.6755 1607.9733 1610.1649 1610.7836 1611.7459 1612.5382 1613.4845 1614.9130 1616.0526 1617.1052 1617.9305 1619.1810 1621.5902 1622.3422 1623.4144 1624.2610 1625.0881 1626.1832 1628.7346 1630.3409 1631.2977 1632.2302 1633.0955 1634.6721 1636.4924 1638.6404 1640.4127 1641.6337 1643.9996 1645.8635 1646.6339 1647.7058 1649.7590 1651.1925 1652.5905 1653.9025 1654.6809 1655.7311 1656.6876 1657.7750 1659.5773 1661.7122 1662.9540 1663.6642 1665.8490 1666.8977 1668.0911 1668.8577 1669.8044 1671.3678 1673.4745 1674.6852 1675.7462 1676.9249 1678.8080 1679.5559 1680.3009 1681.6830 1683.0099 1683.9071 1685.6615 1687.5011 1688.1622 1688.9943 1690.2024 1692.0653 1693.5717 1694.7338 1695.4806 1696.1813 1696.8948 1697.6115 1698.9225 1700.2782 1701.3210 1702.1239 1702.8544 1703.7957 1704.7933 1706.4465 1707.1056 1708.0559 1709.1427 1709.8922 1710.8323 1711.8210 1712.7755 1713.9504 1714.9691 1716.7858 1718.0265 1719.0309 1719.8810 1720.8185 1722.9835 1723.9435 1724.7341 1725.7323 1726.5270 1727.5505 1729.0011 1729.9222 1731.0447 1732.1539 1733.8221 1734.8827 1736.3485 1737.5382 1738.6692 1740.2506 1740.9163 1743.7560 1744.6329 1745.5900 1746.3591 1747.7025 1749.0897 1749.7540 1752.7738 1753.6683 1755.0564 1755.7859 1756.7357 1757.6791 1759.6892 1760.4733 1761.1599 1762.2556 1764.7103 1765.9706 1766.8046 1768.2101 1768.8157 1769.9657 1770.7970 1771.9164 1772.9025 1773.8059 1775.1458 1776.6107 1779.2338 1780.1027 1781.0817 1781.9913 1782.7195 1783.4801 1784.1356 1785.5419 1786.3993 1787.6823 1789.5367 1790.4017 1791.3219 1792.4680 1793.1882 1795.7698 1796.5364 1797.2938 1798.4198 1799.3319 1800.0663 1801.1501 1803.3821 1804.6417 1805.9270 1807.1834 1807.9470 1809.0968 1810.1481 1811.6378 1812.4764 1814.2875 1817.0807 1817.9548 1819.2286 1820.1857 1821.4830 1823.0793 1823.9237 1825.2963 1826.2532 1827.0155 1829.1477 1831.3790 1832.2288 1832.9703 1834.2249 1835.1229 1835.8087 1836.8464 1837.9998 1839.3899 1840.1019 1840.8835 1841.7587 1843.5868 1845.3617 1846.1794 1847.0574 1849.7654 1850.7377 1851.9037 1852.5720 1853.3700 1854.2758 1855.6716 1856.8563 1857.9340 1858.6801 1859.7845 1860.7200 1861.5496 1863.4694 1864.0726 1864.8372 1866.0847 1868.1243 1869.1791 1869.8289 1870.5437 1872.1674 1873.2410 1875.0300 1876.2096 1877.3793 1878.5165 1879.6874 1880.4792 1881.6610 1883.1312 1885.2472 1887.0271 1888.1625 1888.9174 1890.5309 1891.2589 1893.3943 1894.4636 1895.6503 1896.5255 1897.2307 1897.9181 1898.6760 1899.3431 1900.9559 1902.2091 1903.5544 1904.4203 1905.1531 1906.1698 1907.9913 1909.0251 1910.7616 1911.5227 1912.1790 1913.0300 1914.0295 1914.7692 1915.6569 1916.6051 1917.5935 1918.5018 1919.3541 1919.9784 1921.6935 1923.4358 1925.7581 1927.1697 1928.0433 1928.8586 1930.2390 1931.2316 1932.0490 1933.0448 1934.0111 1934.7220 1935.4039 1938.1046 1939.1642 1939.9382 1940.9369 1941.8190 1944.0131 1944.9138 1945.9586 1946.7902 1947.6591 1948.4996 1949.7133 1951.1160 1952.2699 1953.3293 1954.5367 1955.4891 1956.4583 1957.0695 1958.6932 1961.0066 1961.8845 1962.5778 1963.3214 1964.5984 1965.7990 1966.7601 1967.8685 1968.7721 1969.4603 1970.6687 1971.7404 1972.9569 1974.2152 1975.1781 1975.9866 1976.6185 1977.9000 1978.8500 1979.6599 1980.8884 1981.7853 1982.6971 1984.0935 1984.7003 1987.0709 1987.8046 1988.9127 1989.6251 1990.4127 1991.2408 1992.2101 1993.2214 1994.1090 1995.3894 1996.1937 1997.2961 1998.4218 1999.6508",
      "  0.7882   1.2338   1.4690   5.2715   2.7157   0.8665   1.6112   2.9554   1.3347   4.1027   3.8946   3.0272  29.1180  92.1873  18.7286   1.6491   5.1260   0.7851   0.7923 315.9619 110.0733  13.9588   1.7858   4.1573   9.7864   5.5606   1.1364   4.9930   1.6353   1.3330   3.2970  18.8724  13.0217   3.0066   4.0489   1.4200   2.3865   2.7739   1.8415   3.5709   4.6604   2.5864   2.2279   2.9136   3.2431 132.1895  15.9678  29.7801   7.8762   1.8624   0.7930   4.7272   0.7881   1.4935   3.3303   6.1205   1.9669   1.7625   4.8351   4.4149   1.0977   5.5422   2.5520   1.6473   2.1585   0.6312   6.7079   5.8302   3.3232  22.1818   1.5718   3.2131   7.0913   1.3245   1.6089   1.0221   1.5416   7.5623   0.9457   2.2525   5.2898   3.3345 182.4794  45.2480  16.5450  14.5041   2.1931   6.8329   2.8742  10.2193   2.3163   4.6695   3.1755   2.9844   2.3901   3.5107   1.9076   1.1518   0.9497  20.2494   3.3190  11.0620   1.6660   1.7885   2.4268   3.0563   1.5379   0.5539   7.3357  12.4545   1.0707   6.0737   9.9238   7.0426  42.9061   8.9792   0.8669   2.9589   1.8796   3.2521   1.3315 145.5146  37.0337   3.6117   1.4463  74.4829  24.5854  14.8762   8.6619   5.9887  10.8583   0.6368   9.0039  12.2474   8.1285   1.4736   7.2705   2.3549   0.7095  16.7374  25.6815  36.7998   4.8877   2.9292   7.4774   1.7576   2.5155   1.0239 103.7108  22.9241  19.9536  15.3724   6.8771   5.2714  13.2522  12.5899   3.5023   3.2909   4.4146  14.3020  20.8035   8.6278   8.7088  42.2430  88.1366 899.3025 1000.0000 423.5352  29.6849  14.2239   8.9192  21.5592   5.0350   5.6656  10.3790  90.1816  18.8989   7.1451   5.6359  11.6304   7.3775   2.7498   3.9538   4.6622   3.4874   8.9575   3.8797   6.2168   5.9806   3.3273  13.8733   0.9658   6.8611  18.4296 107.5015  11.9635   8.8080   4.7998   9.5301  10.4798   1.1775   1.8836   2.9827   9.3508  12.5125   1.7291   3.7092  32.0875   9.3238   9.1529   1.8324  16.4488   3.1251   3.5275   8.5837   7.7945   0.9962   4.1653  32.9291  21.6273  10.7102   2.1521   2.1954 233.5511  37.5855   4.5931   2.6888   1.1378   9.8956   5.9154   6.4284  12.1675  98.4197   9.7095   2.4615   2.5431   2.3061   3.2045  19.4397   7.8092   4.5308  21.1395  16.7198  40.1602   0.6974   3.6422  18.5252   7.9833  12.8113   4.9480   3.6519   5.0524   5.8375   2.6901   3.4764  15.8601  16.2370  13.5469   2.5415 189.6745  76.8495  28.1688  13.2680  28.1880  26.4373   3.0931   6.9593   3.0615  17.6371   3.2466   7.1994  49.6675  48.7677  54.9408  50.2091   9.6626   7.2094   2.2310   5.4398   5.8776   6.8273  45.3912   2.8537   1.2580   2.3087  32.5015  18.0361   5.5497   1.2442   5.7918   7.9674  24.7011   1.5223   5.7114   4.2552   9.0050  19.2156   3.1398   7.4090   3.3731   2.8531  92.6507   1.0116   0.7883  13.9968   2.8556  44.3379  16.5552  40.5248  27.4875  14.9865  27.5186  11.5219  10.7622  52.8381  24.3738   1.2978   7.5290   6.3854  24.4063  14.3000  15.9755   8.8281   2.9272  58.1891 114.0278  33.1024  14.8397  22.7402   9.4772  11.9262   4.4631   7.3775   7.5562  16.0341  10.6938  29.8320   2.9340  14.5792  21.9621  47.1874   2.1519  28.3740   6.7118  15.1696   0.8143  10.4319  12.5107   6.2136  33.3331   4.9963  16.8548  18.1042   0.9928   8.9262  36.8391  37.8557  22.0204  65.8654  15.4924  51.1154  64.2312   7.5328   3.4474  16.5854   1.8822   9.2501  20.6316   3.7565  12.3188   12.7473   46.1698   28.1419    6.2244    2.8962    6.2451   33.6466   34.6984    6.9332   57.7374   37.9611  100.3497   12.7352   17.6961   49.9136   53.0904   82.0319   66.6688   15.2200    4.2142   10.9943    9.6479   49.1216  124.8409   74.6264   48.5882   21.3874   14.2358  131.5370   20.6068   54.6720  123.8634   59.9938   15.6989   23.2243   84.2466  448.8117  729.3666  415.6860  400.0499  228.7654   10.9615   78.2260   17.9154   23.6912   46.1553   56.7357    7.2724    5.5523    3.1658    7.2333    6.9645    2.5917    8.5677   22.6664    2.9167   12.8866    1.8680    3.8731    1.5003    1.4150    8.2251    6.6937    6.7158   54.6595   40.5610   25.5735   26.5802   25.7058   32.9240    4.7233    7.0271    9.4470  101.3233   61.6267   19.3270    6.7386   27.3538    5.8047   29.4897    8.6042   10.9515    6.7882  108.2355   38.1058   23.2399   23.0468    6.3150    7.3250   10.1624    2.4303    3.1598    9.1313   12.9203   11.4054   16.3637   15.4630   12.6306    8.8983   15.0054    3.1066    2.5454    3.1221    4.8013    8.7342   19.5644   33.7218   53.9973   16.0492    4.1033   12.1449   11.7813    4.3366   11.3464   26.1843   36.2671   17.1257   12.0484   11.4197   27.4982   12.7326    9.1530   16.5634   44.9096    0.5508   34.2772   14.0924   40.9731    4.4487   50.2529   55.8065   12.9221   15.8507   15.6793    4.7133   19.4843   16.7195   16.2017   14.6401    5.1641   43.6599   30.2467   11.0664    2.0359   22.7743   16.6831   47.7048   15.0643   17.2578    8.9148   14.0071    1.7875   24.3035   17.3980   12.3099    3.9429   10.5832   21.7551   11.7909    6.6174   35.9319   45.7622   49.2996   25.8661   24.7929   37.6086    3.4061   10.7790   19.1435   26.6210  133.2804   71.6528   31.1831   22.6257   14.2815   30.2932    9.9040    6.6790    3.3256    7.4046   18.5438   13.9573    9.7036    4.6337    5.2810    6.1957   20.6643   28.9472  177.8307  102.4394   22.3683   16.5090    7.7104    8.2458   25.9384   10.0650    7.4178   10.7713   51.6587    3.0736    2.2825    5.1280   15.0373   16.8868    7.1792   19.0850   20.7129    4.7520   17.2991   25.2128   35.0184   18.6991   28.7550   13.9657    2.6156    4.3171   23.0390   19.3593   41.7318   62.1964   19.0136   41.0846   26.0256   40.3554   35.7827   30.7425   12.2687    1.1788    4.8741    9.9576   26.9510    7.6796   25.3728    2.5439   15.3496  276.8694  140.9403   66.8089   40.1855   28.3261    5.6646   13.2502    1.1692   18.7341   76.7152   22.0055    6.5392    4.1076    3.3208    9.0892    3.4477   15.3705    1.7882   40.8286   13.2536    4.9919   50.0643  179.0947   21.1458   17.4503    6.3555    5.6109   20.3739   31.1409    7.4474    9.5979    0.9318   22.8467    3.0125   12.9197    4.6544   10.1378   14.2818    5.5517   37.4230   49.7872   52.5437   65.4898   67.9736   29.0907    9.5927    1.8327   35.9073   70.7768   40.3347   18.9421   33.7865   38.6511   12.4221   42.6717   48.4038   41.2194   89.6144   33.4612    9.6977   52.9020   84.1777   65.4023    4.8584   15.2862  103.6673   67.3765  163.6603   37.0745   53.5752   34.4623   73.6682  159.8176   70.8152   19.9635   56.5708   20.0970  190.8416   75.6044   42.8530   16.1060  265.4394  357.4647  118.7277   69.0456   98.4367   95.9682  669.7959  272.6677   32.6305   18.6214   22.6137   51.3712   67.0088  114.0852   83.8836   72.7718  163.4914  148.9661  287.9714  209.1821  212.1409  529.3602  734.4439  449.1429   37.1894   15.4887  111.1239   52.6632   29.3436    8.5445    5.8198   18.8332   40.2829   17.3968   12.5103    2.5262    4.6379    5.9340    2.6933   19.9679    5.1856    2.8044    5.4177    4.3256   16.3718   11.9438    5.2611    3.6056    7.9556   17.1140   19.6393   15.6255    1.4935    2.7949   13.1273    5.6584   16.6632   11.6434   22.8122   21.6332   38.5820    5.4408    2.9926    4.1127   16.9628    4.4425   18.9169   59.1490   23.6358   24.3605   18.9449   11.4542   19.6536   10.7939   12.4849    5.1658   45.6344   49.4934   34.1920   19.5189    1.6754    1.8842    2.1514    2.6423   11.7624    3.6703    9.7487   12.9905   35.3189   43.4424    9.1075    8.1898   14.6987   44.8164   82.9347   35.1636   24.0687    9.4809    9.1777   12.7550   18.9022    8.8221    0.9432    9.1233    6.3374    4.8303    9.7621   10.1872    4.1039    3.0005   16.0191   51.7438   66.0449   25.1428   13.0474    1.4431   13.0180    7.0912    5.1896    5.8161    8.8903   28.7951    9.6813   27.5525   25.9896    7.3851   19.4597   26.1919   11.4152    3.3117    0.8425    4.4223   10.9286   12.3103    6.4925    4.7694   17.7922   14.9555   13.8515   54.9117    8.8562    4.2778    9.1613    2.9274   12.6820   19.1942   19.5758   11.1650   10.1294    9.4094    4.2609    1.0210    6.5288    4.5425    5.8404    2.0670   10.2111    4.7946    1.5700    3.4809    7.1907    6.3686   16.7089    5.7932   15.1646    2.5100   25.7024   10.8899   43.2942   21.2989   24.6382   26.8564    1.7809   16.2150    6.0329   53.3263   31.6996   11.1775    7.6595   29.8364   19.2204   14.2775    3.5022   75.2761   93.9829   25.7818   39.4646   12.3446    5.9996    7.6165   10.5357    1.4668   15.1383   17.5517   53.9785   44.2829    3.5110    3.0863    3.6781   13.0000    5.8848   19.1277   15.5125    3.4551    1.3886   22.8519    9.9976   16.2180   13.1197    7.6040   11.3070    9.7544   16.3058    8.9744    9.2886   23.7590    2.4494    5.7015    8.1026    6.1667   26.2368   17.5402    4.2452    6.0370   29.7126   13.3919   40.4039    1.8471   24.4204    4.5108   12.5156   25.9107    3.0860    1.2098    3.1305   21.6278   38.8935   14.6334    4.0924    9.0087   19.2632    4.2845   10.9829    9.6669    4.5447   20.8103    8.2753   11.7439   17.6609    2.6647    3.8683    7.5676   32.2318   15.8165   19.5057   11.8576    9.2740   16.7955   13.6268   58.7846   65.1262   16.6333    6.8582    6.4038   12.0963   27.5998   18.8584   25.1758   27.3996    7.1412   11.1917    0.5174    5.9764   47.9578   90.3362  152.1247   42.3611   57.8727   22.3458    7.5690   16.9555   19.7501    5.3896   17.9180   23.1288   38.3125   57.1665   10.8149    8.8784    6.1971    8.9094   24.1074   13.5064   44.3308  118.6540   51.7940   24.1366   42.5735   62.0631   28.3070   14.8120    2.9579   12.1073   20.8824   54.8472   36.4654   29.6668   26.6935   14.0049    5.8890    4.8282    5.2406    7.0041   10.2295    3.2089   10.2921   10.5749   22.6961   14.4206   24.1620   15.3352   15.9632   18.3046   17.4620   14.6835    2.5325   36.8980   15.0390    7.6405   13.8095    8.5752    9.0806   10.2180    7.5096   19.1217    2.9220   10.7044    3.0632   21.4457    3.9442    3.2078   59.1480   76.8403   30.1047    1.2281    9.2401   14.2531   27.2254   10.0098    4.5519    2.4809    3.0208    7.4779    6.9956   14.3988   12.0743   20.9794    2.3019   23.3461   27.2394   31.7722   12.4663    3.4009   11.4335   42.1608   33.5015    8.9897   14.8191   12.6126   28.3521   45.6324    4.6593    8.9367    4.0243   15.4324   13.4652    6.2845   12.7486   24.3929    3.9183    9.1934   30.8826   33.8580   67.2932   26.6427   32.7887   21.9741   17.6815   22.4838   10.8720   15.7800    1.5712    8.7224    7.2089   13.1418    8.1982    9.2143    3.6575   15.8164   13.5247    5.5736    3.7005    8.7226    5.1440    6.7949    3.3984    8.9614   30.4627   16.8158    6.6020    2.1429   11.3900    8.0702   49.1393   83.4953   26.3869   13.9723   20.6791   13.9586    5.4295    5.1951   10.2817    3.9876    7.3068   30.8753   27.9145   11.0200    2.7721    9.9651   19.3649   32.7778   35.9366   53.2385   13.5144    2.2395    4.4096   17.8227   31.6407   16.3919    5.4519    6.8358   25.7027    1.3013   14.2578    6.1699   10.7177    5.0816   16.1772   12.6636   15.9320    1.7283    8.2588   25.4254   13.7164    3.1962    3.4199   13.4309   14.2145   15.5726   24.2617   17.0439   21.0429   26.2592   82.4623   76.8849   41.5466   15.9525    7.5481   38.1850   49.3495   49.8431   36.6124    2.3115    8.5146   14.4192   17.8063    4.3331   17.4075   16.2087   37.1296   47.7948   70.5014  126.4190   59.3923   13.9014    9.8167   12.2739   12.6826  183.1271  177.8186  161.7365   60.2777   47.0338   30.7163   55.8528   24.6995   10.2564   11.8114   20.8701   45.1977   44.0494   51.2530   31.1729   10.2021    7.9284   74.1734   11.7898   19.4566   40.7594   17.8223   12.2809   10.6742    1.7347   25.0762   35.2629   15.0124    7.4382   32.0495    8.2455   13.6037   37.0519   14.7269   30.9821   29.9967   53.4566   15.8324   16.5273",
      "6", "ETD", 695.04,
      true, 2, 4, 0.9, "3 6", false },

    // 20090326_Yest_LysC_1x6ETD_longGrad_01.10231.10231.5.dta
    { "199.9968 203.3892 217.9639 219.2876 230.3625 231.2264 233.4094 244.2236 246.3432 273.0931 278.3110 287.3306 288.3451 300.9216 301.8600 318.0997 346.9664 357.3155 358.2166 359.1941 365.1911 372.4192 373.1793 374.2338 375.2891 377.2897 378.3042 379.2480 389.3028 402.2855 412.2734 421.2726 430.2226 443.0610 444.0184 445.4998 448.9448 452.1257 460.4586 471.1894 481.1470 487.4255 488.3946 489.5114 493.3804 502.1929 505.7596 514.4465 515.2761 516.4627 529.4846 530.3494 531.5333 533.4164 534.4004 541.1827 542.0539 556.8998 558.8531 564.4246 570.2122 581.1977 587.5244 596.4286 597.0301 601.3938 619.0530 626.8019 629.2461 630.1948 631.3355 632.3667 636.5231 643.1704 645.6285 646.6180 650.0156 656.6390 658.4033 659.6189 660.5480 667.5349 668.4302 669.1138 670.1107 671.8558 683.4437 684.3170 687.5054 690.7796 696.3185 703.2939 708.2760 710.4343 712.9634 713.6234 714.2911 715.2505 743.0220 744.4070 749.9543 750.6201 751.4276 756.4520 759.9512 761.4014 765.2022 768.4483 769.7205 770.4863  771.3599 772.4193 773.5159 774.4064 777.8081 781.2615 784.3503 786.4894 788.3555 792.4168 795.1695 797.5469 798.4954 799.4752 801.4368 803.5952 805.2787 809.6343 810.8914 813.3586 814.2593 815.2007 816.4573 817.2343 820.4378 822.0837 828.1140 829.6492 830.9940 831.6106 832.3195 849.5237 858.2823 859.1848 866.5513 869.4810 870.4755 875.3311 877.2744 879.4274 880.5972 881.2021 883.6060 892.2300 893.5712 894.1797 895.6813 896.3052 897.4622 898.8010 899.5786 901.5259 904.5444 907.5441 908.7720 909.6478 910.6677 915.4683 917.5506 921.2318 925.4321 927.1816 928.4699 929.4986 931.1143 931.7744 937.8880 940.5587 943.7704 947.2046 949.2724 950.2670 954.0981 955.5023 960.2236 961.3193 962.8962 963.6597 964.7088 965.7717 967.6285 968.3437 976.9644 977.6417 979.6945 985.3494 990.3233 991.2648 992.5762 994.2827 996.5015 998.1729 999.5853 1002.1657 1007.8813 1012.2427 1013.4610 1015.3331 1015.9792 1016.7728 1019.3039 1020.4933 1021.3461 1022.1887 1023.8343 1026.4688 1027.1772 1027.9828 1029.1892 1030.2264 1030.9963 1032.0424 1043.7311 1044.8026 1048.3547 1049.2153 1058.7727 1059.5665 1060.7236 1062.0500 1064.5720 1065.1926 1066.1375 1068.6206 1071.0164 1073.6232 1075.5795 1076.2839 1077.5339 1079.2949 1083.4153 1084.9513 1087.4999 1090.6151 1092.1388 1092.8107 1094.6973 1097.0104 1097.8096 1098.5801 1099.2081 1100.0697 1101.7253 1103.4839 1106.8026 1108.9430 1110.1464 1113.0355 1114.6616 1115.4230 1119.2159 1120.3712 1121.6892 1123.3085 1125.1543 1125.9171 1126.9982 1127.8264 1128.7831 1132.0010 1133.7784 1134.5132 1135.3239 1136.3513 1139.2911 1139.9486 1141.6228 1142.4706 1143.4874 1144.6455 1145.4458 1146.6713 1147.3684 1148.4218 1149.5150 1150.4442 1151.0565 1151.6959 1152.4464 1153.3569 1154.6511 1155.6222 1156.5662 1157.2361 1158.1544 1159.3284 1160.2274 1161.2325 1162.2584 1162.9725 1164.6285 1165.3418 1167.0341 1169.7928 1170.7620 1172.5529 1174.3608 1175.3719 1179.5142 1180.5903 1181.2582 1182.0927 1184.7328 1185.8318 1186.5153 1190.8060 1191.4373 1193.6167 1195.8188 1200.8970 1201.8322 1207.1527 1209.3734 1212.1838 1215.0480 1216.0809 1216.9240 1222.0365 1224.6493 1227.3143 1228.1488 1229.5078 1231.5402 1233.1925 1236.9253 1237.7500 1238.9708 1239.5934 1241.8324 1242.6617 1243.5510 1244.4005 1247.3298 1248.8254 1252.3721 1253.1113 1253.7217 1254.5360 1255.6528 1256.8463 1258.4022 1259.0864 1259.7694 1261.0781 1263.7369 1264.7242 1265.6049 1266.2859 1266.8942 1267.5886 1268.6367 1269.3829 1271.0029 1272.0626 1273.1268 1274.6068 1275.7810 1276.8475 1277.6908 1279.2212 1280.4325 1281.3257 1282.7299 1283.8431 1285.4242 1286.3147 1287.2496 1288.3251 1288.9520 1289.9506 1290.5951 1291.6273 1293.1234 1294.0182 1294.6876 1295.4559 1296.6429 1297.3284 1298.3654 1299.7465 1301.5834 1302.5348 1303.1448 1303.7810 1305.4895 1307.0480 1308.5162 1310.6530 1322.2665 1324.6090 1325.3912 1326.1967 1328.6816 1329.4915 1330.1071 1331.4850 1336.3151 1337.8545 1338.7771 1339.5541 1340.5658 1341.6544 1342.3667 1343.5753 1345.5013 1347.2640 1348.8749 1351.1949 1352.0460 1352.9468 1354.1296 1355.9504 1356.7135 1357.3826 1360.4873 1361.4817 1364.7675 1365.8361 1368.3119 1369.3562 1371.2378 1373.0154 1374.3073 1380.1006 1381.2966 1382.1339 1384.5953 1386.6671 1387.4862 1389.6205 1391.6823 1393.8027 1395.3778 1398.7206 1402.5897 1403.2291 1404.2162 1405.7178 1409.6974 1410.8346 1412.0652 1412.7459 1413.8644 1414.7704 1415.4298 1416.4241 1417.8678 1420.3739 1422.0052 1423.2833 1424.9669 1426.4757 1427.4905 1428.6627 1429.4583 1430.9574 1431.9563 1432.7750 1435.6735 1437.2614 1438.4911 1439.8132 1442.8597 1443.7821 1444.4789 1446.1693 1446.8107 1448.7933 1451.0183 1451.9075 1452.9984 1456.4574 1459.8391 1460.5907 1462.3885 1463.1023 1464.5487 1465.4086 1466.1863 1467.3358 1469.4706 1470.6277 1472.2717 1473.3860 1476.2083 1477.3331 1478.2662 1481.0092 1482.1669 1483.3353 1484.6349 1485.6622 1486.5348 1488.6351 1491.1277 1494.1429 1495.1176 1496.6829 1497.4846 1498.6361 1499.4708 1500.1746 1501.0580 1502.7214 1503.7355 1505.5406 1507.7498 1508.9955 1509.6993 1510.8724 1512.0980 1513.1663 1514.1228 1515.2155 1516.0603 1516.9320 1518.0652 1519.5046 1520.1854 1521.4220 1522.3212 1523.5261 1524.3551 1525.6578 1526.7985 1527.6752 1528.8207 1529.9440 1530.7136 1531.3710 1532.3114 1533.4865 1535.0594 1535.9149 1536.6276 1537.2911 1538.9327 1539.7517 1540.5729 1541.2230 1541.9208 1542.8248 1543.6492 1544.6127 1545.4789 1546.5535 1547.3517 1548.8375 1551.9415 1553.3087 1555.7809 1556.7722 1557.4882 1558.1554 1559.6254 1562.8792 1565.8635 1569.8210 1572.8729 1575.5411 1576.7335 1578.6332 1583.7629 1585.2416 1586.1223 1587.2738 1588.0520 1589.9310 1590.7679 1591.4988 1592.5266 1595.0131 1600.4128 1605.4078 1606.3152 1608.2335 1609.0292 1613.7843 1614.4235 1615.0240 1617.4705 1618.2316 1620.2174 1621.1448 1623.4460 1625.0553 1626.7288 1628.3973 1630.0243 1634.9392 1635.9377 1636.8392 1637.9626 1640.2648 1641.5309 1642.1378 1645.0062 1647.3020 1654.0756 1655.5837 1658.4904 1660.6063 1661.4950 1662.2313 1662.9808 1663.8503 1665.6305 1671.3563 1673.4164 1676.8075 1680.4739 1681.5331 1684.1039 1685.0872 1686.1285 1687.0353 1688.6887 1694.0150 1694.7936 1696.5917 1697.3898 1709.8912 1711.3949 1716.2777 1719.2097 1721.3435 1725.4178 1727.8567 1731.3132 1732.1276 1732.9091 1734.1439 1739.8469 1740.8627 1741.9377 1743.1740 1744.0885 1748.9207 1749.7902 1751.4054 1754.9792 1755.6819 1756.9647 1760.7775 1761.9359 1762.9423 1774.4584 1775.7214 1777.2146 1777.9000 1778.6409 1779.6849 1780.4164 1781.5988 1784.6150 1785.4786 1786.4226 1787.0295 1788.9304 1792.2430 1793.5470 1794.6105 1796.8732 1797.5880 1799.4418 1804.2677 1806.3702 1807.0431 1808.5135 1810.1678 1810.9910 1812.2628 1814.0519 1816.0890 1816.8850 1817.4900 1818.8058 1819.6410 1820.7987 1822.8992 1827.8724 1828.9021 1832.9716 1834.5807 1835.5642 1836.9562 1839.7180 1842.3806 1843.4514 1844.7516 1846.0460 1847.1014 1847.8693 1850.0310 1850.6803 1851.8031 1852.6888 1853.3849 1854.0485 1855.2161 1856.7922 1857.8839 1861.8253 1863.5110 1864.6586 1865.8446 1867.0265 1872.2233 1872.8591 1875.1160 1876.8230 1877.8879 1878.9817 1879.9806 1884.8640 1885.9685 1886.8761 1888.2507 1890.1453 1891.0648 1892.5250 1893.5046 1895.5006 1896.6990 1897.9305 1898.8899 1899.9142 1901.0745 1902.1547 1902.9152 1903.9285 1904.9880 1905.6917 1906.3856 1907.1786 1908.0311 1908.7242 1909.7391 1911.7196 1912.5292 1913.4950 1914.2708 1915.0756 1916.1615 1917.4456 1919.4734 1920.6144 1921.8298 1922.7296 1923.7186 1924.6965 1925.8077 1926.8290 1928.0314 1928.9969 1930.0419 1931.1810 1931.9924 1932.7919 1934.0353 1934.7501 1936.2020 1937.0485 1938.4026 1939.9923 1941.0868 1941.8502 1942.6708 1943.8409 1947.2426 1950.8925 1954.6056 1955.6066 1958.3883 1968.1917 1970.5634 1971.3766 1973.0155 1975.0997 1978.9559 1982.6322 1983.3234 1984.2694 1985.1986 1986.3406 1987.0579 1994.9388 1995.9553 1998.4855",
      "  3.6016   2.1781  10.1620   7.6023  10.0603   3.0831   2.3715   4.6235   4.4300   1.8208   2.2721 123.0606  29.0272   9.0489   3.1364   8.1317   1.4704   2.0508   1.4641   2.9302   3.0248   5.6357   2.8385 178.4803  37.6749 108.5945  22.0127   4.4572   1.9127   1.0174   2.3556   3.5009   4.4925   2.2436   3.6303   2.9498   3.3862   2.1287   2.6850   1.7065  12.8303  41.9667   5.8168   2.2576   3.1353   4.8054   9.9327  19.0753   4.8223   5.3841   3.3041   6.4941   2.1635   6.2719   5.0523   6.6585  16.6893   0.5703   6.9696   1.7052   2.2385   1.7965   5.5334   1.2431  10.3704   3.0587   3.7108  22.3982   7.5832  25.9226  11.1733   2.8295   2.1502   5.3442   2.2480   4.3584   7.1040   2.8231   3.2184  11.0678   1.8020   9.9538   2.4955   4.8483  13.8447  10.2946  83.7919  44.7566   2.3803   5.5501   7.2935  10.1587  11.3597   2.7159   6.4879  29.6912  20.7196  17.7341   1.9123   3.1369   4.8179   3.8283   1.2447   8.2909   4.8056  22.9553   5.4880   6.9515  17.9575 714.4054 1000.0000 915.6977  44.1673   6.7532   8.5911  11.7277   3.3898   6.7872   1.3728  11.1750   4.4791  87.0462  62.8194  14.1495   7.7174   8.1923   5.1595   0.8000   0.9046  46.0808  56.2002  12.6137   9.4954   2.0218  22.4530   4.4834   7.7376   4.8174  42.1129   7.5317   5.3538   3.0543  18.4400   0.9088   2.5343  11.0701   3.4334   4.0969   2.6886   7.7208   5.1476   2.6699  16.7045   1.8505   4.2412   2.0402  18.7736   1.0203   7.2720  46.0384  26.9076   5.6503   7.6962   5.4857  40.9232   8.9633  11.5495   7.0634  18.6693   3.8990   2.3285  12.3768  42.3419  24.6978   2.2453  11.7875  15.4995   1.8514  11.6239   3.3570  12.2979  21.4512   2.2591   4.4746   1.7085   1.9108  55.7391  25.0911  21.2561  20.5229   7.0451   7.1804   8.3820   7.8361   5.4943   4.6157   8.6582   3.1558   6.5542   6.5112   8.9484   2.4692   7.4760    8.4992   27.5520   29.6512   13.1910   14.6707   10.4552    5.0611    9.5637   17.3501   75.1421    2.6628    4.8813   32.5452   46.3334   23.0088  119.1303   50.8363    2.9102   12.8323   20.4486    3.6650   27.7376   14.5682   38.0279    3.6371    1.4630    1.4594   16.6963   15.3944   13.4947    6.3783    9.8533   13.6908    1.1291    0.5741    4.3807    6.6031   40.6825   18.8058   12.4767   26.1853   92.1437   12.7664    2.9274    9.8584   33.5797   10.8649    6.0002    1.4197    1.5188    3.4842   10.6102   48.8184   27.8690    2.8027   19.8795    6.0512   12.0446   14.8769    0.9105    9.1493   12.7402   10.7742   66.4134   80.6953   35.3021   48.7407  175.0935  113.9304   89.6744  108.6400    4.4578   22.7156   38.7327   66.5691   28.4701   17.0779   16.4588   92.8622   83.5663   94.0610  137.2564   30.6636    1.6444    3.8182    4.2725   25.2232  351.3492  513.5794  323.5553  530.3840  243.5248  120.0198   49.8522  151.7019   36.0866   12.3856   10.8345   15.9476   14.2057   13.6114    4.0372    5.5699    1.2438    9.5965    6.5430    2.5804   17.5451   15.6703    4.8233   12.7172    1.0194   24.3253    3.5262   22.5319   12.6457   26.2802   39.8092   15.9194   22.2088   22.7952   76.8492   14.0347    6.7183    3.2569    9.8591   10.7724   19.3807    2.0346   18.7453    9.0429    3.1454    3.8070   12.2812    1.2422   14.4762   58.9126   42.0883   14.0934    8.1424   10.1568   59.6108   47.3159   80.4912   33.7870   10.0400   16.9593   14.6570   17.9829    2.5909   43.2117   22.4885   13.5246   31.6376   46.9633   84.9990   44.5988   13.7229    9.5891  274.9896  100.2578   13.8008   50.0539    6.4921    5.5462   35.9377   15.6007  305.8508  251.5432   57.4802   49.1734  564.9909  589.0791  280.8830   45.2754    6.5993   30.6451    2.4397    7.5348   25.4702   31.7668    2.7709    8.1462    8.1616    6.7696   17.5737    5.0442  108.2370   81.4792   55.2033   24.1922    3.1839    3.9067    1.2473    8.4202   10.0765   18.3724   39.7493    9.8224   10.2762    4.7164    4.8282    1.2411    1.3961    8.7339    4.5549   20.1582   54.7656   22.4691    8.8869    6.2968    6.9698    9.0609    5.8200   61.2628   58.5081    3.7403   19.5916   15.3810    2.9041    6.1390   11.5064    2.1780   10.1354    3.9704    1.8637   10.3703    5.6561    6.6086    8.6657    3.8186   10.3917   16.2590   10.2584    2.4232    2.0876    5.9746    5.3846    9.7090    5.8886    1.3666    8.3193   12.0021   12.2612    4.4951   28.9430   34.9571   30.1844   10.7015   72.7141   18.8182    5.1349    4.9341   11.1353   23.3139    2.0227   16.0617   17.6052   11.0747   36.6177   35.3493   27.3406   29.9640    5.9574    2.7009    4.9227    8.4978   15.3323   18.8988   57.8273   16.5461   12.2736   15.6057    5.4379   10.7086    8.6146   10.0282    2.8870    2.8222    6.2616   18.1149    4.4740    4.9275   68.6819   79.5819   36.8275    5.4694   17.7876    1.2396   24.4970    4.1348   14.7964   28.1291    5.2661   13.3220   22.9010   31.1023   11.8355   23.3682   19.9112   19.9869   17.7420    4.1990   33.8551   34.5815   59.6799   51.4557    9.2841    4.3665   14.6125    7.3738    6.0538   18.8061   28.9519   10.3743   30.5676   58.8658   54.3548   20.0509   92.4775   31.4807   28.5316   25.0807   27.5678   27.4990   25.3017   62.0100   77.4487   73.3813   34.3738   29.1524   44.7990   10.1968   55.3135   70.9857   10.5522   44.7691   21.3174   36.2008   38.4463    7.1065    6.3669   14.1117  136.4879  221.7289  371.7863  169.5908  406.2278  256.3649  300.1304  325.9155   76.1877    8.7254    6.1542   10.1097   14.7954    6.4801    6.4860   28.3451   13.4395   12.6126   12.1537   10.1850    1.5757   16.5649    2.0181    5.3720   14.1507    3.8764    3.3603   23.5219   18.6815   93.6118  118.8354    7.1111    9.7143   15.9778    7.5743   18.9328   14.2496    6.2725   19.9236   10.7421   10.6545   49.7266    6.5881   20.4936    3.1739    8.9911    7.9305    7.9815    5.1467   15.9047    3.1738   12.6110    2.2522    5.2648    3.5906    5.5694    7.1450   20.2604   23.4119   11.5469   16.0100   15.6099    7.9545    4.6756   19.7794   16.1550   39.5286   13.3297    7.2781    9.7234    2.8022   17.1476    6.5237   21.1640    3.5790    5.0739    9.6155   16.7855   51.1709   11.3454   17.2556   10.3444    8.1492    9.4472    2.3998    4.6404    4.5918    6.9811   25.8694    3.5074    7.4071    8.7327    2.1682   10.4228   36.2800    5.3763  140.2006  168.7035  207.4277   95.3589   23.9144    9.7164    8.2576   14.7594    6.3800    4.1748    1.1255   17.6303   39.6940   19.5405    9.4882   11.2341    7.5811   15.9337    1.7975    4.5935    1.7524   22.1690   72.7275   84.6949   52.6702    7.8753    1.6874    1.7067    7.2548   17.7820   25.1585   13.7912   10.2170    8.0277    4.9261   11.2626    4.2814    9.3588    8.7016    2.8790   17.3916    7.0336    4.8461   13.6477   25.3355   53.2030    3.3556    3.4744    9.5729    5.4406   10.7160    6.3282   20.6909   16.4194    6.8849   23.8984   33.6639    4.1344    9.9854   16.3974    2.9186   16.1606    5.3541    8.6729    6.2660   10.7086   15.7140   60.4747   25.7030    8.4810    3.6418  136.7862   96.7313   40.5015    4.2546    9.2299    4.5586    5.0291   20.4064   37.9694   13.9030   20.7789   32.6029   12.1402   32.6036    8.4372   25.0322   19.1940   27.4789   10.1319   17.2868    6.6996   93.2260  238.2121  152.4210   49.3212   11.2141    3.4435   11.3247   74.4382   83.1977  153.7884  164.8175   46.1960   21.3199    8.8300  101.6145   54.3686   57.5756   65.0307   20.4811   23.6840   26.2650  363.4641  630.0406  203.9385   49.3090   31.8182   40.6440  122.4566   41.5987  707.0180  989.1421  495.5886  140.0913   32.2434   12.9184   34.8640    8.3428   12.8436    5.7005   12.3609   10.9389   12.8152    4.1081    9.7312    1.6886    2.0328    1.2402    7.3164    9.9376    8.0779   20.2228    8.0435    2.2537    6.2250   10.9980   15.0756    6.3459   15.3282    8.3546   12.9332    3.5723    8.3276   15.5953    8.2561   14.8921",
      "5", "ETD", 771.98,
      true, 2, 4, 0.9, "5", false },

    // 20090326_Yest_LysC_1x6ETD_longGrad_01.10590.10590.3.dta
    { "132.0715 147.0837 218.1497 220.0796 230.4796 232.0339 247.2026 259.3970 288.2270 291.1560 299.8253 318.1685 325.1313 325.8452 342.9795 345.4805 349.1709 372.1323 374.2528 375.1326 375.9820 387.2794 388.4255 389.4038 394.8831 405.3392 406.2431 422.1203 424.0133 427.4456 428.5132 429.2105 430.1854 431.3031 443.0010 448.0626 458.1953 459.0851 467.2625 473.2182 475.3655 476.2024 477.9857 485.3104 488.3539 489.2596 492.0747 493.3931 501.1275 523.2679 524.2704 543.2321 546.2448 558.2181 559.4894 560.9747 562.1701 563.5605 572.1761 573.2726 575.4438 576.7999 585.0765 588.0498 598.0027 630.4970 641.5347 642.3078 643.3769 644.9922 648.2646 653.5646 658.4521 659.3398 661.1553 671.3445 672.2643 673.8300 686.2356 687.2797 688.3865 694.2662 702.9780 717.8011 719.0940 724.1596 735.9033 744.6633 747.7307 752.4191 753.2854 754.1927 754.8959 755.5855 756.2725 757.0895 757.9471 758.7214 759.3835 760.2155 762.5571 771.9141 773.3524 774.0510 775.0845 776.1815 777.1197 786.3012 787.1558 788.6226 790.5959 791.5471 794.0435 796.9368 801.4812 802.4707 805.9221 807.7691 813.2900 815.0375 819.3782 821.2471 826.4383 834.3730 838.5256 845.8716 848.2889 853.5008 854.1682 858.3172 861.9479 869.9322 874.4223 877.4258 878.3083 879.5461 882.4388 884.3433 886.2766 887.4866 890.3177 892.3860 894.1328 896.8228 898.6152 900.3166 901.3664 902.4816 903.2700 904.2283 905.2463 907.5801 908.5921 914.6594 915.6624 917.3672 919.1120 920.0164 921.1387 922.3276 923.3732 927.0599 928.1668 929.3540 933.0705 935.7183 937.1277 938.0724 938.7009 939.4881 944.4953 945.3680 948.3203 949.5163 951.0867 956.7850 960.2507 961.0474 961.7953 962.7480 967.9710 969.8694 971.1596 972.2363 973.5232 975.0296 976.2681 979.4318 981.0421 981.9901 984.7593 985.9832 987.3868 988.4250 989.2796 990.5510 991.8367 992.8450 994.2226 995.0679 997.8716 998.6213 999.3199 1000.5422 1001.5933 1003.1556 1004.0806 1005.2917 1006.1589 1007.3611 1008.6099 1009.6362 1010.5707 1011.6604 1012.4658 1014.5402 1015.9670 1017.7939 1018.6812 1020.3931 1021.3630 1022.9549 1023.6440 1025.9456 1026.6624 1028.3309 1031.8478 1032.5596 1033.6030 1034.4243 1036.3458 1038.7056 1040.3380 1043.4136 1045.9514 1048.1697 1049.0587 1049.9420 1050.6794 1052.7510 1056.5795 1057.4160 1058.3438 1059.0503 1059.7845 1060.4841 1063.3778 1064.2854 1065.3510 1066.1342 1067.3438 1070.7596 1073.7325 1074.5242 1075.1725 1077.2491 1077.9303 1079.5321 1080.5077 1081.6063 1082.7450 1085.4679 1086.6656 1087.3792 1089.4362 1090.0438 1091.5702 1092.6472 1093.3260 1096.6234 1097.3236 1098.6217 1099.5741 1100.5021 1101.1124 1101.7311 1102.8992 1103.8336 1104.7256 1106.4027 1107.7963 1108.7084 1109.4752 1110.7401 1111.8080 1112.7272 1113.4573 1114.3319 1114.9320 1116.0812 1117.0597 1118.2805 1119.8781 1120.8762 1122.2347 1122.9132 1123.8802 1124.6854 1125.7269 1126.9999 1128.0149 1128.7999 1129.7657 1131.0870 1132.8833 1134.1522 1135.5182 1136.3419 1137.5546 1138.4476 1139.1550 1139.8304 1141.1919 1141.8374 1143.2264 1145.2018 1146.2469 1147.3065 1147.9469 1149.5527 1150.4755 1151.9442 1152.6703 1154.1738 1155.1608 1156.5076 1159.3502 1161.2046 1167.1737 1170.2955 1171.2012 1173.1973 1176.2446 1177.4215 1178.2328 1180.5065 1181.3906 1186.8690 1187.8267 1190.8583 1192.4257 1193.9436 1194.8959 1195.5464 1198.4105 1199.5392 1201.9952 1202.5953 1204.9934 1206.0919 1211.7484 1213.1976 1214.5695 1215.7498 1217.5830 1218.2981 1220.3254 1226.4944 1228.7361 1233.7257 1235.6561 1239.3969 1240.0757 1240.8286 1242.0421 1242.7646 1244.9883 1246.4276 1247.7548 1249.3785 1250.7234 1251.9867 1252.7737 1254.0602 1255.4293 1256.5275 1257.6901 1259.8588 1260.7098 1262.3071 1263.2318 1264.1481 1264.9606 1268.8447 1269.7992 1270.6405 1272.4789 1273.1649 1275.6637 1276.6760 1277.8824 1279.0483 1282.3561 1284.8334 1287.7657 1289.6763 1290.7190 1291.4702 1292.7799 1293.5560 1294.5465 1296.1893 1300.2904 1301.1901 1302.6261 1303.7428 1305.5427 1306.1834 1307.2145 1308.1500 1314.6536 1315.8959 1319.4083 1321.2302 1330.2814 1334.6656 1335.7017 1340.0239 1340.7843 1341.6863 1342.6956 1346.4484 1347.0558 1347.8868 1348.8137 1350.6993 1353.6053 1355.2299 1356.2266 1357.1666 1358.0416 1361.2682 1362.0555 1363.2323 1367.6686 1368.4825 1371.4366 1372.5746 1374.9340 1375.9182 1378.2938 1379.1589 1379.7727 1380.7631 1382.4111 1385.3702 1386.5568 1390.0216 1390.9486 1391.7754 1392.7571 1393.8690 1394.8009 1396.9442 1398.2675 1398.8807 1400.2094 1401.3044 1402.8741 1403.8854 1407.2385 1408.1870 1411.1185 1411.8433 1416.5988 1418.4542 1419.4943 1420.5707 1422.6476 1424.9674 1426.5623 1427.3171 1428.0719 1429.7015 1431.2341 1432.2362 1432.8817 1434.8376 1436.6995 1438.9518 1440.3187 1441.0372 1441.6425 1442.3195 1443.3434 1444.0919 1444.7957 1445.6251 1447.6378 1448.2841 1449.4817 1450.7064 1451.7474 1452.6267 1455.0867 1456.0149 1457.7488 1459.4772 1460.3107 1461.2782 1462.2648 1463.1086 1463.8396 1464.6327 1465.9381 1466.7299 1467.3907 1468.3772 1469.4128 1470.5614 1471.5663 1472.3510 1473.6410 1474.4960 1475.5295 1476.5924 1478.0243 1479.1256 1480.2164 1481.1080 1482.4667 1483.4752 1484.2919 1484.9777 1486.3055 1487.0165 1487.6227 1488.5546 1489.5608 1490.3602 1491.2828 1492.1643 1492.8318 1494.0306 1495.2540 1496.0427 1497.1150 1497.9700 1498.8796 1499.6732 1500.5129 1501.2882 1502.2040 1503.4967 1504.5358 1505.5095 1506.4403 1507.5525 1508.5974 1509.6749 1510.7125 1513.4476 1514.1985 1516.1212 1517.1346 1518.1415 1519.6188 1524.7462 1530.5406 1531.6271 1532.9105 1533.7650 1536.1287 1538.3522 1540.5753 1545.5656 1548.4792 1552.3759 1555.9591 1556.8051 1557.4430 1558.6722 1560.5824 1564.6781 1565.8656 1570.8792 1573.4445 1574.9510 1575.6534 1576.5870 1577.6252 1578.5360 1586.2131 1588.5458 1589.9314 1591.4358 1592.6217 1593.6790 1594.6000 1595.3712 1597.6683 1598.7733 1599.6034 1601.7477 1602.5172 1603.6656 1604.5775 1605.7361 1606.5272 1607.3821 1608.0726 1608.7301 1610.0432 1611.7487 1613.7721 1614.9047 1616.7120 1617.9666 1619.5499 1620.2096 1626.8650 1627.6273 1629.4537 1632.9762 1633.6058 1639.7977 1641.3356 1642.4802 1646.5656 1654.2638 1655.7179 1657.8710 1659.6442 1660.9542 1662.6053 1663.8749 1668.2489 1669.1154 1672.5940 1677.8602 1678.4950 1679.4412 1684.5973 1685.4513 1688.3505 1689.4299 1690.5846 1691.5455 1694.7162 1695.5563 1697.2151 1700.8923 1702.3458 1705.5192 1706.1414 1711.7404 1716.2643 1717.3661 1718.1285 1720.4288 1723.7699 1725.2262 1725.8705 1727.8346 1731.9359 1736.0670 1736.7618 1738.4319 1739.3585 1740.3849 1754.4172 1755.6169 1760.1327 1761.8771 1766.0066 1766.9148 1771.9343 1773.3990 1774.6653 1775.5326 1780.0551 1780.6740 1781.7604 1786.6141 1787.5883 1788.3990 1790.8025 1792.9210 1793.9456 1794.7343 1795.9086 1797.9089 1799.9230 1801.6803 1803.5555 1804.6731 1805.7235 1809.8777 1811.1664 1812.3549 1813.0197 1813.8759 1815.0143 1826.7389 1831.6859 1832.8005 1833.6959 1837.2887 1838.3933 1844.3966 1846.0774 1847.7430 1849.8075 1852.8240 1853.5051 1854.5183 1855.1476 1857.3937 1858.4646 1861.0085 1862.3038 1863.0139 1865.0543 1865.8370 1869.4540 1870.1364 1871.4215 1873.7247 1874.9348 1875.8529 1876.8010 1877.6287 1878.7743 1880.5300 1881.7614 1882.4681 1883.4973 1884.7552 1885.6310 1886.4500 1887.1340 1892.7078 1893.5094 1894.6998 1895.7473 1898.3099 1901.4218 1902.3488 1903.1700 1907.8961 1908.7579 1910.9142 1914.8903 1916.6761 1918.8949 1921.9607 1931.4806 1933.1414 1936.7150 1937.5914 1938.5076 1939.6466 1940.3966 1941.0419 1944.1234 1945.2714 1948.6447 1951.8010 1952.5461 1953.8221 1954.7272 1955.5040 1956.8084 1958.7610 1960.0621 1961.2137 1962.7708 1964.9230 1967.0609 1975.5131 1978.1849 1981.8810 1983.2643 1984.0470 1984.9055 1985.7872 1989.3231 1990.1945 1991.3435 1993.7008 1996.1595 1997.3917",
      "  1.2076   0.2140   2.3574   0.2235   0.7150   1.5416   0.5106   0.2808   1.4575   1.2365   1.5764   1.0484   2.0898   0.3594   0.3937   0.5057   1.1949   0.4417   0.5184   0.7567   1.7286   2.5776   1.0447   0.6475   0.4878   0.5413   0.7210   0.8683   1.5805   1.2481   0.9224   0.4397   2.7264   2.9179   0.4282   3.3980   8.2613   1.4577   0.7768   0.8242   0.9273   0.2600   1.0749   0.8343   6.5725   1.3257   1.4782   1.7275   2.0267   6.2333   4.6120   0.2138   2.1553   0.6762   0.3060   0.5197   1.1004   0.5557  13.5331   3.3736   1.0484   0.2206   1.8039   0.7566   0.8574   0.6322   1.7819   6.2294   1.5022   0.4449   1.0625   1.0556   1.0846   8.6348   0.9826   1.1124   1.1257   0.7346  49.9103  17.2452   0.8687   2.5821   0.8434   1.0176   0.9266   1.3142   0.8698   0.2695   2.9848  95.5053 123.8437 415.3209 162.7748  29.4583  11.2904   4.5553   6.4016   3.3487   2.8605   1.5501   2.8116   1.2292  11.5321 164.9001  52.5358   1.8989   2.7205   7.1684   1.1077   1.7213   4.1389   0.4881   3.1712   2.0489  14.5531   1.2746   1.2662   1.4230   0.2810   0.3932   0.8920   0.6513   1.0045   1.2514   0.5283   1.1155   0.4214   1.7106   0.5640   3.0417   1.7984   1.2852   1.4770   0.3818   2.0609   3.2723   0.3714   0.7191   1.1172   1.7947   3.7536   0.7794   0.3932   4.5500   2.4211   3.7334   0.5021   3.0261   1.2865   1.7746   1.0857   0.1931   2.1706   1.3030   1.3275   1.7470   3.6695   0.7659  32.0758  22.9650   0.7790   2.6867   4.7843   0.5964   2.1821   4.6095   7.3233   3.4811   0.5895   1.5527   0.3956   0.3037   0.6638   0.8059   1.8832   0.7582   7.0576   2.7757   1.7917   0.4221   0.2137   1.1865   9.6860   2.5186   3.0589   1.8420   0.3490   3.8613   2.9495   2.1801   7.8167   2.0074   0.8264   5.8261  19.8522   3.2070   8.9683   7.8701   9.8508   6.2420   7.1219   9.3822   3.4960    3.8728    0.2815   66.7999   72.3691   56.9526   18.8768    5.7605    0.3263    1.4310    0.7763    0.8309    0.4642    1.1406    1.7456    4.3327    0.8780    1.7747    1.7727    3.3483    0.5395    2.4342    2.8337    1.0952    1.4446    5.2073    2.3888    0.3338    0.3789    0.6679    1.2664    0.8963    1.2939    4.4210    2.6471    2.5197    1.9282    1.4689    1.5747    0.1930    0.9260    2.8797    0.5058    1.4175   12.8377    4.1711    1.7514    2.1085    3.2747    3.7872   14.8720   11.9092    1.9080    1.9879    4.6735    4.8620    1.5929    2.5621    0.7800    6.2992    5.7929    0.3338    2.4582    2.1743    2.2921    3.1369    1.3672    3.2963    2.1328    6.6508    6.0206   14.9452   42.7227   66.9737    4.4493    3.8855    1.3761   18.6998   34.3735  338.7904  131.1618    5.8179    5.1552    7.6294    2.0181   11.8245    9.4991   22.9782   24.9889    7.7356   18.3284   18.1433   33.3393   27.6002   10.2412    0.1801    0.4443   10.1228   32.5019   82.3745  101.4169 1000.0000   23.3991   28.4881    4.4190   11.2680    0.7131    1.9087    0.6864    0.8469    3.8191    1.6547    3.0529    0.7080    0.5606    1.9551    1.5675    2.4145    1.4108    0.8076    3.4161    3.3622    3.2567    0.6298    1.9713    1.8804    6.0610    2.1832    0.7576    4.2365    9.0074    6.9004    0.7836    0.6294    0.6448    1.2883    0.8924    1.6529    2.3994    4.0579    0.3263    0.4654    9.7691    9.8065    1.9246    0.6522    0.5289    2.3048    2.4125    2.2490    2.0460    2.2836    0.3336    0.4194    2.9350    1.1824    2.5262    5.0736    1.9755    4.3466    4.5507    4.3216    0.4441    1.9634    2.5902    1.3023    0.8350    0.6009    1.0725    2.3361    0.9471    2.1322    7.5486    4.3870    6.5467    3.8657    0.5319    5.8366    7.9936   10.2615    2.2668    3.9705    1.0091    3.9796    1.0987    2.9872    1.0362    2.0512    2.5960    0.2360    0.4616    2.2687    3.4097    0.7655   19.5894    8.5281    3.0799    0.6448    0.9140    1.6849    2.7325    3.6269    1.5008    8.7271    1.8260    1.8703    2.0504    3.2562    2.1932    3.7697    1.0014    1.0892    2.0676   10.9117    2.6939   16.5948   47.8316   19.5815    3.9960    0.5061    2.4257    0.6470    1.6296    4.4909    0.6711    2.5167    0.9956    3.9846    0.6229    0.6300    1.2317    0.3488    1.7704    1.2194    5.4718    3.0694    8.8268    2.6515    3.4444    1.8046    5.2007    4.2030    1.2666    0.5955    1.4315    3.3365   10.9769    9.6559    3.3541    1.3873    0.8242    1.6417    0.3664    2.5252    4.5472    1.3094    4.0787    2.2106    0.7089    1.8252    1.8290    1.6118    4.9864    3.6972    0.8483    0.9548    1.3124    1.9545    0.9277    0.9933    0.6038    0.5283    2.5788    1.2767    0.2186    5.1594    4.6538    3.5988   14.1830   11.9669   18.5134    5.8140    4.3164    3.9043    5.2363    5.3522    5.2094    1.1306    0.5130    4.5587    1.7590    1.2293    4.0148    0.6274    4.5225    6.6950   26.7434   22.2381    7.7080    2.8686    1.9441    4.3427    2.3264    0.6434    0.4201   11.3562    2.3697    7.8335    1.6202    2.4843    2.9855    5.8517   17.8666   21.4525    8.2153   10.9946    5.5324    6.2264   21.3605   43.6459   14.6577    5.5590   21.0449    4.5376  160.1788   97.0698   45.0015    7.0711   16.9170    9.7769    5.0658    5.5334    7.2382   11.1346   45.0870   37.9341   15.3358    8.7210    6.6676    1.6461    7.6088   22.8921  127.6581  182.2985  118.2491   74.3847   98.4043   54.3199   14.8665    1.2609    0.6863    4.4409    0.6221    2.5149    8.1892    1.9566    2.8118    8.4393    1.9692    0.9676    0.6976    1.0319    2.0317    1.4544    0.4874    2.3965    0.3256    0.6893    0.4004    1.5528    1.9372    1.6881    2.9417    0.2133    0.7383   11.3397   58.1341   25.1474   11.6925    2.1586    1.6553    1.4886    1.6212   14.0345   16.5649    7.0838    2.6957    2.2239    2.1454    1.4155    2.4047    2.3002    3.8509    5.0052    7.1986    4.2648    1.6357    3.1159    0.9122    1.6793    5.0820    2.2470    0.5356    1.1527    2.3515    1.0578    4.9751    3.9399    0.6701    0.7594    1.0517    0.4015    0.9613    2.1521    0.1929    1.6375   10.1434    1.9662    0.4823    1.0296    1.1979    0.5108    0.4227    0.6855    3.1168    1.8406    1.2817    2.3289    4.3205    0.5351    0.7420    2.1783    4.4910   22.3696   22.1857    4.5112    0.5995    1.3008    0.6007    1.7847    1.0353    4.0656    0.4198    3.4135    1.7383    1.2329    2.7071    1.1877    1.0536    2.2255    0.3713    1.2426    0.3756    4.0904    0.4890   26.9088   15.2575    3.2073    8.5312    2.3647    0.5726    1.0939    0.4167    0.2364    0.3041   10.7180    0.7391    6.3138    0.5006    1.5169    1.9374    0.2268    5.1699    1.6165    0.4959    6.7578    0.8952    4.6251    0.2976    1.3892    1.4037    3.5373    8.7896   10.3457    2.9930    1.5991    1.4395    0.9617    1.5030    2.8196    2.5663    0.3422    4.9941    0.5507    2.4602    2.6109    2.5353    1.3524    1.3714    2.4542    2.6043    2.6284    0.9234    2.4981    0.8737    1.5071    0.4602    2.1118    3.2074    1.0979    0.8767    0.6194    0.3803    2.0895    2.7964    2.3609    6.3509    3.9330    0.6710    1.5431    0.2822    3.5670    0.3267    2.8275    6.0304    5.4242    5.4718    3.1268    3.0904    0.8519    1.1704    0.7215    0.2140    0.9974    0.8218    0.6193    1.7848    0.7198    0.9822    1.8476    0.1490    0.9795    1.1393    2.2994    0.5434    4.0934   15.5582   22.2507   13.9339    5.9831    2.6821    1.1616    3.2216    1.2479    4.5594    2.1396    0.8027    0.6561    2.5058    3.0038    1.4703    0.6191    2.6715    3.6273    3.9598    1.1740    0.6479    1.1190    0.8267    0.9587    3.9114    2.1504    0.8505    1.3656    0.6370    0.8712    0.7767    1.1165    2.0315    1.8604",
      "3", "ETD", 754.26,
      true, 2, 4, 0.9, "2 3 4 6", false },

    // 20090326_Yest_LysC_1x6ETD_longGrad_01.10944.10944.4.dta
    { "175.8720 199.8941 218.1583 245.2375 273.0284 274.2396 289.3082 346.2828 358.1471 359.0504 362.9637 385.9947 387.0605 387.6643 390.0204 403.3923 416.9887 422.3281 438.4811 449.2099 456.9319 458.2459 471.9123 473.2959 496.1605 510.5996 534.8668 558.6778 564.4798 586.2914 587.4955 593.1797 612.1970 642.2539 645.4006 673.3190 692.7047 693.7328 694.3719 707.2170 729.7064 736.3296 738.6448 774.5779 785.3604 786.3757 791.3216 792.3455 821.2388 822.4852 823.8489 824.7068 832.6044 859.9987 885.7700 895.0964 896.4106 897.1700 898.6160 899.4789 900.9644 901.8069 902.9141 904.2317 905.2639 906.5930 935.1630 936.8126 937.6146 938.2476 944.3350 949.3759 954.8591 974.3253 975.4271 976.2615 977.1982 994.7823 1010.4415 1018.3923 1019.4538 1021.3811 1028.0819 1033.3838 1039.5527 1040.4869 1041.1951 1042.4680 1049.5273 1058.4209 1069.6090 1090.3763 1104.6752 1111.2748 1121.3943 1131.6147 1132.6053 1146.3213 1147.4335 1153.2089 1155.8812 1164.2150 1165.4015 1166.3696 1170.2813 1175.3116 1176.0127 1176.8715 1178.0538 1179.4586 1180.1742 1180.9137 1182.0205 1183.0919 1185.4933 1186.2325 1187.3251 1188.1422 1189.1251 1190.2260 1191.2662 1192.4812 1193.5873 1194.2908 1195.0613 1195.8481 1196.9855 1198.0087 1198.7291 1207.3302 1211.6921 1223.9277 1224.6180 1233.6984 1235.6779 1241.8362 1251.8188 1257.3807 1259.4963 1262.5111 1263.4620 1264.1301 1264.8878 1267.9177 1273.4928 1275.4266 1277.7567 1279.6074 1282.1769 1283.6276 1284.6364 1290.4642 1293.4266 1295.2639 1296.6180 1302.2794 1307.9474 1311.6423 1313.4198 1314.0905 1315.3881 1316.1901 1316.9230 1317.6395 1319.8770 1321.4464 1322.1534 1323.1881 1324.1664 1325.4769 1328.3777 1329.6705 1330.3971 1331.5485 1333.1272 1334.5436 1335.4484 1336.4099 1337.3990 1340.6471 1341.3518 1341.9587 1342.8571 1344.1476 1344.7744 1345.5146 1346.7733 1348.1099 1348.9272 1351.2007 1356.2775 1358.0707 1359.5850 1363.6600 1369.3317 1375.6279 1377.6879 1378.3973 1381.5529 1382.1711 1387.8429 1396.8002 1397.4147 1401.4276 1404.5148 1405.6346 1407.8031 1409.7621 1418.9779 1425.5529 1430.6195 1435.7096 1437.9529 1444.8386 1446.1393 1447.4783 1453.4469 1454.5709 1457.5491 1458.9441 1464.5022 1465.4081 1469.3082 1469.9724 1470.9104 1472.9240 1477.8826 1482.6809 1489.5529 1490.4714 1492.0417 1492.9250 1493.9073 1494.7794 1501.0922 1505.6737 1506.8972 1507.7159 1510.3519 1511.5221 1514.2782 1516.4744 1518.1965 1519.8464 1522.2704 1525.4772 1529.8956 1531.4729 1534.4573 1540.1346 1546.6332 1552.2518 1556.7091 1564.0348 1564.7952 1565.6415 1566.2545 1567.6869 1568.7028 1569.5258 1578.6302 1582.7854 1594.2843 1596.6448 1599.3793 1600.3068 1601.5730 1602.5927 1604.0251 1615.4640 1616.6613 1619.0719 1621.8811 1623.5704 1626.0370 1629.8317 1634.1423 1634.9856 1636.9403 1638.3601 1639.5543 1646.2424 1650.5768 1656.0638 1657.2604 1658.0907 1660.0883 1661.8132 1662.5750 1664.4910 1671.9044 1673.8621 1674.5935 1675.6832 1676.7406 1679.4630 1680.6952 1682.0568 1683.6165 1689.6736 1692.4982 1698.9215 1699.9553 1702.7340 1703.9652 1704.8475 1706.7195 1707.3943 1708.2800 1709.0286 1713.8107 1715.9885 1717.3331 1718.0604 1720.2208 1722.3376 1724.1884 1726.0417 1726.9645 1728.1034 1728.9226 1730.5367 1731.3829 1732.3666 1733.3103 1734.3645 1735.3290 1736.6276 1738.5490 1740.6384 1741.9008 1743.9740 1744.8041 1745.9012 1747.2345 1748.1301 1749.0778 1750.0002 1750.9318 1754.7299 1755.8284 1756.8458 1758.7975 1761.0399 1761.9836 1763.1096 1764.1193 1765.1483 1765.9137 1766.7489 1767.9734 1769.0042 1769.8815 1770.6935 1771.5680 1772.3074 1773.7887 1774.8766 1775.6345 1776.3323 1777.7008 1778.6428 1779.3386 1780.2023 1781.1492 1781.9978 1783.4535 1784.4506 1785.1273 1786.0966 1787.0409 1788.2284 1789.2784 1790.3038 1791.4622 1792.3827 1793.1508 1793.9470 1794.7955 1795.7273 1796.4738 1797.2253 1798.2062 1799.4427 1800.9071 1801.6083 1802.4524 1803.4679 1805.7606 1808.3549 1809.5353 1815.8075 1818.6710 1821.8365 1830.9684 1832.8890 1840.4708 1851.7019 1867.7532 1869.0187 1871.7620 1873.1697 1878.4259 1886.9445 1890.1262 1892.5468 1893.3860 1895.0382 1896.6545 1898.0157 1900.5481 1906.9015 1910.6238 1911.7203 1913.1337 1916.8942 1917.8323 1921.5272 1933.0487 1933.7792 1934.9421 1949.1449 1949.8992 1950.8385 1952.0890 1953.9465 1956.6921 1961.7362 1967.0646 1974.6322 1982.1481 1986.6511 1987.3944 1996.0720",
      "  4.3624   4.7879  20.4527   6.6807  35.8763   5.0918  15.6105   1.3454   3.6138   2.1448   3.7193  27.7317  15.6468   3.0271   2.8459   3.5915   2.9039   4.4288   8.2573  33.4606  22.5337  10.9575   3.8657   1.8491   2.2616   1.5618   1.5590   2.6759  36.8597   7.2591  13.4960   3.7699   6.0739   5.3025   5.4323   1.5584  13.8957  11.9132   8.8741   5.7557   3.9342   2.6969   7.6398   1.1519   1.1325   5.2826   6.6666  35.6866   4.2547  17.8581   4.8340   2.0095   4.3716   6.5766   9.4939 402.0601 464.8441 537.4277  17.9156  11.4710   6.2864  10.5367  10.8416  31.2116  20.9992   1.9120   1.9416   4.2069  16.4157   5.3653   5.5731   3.7154   2.7373  12.0522  22.7246  10.4559   7.8981  22.9325    5.0917   70.2356   15.1801   16.5597    6.3569    7.2440   97.9662   59.2950    4.9781    3.7180    6.7488    4.8467    4.8074   25.5973    0.7196    1.2403   20.9554    2.9501    6.9573   12.7672    6.1440   25.8016    3.8721   13.1331   24.5403   15.3066    8.4548   14.2019   97.2123   61.2889   10.7561   27.8591  261.0292   81.1214   33.4735   13.0577   55.6080    1.8582    3.8286    3.2263   98.7597   78.0330   29.2879   26.7894   61.2896  113.8111  471.9098  358.8193   69.3440   15.7886   11.8259   12.7191    3.5384    4.0754    2.6276   14.6821    9.7790   15.8430    2.7378   18.5379   13.0401    4.4782    2.5456    4.4677   15.8584   13.9536    6.7222   12.2559    8.2438   26.7612   20.6164   54.1877   21.4669   14.5228    3.0030    5.9903    3.1915    5.5731    8.8791    3.9608   42.3983   26.5532   32.7512   45.2916   22.6209    7.5070   26.7845   89.1145   17.6498   63.5447   23.8606   14.8505   63.5104   19.3959    1.2831   32.9884   32.3640   28.6556   36.8855   25.8593   53.3514   40.6725   70.8280  107.3617  291.0123  192.9853  203.6091  417.3050   45.1871   20.7725   10.4230    7.1783   14.1885    2.8451    9.7166    2.9932    3.7233   20.7696    1.9156    2.9814    8.0314    9.7279    5.3751    8.3862   20.9312    7.1718   16.5633   17.5793    5.9896    2.9916    5.4384    4.1460    3.5051    8.0391    7.0511    4.4688    8.4119   14.0447    2.2393    6.5569   23.8706    5.3554    1.4536    3.4873   15.1218    2.6706    6.1644    8.7950   16.1862    4.2813    6.6045    6.5178   13.1938    1.7797   22.7762   40.2348    5.9908   47.1946   33.6723    2.1450   33.1998   12.1067    9.6527   11.6886   23.3409    1.1411    6.8669    3.0980    1.1323    1.3458    2.9472    4.8933   40.1039    9.7925    4.2470   35.5044   18.2576   12.7848    6.3117   76.5752   48.8269   10.3323   12.9210   12.3009    7.7026    5.8583   11.2033   15.7370   12.5100   10.7524    6.1776    3.9357    2.9531    9.2800    4.9446    5.0280   12.8576    1.4519   16.9346   18.0947    7.4970    1.0260    2.1293    8.9426    5.8193   29.4846   18.4077   24.6667   16.0103    2.7393    9.7548    3.7904    1.5611    1.4558    6.3051    2.4478   10.7336   14.2377    1.7755    6.2844   23.8180   12.0611    9.3421   11.4608    7.1515    3.2196   12.6391    6.7725   67.7098   32.5935   21.0991    4.1655   11.3855    4.6931   16.1719    5.1260   10.0803    8.6224    8.5366   23.9460   25.0061   51.9124   32.0715   10.3749   18.8416    9.6547   13.8276   33.7576   76.0382   30.7797    1.3435  101.8303   28.4178   24.8342   12.7440   22.9802  118.5810   24.5601   61.0085   64.8762    8.8232   55.3785   26.0974   11.1268    0.6121   73.7481   72.0108  424.9133  349.1756  131.7968   47.4989   19.6911   28.8521  129.5987  344.8895  421.7938  209.4594   37.1212   12.8375   31.8070   21.4200   38.0732   31.5828  103.0884  126.7100   17.9557   32.8707   12.6995  159.1095  188.3956  161.5748  172.3210    4.0091   28.4575  170.2840  222.2034  531.8163 1000.0000  583.4736  747.9631  198.2331   81.7858   22.8003   28.8037    5.5438   19.4686   17.0368    6.0680   11.5053   19.6225   17.7406    2.8740    1.1310    3.1348   13.2612    7.4114    2.8418    7.8868    6.3045    3.4019    2.3112   16.0399   13.7013    5.9530   12.2208   15.6279    8.3215    2.3035   18.6413   24.7575    4.8850    2.3112    7.4741    3.9194   12.8287    4.8486   10.3892   19.0347   14.9629    6.1971    3.2175    3.5099    6.7257    9.0118   20.0309    5.3249    4.1317    1.3497   11.0168    6.1148    2.7018   10.4698   11.6874    4.3419    1.5923    3.3443",
      "4", "ETD", 896.93,
      true, 2, 4, 0.9, "4", false },

    // 20090326_Yest_LysC_1x6ETDsa_01.02755.02755.2.dta
    { "230.0981 286.1379 318.3454 356.3785 371.5393 373.5950 400.2392 401.3641 402.4094 427.4315 455.0078 456.1883 457.6418 471.3203 472.2448 473.3338 474.3383 483.2625 484.4883 516.1983 527.4332 530.3140 539.4369 548.9628 554.0997 557.0588 561.3072 572.1891 573.3624 574.4203 575.5069 580.4650 586.4937  617.1796 617.9493 618.8731 620.1948 621.2181 621.9984 622.8047 623.8954 625.2957 630.2192 632.0173 633.7305 635.7570 641.9611 653.3010 657.3339 659.5907 675.3161 676.1430 677.2904 678.5961 679.3595 703.5071 704.3304 705.3985 740.1729 762.3950 766.2808 768.6506 770.0381 798.4349 800.6177 808.5179 816.2749 817.4937 818.4566 819.4445 820.8203 822.4968 823.1750 824.3505 829.4699 830.1567 832.6044 834.6618 846.2754 850.5706 854.6957 857.3557 858.9043 859.8839 862.1486 873.2971 874.3799 875.4854 876.4969 878.7475 879.3759 885.4304 890.4575 903.4590 905.9922 912.6687 913.3745 914.5020 916.3308 922.7803 925.0191 925.7469 926.6190 927.3928 928.4533 929.5112 930.6172 933.9146 937.4092 943.1749 944.7322 947.3606 954.5606 956.4565 957.6349 958.7437 961.3524 963.2213 964.5411 966.5060 972.8018 973.5627 974.4982 975.5117 976.7676 979.9922 986.5786 991.7271 993.4317 1006.2110 1010.6760 1018.1799 1019.1472 1027.8459 1029.8701 1031.1293 1034.0077 1042.6014 1047.3507 1050.4709 1053.0702 1056.7209 1057.4386 1058.3582 1059.1672 1084.7484 1085.4883 1086.6018 1092.6569 1101.5668 1102.5316 1103.4456 1104.5449 1105.4836 1106.2820 1107.3219 1108.6346 1118.7115 1119.9362 1134.4797 1136.1908 1138.3112 1143.4336 1144.3865 1145.3591 1146.4233 1147.3323 1148.7775 1156.2939 1158.3121 1159.5941 1160.4716 1161.4147 1162.6970 1163.3986 1164.3463 1165.4460 1167.3688 1169.7459 1170.3529 1173.8286 1175.2434 1176.2880 1177.2156 1178.3756 1184.8765 1186.2137 1187.5798 1188.6298 1189.5321 1190.4587 1191.2911 1192.2899 1193.4386 1198.7231 1204.5912 1205.5559 1206.5072 1207.3990 1208.5984 1209.4135 1210.0560 1214.8812 1215.6157 1216.6219 1217.4187 1218.3597 1219.3088 1220.2905 1221.2997 1225.5035 1228.3008 1229.3121 1233.5978 1234.5568 1235.5013 1236.5106 1237.4996 1238.3781 1265.4860 1268.3812 1440.9069 1466.6534 1467.7113 1484.6107 1490.2760 1494.6941 1496.7672 1520.9896 1532.8058 1537.6378 1569.5238 1596.6863 1607.9568 1621.5748 1622.5538 1650.2029 1660.3529 1668.5236 1676.0067 1677.2543 1737.1874 1740.8645 1793.6110 1800.9139 1808.6078 1820.8539 1823.1715 1832.6832 1834.9177 1836.3197 1838.1199 1849.7692 1851.6238 1852.4869 1853.9988 1856.8741",
      "  0.4833   2.8457   1.3471   1.5642   4.1266   2.0825  18.2947  11.9255   3.2442   1.8652   1.0598   1.7549   1.4362  23.5484  14.5760   2.9233   0.3836   0.7490   0.6133   1.0514   1.0303   0.9975   1.4244   1.7969   1.3435   1.2036   0.7670  22.5997  24.3054   3.8169   0.5100   0.6350   0.4580 1000.0000 950.0541 146.4767  20.2407  10.1327   5.6393   1.6020   2.0908   4.8558   0.7916   2.3856   1.5659   0.3025   1.3060   1.4838   0.6670   2.7304   4.8081  11.3607  15.2884   5.1882   1.5464   5.8153   1.3453   5.4665   0.6211   1.4106   0.5344   1.2328   0.7641   0.4056   0.7916   1.1898  51.8903  45.0533  20.9658   2.8625   0.6900   5.5034   1.1943   0.9176   0.7222   0.8172   1.0576   1.1637   2.8884   2.3396   1.5933   1.3111   1.4198   2.2153   1.3822  36.6191  93.2001  31.6534  13.9696   0.5607   1.0470   1.1894   1.3608   0.4800   1.2049   1.2550   0.9972   0.4859   3.2288   2.1349   4.6879   9.9833   3.0879   3.9000   9.7938   7.9775   3.2508   1.2384   0.2507   1.4634   0.8810   0.6898   2.2602  24.2051  10.3956   2.2402   1.0667   2.7753   1.0992   1.0642  23.1131  80.2114  41.9553  18.8793   6.8117   0.8456   0.8240   5.1849   2.9205    0.4809    2.4141    1.8661    0.9377    2.9857    0.4307    0.8422    0.9059    2.4583    2.0787    0.2538    1.9636    4.3802   13.6931   13.2065    3.3515    0.3303    4.0916    4.8061    1.1273  273.3655  164.5871   57.7409  117.6905   17.3177   16.4939    6.5438    1.7975    1.2035    0.6403    2.4150    1.9684    0.8133    8.3222    6.5229    3.3791    3.8109    0.6942    3.5036    0.2754    1.8501   23.3673   99.5213   47.5771   18.0218    8.4184    1.8678    4.3590    2.1376    0.7651    1.3611    2.3389   71.5708   39.1599   15.5463    5.5008    1.4616    0.7630    4.8375   22.6568   72.4867   18.4484   43.0253   10.7310    0.9463    2.4418   12.0548   48.5096   85.5242   32.0147   10.2352    0.3914    1.5637    2.8594    3.5130   47.4181  422.0222  241.4819  118.3296   40.2156    5.6453    3.3134    0.6608    2.7144  354.3168  331.3383  239.9229  102.3721   26.2857    5.2251    0.9337    1.4099    2.2105    2.6525    2.4697    0.6425    1.5885    1.0301    0.9003    2.7271    0.9299    0.4291    1.0306    1.2106    1.2834    1.1393    3.0377    1.8045    2.0633    3.3763    2.2776    2.2298    2.1149    0.7932    1.2733    0.9688    1.4401    1.2338    1.2547    2.6372    0.3283    2.3958    5.4447    2.5306    2.2378    0.3022    1.5756    3.1085",
      "2", "CID ETD", 617.86,
      true, 2, 4, 0.9, "2", false },

    // 20090326_Yest_LysC_1x6ETDsa_01.02929.02929.2.dta
    { "389.4117 391.0946 505.2968 507.3191 531.2183 532.9875 586.2795 604.4263 605.3598 606.4113 627.1395 632.1246 642.8731 647.7056 657.8644 661.1228 662.4505 663.5219 684.4991 691.8132 700.9722 756.3806 784.6460 789.4070 791.5776 804.7153 824.1043 825.2573 827.2163  828.6158 829.5567 831.2430 832.2522 833.7403 834.8653 835.9374 836.9997 838.7095 840.0015 841.4807 852.7488 855.3260 857.3301 862.5591 884.5059 888.2957 893.1559 916.4060 921.8802 924.5466 933.4266 937.3484 938.4031 945.7426 946.4650 948.7126 961.4505 962.6267 963.4643 964.3229 965.4019 972.7449 982.4905 990.2189 991.5987 992.4919 993.4343 994.4797 996.4581 1004.6815 1008.6563 1009.3945 1011.1580 1023.3934 1024.8807 1052.1252 1053.6049 1057.2513 1064.6522 1072.8944 1077.9215 1082.5858 1091.8741 1094.1873 1095.7383 1110.5525 1111.6389 1112.2892 1119.4299 1126.7136 1129.5619 1136.2286 1139.2841 1140.5375 1141.4402 1153.1908 1154.3090 1155.6510 1176.2434 1177.2328 1178.3448 1181.4806 1182.1593 1182.8680 1187.5962 1194.3029 1197.2499 1198.1327 1204.5625 1210.7902 1212.1400 1214.1775 1215.5211 1218.1316 1219.9628 1221.1005 1222.3605 1223.3898 1224.5499 1225.6085 1226.4681 1227.9926 1228.5979 1232.2542 1233.2992 1234.5721 1235.2050 1237.5599 1239.4896 1240.2415 1241.2290 1242.0975 1242.7695 1243.6515 1244.5927 1245.5897 1248.1522 1248.7697 1249.4774 1250.6180 1252.3149 1253.7596 1254.4711 1256.7651 1261.3907 1266.1443 1267.2919 1268.2922 1269.4253 1270.6189 1271.5963 1273.5363 1277.1300 1285.0336 1285.6952 1286.5616 1297.1799 1301.2589 1303.4088 1307.3865 1310.0802 1311.4427 1312.3468 1314.4532 1329.3843 1330.9202 1335.6158 1338.5448 1339.6599 1340.9244 1342.0405 1344.7380 1345.3879 1349.2100 1353.3885 1365.1899 1366.8412 1367.4452 1368.3197 1369.0911 1369.9567 1372.1359 1377.0958 1382.3802 1382.9803 1384.1898 1385.5289 1387.5792 1395.8666 1397.2480 1398.4102 1399.5603 1400.6278 1406.8954 1410.8093 1412.3290 1413.1520 1414.3805 1415.6887 1424.1570 1425.0261 1428.8671 1439.1027 1440.6417 1441.3064 1442.0126 1454.3318 1455.3502 1456.5426 1457.4420 1459.7175 1460.5052 1463.5647 1470.8596 1471.6796 1476.4039 1479.8065 1482.5315 1483.9215 1493.7325 1500.4481 1502.0992 1509.7078 1510.9183 1512.7770 1514.0159 1514.8826 1515.6151 1523.6542 1524.5192 1525.2487 1526.3709 1527.4846 1528.7477 1537.2732 1543.2845 1546.3744 1547.9034 1548.9309 1552.8575 1553.6243 1555.9164 1558.4528 1559.4818 1564.1635 1565.4012 1568.0790 1569.1183 1574.3885 1575.6483 1582.9139 1584.2750 1584.9357 1585.6139 1587.0966 1589.9462 1593.6754 1594.7550 1596.2111 1597.6212 1598.4812 1599.9320 1600.7953 1601.7797 1608.1990 1609.5865 1610.4445 1611.3497 1612.9202 1613.6732 1623.3538 1624.4572 1625.3722 1626.7540 1627.5168 1628.2701 1629.2654 1630.1409 1631.0465 1632.6423 1634.2019 1635.4436 1636.6576 1637.9016 1638.6970 1639.4714 1640.4979 1641.4641 1642.3527 1646.7335 1651.0729 1651.7736 1652.8461 1653.8489 1655.0579 1655.7216 1656.6390 1657.7958 1659.1790 1660.0687 1670.9275 1674.4968 1676.9290 1680.0575 1685.6888 1692.5092 1746.4464 1792.5345 1795.4675 1849.9757 1859.6594 1887.2311 1893.0343 1907.1237 1951.7926 1968.7068 1976.0979 1993.9811",
      "  0.4432   0.4417   0.9354   0.8766   0.1712   0.3226   0.5044   9.6652  44.2951  11.0255   0.3226   0.1435   0.5594   0.6825   0.3529   0.2423  20.8132   5.4826   0.7176   0.6450   0.2833   0.1577   0.5876   0.3551   0.7929   1.5588   5.9055   7.9331  94.5543 1000.0000 279.5447  13.4164  16.5519   4.1069   4.3171   1.4572   0.8147   0.5594   0.5766   0.3814   0.5656   0.1724   0.5781   2.1148   0.1717   0.3526   0.1161   0.6280   0.3660   0.4753   0.3448   2.2703   0.3503   0.6154   0.5865   1.0615   0.5179   2.1165   1.1192   1.8186   0.9755   0.9463   0.3675   2.6630   1.6269   4.0809  31.7227  12.0026   1.5629    0.6499    1.2754    0.8833    0.7261    0.1851    0.1983    5.4962    5.1440    0.3706    0.4881    0.6551    1.5435    0.6472    1.7838    0.6593    0.7380    2.0398    0.6327    1.2175    1.7458    0.5819    0.2884    0.2446   21.3980   19.5155    2.7173    2.2491    4.5808    1.1293    0.2402    2.2120    0.4669    2.6793    0.9654    0.7541    0.1891    0.1845    0.4222    2.3650    1.8310    1.2977    3.1053    1.1177    2.5762    0.9639    3.7044    0.1880    1.2117    2.3801    0.5720    2.9481    3.3816    2.9527    0.2321    1.7355    3.1817    1.7169    1.0149    1.3746   17.7523   20.2252   27.3089    4.6224   21.3270    0.7230    2.2827    1.7491    1.5890    0.9138    0.7536    0.9966    0.8261    0.4897    0.2551    0.7031    0.2696   15.5479   15.3047   38.6617   59.0178   15.4684    2.6112    0.4600    0.1848    0.3089    1.4164    0.1486    1.5678    0.4534    1.0769    0.4082    0.6693    1.3501    1.3779    0.2397    0.5309    1.0835    0.1988    1.7497    0.8761    2.4669    3.7735    0.1852    2.0049    1.8926    0.4897    0.1716   24.5692   52.9824   23.6798    2.4613    0.7268    0.9052    0.9184    2.1461    3.2504    2.5592    0.4701    0.7751    0.1299   43.3387   43.6989    5.6293    0.9027    0.2285    0.7651    0.9710    2.5743    2.5314    1.0250    0.5931    0.4779    0.1987    1.1296    2.4663    0.1482    0.3540   10.8254   18.9384    6.9893    2.5497    0.9191    0.7696    0.2815    0.8776    1.5641    0.8658    0.5190    1.4253    0.1582    2.3857    0.2170    1.0275    1.3075    0.4332    0.9517    0.9760    0.6412    0.2401    3.2072    2.0572    3.5485   49.1692   56.2148   13.3726    0.2985    1.6213    0.8056    0.9215    0.2296    0.7948    0.1711    1.0435    0.7248    1.0573    0.7018    0.4461    0.6351    0.1712    0.1220    0.8497    0.6124    3.0896    3.0315    2.7034    0.7005    0.5733    4.2601    2.0835    4.6959    4.6664    0.8142    3.0912    3.7012    2.1484    0.2718    6.9363    8.1365    5.7538    2.7637    5.9932    2.9869    5.4791    7.3112   34.1085   29.4706  101.2335   66.5634   19.2596    0.5331    2.4087    1.1253    0.5604   10.6444    6.6425   11.4832  118.5447   94.5447   54.5129    0.4179    1.3111    0.6019    4.5231   10.5781    5.7964   30.3947  198.2271  336.4221  387.8374    4.3948    1.2012    0.2840    1.2967    0.6841    0.7698    0.6992    0.9044    0.1992    0.4794    0.4203    0.2556    0.9895    0.2512    0.6975    0.1596    0.2560    0.3940    0.4488    0.6522",
      "2", "CID ETD", 828.69,
      true, 2, 4, 0.9, "2", false },

    // 20090326_Yest_LysC_1x6ETDsa_01.03022.03022.3.dta
    { "174.1554 179.0752 188.1227 189.3757 199.0582 204.2504 205.0559 218.2308 230.1408 231.3153 244.1837 246.2132 259.3337 260.2889 261.8342 277.0529 278.3394 302.2882 303.2218 304.1876 305.0680 319.1481 321.0120 322.0627 325.8947 332.1807 333.9844 337.2158 338.3555 339.0264 343.1706 344.0677 359.2288 360.6847 367.9915 378.2617 379.3112 389.2911 405.3777 406.0213 407.3484 416.2676 431.4978 432.1053 432.7890 433.4138 434.3278 437.5497 440.1097 450.2363 451.2313 452.2657 471.0859 472.1441 473.2143 474.2384 476.4431 481.1189 487.2791 488.3115 489.4159 495.2060 496.0844 500.8728 502.1552 506.3493 508.9581 514.4097 515.2046 515.8235 516.4713 517.3395 518.2429 519.0676 519.7510 520.9240 522.0898 522.8770 525.3414 526.3989 527.1591 527.9026 531.2535 533.2774 534.0975 535.9919 538.3132 539.1142 541.3626 542.7073 549.4869 550.2532 553.2217 565.2968 566.1797 566.8476 568.3304 569.6433 570.2741 571.2880 572.3028 573.0214 586.6914 589.3284 597.0137 598.1807 600.0079 604.0874 610.3766 612.4822 613.3114 619.8691 622.3531 625.0349 625.8797 629.3589 630.4100 632.1529 634.1899 635.3845 640.3384 645.5893 648.1205 652.3812 655.3437 656.3964 657.2783 658.4047 659.1335 669.4590 670.4255 671.4442 673.9225 676.9805 678.3733 679.5092 682.0251 683.4218 684.3719 685.2919 686.7169 687.3264 697.1274 698.0479 700.1506 703.7377 704.9965 707.8172 708.5850 709.8153 711.1500 712.8489 713.4771 714.4347 726.3566 727.7513 728.9944 733.5585 735.3016 736.4961 737.6189 738.2994 739.2595 740.4065 741.9040 743.1981 744.1530 745.2916 746.5226 748.9809 750.2281 750.9960 752.3406 752.9756 754.3780 755.5276 756.3692 758.2634 759.0110 759.7686 761.7732 762.9434  764.4741 765.3917 766.6286 767.4312 770.1957 771.1882 772.7053 773.5267 774.7447 776.6511 777.8210 778.7165 779.6204 780.7280 782.1320 782.9822 784.2060 785.6016 786.4901 787.9463 790.5060 792.1089 798.3909 800.2761 801.1783 807.3977 808.4180 809.3495 811.3911 815.3892 816.1534 823.2515 828.4472 835.0657 841.3511 845.4922 847.7346 857.4641 858.6213 861.3310 862.4926 863.5303 867.1469 868.1398 869.3214 871.2722 873.0120 876.2525 877.3261 883.4003 884.2711 886.6406 892.3677 893.9593 897.2706 898.3701 899.4016 900.1539 901.5656 902.4991 903.3836 904.6151 913.5021 915.0329 916.4500 917.8973 924.1539 925.5574 932.6861 933.3383 934.5273 935.4116 937.0532 937.8554 938.4962 940.7910 943.3398 948.5141 949.4937 950.4376 951.5229 957.2259 962.9388 963.7307 966.1238 968.4544 969.2480 969.8521 970.5481 971.1883 972.3713 973.9487 975.5293 976.4604 977.3992 978.0324 980.2961 981.3935 982.3759 984.0764 985.4116 987.2023 988.3969 992.0993 996.1435 997.1547 998.5941 999.2305 1000.3836 1001.2307 1002.1909 1003.0289 1004.7115 1009.8257 1010.6499 1011.4964 1012.4798 1013.3533 1014.4931 1015.3984 1016.2422 1018.9663 1020.0115 1021.2588 1022.4805 1023.6997 1026.5995 1027.5355 1028.6578 1030.4266 1031.3927 1032.4086 1033.5220 1036.5186 1037.5435 1038.3613 1040.8837 1043.6375 1047.5848 1049.6929 1053.3673 1056.1361 1057.4950 1058.4175 1059.7382 1061.5444 1062.6033 1069.4982 1070.5929 1074.5464 1075.5112 1076.5323 1094.0470 1094.7539 1095.4060 1096.3177 1097.1804 1098.6792 1106.3046 1111.6324 1112.7260 1113.5006 1114.6757 1115.6527 1116.7783 1120.6261 1127.2771 1128.2255 1130.9427 1142.5581 1149.3495 1150.4750 1156.3373 1158.3846 1159.3934 1166.5638 1167.3893 1168.3226 1179.4652 1181.1132 1182.8070 1184.4824 1185.6229 1186.8967 1188.7745 1198.0908 1202.0013 1202.7391 1203.6868 1204.6486 1207.7235 1208.3269 1209.3125 1210.6682 1222.6799 1224.0740 1224.9460 1225.7700 1226.5751 1232.3053 1233.6705 1240.3016 1242.0094 1242.6102 1243.5892 1244.6050 1245.5419 1255.7289 1257.0131 1258.3273 1263.4673 1265.6155 1270.8903 1271.6210 1272.5895 1274.6886 1277.0499 1278.8348 1281.0328 1283.3326 1298.9988 1300.7955 1312.8987 1313.9017 1315.6721 1316.6818 1317.7777 1327.3593 1328.6724 1334.4410 1350.9503 1354.4154 1355.7457 1356.7155 1357.5637 1358.3634 1359.5591 1360.3795 1370.4689 1371.5935 1372.5089 1373.5098 1374.3971 1382.8084 1384.6149 1385.2179 1385.9840 1387.7052 1388.7964 1398.0221 1398.7335 1399.5337 1400.5376 1401.2437 1403.0814 1405.5397 1406.5472 1414.0992 1414.7181 1415.6832 1416.5590 1417.3118 1439.9700 1440.9714 1441.7449 1444.8362 1446.7238 1452.5648 1455.9109 1457.6229 1458.2463 1458.9991 1464.7961 1467.6711 1468.9851 1469.7860 1471.3777 1472.6039 1473.4760 1474.5516 1475.6071 1476.4310 1481.3157 1484.0294 1484.7820 1485.6501 1486.7737 1487.8479 1488.8738 1494.5118 1495.4846 1496.9874 1498.2531 1499.5580 1500.6381 1501.6008 1502.8416 1503.5187 1510.0199 1510.7570 1511.7748 1512.9452 1513.8280 1514.9672 1516.0631 1516.9602 1518.3619 1519.9796 1526.1310 1527.5311 1528.7311 1529.7216 1530.7430 1531.9368 1540.5118 1541.5272 1542.5696 1543.8453 1544.6794 1545.3503 1545.9773 1546.7645 1547.4601 1548.0663 1549.9073 1559.0773",
      " 30.7727   2.0161   1.6459   4.3806   1.1425   1.1789   0.8878   2.6694  66.7722   5.0787   2.6722   3.9851   1.1518   1.2032   2.4317   4.6574   1.6702   4.2097 117.4776  12.4575   1.2703   3.0528  27.1081   3.9674   3.6453   2.3061   2.1601  59.6653  11.9821   3.2825  75.5054   8.0337   8.7827   1.9702   1.3510  41.6213   2.1465   6.2814   3.5642  12.8342   4.9883   1.0159   8.8457  50.0453   1.3800   2.6002   5.8939   1.5987   0.9524 305.1002  51.2715  13.4841 484.2204 127.2207  19.2103   1.3571   4.1935   5.1572 257.3338  93.9001  17.8799   9.0634   2.8279   1.2680   2.7964   2.5447   2.0365 158.8841 299.0704 481.5560 115.0965   5.6347   6.5330   9.8330   1.4780   8.6310   6.5452   0.5629  35.5242   9.4683   3.4381   3.4979   3.0431  19.5555   9.6309   5.5580   6.4886   5.0219   4.3135   3.9848  11.5080   0.9534   4.7663  98.4411  20.8500   0.3967   2.5985  19.5768 238.1391  75.5913   5.2974   1.8430   4.0059   4.8391   8.7758   5.8762   5.8455   2.2052   2.9844  21.6526  13.9778   3.7561   2.6716  11.6179   7.8719   4.0101   1.2784  25.2776   1.1432   8.4523   3.6400   3.2014   2.6722   2.8581   3.0325   4.6598   1.9732   9.7426   2.5893  49.3025  26.8525   6.6824   2.0560   4.3248  45.3667  19.9832   1.5250  10.0611   2.7264   3.5644  18.5327   6.7304   2.1628   2.2221   5.9178   1.7232   0.8251 270.8201 295.9683   3.7999   8.2268   4.1318   9.0103   6.8250   3.9068   2.2854   8.9261   1.9069   1.4708  39.8080  44.8511 159.0369  70.2632  11.9191   3.6196 181.3270  26.1012  43.0580  11.7856   8.9266 130.4149 133.3700  61.0688   1.7780  76.7575  51.7504  23.3879  39.1915  31.7174   6.8720   4.4801  96.5574 1000.0000 312.7447  24.9426   6.9890  75.4365  43.6599 953.2108 402.4480  19.2552   5.4729   4.6419   4.7435   3.5571  12.3355   2.4764  12.4966  16.7230   2.0993   1.5892   4.2731   4.1348   1.2068   2.8595  18.4958   2.6202 269.7065  67.5150  31.2379   1.7309   2.6001   1.1432   2.4499   6.6494   2.0343   6.2935   1.6530   3.9444  12.9029   4.0644   2.9379   3.9357   3.5597  60.0388  12.8828   8.2587   2.3077   5.8863  58.8523  39.2800   4.3824   1.4790   2.0320   7.5404   8.8032   2.6946  18.7148  12.3221   6.6297 118.5322  33.3893   1.4592   3.6276   8.6612   8.8691   3.9989  25.9733   0.7611   4.5409   2.1792  11.1640   9.2868   5.3895   1.2807   3.0070   3.0537  10.7582   0.9533   8.0324  14.9159  10.3385   5.3296   3.6935  11.8476   6.1669   5.0133  32.8007   1.6640   3.4646   1.2114   7.0836  22.4836  15.3114 222.6936 130.4593  56.0358   2.1928  93.1751  51.1742  22.4876   2.4862   2.8197  10.0822   2.3525   7.5709   3.0341   6.7221   8.6199   2.8012    5.3345    4.4545   20.9498    7.7847    3.4295   11.7081    7.0559   12.4439   50.6903   13.2968   61.2776   29.6710    3.4264    0.5693   54.4382   25.1104    1.1427    4.3269    9.3992   24.5318   39.7347  473.4945  213.1830   67.0806    8.3608    7.5046    4.5102    2.8610    2.1589    1.2126    1.5994    3.1353    4.7299    1.4756    7.6072    3.3712    3.5634    0.7592    1.0165    5.8512    2.4799  503.8126  292.1578   59.3837   11.2216   21.5553  313.6450  154.7770   60.7759    1.1587    2.4143   27.2744   18.1139   43.7092   42.0509   10.8123    1.5881    3.6201    1.6662    1.3973    2.1608    0.4417    1.1521    5.8088    2.7345   13.5170    4.7015    9.1603    6.5711    9.4365    7.4895    5.1638    3.7094    9.7980    3.5850    1.9021   12.9247    0.6993   21.0758   59.9580   74.1291   12.3582    7.6579   85.4680   50.8627   18.1032    1.7348    7.7953   14.2360    6.4149    5.0288   11.9227    0.8295    7.1470   12.9598   76.9328   65.2442   23.9005    2.3542    4.8173    3.6250    3.2443   11.6651    6.1660   11.6351    8.2687   24.0939    2.5337    0.9496    1.6613    5.7392    3.6960   10.9175    7.2121    7.2563    6.2668   61.5208   76.7468    8.1182    2.2901    2.0347    5.9157    6.3640   24.8738   27.1733   25.1642   12.6795   12.6037    9.9568    4.0782   97.9387  238.7174  126.1478   80.0853   18.7889    3.4394    3.3667    4.1946    1.1294    9.1245    4.7423   22.6290   43.2759   27.8529   12.3748    9.5206    3.0884    2.8751    5.0830   85.2423  221.9227  323.0743  101.9702   47.9052    0.9774    3.6228    5.1394    4.5138    0.8336    2.7239    6.8520   16.2821    1.1911    2.1605    1.6688    2.2945   11.0464   11.9652   23.7036   23.3097   23.9616   28.4958    2.7745    3.5857    1.5495   27.6622   48.3842   93.3411   86.9475   38.9364   13.8767    7.2797    5.9672    2.1445   18.0167   45.0836  137.9359   62.3037   39.9054    9.9711   13.6787   23.5004   73.1363   26.9701   34.7489    1.5825   21.2115   13.4220    4.8053    1.9705    8.0361  205.9319  775.8333  524.7843  204.9251   12.5449    2.0510    2.0342   10.6749  162.8668  214.5419  183.2698  124.4973  114.2632   13.9038   50.7772    1.5754    2.8177",
      "3", "CID ETD", 515.67,
      true, 2, 4, 0.9, "3", false },

    // 20090326_Yest_LysC_1x6ETDsa_01.03090.03090.4.dta
    { "243.7879 248.2352 256.2038 260.0671 260.8535 261.9995 275.7320 290.2630 303.2579 324.1052 331.1282 332.3542 340.1091 349.3828 392.1447 402.2183 403.2556 404.0996 405.1668 406.0318 418.1470 418.9594 420.7508 431.1273 432.2661 470.3546 475.0428 485.4223 487.2188 517.3171 518.4039 519.0720 520.1177 521.3829 530.0964 531.0302 532.0327 533.2005 534.2474 536.0812 543.2462 547.2946 548.3525 559.3591 562.9785 566.3115 591.8767 600.1329 602.4547 604.0406 625.2937 626.4370 634.4691 635.4151 645.2241 646.2679 650.5569 654.5934 664.5505 665.5209 666.2430 666.9123 667.6990 668.6552 669.8634 673.0308 675.1030 679.1826 682.6284 687.2708 688.2209 689.3478 700.4522 701.2593 702.0752 715.2581 716.0817 717.2546 723.1075 728.2315 747.5339 748.2724 750.2634 755.8646 762.2759 769.3102 770.0140 772.0799 774.6414 778.6964 779.3589 800.2402 801.3123 802.7204 803.6292 806.2991 809.4399 815.0102 816.4662 817.3740 818.2971 819.5718 823.5381 824.3093 827.9058 829.3743 831.2183 837.5764 842.1263 843.3616 844.1660 844.8753 845.6047 846.7016 848.9698 852.7267 856.3286 859.3588 860.1613 863.6732 864.4836 868.1258 869.7119 871.5098 872.4384 873.0388 873.9880 875.3660 876.2717 877.9868 879.1215 881.7221 882.9334 884.0421 886.0036  887.2063 888.0679 889.4487 890.5137 891.7407 893.3141 896.2202 897.0464 900.6588 902.3458 903.5149 904.9387 905.5971 908.9622 909.7800 910.5516 912.8056 915.3342 918.0927 922.3116 924.2397 926.3204 928.4907 929.4748 931.4737 932.1279 932.9814 935.2791 936.5615 939.3074 941.0369 944.0950 944.9709 950.6240 953.5076 955.1049 957.3259 958.0078 958.6191 959.3874 960.2178 961.4774 968.2621 969.0814 970.3516 971.2019 972.6970 973.8386 974.9577 975.7839 976.6500 977.2662 978.0444 979.8441 980.4866 981.5150 983.2332 983.9152 984.5562 985.8335 987.9278 989.2733 990.5781 991.6534 992.2967 994.0283 995.1682 996.0893 996.7576 997.6959 998.7651 1000.0078 1001.1967 1002.4177 1003.2166 1004.1922 1005.6064 1006.3274 1008.5909 1014.5851 1015.5349 1016.4651 1017.0720 1020.1202 1021.8987 1023.5038 1028.3291 1037.6019 1043.4714 1044.4396 1046.3817 1050.5145 1053.8496 1054.4580 1055.6554 1060.5215 1064.1461 1065.7567 1066.6307 1067.9528 1068.8159 1071.6482 1072.4299 1073.6187 1075.8084 1078.6600 1080.1945 1082.4708 1083.6788 1085.0514 1085.7885 1086.6343 1087.7355 1088.8904 1090.5883 1091.8010 1093.0393 1094.9630 1098.5443 1100.2257 1103.1823 1105.4001 1106.2609 1108.2299 1109.0792 1111.3062 1112.1854 1115.1538 1119.3478 1121.1239 1122.1865 1126.6741 1128.5758 1129.3092 1130.2064 1131.1403 1132.1752 1133.1498 1134.8640 1136.5469 1137.8358 1139.6202 1140.4686 1142.4139 1144.0797 1145.3140 1146.2294 1146.8634 1148.9181 1149.6946 1151.3239 1152.7325 1156.8146 1157.7513 1158.5095 1159.2571 1161.7758 1162.9086 1165.7037 1166.8081 1170.6285 1173.0693 1174.6534 1175.6210 1176.5850 1178.1290 1179.8569 1180.7069 1182.9113 1183.5262 1184.5698 1187.1746 1188.8331 1194.1100 1195.0675 1196.4001 1197.1163 1198.5291 1199.5568 1200.5071 1201.4010 1202.3007 1203.1454 1203.7914 1204.9648 1205.6096 1207.7416 1208.7045 1209.6604 1211.3644 1212.3815 1214.2264 1215.3739 1216.5845 1218.3002 1236.0392 1237.8418 1239.5880 1240.3123 1243.9308 1244.8210 1245.7513 1246.8600 1248.2845 1252.9122 1255.3536 1256.6062 1257.5916 1258.2028 1259.1376 1259.7753 1261.5348 1262.6210 1263.3873 1264.7200 1265.6898 1266.5570 1267.4847 1268.2734 1268.9828 1269.6935 1270.5500 1271.6449 1272.3439 1273.3456 1274.0927 1275.4904 1276.1138 1276.8590 1281.9656 1283.2112 1284.5497 1285.7233 1286.5983 1287.5159 1288.8540 1290.6437 1292.5269 1293.1344 1294.3699 1295.1715 1296.9884 1298.2716 1299.4836 1300.2164 1301.3239 1302.3589 1303.3157 1304.2849 1304.9520 1306.1122 1306.9321 1308.2411 1309.1156 1310.1351 1311.0634 1312.0157 1312.7286 1313.4656 1314.6344 1315.7052 1316.8828 1317.8702 1318.6637 1319.6012 1320.5795 1321.5009 1322.3763 1323.2101 1324.1195 1325.1630 1326.0287 1327.5521 1328.6740 1329.5428 1330.4265 1331.3120 1332.1520 1332.8768 1333.6855 1334.6652 1336.0970 1337.1306 1338.5605 1341.4021 1342.0756 1342.7965 1345.4364 1347.1927 1348.1262 1349.5397 1352.6696 1354.5559 1356.7841 1368.4066 1369.6239 1373.8602 1374.4603 1375.4882 1380.6715 1383.9158 1386.2748 1390.6932 1392.5404 1393.6093 1394.6329 1395.7740 1397.7992 1398.6327 1399.6852 1400.6708 1401.7643 1402.6301 1403.5635 1405.9886 1415.6130 1421.7067 1425.9807 1433.7421 1436.1149 1453.0121 1458.2045 1458.8499 1463.9680 1467.4247 1475.7245 1476.9900 1481.4274 1487.9906 1490.0920 1497.4567 1507.5260 1508.8215 1510.6302 1514.5507 1521.4270 1529.1095 1532.1146 1532.7543 1533.8010 1535.5734 1536.7006 1538.9001 1539.7325 1541.3253 1545.6788 1555.6199 1556.8606 1557.7072 1558.4188 1562.3300 1564.2053 1565.2517 1566.3585 1568.6472 1570.0729 1570.9081 1574.8679 1576.5817 1577.8307 1578.7906 1579.9139 1580.7203 1585.5433 1588.2838 1588.9833 1590.9456 1594.6622 1595.4481 1596.4623 1600.3292 1601.5031 1602.5115 1608.5905 1611.9725 1613.0826 1615.1105 1616.5769 1629.3666 1634.1407 1635.8585 1641.7272 1642.5457 1648.6711 1651.9296 1652.9420 1655.5812 1656.7283 1659.8629 1660.8605 1661.6638 1662.3485 1663.2076 1664.7709 1665.5188 1671.5364 1672.9659 1673.9637 1674.6150 1675.7322 1678.1320 1680.3798 1683.6414 1688.7131 1689.8280 1693.2284 1694.5199 1695.3002 1695.9480 1696.5748 1703.6605 1704.6127 1705.6389 1706.7621 1707.8983 1708.5499 1710.3943 1711.4486 1720.8822 1727.2303 1727.8524 1737.8192 1739.3956 1746.6827 1758.8782 1764.5946 1775.1521 1776.0402 1784.8909 1786.1140 1786.9799 1787.7216 1788.8445 1800.1788 1800.9778 1801.8905 1803.6886 1806.7655 1807.7987 1808.5040 1809.6721 1817.1617 1818.5787 1819.2245 1820.5875 1821.6638 1823.6163 1824.9684 1830.0878 1832.4257 1844.2258 1846.2727 1848.7760 1850.9319 1852.9744 1857.5016 1860.8910 1861.9016 1862.9977 1864.8854 1865.8596 1867.2062 1867.9637 1869.0101 1877.6544 1882.0580 1901.7526 1903.9530 1904.9755 1907.1428 1908.0741 1909.6555 1910.3221 1912.5148 1914.8358 1916.1473 1917.4291 1918.5476 1931.0032 1931.8480 1933.0316 1934.2292 1936.1769 1937.0603 1938.1121 1939.8079 1940.8578 1946.8407 1950.1467 1950.9208 1952.0140 1953.0594 1955.9314 1957.8478 1961.9181 1963.7831 1964.6874 1965.3129 1966.2650 1967.3536 1968.3961 1969.2345 1971.4152 1974.0541 1975.0831 1976.0770 1977.0219 1978.4110 1979.8290 1980.8768 1981.8845 1983.1393 1990.7211 1992.0797 1992.8196 1994.0579 1995.0470 1995.8445 1997.1169 1998.4944 1999.1747 1999.9548",
      "  2.5076  18.9266   3.2094  49.4415  11.1240   3.1239   4.4428   2.3743   8.4930   1.5319 116.0403   9.2358   5.1740   1.9491   8.2785 250.2189  90.3265   9.7607  10.4975   3.3480  29.8301   6.2930   5.4928  33.4564   3.1056   3.4949   7.2611  11.4222   8.1265   3.9918  11.0943 111.8997  80.7320  12.8881   1.6924 163.0821  59.2097  11.5111  13.1041  11.4829  43.7376  18.5996  24.1879   9.4563   6.9902   5.1199   1.6691   4.4790   8.0981   2.0882   5.3213  19.1654  47.6864  38.9337   8.3869   9.7502   8.0166   6.9817 510.2017 712.8604 490.7755 237.6616  47.8276  21.1665   5.7600   3.7721   8.7153   5.3138   9.1014  39.9138  31.6539  17.3705 268.0664  14.8031   8.1302   4.1933   4.0638  10.3567   1.5317   3.7257  46.7505  42.0271   2.6775  16.5575   3.2252  18.1619  18.9057  18.8581   6.5679   8.4362   4.1945  60.4761  52.8047   9.8821  11.6152   4.3330   7.1869   2.6757  43.8517  21.5576   7.9495   1.9516   5.8711   6.4767  16.3193  10.0589   2.5057   5.7654   6.8518  14.8569  77.5502  58.4654   8.8919   3.6298  88.5366   7.0855  53.4351   1.3915  21.5080   8.2630   4.0534  25.5232  14.2767   1.9576 197.6710 250.7894  78.0244   2.9596  50.7061 170.0106   8.1694 272.8869 253.9160   4.1857  87.1401 1000.0000 526.3197  64.7075  39.7636  17.3089  10.2896   7.5661   5.1657  26.6056   9.5840  10.7909  43.7524   3.4637  50.6477  24.6092   3.4639  29.0624   2.2321  23.3793   7.1294  15.3133   6.0014   4.9342   1.9994  33.9224  38.0253  32.7196  14.3473  10.4929   4.2926  27.0133  17.0891  23.0027   6.4361   8.6253   9.2514 171.9624  47.5352 125.9794  10.9860   1.5289  11.1903  60.2448  34.3771  22.3051   4.1909  10.1036  24.5529  14.4163  20.5782   5.9393  10.2954   9.0187   7.4659   2.3401  18.8896  33.2652  35.2743  10.2916   8.9569 128.8345 195.4695 162.2259  11.3918   7.8248  17.0745  66.5099 148.9855 181.8809 570.1615 436.6480   65.4046   25.1590    6.1581   10.6282    1.6718   16.7032    4.1507    3.4973  318.0643   97.9090   18.1858   12.7225   10.9274    1.3910    5.0489    6.4388   18.6934   45.5914   15.3617    5.3042   13.4612   39.2124   48.8572    4.8387    9.8630    8.3484  205.8090   49.0299   29.8948    6.5187  193.9734  189.5334   72.9852    1.1134    8.7717   32.6508   34.8928   24.1992   61.1739  150.9486   97.6246   13.0581   16.9272    5.7557    5.0259   14.8936   27.8568   13.4189    2.7282   11.4069   29.6101   16.0605   82.8358   13.4283   19.5077    1.8127    8.5283    6.3117    7.5451    9.5664    1.6779  183.3642   98.0382  415.0550  146.1843    7.2957    7.4197    6.4443   92.8868   27.0742    3.0738   29.7548   13.9411   10.3046   19.0847    2.6452    2.3885   12.5378    3.9204   18.8801   10.4848   67.6086  118.3790   23.9461    9.1204   18.1228    8.4828  234.8195   89.1072    5.8637    6.2577   15.8360    8.9748    4.0529   14.9671   20.9793    8.3787   39.5979   68.9838   60.0038    3.7880   27.3046   49.5608    2.2327   11.6102    5.1699    6.6042   38.9037   79.8768  362.8020  135.8894   28.7132   17.5083    3.9135   15.2511  223.2500  182.6142    1.9003    3.6386   17.5666    6.2038   44.5620   10.5063    6.2867   26.5601   18.6813    9.6625    6.6326   64.7087  107.9688   29.7563   16.6371   32.8420    8.0484   29.7604    8.0878   62.2472   34.1126   13.3664   14.8972  303.8790   64.7862   50.1557   28.0513  150.3060  244.6863   75.4416   56.2096   43.2100   84.1846   79.0920  140.3058  249.1291  129.4778  108.6126   11.4286   18.1975    4.0765   37.5395   32.7084    5.1724   61.5775    2.5167  106.5414   48.5900    5.5421   38.9133    4.8854  162.9593   91.7801   21.3603    7.4532   51.1259   87.4694  324.2788  481.0876  242.3429   89.6649    8.6949   22.5628   42.8109  395.4991  906.8211  351.9747   45.2140   42.2932  131.3575  143.6703  279.8663  101.1041  108.7031   64.1748   14.4714   14.8446   59.5402  132.8098  542.4203  605.4359  138.1729   59.9986    7.7087   46.8506  184.1397  169.4386  209.7629  580.2321  303.5908  383.0007   38.1204    5.1781   14.4704    2.6467   10.4877    3.2835    4.7570    6.5749    6.1762    2.6568   11.5992   14.8157   13.4205   20.5046    3.3428   19.7544    6.5771    7.0176    7.4121    6.2113   10.0306    6.3449    2.6622    4.4946   66.9666  101.4519    5.9236    9.2178    8.6639   11.3079   32.9729  149.7118  115.2870   89.5708    1.9438    6.1423   10.6356   11.0249   11.3324    5.1689   12.6315   12.0286   13.7912   12.4758    5.8607   14.9925   10.4446    1.3918    6.5768   10.0995    2.0937    7.2864    2.0943    2.8438    5.2996   16.6985    3.6360    9.9587    2.7950    5.1792    6.5680    9.0599    2.3702    7.8762    2.4463   17.1443    4.7590   19.5622   53.1501   12.3967    5.1623    2.7934   12.9506    8.9373   13.9133    4.1008    5.7210    2.5134   15.5638  171.8266   63.7603   41.9141   40.0678    3.5398    4.3739   10.3743    3.6293    7.4224    7.1716    1.8108   22.0531    2.7937    6.8615    8.5185   10.7562    6.4110   25.5629   19.1717    7.2008    1.5301    6.7089    6.5802   10.8492    7.2661    6.8547   26.7061   12.6254   25.8981    8.6692   20.8036   27.5724   24.8403   22.5842   28.0092   31.8056   12.7479    2.9316    6.0303    4.4677   12.7609   15.2388    4.1883    4.8802    4.4712    4.8785    6.6209   25.0085   17.3346   15.8456   12.2698   13.7656    3.5826  121.5698   90.2465   59.1607   22.9261    3.0484    3.2337   22.3640    7.2639   27.9444    1.2528    1.0145   10.4351    7.6937    4.6031    6.1586   14.8440    4.4704    1.3903   61.6618   36.3846   17.9140   23.4125    9.6047   16.0947   22.8557   10.3551    8.4454    3.0415    3.8349   49.2699   18.4090   43.2297   15.9098   10.8577   13.5819    8.0912   21.4863    8.6807    2.6542   15.9513   11.3206   10.4575    8.6845    2.5100    2.8714   63.3186   32.1922   34.7627   43.9835   25.1937    7.8191    4.0462    7.5328   10.7146    6.7111    4.1300    6.4390    6.8518   17.0808   19.2425    5.7076    6.6691    6.9303   31.3079   31.3799   10.9830   16.0658   11.8404    3.4938    6.9670   21.4177   54.3157    5.1703    5.7322   14.2698    3.2217   11.2535   26.3494   33.8623   23.5860   10.0005    7.4197    4.1937   22.3184   17.1360   12.8346    7.0810   34.6404   14.0404   20.1090   24.0768   11.7454    7.1177   23.8897   68.5088   74.6106   83.0404   36.4579   22.8864   29.4717    8.4930   38.9218   67.0819   72.2918  129.4568  221.4474  207.6776  270.2602   93.7133    1.0129    4.3882",
      "4", "CID ETD", 665.96,
      true, 2, 4, 0.9, "2 3 4 6", false },

    // 20090326_Yest_LysC_1x6ETDsa_02.02321.02321.2.dta
    { "943.4871 951.7823 1064.8960 1066.7097 1067.6230 1068.8214 1069.5854 1070.6409 1071.4691 1073.5938 1074.4456 1075.2776 1076.1846 1088.1877 1089.7196 1090.7126 1156.2903 1169.5214 1174.4698 1186.7987 1200.3717 1201.4275 1202.6658 1217.5098 1256.7128 1287.5265 1288.6165 1289.6875 1291.0149 1313.4830 1316.3944 1317.5052 1318.4432 1358.7137 1371.3746 1383.7867 1386.4844 1387.4824 1402.4170 1403.4615 1404.3643 1405.4310 1426.9939 1427.6129 1428.4337 1429.6627 1430.7828 1431.5824 1432.5782 1457.5984 1473.4108 1474.6364 1475.4916 1499.4441 1500.5341 1501.4215 1502.5790 1503.3033 1504.8606 1517.4626 1518.6984 1519.6815 1520.9386 1524.8741 1530.2655 1543.8617 1569.6876 1592.7599 1597.5741 1599.8959 1600.7618 1627.9799 1628.6863 1629.5306 1630.5973 1631.6581 1632.6907 1634.9969 1643.8317 1664.1378 1685.4745 1686.7106 1687.7426 1688.4935 1701.6852 1702.4674 1703.5613 1704.6144 1705.5582 1743.1940 1756.2999 1778.0231 1783.6737 1785.4012 1786.9049 1787.7581 1788.8737 1789.8173 1803.2833 1804.8668 1817.6497 1818.5804 1819.7025 1820.7083 1830.7328 1831.6576 1871.6897 1885.7474 1886.8210 1887.6454 1888.6915 1889.7943 1890.6288 1892.1139 1906.2728 1912.0148 1956.3971 1958.4529 1959.4286 1974.0027 1986.7545 1999.7296",
      "  0.5446   1.8406   12.4060 1000.0000  380.4662    8.4849    3.0875    6.6752    5.4134    1.9659    1.0996    1.3749    2.7091    2.3103    1.1925    0.8041    1.2601    0.6269    0.9139    0.3304    7.1713    8.2900    2.3422    3.2041    0.3247    7.2178    7.8656    3.9263    1.4840    0.7850    3.8859    0.6056    1.1658    0.8953    1.6970    0.7994    0.7038    0.2345   11.9077   17.7421    6.6932    3.8471    0.4160    0.7663    6.1038   22.5824    5.8907    6.3597    1.2397    1.1754    0.6179    2.7984    0.5540   12.9965   24.0521    7.2658    6.2846    0.7290    2.5242    9.1656   11.7215    4.6917    0.5513    1.3358    0.4055    0.4577    1.9369    0.4230    1.0108    3.9809    1.6382    2.3505   13.1796   15.7449   11.9472   11.9523    3.0888    0.5268    0.1617    0.4460    8.2503   11.0468    5.8862    0.8511    1.7488   20.8060   17.6306    6.7047    1.5912    0.6065    3.5587    3.5022    0.2711    1.0538   27.4272   19.9590    9.9348    2.1255    1.5608    0.7332   24.3087   22.5312   14.8691    5.0499    0.6407    1.4617    0.6787   15.8553   14.5014    6.4299    5.6243    5.2153    1.6044    0.4411    1.4996    0.1980    1.3020    0.4044    1.2132    3.6717    0.5728    0.7156",
      "2", "CID ETD", 1066.72,
      true, 2, 4, 0.9, "2 3 4 6", false },

    // MakeMS2 behavior
    { "126.9657 129.2652 135.9172 141.9427 143.1286 147.1841 149.0127 150.2528 158.0455 163.0535 167.0655 177.1013 192.9888 203.2619 204.075 206.9989 208.9082 209.8115 211.9883 221.0683 225.0675 226.106 227.2834 228.0121 234.9963 237.0106 243.0448 247.1461 249.2151 253.3426 256.2594 257.1935 259.2448 261.0975 263.2516 268.0759 269.1938 270.3783 271.2178 277.2914 280.9584 283.1599 285.0426 288.2384 289.4238 291.1711 293.0654 294.0734 295.0374 297.267 299.9413 301.1416 301.9647 303.0014 304.1252 305.7378 308.8322 309.4466 311.3066 313.0665 313.9782 315.1105 317.1343 318.1869 318.9812 319.9492 321.1036 322.9846 324.2249 325.0501 326.029 327.1547 331.18 333.1406 334.1884 337.0988 341.0733 342.0545 343.0352 344.0117 345.1031 346.165 347.1043 349.1555 350.0966 353.1221 357.0199 357.921 359.0819 360.0584 361.1577 362.3141 363.2353 364.0934 365.3022 366.2242 367.0972 368.0494 369.3801 370.5183 371.2664 372.2919 373.0294 373.9725 375.0907 376.1542 377.2523 379.4164 382.4076 383.3729 385.1636 386.0052 387.1495 389.0345 390.0276 390.9678 391.8777 393.0433 393.9705 400.3051 401.1996 402.1906 402.9228 409.0102 410.9843 412.0966 413.1009 415.1218 417.7721 418.8526 420.6482 427.3444 428.114 429.0626 429.9189 446.916 474.9332 500.6554 529.2106 554.9348 607.3796",
      "11.65686 5.973924 6.336542 9.801903 16.33596 43.81474 351.6202 17.81902 13.21312 43.67112 43.36504 31.75116 9.61234 29.10818 5.006023 45.31382 1384.851 49.05672 22.97882 48.93507 168.9132 44.21313 11.85654 11.1582 18.42228 14.85892 22.64454 14.755 30.44192 6.76189 6.028577 15.03836 39.42986 9.79327 13.60227 6.030219 17.97449 119.7174 136.11 9.588016 48.16204 6.028051 26.51938 50.67508 5.54958 31.8689 56.91853 18.76982 86.78511 47.78939 21.39392 44.05758 43.10071 12.93362 15.29362 22.67178 9.169868 22.70414 36.61877 36.40493 38.88861 62.4144 18.80965 73.55255 8358.035 616.7208 59.05638 41.89165 18.19917 70.24268 11.38647 158.0707 318.9193 38.72861 9.516384 32.10003 17329.54 1931.043 1095.633 146.2284 60.64346 8.186502 95.90353 213.953 48.77734 68.7878 65.71793 25.57602 47967.95 5397.058 248.9106 130.2801 257.1089 159.5846 89.56769 122.899 215.5294 149.6132 187.0416 115.1558 154.1615 19.63517 111.0804 45.56202 52.11806 74.48772 33.20785 22.3274 52.99072 65.79727 281.6389 28.11442 58.28289 73.20836 76.31909 141.2777 9.116022 395.3716 99.41208 56.91262 69.17138 64.92921 7.586085 75.60397 907.0748 208.3159 189.4094 4.653344 22.02694 21.07499 79.46523 21.16875 9.565468 127.7872 46.66726 49.75043 14.75244 30.76107 17.54733 12.85513 22.61628",
      "1", "CID", 429.03,
      false, 2, 4, 0.2, "1", true },

    { "126.9657 129.2652 135.9172 141.9427 143.1286 147.1841 149.0127 150.2528 158.0455 163.0535 167.0655 177.1013 192.9888 203.2619 204.075 206.9989 208.9082 209.8115 211.9883 221.0683 225.0675 226.106 227.2834 228.0121 234.9963 237.0106 243.0448 247.1461 249.2151 253.3426 256.2594 257.1935 259.2448 261.0975 263.2516 268.0759 269.1938 270.3783 271.2178 277.2914 280.9584 283.1599 285.0426 288.2384 289.4238 291.1711 293.0654 294.0734 295.0374 297.267 299.9413 301.1416 301.9647 303.0014 304.1252 305.7378 308.8322 309.4466 311.3066 313.0665 313.9782 315.1105 317.1343 318.1869 318.9812 319.9492 321.1036 322.9846 324.2249 325.0501 326.029 327.1547 331.18 333.1406 334.1884 337.0988 341.0733 342.0545 343.0352 344.0117 345.1031 346.165 347.1043 349.1555 350.0966 353.1221 357.0199 357.921 359.0819 360.0584 361.1577 362.3141 363.2353 364.0934 365.3022 366.2242 367.0972 368.0494 369.3801 370.5183 371.2664 372.2919 373.0294 373.9725 375.0907 376.1542 377.2523 379.4164 382.4076 383.3729 385.1636 386.0052 387.1495 389.0345 390.0276 390.9678 391.8777 393.0433 393.9705 400.3051 401.1996 402.1906 402.9228 409.0102 410.9843 412.0966 413.1009 415.1218 417.7721 418.8526 420.6482 427.3444 428.114 429.0626 429.9189 446.916 474.9332 500.6554 529.2106 554.9348 607.3796",
      "11.65686 5.973924 6.336542 9.801903 16.33596 43.81474 351.6202 17.81902 13.21312 43.67112 43.36504 31.75116 9.61234 29.10818 5.006023 45.31382 1384.851 49.05672 22.97882 48.93507 168.9132 44.21313 11.85654 11.1582 18.42228 14.85892 22.64454 14.755 30.44192 6.76189 6.028577 15.03836 39.42986 9.79327 13.60227 6.030219 17.97449 119.7174 136.11 9.588016 48.16204 6.028051 26.51938 50.67508 5.54958 31.8689 56.91853 18.76982 86.78511 47.78939 21.39392 44.05758 43.10071 12.93362 15.29362 22.67178 9.169868 22.70414 36.61877 36.40493 38.88861 62.4144 18.80965 73.55255 8358.035 616.7208 59.05638 41.89165 18.19917 70.24268 11.38647 158.0707 318.9193 38.72861 9.516384 32.10003 17329.54 1931.043 1095.633 146.2284 60.64346 8.186502 95.90353 213.953 48.77734 68.7878 65.71793 25.57602 47967.95 5397.058 248.9106 130.2801 257.1089 159.5846 89.56769 122.899 215.5294 149.6132 187.0416 115.1558 154.1615 19.63517 111.0804 45.56202 52.11806 74.48772 33.20785 22.3274 52.99072 65.79727 281.6389 28.11442 58.28289 73.20836 76.31909 141.2777 9.116022 395.3716 99.41208 56.91262 69.17138 64.92921 7.586085 75.60397 907.0748 208.3159 189.4094 4.653344 22.02694 21.07499 79.46523 21.16875 9.565468 127.7872 46.66726 49.75043 14.75244 30.76107 17.54733 12.85513 22.61628",
      "2 3", "CID", 429.03,
      false, 2, 4, 0.2, "1 2 3", true },

    { "126.9657 129.2652 135.9172 141.9427 143.1286 147.1841 149.0127 150.2528 158.0455 163.0535 167.0655 177.1013 192.9888 203.2619 204.075 206.9989 208.9082 209.8115 211.9883 221.0683 225.0675 226.106 227.2834 228.0121 234.9963 237.0106 243.0448 247.1461 249.2151 253.3426 256.2594 257.1935 259.2448 261.0975 263.2516 268.0759 269.1938 270.3783 271.2178 277.2914 280.9584 283.1599 285.0426 288.2384 289.4238 291.1711 293.0654 294.0734 295.0374 297.267 299.9413 301.1416 301.9647 303.0014 304.1252 305.7378 308.8322 309.4466 311.3066 313.0665 313.9782 315.1105 317.1343 318.1869 318.9812 319.9492 321.1036 322.9846 324.2249 325.0501 326.029 327.1547 331.18 333.1406 334.1884 337.0988 341.0733 342.0545 343.0352 344.0117 345.1031 346.165 347.1043 349.1555 350.0966 353.1221 357.0199 357.921 359.0819 360.0584 361.1577 362.3141 363.2353 364.0934 365.3022 366.2242 367.0972 368.0494 369.3801 370.5183 371.2664 372.2919 373.0294 373.9725 375.0907 376.1542 377.2523 379.4164 382.4076 383.3729 385.1636 386.0052 387.1495 389.0345 390.0276 390.9678 391.8777 393.0433 393.9705 400.3051 401.1996 402.1906 402.9228 409.0102 410.9843 412.0966 413.1009 415.1218 417.7721 418.8526 420.6482 427.3444 428.114 429.0626 429.9189 446.916 474.9332 500.6554 529.2106 554.9348 607.3796",
      "11.65686 5.973924 6.336542 9.801903 16.33596 43.81474 351.6202 17.81902 13.21312 43.67112 43.36504 31.75116 9.61234 29.10818 5.006023 45.31382 1384.851 49.05672 22.97882 48.93507 168.9132 44.21313 11.85654 11.1582 18.42228 14.85892 22.64454 14.755 30.44192 6.76189 6.028577 15.03836 39.42986 9.79327 13.60227 6.030219 17.97449 119.7174 136.11 9.588016 48.16204 6.028051 26.51938 50.67508 5.54958 31.8689 56.91853 18.76982 86.78511 47.78939 21.39392 44.05758 43.10071 12.93362 15.29362 22.67178 9.169868 22.70414 36.61877 36.40493 38.88861 62.4144 18.80965 73.55255 8358.035 616.7208 59.05638 41.89165 18.19917 70.24268 11.38647 158.0707 318.9193 38.72861 9.516384 32.10003 17329.54 1931.043 1095.633 146.2284 60.64346 8.186502 95.90353 213.953 48.77734 68.7878 65.71793 25.57602 47967.95 5397.058 248.9106 130.2801 257.1089 159.5846 89.56769 122.899 215.5294 149.6132 187.0416 115.1558 154.1615 19.63517 111.0804 45.56202 52.11806 74.48772 33.20785 22.3274 52.99072 65.79727 281.6389 28.11442 58.28289 73.20836 76.31909 141.2777 9.116022 395.3716 99.41208 56.91262 69.17138 64.92921 7.586085 75.60397 907.0748 208.3159 189.4094 4.653344 22.02694 21.07499 79.46523 21.16875 9.565468 127.7872 46.66726 49.75043 14.75244 30.76107 17.54733 12.85513 22.61628",
      "2 3", "ETD", 429.03,
      false, 2, 4, 0.2, "1 2 3", true },

    { "126.9657 129.2652 135.9172 141.9427 143.1286 147.1841 149.0127 150.2528 158.0455 163.0535 167.0655 177.1013 192.9888 203.2619 204.075 206.9989 208.9082 209.8115 211.9883 221.0683 225.0675 226.106 227.2834 228.0121 234.9963 237.0106 243.0448 247.1461 249.2151 253.3426 256.2594 257.1935 259.2448 261.0975 263.2516 268.0759 269.1938 270.3783 271.2178 277.2914 280.9584 283.1599 285.0426 288.2384 289.4238 291.1711 293.0654 294.0734 295.0374 297.267 299.9413 301.1416 301.9647 303.0014 304.1252 305.7378 308.8322 309.4466 311.3066 313.0665 313.9782 315.1105 317.1343 318.1869 318.9812 319.9492 321.1036 322.9846 324.2249 325.0501 326.029 327.1547 331.18 333.1406 334.1884 337.0988 341.0733 342.0545 343.0352 344.0117 345.1031 346.165 347.1043 349.1555 350.0966 353.1221 357.0199 357.921 359.0819 360.0584 361.1577 362.3141 363.2353 364.0934 365.3022 366.2242 367.0972 368.0494 369.3801 370.5183 371.2664 372.2919 373.0294 373.9725 375.0907 376.1542 377.2523 379.4164 382.4076 383.3729 385.1636 386.0052 387.1495 389.0345 390.0276 390.9678 391.8777 393.0433 393.9705 400.3051 401.1996 402.1906 402.9228 409.0102 410.9843 412.0966 413.1009 415.1218 417.7721 418.8526 420.6482 427.3444 428.114 429.0626 429.9189 446.916 474.9332 500.6554 529.2106 554.9348 607.3796",
      "11.65686 5.973924 6.336542 9.801903 16.33596 43.81474 351.6202 17.81902 13.21312 43.67112 43.36504 31.75116 9.61234 29.10818 5.006023 45.31382 1384.851 49.05672 22.97882 48.93507 168.9132 44.21313 11.85654 11.1582 18.42228 14.85892 22.64454 14.755 30.44192 6.76189 6.028577 15.03836 39.42986 9.79327 13.60227 6.030219 17.97449 119.7174 136.11 9.588016 48.16204 6.028051 26.51938 50.67508 5.54958 31.8689 56.91853 18.76982 86.78511 47.78939 21.39392 44.05758 43.10071 12.93362 15.29362 22.67178 9.169868 22.70414 36.61877 36.40493 38.88861 62.4144 18.80965 73.55255 8358.035 616.7208 59.05638 41.89165 18.19917 70.24268 11.38647 158.0707 318.9193 38.72861 9.516384 32.10003 17329.54 1931.043 1095.633 146.2284 60.64346 8.186502 95.90353 213.953 48.77734 68.7878 65.71793 25.57602 47967.95 5397.058 248.9106 130.2801 257.1089 159.5846 89.56769 122.899 215.5294 149.6132 187.0416 115.1558 154.1615 19.63517 111.0804 45.56202 52.11806 74.48772 33.20785 22.3274 52.99072 65.79727 281.6389 28.11442 58.28289 73.20836 76.31909 141.2777 9.116022 395.3716 99.41208 56.91262 69.17138 64.92921 7.586085 75.60397 907.0748 208.3159 189.4094 4.653344 22.02694 21.07499 79.46523 21.16875 9.565468 127.7872 46.66726 49.75043 14.75244 30.76107 17.54733 12.85513 22.61628",
      "2 3", "CID", 429.03,
      false, 2, 2, 0.2, "1 2 3", true },

    { "126.9657 129.2652 135.9172 141.9427 143.1286 147.1841 149.0127 150.2528 158.0455 163.0535 167.0655 177.1013 192.9888 203.2619 204.075 206.9989 208.9082 209.8115 211.9883 221.0683 225.0675 226.106 227.2834 228.0121 234.9963 237.0106 243.0448 247.1461 249.2151 253.3426 256.2594 257.1935 259.2448 261.0975 263.2516 268.0759 269.1938 270.3783 271.2178 277.2914 280.9584 283.1599 285.0426 288.2384 289.4238 291.1711 293.0654 294.0734 295.0374 297.267 299.9413 301.1416 301.9647 303.0014 304.1252 305.7378 308.8322 309.4466 311.3066 313.0665 313.9782 315.1105 317.1343 318.1869 318.9812 319.9492 321.1036 322.9846 324.2249 325.0501 326.029 327.1547 331.18 333.1406 334.1884 337.0988 341.0733 342.0545 343.0352 344.0117 345.1031 346.165 347.1043 349.1555 350.0966 353.1221 357.0199 357.921 359.0819 360.0584 361.1577 362.3141 363.2353 364.0934 365.3022 366.2242 367.0972 368.0494 369.3801 370.5183 371.2664 372.2919 373.0294 373.9725 375.0907 376.1542 377.2523 379.4164 382.4076 383.3729 385.1636 386.0052 387.1495 389.0345 390.0276 390.9678 391.8777 393.0433 393.9705 400.3051 401.1996 402.1906 402.9228 409.0102 410.9843 412.0966 413.1009 415.1218 417.7721 418.8526 420.6482 427.3444 428.114 429.0626 429.9189 446.916 474.9332 500.6554 529.2106 554.9348 607.3796",
      "11.65686 5.973924 6.336542 9.801903 16.33596 43.81474 351.6202 17.81902 13.21312 43.67112 43.36504 31.75116 9.61234 29.10818 5.006023 45.31382 1384.851 49.05672 22.97882 48.93507 168.9132 44.21313 11.85654 11.1582 18.42228 14.85892 22.64454 14.755 30.44192 6.76189 6.028577 15.03836 39.42986 9.79327 13.60227 6.030219 17.97449 119.7174 136.11 9.588016 48.16204 6.028051 26.51938 50.67508 5.54958 31.8689 56.91853 18.76982 86.78511 47.78939 21.39392 44.05758 43.10071 12.93362 15.29362 22.67178 9.169868 22.70414 36.61877 36.40493 38.88861 62.4144 18.80965 73.55255 8358.035 616.7208 59.05638 41.89165 18.19917 70.24268 11.38647 158.0707 318.9193 38.72861 9.516384 32.10003 17329.54 1931.043 1095.633 146.2284 60.64346 8.186502 95.90353 213.953 48.77734 68.7878 65.71793 25.57602 47967.95 5397.058 248.9106 130.2801 257.1089 159.5846 89.56769 122.899 215.5294 149.6132 187.0416 115.1558 154.1615 19.63517 111.0804 45.56202 52.11806 74.48772 33.20785 22.3274 52.99072 65.79727 281.6389 28.11442 58.28289 73.20836 76.31909 141.2777 9.116022 395.3716 99.41208 56.91262 69.17138 64.92921 7.586085 75.60397 907.0748 208.3159 189.4094 4.653344 22.02694 21.07499 79.46523 21.16875 9.565468 127.7872 46.66726 49.75043 14.75244 30.76107 17.54733 12.85513 22.61628",
      "2 3", "CID", 429.03,
      false, 2, 4, 0.00001, "2 3 4", true },

    { "126.9657 129.2652 135.9172 141.9427 143.1286 147.1841 149.0127 150.2528 158.0455 163.0535 167.0655 177.1013 192.9888 203.2619 204.075 206.9989 208.9082 209.8115 211.9883 221.0683 225.0675 226.106 227.2834 228.0121 234.9963 237.0106 243.0448 247.1461 249.2151 253.3426 256.2594 257.1935 259.2448 261.0975 263.2516 268.0759 269.1938 270.3783 271.2178 277.2914 280.9584 283.1599 285.0426 288.2384 289.4238 291.1711 293.0654 294.0734 295.0374 297.267 299.9413 301.1416 301.9647 303.0014 304.1252 305.7378 308.8322 309.4466 311.3066 313.0665 313.9782 315.1105 317.1343 318.1869 318.9812 319.9492 321.1036 322.9846 324.2249 325.0501 326.029 327.1547 331.18 333.1406 334.1884 337.0988 341.0733 342.0545 343.0352 344.0117 345.1031 346.165 347.1043 349.1555 350.0966 353.1221 357.0199 357.921 359.0819 360.0584 361.1577 362.3141 363.2353 364.0934 365.3022 366.2242 367.0972 368.0494 369.3801 370.5183 371.2664 372.2919 373.0294 373.9725 375.0907 376.1542 377.2523 379.4164 382.4076 383.3729 385.1636 386.0052 387.1495 389.0345 390.0276 390.9678 391.8777 393.0433 393.9705 400.3051 401.1996 402.1906 402.9228 409.0102 410.9843 412.0966 413.1009 415.1218 417.7721 418.8526 420.6482 427.3444 428.114 429.0626 429.9189 446.916 474.9332 500.6554 529.2106 554.9348 607.3796",
      "11.65686 5.973924 6.336542 9.801903 16.33596 43.81474 351.6202 17.81902 13.21312 43.67112 43.36504 31.75116 9.61234 29.10818 5.006023 45.31382 1384.851 49.05672 22.97882 48.93507 168.9132 44.21313 11.85654 11.1582 18.42228 14.85892 22.64454 14.755 30.44192 6.76189 6.028577 15.03836 39.42986 9.79327 13.60227 6.030219 17.97449 119.7174 136.11 9.588016 48.16204 6.028051 26.51938 50.67508 5.54958 31.8689 56.91853 18.76982 86.78511 47.78939 21.39392 44.05758 43.10071 12.93362 15.29362 22.67178 9.169868 22.70414 36.61877 36.40493 38.88861 62.4144 18.80965 73.55255 8358.035 616.7208 59.05638 41.89165 18.19917 70.24268 11.38647 158.0707 318.9193 38.72861 9.516384 32.10003 17329.54 1931.043 1095.633 146.2284 60.64346 8.186502 95.90353 213.953 48.77734 68.7878 65.71793 25.57602 47967.95 5397.058 248.9106 130.2801 257.1089 159.5846 89.56769 122.899 215.5294 149.6132 187.0416 115.1558 154.1615 19.63517 111.0804 45.56202 52.11806 74.48772 33.20785 22.3274 52.99072 65.79727 281.6389 28.11442 58.28289 73.20836 76.31909 141.2777 9.116022 395.3716 99.41208 56.91262 69.17138 64.92921 7.586085 75.60397 907.0748 208.3159 189.4094 4.653344 22.02694 21.07499 79.46523 21.16875 9.565468 127.7872 46.66726 49.75043 14.75244 30.76107 17.54733 12.85513 22.61628",
      "2 3 4", "CID", 429.03,
      false, 2, 3, 0.00001, "2 3 4", true },

    { "126.9657 129.2652 135.9172 141.9427 143.1286 147.1841 149.0127 150.2528 158.0455 163.0535 167.0655 177.1013 192.9888 203.2619 204.075 206.9989 208.9082 209.8115 211.9883 221.0683 225.0675 226.106 227.2834 228.0121 234.9963 237.0106 243.0448 247.1461 249.2151 253.3426 256.2594 257.1935 259.2448 261.0975 263.2516 268.0759 269.1938 270.3783 271.2178 277.2914 280.9584 283.1599 285.0426 288.2384 289.4238 291.1711 293.0654 294.0734 295.0374 297.267 299.9413 301.1416 301.9647 303.0014 304.1252 305.7378 308.8322 309.4466 311.3066 313.0665 313.9782 315.1105 317.1343 318.1869 318.9812 319.9492 321.1036 322.9846 324.2249 325.0501 326.029 327.1547 331.18 333.1406 334.1884 337.0988 341.0733 342.0545 343.0352 344.0117 345.1031 346.165 347.1043 349.1555 350.0966 353.1221 357.0199 357.921 359.0819 360.0584 361.1577 362.3141 363.2353 364.0934 365.3022 366.2242 367.0972 368.0494 369.3801 370.5183 371.2664 372.2919 373.0294 373.9725 375.0907 376.1542 377.2523 379.4164 382.4076 383.3729 385.1636 386.0052 387.1495 389.0345 390.0276 390.9678 391.8777 393.0433 393.9705 400.3051 401.1996 402.1906 402.9228 409.0102 410.9843 412.0966 413.1009 415.1218 417.7721 418.8526 420.6482 427.3444 428.114 429.0626 429.9189 446.916 474.9332 500.6554 529.2106 554.9348 607.3796",
      "11.65686 5.973924 6.336542 9.801903 16.33596 43.81474 351.6202 17.81902 13.21312 43.67112 43.36504 31.75116 9.61234 29.10818 5.006023 45.31382 1384.851 49.05672 22.97882 48.93507 168.9132 44.21313 11.85654 11.1582 18.42228 14.85892 22.64454 14.755 30.44192 6.76189 6.028577 15.03836 39.42986 9.79327 13.60227 6.030219 17.97449 119.7174 136.11 9.588016 48.16204 6.028051 26.51938 50.67508 5.54958 31.8689 56.91853 18.76982 86.78511 47.78939 21.39392 44.05758 43.10071 12.93362 15.29362 22.67178 9.169868 22.70414 36.61877 36.40493 38.88861 62.4144 18.80965 73.55255 8358.035 616.7208 59.05638 41.89165 18.19917 70.24268 11.38647 158.0707 318.9193 38.72861 9.516384 32.10003 17329.54 1931.043 1095.633 146.2284 60.64346 8.186502 95.90353 213.953 48.77734 68.7878 65.71793 25.57602 47967.95 5397.058 248.9106 130.2801 257.1089 159.5846 89.56769 122.899 215.5294 149.6132 187.0416 115.1558 154.1615 19.63517 111.0804 45.56202 52.11806 74.48772 33.20785 22.3274 52.99072 65.79727 281.6389 28.11442 58.28289 73.20836 76.31909 141.2777 9.116022 395.3716 99.41208 56.91262 69.17138 64.92921 7.586085 75.60397 907.0748 208.3159 189.4094 4.653344 22.02694 21.07499 79.46523 21.16875 9.565468 127.7872 46.66726 49.75043 14.75244 30.76107 17.54733 12.85513 22.61628",
      "2 3", "CID", 429.03,
      false, 2, 2, 0.00001, "2 3", true },

    { "126.9657 129.2652 135.9172 141.9427 143.1286 147.1841 149.0127 150.2528 158.0455 163.0535 167.0655 177.1013 192.9888 203.2619 204.075 206.9989 208.9082 209.8115 211.9883 221.0683 225.0675 226.106 227.2834 228.0121 234.9963 237.0106 243.0448 247.1461 249.2151 253.3426 256.2594 257.1935 259.2448 261.0975 263.2516 268.0759 269.1938 270.3783 271.2178 277.2914 280.9584 283.1599 285.0426 288.2384 289.4238 291.1711 293.0654 294.0734 295.0374 297.267 299.9413 301.1416 301.9647 303.0014 304.1252 305.7378 308.8322 309.4466 311.3066 313.0665 313.9782 315.1105 317.1343 318.1869 318.9812 319.9492 321.1036 322.9846 324.2249 325.0501 326.029 327.1547 331.18 333.1406 334.1884 337.0988 341.0733 342.0545 343.0352 344.0117 345.1031 346.165 347.1043 349.1555 350.0966 353.1221 357.0199 357.921 359.0819 360.0584 361.1577 362.3141 363.2353 364.0934 365.3022 366.2242 367.0972 368.0494 369.3801 370.5183 371.2664 372.2919 373.0294 373.9725 375.0907 376.1542 377.2523 379.4164 382.4076 383.3729 385.1636 386.0052 387.1495 389.0345 390.0276 390.9678 391.8777 393.0433 393.9705 400.3051 401.1996 402.1906 402.9228 409.0102 410.9843 412.0966 413.1009 415.1218 417.7721 418.8526 420.6482 427.3444 428.114 429.0626 429.9189 446.916 474.9332 500.6554 529.2106 554.9348 607.3796",
      "11.65686 5.973924 6.336542 9.801903 16.33596 43.81474 351.6202 17.81902 13.21312 43.67112 43.36504 31.75116 9.61234 29.10818 5.006023 45.31382 1384.851 49.05672 22.97882 48.93507 168.9132 44.21313 11.85654 11.1582 18.42228 14.85892 22.64454 14.755 30.44192 6.76189 6.028577 15.03836 39.42986 9.79327 13.60227 6.030219 17.97449 119.7174 136.11 9.588016 48.16204 6.028051 26.51938 50.67508 5.54958 31.8689 56.91853 18.76982 86.78511 47.78939 21.39392 44.05758 43.10071 12.93362 15.29362 22.67178 9.169868 22.70414 36.61877 36.40493 38.88861 62.4144 18.80965 73.55255 8358.035 616.7208 59.05638 41.89165 18.19917 70.24268 11.38647 158.0707 318.9193 38.72861 9.516384 32.10003 17329.54 1931.043 1095.633 146.2284 60.64346 8.186502 95.90353 213.953 48.77734 68.7878 65.71793 25.57602 47967.95 5397.058 248.9106 130.2801 257.1089 159.5846 89.56769 122.899 215.5294 149.6132 187.0416 115.1558 154.1615 19.63517 111.0804 45.56202 52.11806 74.48772 33.20785 22.3274 52.99072 65.79727 281.6389 28.11442 58.28289 73.20836 76.31909 141.2777 9.116022 395.3716 99.41208 56.91262 69.17138 64.92921 7.586085 75.60397 907.0748 208.3159 189.4094 4.653344 22.02694 21.07499 79.46523 21.16875 9.565468 127.7872 46.66726 49.75043 14.75244 30.76107 17.54733 12.85513 22.61628",
      "1", "CID", 429.03,
      false, 2, 4, 0.00001, "1", true },

    { "126.9657 129.2652 135.9172 141.9427 143.1286 147.1841 149.0127 150.2528 158.0455 163.0535 167.0655 177.1013 192.9888 203.2619 204.075 206.9989 208.9082 209.8115 211.9883 221.0683 225.0675 226.106 227.2834 228.0121 234.9963 237.0106 243.0448 247.1461 249.2151 253.3426 256.2594 257.1935 259.2448 261.0975 263.2516 268.0759 269.1938 270.3783 271.2178 277.2914 280.9584 283.1599 285.0426 288.2384 289.4238 291.1711 293.0654 294.0734 295.0374 297.267 299.9413 301.1416 301.9647 303.0014 304.1252 305.7378 308.8322 309.4466 311.3066 313.0665 313.9782 315.1105 317.1343 318.1869 318.9812 319.9492 321.1036 322.9846 324.2249 325.0501 326.029 327.1547 331.18 333.1406 334.1884 337.0988 341.0733 342.0545 343.0352 344.0117 345.1031 346.165 347.1043 349.1555 350.0966 353.1221 357.0199 357.921 359.0819 360.0584 361.1577 362.3141 363.2353 364.0934 365.3022 366.2242 367.0972 368.0494 369.3801 370.5183 371.2664 372.2919 373.0294 373.9725 375.0907 376.1542 377.2523 379.4164 382.4076 383.3729 385.1636 386.0052 387.1495 389.0345 390.0276 390.9678 391.8777 393.0433 393.9705 400.3051 401.1996 402.1906 402.9228 409.0102 410.9843 412.0966 413.1009 415.1218 417.7721 418.8526 420.6482 427.3444 428.114 429.0626 429.9189 446.916 474.9332 500.6554 529.2106 554.9348 607.3796",
      "11.65686 5.973924 6.336542 9.801903 16.33596 43.81474 351.6202 17.81902 13.21312 43.67112 43.36504 31.75116 9.61234 29.10818 5.006023 45.31382 1384.851 49.05672 22.97882 48.93507 168.9132 44.21313 11.85654 11.1582 18.42228 14.85892 22.64454 14.755 30.44192 6.76189 6.028577 15.03836 39.42986 9.79327 13.60227 6.030219 17.97449 119.7174 136.11 9.588016 48.16204 6.028051 26.51938 50.67508 5.54958 31.8689 56.91853 18.76982 86.78511 47.78939 21.39392 44.05758 43.10071 12.93362 15.29362 22.67178 9.169868 22.70414 36.61877 36.40493 38.88861 62.4144 18.80965 73.55255 8358.035 616.7208 59.05638 41.89165 18.19917 70.24268 11.38647 158.0707 318.9193 38.72861 9.516384 32.10003 17329.54 1931.043 1095.633 146.2284 60.64346 8.186502 95.90353 213.953 48.77734 68.7878 65.71793 25.57602 47967.95 5397.058 248.9106 130.2801 257.1089 159.5846 89.56769 122.899 215.5294 149.6132 187.0416 115.1558 154.1615 19.63517 111.0804 45.56202 52.11806 74.48772 33.20785 22.3274 52.99072 65.79727 281.6389 28.11442 58.28289 73.20836 76.31909 141.2777 9.116022 395.3716 99.41208 56.91262 69.17138 64.92921 7.586085 75.60397 907.0748 208.3159 189.4094 4.653344 22.02694 21.07499 79.46523 21.16875 9.565468 127.7872 46.66726 49.75043 14.75244 30.76107 17.54733 12.85513 22.61628",
      "1", "ETD", 429.03,
      false, 2, 4, 0.00001, "1", true },

    { "126.9657 129.2652 135.9172 141.9427 143.1286 147.1841 149.0127 150.2528 158.0455 163.0535 167.0655 177.1013 192.9888 203.2619 204.075 206.9989 208.9082 209.8115 211.9883 221.0683 225.0675 226.106 227.2834 228.0121 234.9963 237.0106 243.0448 247.1461 249.2151 253.3426 256.2594 257.1935 259.2448 261.0975 263.2516 268.0759 269.1938 270.3783 271.2178 277.2914 280.9584 283.1599 285.0426 288.2384 289.4238 291.1711 293.0654 294.0734 295.0374 297.267 299.9413 301.1416 301.9647 303.0014 304.1252 305.7378 308.8322 309.4466 311.3066 313.0665 313.9782 315.1105 317.1343 318.1869 318.9812 319.9492 321.1036 322.9846 324.2249 325.0501 326.029 327.1547 331.18 333.1406 334.1884 337.0988 341.0733 342.0545 343.0352 344.0117 345.1031 346.165 347.1043 349.1555 350.0966 353.1221 357.0199 357.921 359.0819 360.0584 361.1577 362.3141 363.2353 364.0934 365.3022 366.2242 367.0972 368.0494 369.3801 370.5183 371.2664 372.2919 373.0294 373.9725 375.0907 376.1542 377.2523 379.4164 382.4076 383.3729 385.1636 386.0052 387.1495 389.0345 390.0276 390.9678 391.8777 393.0433 393.9705 400.3051 401.1996 402.1906 402.9228 409.0102 410.9843 412.0966 413.1009 415.1218 417.7721 418.8526 420.6482 427.3444 428.114 429.0626 429.9189 446.916 474.9332 500.6554 529.2106 554.9348 607.3796",
      "11.65686 5.973924 6.336542 9.801903 16.33596 43.81474 351.6202 17.81902 13.21312 43.67112 43.36504 31.75116 9.61234 29.10818 5.006023 45.31382 1384.851 49.05672 22.97882 48.93507 168.9132 44.21313 11.85654 11.1582 18.42228 14.85892 22.64454 14.755 30.44192 6.76189 6.028577 15.03836 39.42986 9.79327 13.60227 6.030219 17.97449 119.7174 136.11 9.588016 48.16204 6.028051 26.51938 50.67508 5.54958 31.8689 56.91853 18.76982 86.78511 47.78939 21.39392 44.05758 43.10071 12.93362 15.29362 22.67178 9.169868 22.70414 36.61877 36.40493 38.88861 62.4144 18.80965 73.55255 8358.035 616.7208 59.05638 41.89165 18.19917 70.24268 11.38647 158.0707 318.9193 38.72861 9.516384 32.10003 17329.54 1931.043 1095.633 146.2284 60.64346 8.186502 95.90353 213.953 48.77734 68.7878 65.71793 25.57602 47967.95 5397.058 248.9106 130.2801 257.1089 159.5846 89.56769 122.899 215.5294 149.6132 187.0416 115.1558 154.1615 19.63517 111.0804 45.56202 52.11806 74.48772 33.20785 22.3274 52.99072 65.79727 281.6389 28.11442 58.28289 73.20836 76.31909 141.2777 9.116022 395.3716 99.41208 56.91262 69.17138 64.92921 7.586085 75.60397 907.0748 208.3159 189.4094 4.653344 22.02694 21.07499 79.46523 21.16875 9.565468 127.7872 46.66726 49.75043 14.75244 30.76107 17.54733 12.85513 22.61628",
      "", "CID", 429.03,
      false, 2, 3, 0.2, "1", true },

    { "126.9657 129.2652 135.9172 141.9427 143.1286 147.1841 149.0127 150.2528 158.0455 163.0535 167.0655 177.1013 192.9888 203.2619 204.075 206.9989 208.9082 209.8115 211.9883 221.0683 225.0675 226.106 227.2834 228.0121 234.9963 237.0106 243.0448 247.1461 249.2151 253.3426 256.2594 257.1935 259.2448 261.0975 263.2516 268.0759 269.1938 270.3783 271.2178 277.2914 280.9584 283.1599 285.0426 288.2384 289.4238 291.1711 293.0654 294.0734 295.0374 297.267 299.9413 301.1416 301.9647 303.0014 304.1252 305.7378 308.8322 309.4466 311.3066 313.0665 313.9782 315.1105 317.1343 318.1869 318.9812 319.9492 321.1036 322.9846 324.2249 325.0501 326.029 327.1547 331.18 333.1406 334.1884 337.0988 341.0733 342.0545 343.0352 344.0117 345.1031 346.165 347.1043 349.1555 350.0966 353.1221 357.0199 357.921 359.0819 360.0584 361.1577 362.3141 363.2353 364.0934 365.3022 366.2242 367.0972 368.0494 369.3801 370.5183 371.2664 372.2919 373.0294 373.9725 375.0907 376.1542 377.2523 379.4164 382.4076 383.3729 385.1636 386.0052 387.1495 389.0345 390.0276 390.9678 391.8777 393.0433 393.9705 400.3051 401.1996 402.1906 402.9228 409.0102 410.9843 412.0966 413.1009 415.1218 417.7721 418.8526 420.6482 427.3444 428.114 429.0626 429.9189 446.916 474.9332 500.6554 529.2106 554.9348 607.3796",
      "11.65686 5.973924 6.336542 9.801903 16.33596 43.81474 351.6202 17.81902 13.21312 43.67112 43.36504 31.75116 9.61234 29.10818 5.006023 45.31382 1384.851 49.05672 22.97882 48.93507 168.9132 44.21313 11.85654 11.1582 18.42228 14.85892 22.64454 14.755 30.44192 6.76189 6.028577 15.03836 39.42986 9.79327 13.60227 6.030219 17.97449 119.7174 136.11 9.588016 48.16204 6.028051 26.51938 50.67508 5.54958 31.8689 56.91853 18.76982 86.78511 47.78939 21.39392 44.05758 43.10071 12.93362 15.29362 22.67178 9.169868 22.70414 36.61877 36.40493 38.88861 62.4144 18.80965 73.55255 8358.035 616.7208 59.05638 41.89165 18.19917 70.24268 11.38647 158.0707 318.9193 38.72861 9.516384 32.10003 17329.54 1931.043 1095.633 146.2284 60.64346 8.186502 95.90353 213.953 48.77734 68.7878 65.71793 25.57602 47967.95 5397.058 248.9106 130.2801 257.1089 159.5846 89.56769 122.899 215.5294 149.6132 187.0416 115.1558 154.1615 19.63517 111.0804 45.56202 52.11806 74.48772 33.20785 22.3274 52.99072 65.79727 281.6389 28.11442 58.28289 73.20836 76.31909 141.2777 9.116022 395.3716 99.41208 56.91262 69.17138 64.92921 7.586085 75.60397 907.0748 208.3159 189.4094 4.653344 22.02694 21.07499 79.46523 21.16875 9.565468 127.7872 46.66726 49.75043 14.75244 30.76107 17.54733 12.85513 22.61628",
      "", "ETD", 429.03,
      true, 2, 3, 0.2, "1", true },

    { "126.9657 129.2652 135.9172 141.9427 143.1286 147.1841 149.0127 150.2528 158.0455 163.0535 167.0655 177.1013 192.9888 203.2619 204.075 206.9989 208.9082 209.8115 211.9883 221.0683 225.0675 226.106 227.2834 228.0121 234.9963 237.0106 243.0448 247.1461 249.2151 253.3426 256.2594 257.1935 259.2448 261.0975 263.2516 268.0759 269.1938 270.3783 271.2178 277.2914 280.9584 283.1599 285.0426 288.2384 289.4238 291.1711 293.0654 294.0734 295.0374 297.267 299.9413 301.1416 301.9647 303.0014 304.1252 305.7378 308.8322 309.4466 311.3066 313.0665 313.9782 315.1105 317.1343 318.1869 318.9812 319.9492 321.1036 322.9846 324.2249 325.0501 326.029 327.1547 331.18 333.1406 334.1884 337.0988 341.0733 342.0545 343.0352 344.0117 345.1031 346.165 347.1043 349.1555 350.0966 353.1221 357.0199 357.921 359.0819 360.0584 361.1577 362.3141 363.2353 364.0934 365.3022 366.2242 367.0972 368.0494 369.3801 370.5183 371.2664 372.2919 373.0294 373.9725 375.0907 376.1542 377.2523 379.4164 382.4076 383.3729 385.1636 386.0052 387.1495 389.0345 390.0276 390.9678 391.8777 393.0433 393.9705 400.3051 401.1996 402.1906 402.9228 409.0102 410.9843 412.0966 413.1009 415.1218 417.7721 418.8526 420.6482 427.3444 428.114 429.0626 429.9189 446.916 474.9332 500.6554 529.2106 554.9348 607.3796",
      "11.65686 5.973924 6.336542 9.801903 16.33596 43.81474 351.6202 17.81902 13.21312 43.67112 43.36504 31.75116 9.61234 29.10818 5.006023 45.31382 1384.851 49.05672 22.97882 48.93507 168.9132 44.21313 11.85654 11.1582 18.42228 14.85892 22.64454 14.755 30.44192 6.76189 6.028577 15.03836 39.42986 9.79327 13.60227 6.030219 17.97449 119.7174 136.11 9.588016 48.16204 6.028051 26.51938 50.67508 5.54958 31.8689 56.91853 18.76982 86.78511 47.78939 21.39392 44.05758 43.10071 12.93362 15.29362 22.67178 9.169868 22.70414 36.61877 36.40493 38.88861 62.4144 18.80965 73.55255 8358.035 616.7208 59.05638 41.89165 18.19917 70.24268 11.38647 158.0707 318.9193 38.72861 9.516384 32.10003 17329.54 1931.043 1095.633 146.2284 60.64346 8.186502 95.90353 213.953 48.77734 68.7878 65.71793 25.57602 47967.95 5397.058 248.9106 130.2801 257.1089 159.5846 89.56769 122.899 215.5294 149.6132 187.0416 115.1558 154.1615 19.63517 111.0804 45.56202 52.11806 74.48772 33.20785 22.3274 52.99072 65.79727 281.6389 28.11442 58.28289 73.20836 76.31909 141.2777 9.116022 395.3716 99.41208 56.91262 69.17138 64.92921 7.586085 75.60397 907.0748 208.3159 189.4094 4.653344 22.02694 21.07499 79.46523 21.16875 9.565468 127.7872 46.66726 49.75043 14.75244 30.76107 17.54733 12.85513 22.61628",
      "2 3", "CID", 429.03,
      true, 2, 4, 0.2, "1", true },

    { "126.9657 129.2652 135.9172 141.9427 143.1286 147.1841 149.0127 150.2528 158.0455 163.0535 167.0655 177.1013 192.9888 203.2619 204.075 206.9989 208.9082 209.8115 211.9883 221.0683 225.0675 226.106 227.2834 228.0121 234.9963 237.0106 243.0448 247.1461 249.2151 253.3426 256.2594 257.1935 259.2448 261.0975 263.2516 268.0759 269.1938 270.3783 271.2178 277.2914 280.9584 283.1599 285.0426 288.2384 289.4238 291.1711 293.0654 294.0734 295.0374 297.267 299.9413 301.1416 301.9647 303.0014 304.1252 305.7378 308.8322 309.4466 311.3066 313.0665 313.9782 315.1105 317.1343 318.1869 318.9812 319.9492 321.1036 322.9846 324.2249 325.0501 326.029 327.1547 331.18 333.1406 334.1884 337.0988 341.0733 342.0545 343.0352 344.0117 345.1031 346.165 347.1043 349.1555 350.0966 353.1221 357.0199 357.921 359.0819 360.0584 361.1577 362.3141 363.2353 364.0934 365.3022 366.2242 367.0972 368.0494 369.3801 370.5183 371.2664 372.2919 373.0294 373.9725 375.0907 376.1542 377.2523 379.4164 382.4076 383.3729 385.1636 386.0052 387.1495 389.0345 390.0276 390.9678 391.8777 393.0433 393.9705 400.3051 401.1996 402.1906 402.9228 409.0102 410.9843 412.0966 413.1009 415.1218 417.7721 418.8526 420.6482 427.3444 428.114 429.0626 429.9189 446.916 474.9332 500.6554 529.2106 554.9348 607.3796",
      "11.65686 5.973924 6.336542 9.801903 16.33596 43.81474 351.6202 17.81902 13.21312 43.67112 43.36504 31.75116 9.61234 29.10818 5.006023 45.31382 1384.851 49.05672 22.97882 48.93507 168.9132 44.21313 11.85654 11.1582 18.42228 14.85892 22.64454 14.755 30.44192 6.76189 6.028577 15.03836 39.42986 9.79327 13.60227 6.030219 17.97449 119.7174 136.11 9.588016 48.16204 6.028051 26.51938 50.67508 5.54958 31.8689 56.91853 18.76982 86.78511 47.78939 21.39392 44.05758 43.10071 12.93362 15.29362 22.67178 9.169868 22.70414 36.61877 36.40493 38.88861 62.4144 18.80965 73.55255 8358.035 616.7208 59.05638 41.89165 18.19917 70.24268 11.38647 158.0707 318.9193 38.72861 9.516384 32.10003 17329.54 1931.043 1095.633 146.2284 60.64346 8.186502 95.90353 213.953 48.77734 68.7878 65.71793 25.57602 47967.95 5397.058 248.9106 130.2801 257.1089 159.5846 89.56769 122.899 215.5294 149.6132 187.0416 115.1558 154.1615 19.63517 111.0804 45.56202 52.11806 74.48772 33.20785 22.3274 52.99072 65.79727 281.6389 28.11442 58.28289 73.20836 76.31909 141.2777 9.116022 395.3716 99.41208 56.91262 69.17138 64.92921 7.586085 75.60397 907.0748 208.3159 189.4094 4.653344 22.02694 21.07499 79.46523 21.16875 9.565468 127.7872 46.66726 49.75043 14.75244 30.76107 17.54733 12.85513 22.61628",
      "", "CID", 429.03,
      false, 2, 4, 0.2, "1", true },

    { "126.9657 129.2652 135.9172 141.9427 143.1286 147.1841 149.0127 150.2528 158.0455 163.0535 167.0655 177.1013 192.9888 203.2619 204.075 206.9989 208.9082 209.8115 211.9883 221.0683 225.0675 226.106 227.2834 228.0121 234.9963 237.0106 243.0448 247.1461 249.2151 253.3426 256.2594 257.1935 259.2448 261.0975 263.2516 268.0759 269.1938 270.3783 271.2178 277.2914 280.9584 283.1599 285.0426 288.2384 289.4238 291.1711 293.0654 294.0734 295.0374 297.267 299.9413 301.1416 301.9647 303.0014 304.1252 305.7378 308.8322 309.4466 311.3066 313.0665 313.9782 315.1105 317.1343 318.1869 318.9812 319.9492 321.1036 322.9846 324.2249 325.0501 326.029 327.1547 331.18 333.1406 334.1884 337.0988 341.0733 342.0545 343.0352 344.0117 345.1031 346.165 347.1043 349.1555 350.0966 353.1221 357.0199 357.921 359.0819 360.0584 361.1577 362.3141 363.2353 364.0934 365.3022 366.2242 367.0972 368.0494 369.3801 370.5183 371.2664 372.2919 373.0294 373.9725 375.0907 376.1542 377.2523 379.4164 382.4076 383.3729 385.1636 386.0052 387.1495 389.0345 390.0276 390.9678 391.8777 393.0433 393.9705 400.3051 401.1996 402.1906 402.9228 409.0102 410.9843 412.0966 413.1009 415.1218 417.7721 418.8526 420.6482 427.3444 428.114 429.0626 429.9189 446.916 474.9332 500.6554 529.2106 554.9348 607.3796",
      "11.65686 5.973924 6.336542 9.801903 16.33596 43.81474 351.6202 17.81902 13.21312 43.67112 43.36504 31.75116 9.61234 29.10818 5.006023 45.31382 1384.851 49.05672 22.97882 48.93507 168.9132 44.21313 11.85654 11.1582 18.42228 14.85892 22.64454 14.755 30.44192 6.76189 6.028577 15.03836 39.42986 9.79327 13.60227 6.030219 17.97449 119.7174 136.11 9.588016 48.16204 6.028051 26.51938 50.67508 5.54958 31.8689 56.91853 18.76982 86.78511 47.78939 21.39392 44.05758 43.10071 12.93362 15.29362 22.67178 9.169868 22.70414 36.61877 36.40493 38.88861 62.4144 18.80965 73.55255 8358.035 616.7208 59.05638 41.89165 18.19917 70.24268 11.38647 158.0707 318.9193 38.72861 9.516384 32.10003 17329.54 1931.043 1095.633 146.2284 60.64346 8.186502 95.90353 213.953 48.77734 68.7878 65.71793 25.57602 47967.95 5397.058 248.9106 130.2801 257.1089 159.5846 89.56769 122.899 215.5294 149.6132 187.0416 115.1558 154.1615 19.63517 111.0804 45.56202 52.11806 74.48772 33.20785 22.3274 52.99072 65.79727 281.6389 28.11442 58.28289 73.20836 76.31909 141.2777 9.116022 395.3716 99.41208 56.91262 69.17138 64.92921 7.586085 75.60397 907.0748 208.3159 189.4094 4.653344 22.02694 21.07499 79.46523 21.16875 9.565468 127.7872 46.66726 49.75043 14.75244 30.76107 17.54733 12.85513 22.61628",
      "", "CID", 429.03,
      false, 2, 4, 0.00001, "2 3 4", true },

    { "126.9657 129.2652 135.9172 141.9427 143.1286 147.1841 149.0127 150.2528 158.0455 163.0535 167.0655 177.1013 192.9888 203.2619 204.075 206.9989 208.9082 209.8115 211.9883 221.0683 225.0675 226.106 227.2834 228.0121 234.9963 237.0106 243.0448 247.1461 249.2151 253.3426 256.2594 257.1935 259.2448 261.0975 263.2516 268.0759 269.1938 270.3783 271.2178 277.2914 280.9584 283.1599 285.0426 288.2384 289.4238 291.1711 293.0654 294.0734 295.0374 297.267 299.9413 301.1416 301.9647 303.0014 304.1252 305.7378 308.8322 309.4466 311.3066 313.0665 313.9782 315.1105 317.1343 318.1869 318.9812 319.9492 321.1036 322.9846 324.2249 325.0501 326.029 327.1547 331.18 333.1406 334.1884 337.0988 341.0733 342.0545 343.0352 344.0117 345.1031 346.165 347.1043 349.1555 350.0966 353.1221 357.0199 357.921 359.0819 360.0584 361.1577 362.3141 363.2353 364.0934 365.3022 366.2242 367.0972 368.0494 369.3801 370.5183 371.2664 372.2919 373.0294 373.9725 375.0907 376.1542 377.2523 379.4164 382.4076 383.3729 385.1636 386.0052 387.1495 389.0345 390.0276 390.9678 391.8777 393.0433 393.9705 400.3051 401.1996 402.1906 402.9228 409.0102 410.9843 412.0966 413.1009 415.1218 417.7721 418.8526 420.6482 427.3444 428.114 429.0626 429.9189 446.916 474.9332 500.6554 529.2106 554.9348 607.3796",
      "11.65686 5.973924 6.336542 9.801903 16.33596 43.81474 351.6202 17.81902 13.21312 43.67112 43.36504 31.75116 9.61234 29.10818 5.006023 45.31382 1384.851 49.05672 22.97882 48.93507 168.9132 44.21313 11.85654 11.1582 18.42228 14.85892 22.64454 14.755 30.44192 6.76189 6.028577 15.03836 39.42986 9.79327 13.60227 6.030219 17.97449 119.7174 136.11 9.588016 48.16204 6.028051 26.51938 50.67508 5.54958 31.8689 56.91853 18.76982 86.78511 47.78939 21.39392 44.05758 43.10071 12.93362 15.29362 22.67178 9.169868 22.70414 36.61877 36.40493 38.88861 62.4144 18.80965 73.55255 8358.035 616.7208 59.05638 41.89165 18.19917 70.24268 11.38647 158.0707 318.9193 38.72861 9.516384 32.10003 17329.54 1931.043 1095.633 146.2284 60.64346 8.186502 95.90353 213.953 48.77734 68.7878 65.71793 25.57602 47967.95 5397.058 248.9106 130.2801 257.1089 159.5846 89.56769 122.899 215.5294 149.6132 187.0416 115.1558 154.1615 19.63517 111.0804 45.56202 52.11806 74.48772 33.20785 22.3274 52.99072 65.79727 281.6389 28.11442 58.28289 73.20836 76.31909 141.2777 9.116022 395.3716 99.41208 56.91262 69.17138 64.92921 7.586085 75.60397 907.0748 208.3159 189.4094 4.653344 22.02694 21.07499 79.46523 21.16875 9.565468 127.7872 46.66726 49.75043 14.75244 30.76107 17.54733 12.85513 22.61628",
      "", "CID", 429.03,
      true, 2, 3, 0.00001, "2 3", true },

    { "126.9657 129.2652 135.9172 141.9427 143.1286 147.1841 149.0127 150.2528 158.0455 163.0535 167.0655 177.1013 192.9888 203.2619 204.075 206.9989 208.9082 209.8115 211.9883 221.0683 225.0675 226.106 227.2834 228.0121 234.9963 237.0106 243.0448 247.1461 249.2151 253.3426 256.2594 257.1935 259.2448 261.0975 263.2516 268.0759 269.1938 270.3783 271.2178 277.2914 280.9584 283.1599 285.0426 288.2384 289.4238 291.1711 293.0654 294.0734 295.0374 297.267 299.9413 301.1416 301.9647 303.0014 304.1252 305.7378 308.8322 309.4466 311.3066 313.0665 313.9782 315.1105 317.1343 318.1869 318.9812 319.9492 321.1036 322.9846 324.2249 325.0501 326.029 327.1547 331.18 333.1406 334.1884 337.0988 341.0733 342.0545 343.0352 344.0117 345.1031 346.165 347.1043 349.1555 350.0966 353.1221 357.0199 357.921 359.0819 360.0584 361.1577 362.3141 363.2353 364.0934 365.3022 366.2242 367.0972 368.0494 369.3801 370.5183 371.2664 372.2919 373.0294 373.9725 375.0907 376.1542 377.2523 379.4164 382.4076 383.3729 385.1636 386.0052 387.1495 389.0345 390.0276 390.9678 391.8777 393.0433 393.9705 400.3051 401.1996 402.1906 402.9228 409.0102 410.9843 412.0966 413.1009 415.1218 417.7721 418.8526 420.6482 427.3444 428.114 429.0626 429.9189 446.916 474.9332 500.6554 529.2106 554.9348 607.3796",
      "11.65686 5.973924 6.336542 9.801903 16.33596 43.81474 351.6202 17.81902 13.21312 43.67112 43.36504 31.75116 9.61234 29.10818 5.006023 45.31382 1384.851 49.05672 22.97882 48.93507 168.9132 44.21313 11.85654 11.1582 18.42228 14.85892 22.64454 14.755 30.44192 6.76189 6.028577 15.03836 39.42986 9.79327 13.60227 6.030219 17.97449 119.7174 136.11 9.588016 48.16204 6.028051 26.51938 50.67508 5.54958 31.8689 56.91853 18.76982 86.78511 47.78939 21.39392 44.05758 43.10071 12.93362 15.29362 22.67178 9.169868 22.70414 36.61877 36.40493 38.88861 62.4144 18.80965 73.55255 8358.035 616.7208 59.05638 41.89165 18.19917 70.24268 11.38647 158.0707 318.9193 38.72861 9.516384 32.10003 17329.54 1931.043 1095.633 146.2284 60.64346 8.186502 95.90353 213.953 48.77734 68.7878 65.71793 25.57602 47967.95 5397.058 248.9106 130.2801 257.1089 159.5846 89.56769 122.899 215.5294 149.6132 187.0416 115.1558 154.1615 19.63517 111.0804 45.56202 52.11806 74.48772 33.20785 22.3274 52.99072 65.79727 281.6389 28.11442 58.28289 73.20836 76.31909 141.2777 9.116022 395.3716 99.41208 56.91262 69.17138 64.92921 7.586085 75.60397 907.0748 208.3159 189.4094 4.653344 22.02694 21.07499 79.46523 21.16875 9.565468 127.7872 46.66726 49.75043 14.75244 30.76107 17.54733 12.85513 22.61628",
      "", "ETD", 429.03,
      true, 2, 3, 0.00001, "2 3", true },

    { "126.9657 129.2652 135.9172 141.9427 143.1286 147.1841 149.0127 150.2528 158.0455 163.0535 167.0655 177.1013 192.9888 203.2619 204.075 206.9989 208.9082 209.8115 211.9883 221.0683 225.0675 226.106 227.2834 228.0121 234.9963 237.0106 243.0448 247.1461 249.2151 253.3426 256.2594 257.1935 259.2448 261.0975 263.2516 268.0759 269.1938 270.3783 271.2178 277.2914 280.9584 283.1599 285.0426 288.2384 289.4238 291.1711 293.0654 294.0734 295.0374 297.267 299.9413 301.1416 301.9647 303.0014 304.1252 305.7378 308.8322 309.4466 311.3066 313.0665 313.9782 315.1105 317.1343 318.1869 318.9812 319.9492 321.1036 322.9846 324.2249 325.0501 326.029 327.1547 331.18 333.1406 334.1884 337.0988 341.0733 342.0545 343.0352 344.0117 345.1031 346.165 347.1043 349.1555 350.0966 353.1221 357.0199 357.921 359.0819 360.0584 361.1577 362.3141 363.2353 364.0934 365.3022 366.2242 367.0972 368.0494 369.3801 370.5183 371.2664 372.2919 373.0294 373.9725 375.0907 376.1542 377.2523 379.4164 382.4076 383.3729 385.1636 386.0052 387.1495 389.0345 390.0276 390.9678 391.8777 393.0433 393.9705 400.3051 401.1996 402.1906 402.9228 409.0102 410.9843 412.0966 413.1009 415.1218 417.7721 418.8526 420.6482 427.3444 428.114 429.0626 429.9189 446.916 474.9332 500.6554 529.2106 554.9348 607.3796",
      "11.65686 5.973924 6.336542 9.801903 16.33596 43.81474 351.6202 17.81902 13.21312 43.67112 43.36504 31.75116 9.61234 29.10818 5.006023 45.31382 1384.851 49.05672 22.97882 48.93507 168.9132 44.21313 11.85654 11.1582 18.42228 14.85892 22.64454 14.755 30.44192 6.76189 6.028577 15.03836 39.42986 9.79327 13.60227 6.030219 17.97449 119.7174 136.11 9.588016 48.16204 6.028051 26.51938 50.67508 5.54958 31.8689 56.91853 18.76982 86.78511 47.78939 21.39392 44.05758 43.10071 12.93362 15.29362 22.67178 9.169868 22.70414 36.61877 36.40493 38.88861 62.4144 18.80965 73.55255 8358.035 616.7208 59.05638 41.89165 18.19917 70.24268 11.38647 158.0707 318.9193 38.72861 9.516384 32.10003 17329.54 1931.043 1095.633 146.2284 60.64346 8.186502 95.90353 213.953 48.77734 68.7878 65.71793 25.57602 47967.95 5397.058 248.9106 130.2801 257.1089 159.5846 89.56769 122.899 215.5294 149.6132 187.0416 115.1558 154.1615 19.63517 111.0804 45.56202 52.11806 74.48772 33.20785 22.3274 52.99072 65.79727 281.6389 28.11442 58.28289 73.20836 76.31909 141.2777 9.116022 395.3716 99.41208 56.91262 69.17138 64.92921 7.586085 75.60397 907.0748 208.3159 189.4094 4.653344 22.02694 21.07499 79.46523 21.16875 9.565468 127.7872 46.66726 49.75043 14.75244 30.76107 17.54733 12.85513 22.61628",
      "1", "CID", 429.03,
      true, 2, 4, 0.00001, "2 3 4", true },

    { "1218.258 1244.477 1354.132 1391.253",
      "29.83101 15.71422 9.135175 6.936273",
      "", "CID", 1390.47,
      false, 2, 3, 0.2, "1", true },

    { "1218.258 1244.477 1354.132 1391.253",
      "29.83101 15.71422 9.135175 6.936273",
      "", "CID", 1390.47,
      false, 2, 3, 0.00001, "1", true },
};

const size_t testChargeStateCalculatorsSize = sizeof(testChargeStateCalculators) / sizeof(TestChargeStateCalculator);

vector<double> parseDoubleArray(const string& doubleArray)
{
    vector<double> doubleVector;
    vector<string> tokens;
    bal::split(tokens, doubleArray, bal::is_space(), bal::token_compress_on);
    if (!tokens.empty())
        for (size_t i=0; i < tokens.size(); ++i)
            if (!tokens[i].empty())
                doubleVector.push_back(lexical_cast<double>(tokens[i]));
    return doubleVector;
}

vector<CVID> parseCVTermArray(const string& cvTermArray)
{
    static pwiz::data::CVTranslator cvTranslator;
    vector<CVID> cvTermVector;
    vector<string> tokens;
    bal::split(tokens, cvTermArray, bal::is_space());
    if (!tokens.empty() && !tokens[0].empty())
        for (size_t i=0; i < tokens.size(); ++i)
            cvTermVector.push_back(cvTranslator.translate(tokens[i]));
    return cvTermVector;
}

int test()
{
    int failedTests = 0;
    for (size_t i=0; i < testChargeStateCalculatorsSize; ++i)
    {
        SpectrumListSimple* sl = new SpectrumListSimple;
        SpectrumListPtr originalList(sl);
        SpectrumPtr s(new Spectrum);
        s->set(MS_MSn_spectrum);
        s->set(MS_ms_level, 2);
        sl->spectra.push_back(s);

        try
        {
            TestChargeStateCalculator& t = testChargeStateCalculators[i];

            vector<double> inputMZArray = parseDoubleArray(t.inputMZArray);
            vector<double> inputIntensityArray = parseDoubleArray(t.inputIntensityArray);
            s->setMZIntensityArrays(inputMZArray, inputIntensityArray, MS_number_of_detector_counts);

            s->precursors.push_back(Precursor(t.inputPrecursorMZ));
            vector<double> inputChargeStateArray = parseDoubleArray(t.inputChargeStateArray);
            CVID inputChargeStateTerm = inputChargeStateArray.size() > 1 ? MS_possible_charge_state : MS_charge_state;
            BOOST_FOREACH(int z, inputChargeStateArray)
                s->precursors[0].selectedIons[0].cvParams.push_back(CVParam(inputChargeStateTerm, z));

            vector<CVID> inputActivationTypes = parseCVTermArray(t.inputActivationTypeArray);
            BOOST_FOREACH(CVID cvid, inputActivationTypes)
                s->precursors[0].activation.set(cvid);

            SpectrumListPtr calculator(new SpectrumList_ChargeStateCalculator(
                originalList,
                t.overrideExistingChargeState,
                t.maxMultipleCharge,
                t.minMultipleCharge,
                t.singlyChargedFraction,
                t.useMakeMS2Behavior));

            vector<double> outputChargeStateArray = parseDoubleArray(t.chargeStateArray);
            CVID outputChargeStateTerm = outputChargeStateArray.size() > 1 ? MS_possible_charge_state : MS_charge_state;
            SpectrumPtr calculatedSpectrum = calculator->spectrum(0, true);
            BOOST_FOREACH(const CVParam& cvParam, calculatedSpectrum->precursors[0].selectedIons[0].cvParams)
            {
                if (cvParam.cvid != MS_charge_state && cvParam.cvid != MS_possible_charge_state)
                    continue;
                unit_assert_operator_equal(outputChargeStateTerm, cvParam.cvid);
                unit_assert(find(outputChargeStateArray.begin(), outputChargeStateArray.end(), cvParam.valueAs<int>()) != outputChargeStateArray.end());
            }
        }
        catch (exception& e)
        {
            cerr << "Test case " << (i+1) << " failed:\n" << e.what() << endl;
            ++failedTests;
        }
    }
    return failedTests;
}


int main(int argc, char* argv[])
{
    TEST_PROLOG(argc, argv)

    try
    {
        if (argc>1 && !strcmp(argv[1],"-v")) os_ = &cout;
        int failedTests = test();
        unit_assert_operator_equal(0, failedTests);
    }
    catch (exception& e)
    {
        TEST_FAILED(e.what())
    }
    catch (...)
    {
        TEST_FAILED("Caught unknown exception.")
    }

    TEST_EPILOG
}
