#include "osl/move_classifier/safeMove.h"
#include "osl/move_action/store.h"
#include "osl/move_generator/allMoves.h"
#include "osl/record/csaRecord.h"
#include "osl/record/csaString.h"
#include "osl/container/moveVector.h"
#include "osl/state/numEffectState.h"
#include "osl/apply_move/applyMove.h"
#include "osl/oslConfig.h"

#include <iostream>
#include <fstream>
#include <cppunit/TestCase.h>
#include <cppunit/extensions/HelperMacros.h>
typedef osl::NumEffectState test_state_t;

class SafeMoveTest : public CppUnit::TestFixture 
{
  CPPUNIT_TEST_SUITE( SafeMoveTest );
  CPPUNIT_TEST( testIsSafe );
  CPPUNIT_TEST( testFile );
  CPPUNIT_TEST_SUITE_END();
public:
  void testIsSafe();
  void testFile();
};

CPPUNIT_TEST_SUITE_REGISTRATION(SafeMoveTest);

using namespace osl;
using namespace osl::move_action;
using namespace osl::move_generator;
using namespace osl::move_classifier;

void SafeMoveTest::testIsSafe(){
  {
    NumEffectState state=CsaString(
"P1+NY+TO *  *  *  * -OU-KE-KY\n"
"P2 *  *  *  *  * -GI-KI *  *\n"
"P3 * -RY *  *  *  * -KI-FU-FU\n"
"P4 * +FU * -FU *  *  *  *  *\n"
"P5 *  * -KE * +FU *  * +FU *\n"
"P6+KE *  * +FU+GI-FU *  * +FU\n"
"P7 *  * -UM-KA *  *  *  *  *\n"
"P8 *  * +FU *  *  *  *  *  * \n"
"P9 * +OU * -GI *  *  *  * -NG\n"
"P+00HI00KI00KE00KY00FU00FU00FU00FU00FU\n"
"P-00KI00KY00FU00FU\n"
"P-00AL\n"
"+\n"
).getInitialState();
    // 王が動くが安全
    CPPUNIT_ASSERT((SafeMove<BLACK>::isMember(state,KING,Position(8,9),Position(7,9))));
    // 王が動いて危険
    CPPUNIT_ASSERT((!SafeMove<BLACK>::isMember(state,KING,Position(8,9),Position(8,8))));
    // 龍をblockしている歩が動くが相手の龍を取るので安全
    CPPUNIT_ASSERT((SafeMove<BLACK>::isMember(state,PAWN,Position(8,4),Position(8,3))));
    // 角をblockしている歩が動いて危険
    CPPUNIT_ASSERT(!(SafeMove<BLACK>::isMember(state,PAWN,Position(7,8),Position(7,7))));
  }
  {
    NumEffectState state=CsaString(
"P1-KY-KE-GI-KI-OU-KI-GI-KE-KY\n"
"P2 * -HI *  *  *  *  * -KA * \n"
"P3-FU-FU-FU-FU-FU-FU-FU-FU-FU\n"
"P4 *  *  *  *  *  *  *  *  * \n"
"P5 *  *  *  *  *  *  *  *  * \n"
"P6 *  * +FU *  *  *  *  *  * \n"
"P7+FU+FU * +FU+FU+FU+FU+FU+FU\n"
"P8 * +KA *  *  *  *  * +HI * \n"
"P9+KY+KE+GI+KI+OU+KI+GI+KE+KY\n"
"-\n"
).getInitialState();
    // 安全
    CPPUNIT_ASSERT((SafeMove<WHITE>::isMember(state,PAWN,Position(4,3),Position(4,4))));
  }
  {
    NumEffectState state=CsaString(
"P1-KY-KE * -KI-OU * -GI+RY-KY\n"
"P2 *  *  *  *  *  * -KI *  * \n"
"P3-FU * -GI-FU-FU-FU *  * -FU\n"
"P4 *  * -FU *  *  *  *  *  * \n"
"P5 *  *  *  * +KA *  *  *  * \n"
"P6 *  *  *  *  *  *  *  *  * \n"
"P7+FU * +FU+FU+FU+FU+FU * +FU\n"
"P8 * +GI+KI *  *  *  *  *  * \n"
"P9+KY+KE *  * +OU+KI+GI * +KY\n"
"P+00FU\n"
"P+00FU\n"
"P-00FU\n"
"P-00FU\n"
"P-00FU\n"
"P+00KE\n"
"P-00KE\n"
"P-00KA\n"
"P-00HI\n"
"-\n"
).getInitialState();
    // 安全ではない
    CPPUNIT_ASSERT(!(SafeMove<WHITE>::isMember(state,SILVER,Position(3,1),Position(2,2))));
  }
  {
    NumEffectState state=CsaString(
      "P1 *  *  *  *  *  *  *  * -KY\n"
      "P2 *  *  * +TO * -GI *  * -GI\n"
      "P3-OU+FU *  *  * -KI *  * -FU\n"
      "P4 *  *  * -GI-FU * -FU-FU * \n"
      "P5 * -HI *  *  *  *  *  *  * \n"
      "P6 *  * +OU+FU+FU * +FU *  * \n"
      "P7+FU *  *  * +KI *  *  * +FU\n"
      "P8 * -NY * -NK *  *  *  *  * \n"
      "P9+KY *  *  *  *  *  * +KE+KY\n"
      "P+00FU00FU\n"
      "P-00HI00KA00KA00KI00KI00GI00KE00KE00FU00FU00FU00FU00FU\n"
      "+\n").getInitialState();
    CPPUNIT_ASSERT(!(SafeMove<BLACK>::isMember(state,KING,Position(7,6),Position(8,6))));    
  }
}
template<typename State> 
static bool isSafeMove(const State& state_org,Move move){
  State state = state_org;
  ApplyMoveOfTurn::doMove(state, move);
  return !state.hasEffectBy(alt(move.player()),state.getKingPosition(move.player()));
}

template<class State>
static void testMoveFile(const std::string& fileName){
  Record rec=CsaFile(fileName).getRecord();
  NumEffectState sState(rec.getInitialState());
  State state(sState);
  vector<osl::Move> moves=rec.getMoves();
  for(unsigned int i=0;i<moves.size();i++){
    MoveVector allMoves;
    Player turn=state.getTurn();
    // 敵から王手がかかっている時は扱わない
    if(state.hasEffectBy(alt(turn),state.getKingPosition(turn)))
      continue;
    {
      Store store(allMoves);
      AllMoves<Store>::
	generate(turn,state,store);
    }
    for(size_t j=0;j<allMoves.size();j++){
      Move move=allMoves[j];
      // dropmoveは扱わない
      if (move.isDrop()) continue;
      if(turn==BLACK)
	CPPUNIT_ASSERT(isSafeMove(state,move) ==
		       (SafeMove<BLACK>::isMember(state,move.ptype(),move.from(),move.to())));
      else
	CPPUNIT_ASSERT(isSafeMove(state,move) ==
		       (SafeMove<WHITE>::isMember(state,move.ptype(),move.from(),move.to())));
    }
    ApplyMoveOfTurn::doMove(state, moves[i]);
  }
}
void SafeMoveTest::testFile(){
  extern bool isShortTest;
  std::ifstream ifs(OslConfig::testCsaFile("FILES"));
  CPPUNIT_ASSERT(ifs);
  int i=0;
  int count=50;
  if (isShortTest) 
    count=5;
  std::string fileName;
  while((ifs >> fileName) && ++i<count){
    if(fileName == "") 
      break;
    if (! isShortTest)
      std::cerr << fileName << " ";
    testMoveFile<test_state_t>(OslConfig::testCsaFile(fileName));
  }
}
// ;;; Local Variables:
// ;;; mode:c++
// ;;; c-basic-offset:2
// ;;; End:
