/* proofNumberTable.t.cc
 */
#include "osl/checkmate/proofNumberTable.h"
#include "osl/state/numEffectState.h"
#include "osl/record/csaString.h"
#include <cppunit/TestCase.h>
#include <cppunit/extensions/HelperMacros.h>

namespace osl
{
  namespace checkmate
  {
    class ProofNumberTableTest : public CppUnit::TestFixture 
    {
      CPPUNIT_TEST_SUITE(ProofNumberTableTest);
      CPPUNIT_TEST(testConstruction);
      CPPUNIT_TEST(testCountLiberty);
      CPPUNIT_TEST(testLibertyAfterAllDrop);
      CPPUNIT_TEST(testLibertyAfterAllMove);
      CPPUNIT_TEST_SUITE_END();
    public:
      void testConstruction() { 
	ProofNumberTable table;
      }
      void testCountLiberty();
      void testLibertyAfterAllDrop();
      void testLibertyAfterAllMove();
    };
  }
}

CPPUNIT_TEST_SUITE_REGISTRATION(osl::checkmate::ProofNumberTableTest);

void osl::checkmate::
ProofNumberTableTest::testCountLiberty()
{
  const ProofNumberTable& table = Proof_Number_Table;
  {
    unsigned int liberty = 0xff;
    CPPUNIT_ASSERT_EQUAL((uint8_t)7, table.countLiberty(PAWN, U, liberty).liberty);
    CPPUNIT_ASSERT_EQUAL((uint8_t)6, table.countLiberty(LANCE, U, liberty).liberty);
    CPPUNIT_ASSERT_EQUAL((uint8_t)5, table.countLiberty(SILVER, U, liberty).liberty);
    CPPUNIT_ASSERT_EQUAL((uint8_t)6, table.countLiberty(SILVER, UL, liberty).liberty);
    CPPUNIT_ASSERT_EQUAL((uint8_t)3, table.countLiberty(GOLD, U, liberty).liberty);
    CPPUNIT_ASSERT_EQUAL((uint8_t)2, table.countLiberty(PROOK, U, liberty).liberty);
    CPPUNIT_ASSERT_EQUAL((uint8_t)5, table.countLiberty(GOLD, D, liberty).liberty);
    // 王手ではない	
    CPPUNIT_ASSERT_EQUAL((uint8_t)7, table.countLiberty(PAWN, D, liberty).liberty);
  }
  {
    NumEffectState state(CsaString(
			   "P1 *  *  *  *  *  * -OU *  * \n"
			   "P2 *  *  *  *  *  *  * -FU * \n"
			   "P3 *  *  *  *  *  *  *  *  * \n"
			   "P4 *  *  *  *  *  *  *  *  * \n"
			   "P5 *  *  *  *  *  *  *  *  * \n"
			   "P6 *  *  *  *  *  *  *  *  * \n"
			   "P7 *  *  *  *  *  *  *  *  * \n"
			   "P8 * +FU *  *  *  *  *  *  * \n"
			   "P9 *  * +OU *  *  *  *  *  * \n"
			   "P-00AL\n"
			   "+\n").getInitialState());
    const Move m42ki(Position(4,2),GOLD,BLACK);
    const Move m42gi(Position(4,2),SILVER,BLACK);

    CPPUNIT_ASSERT_EQUAL(3, table.countLiberty(state, m42gi));
    CPPUNIT_ASSERT_EQUAL(2, table.countLiberty(state, m42ki));

    const Move m68ki(Position(6,8),GOLD,  WHITE);
    const Move m68gi(Position(6,8),SILVER,WHITE);

    CPPUNIT_ASSERT_EQUAL(3, table.countLiberty(state, m68gi));
    CPPUNIT_ASSERT_EQUAL(2, table.countLiberty(state, m68ki));

    const Move m71hi(Position(7,1),ROOK,BLACK);
    const Move m53ka(Position(5,3),BISHOP,BLACK);

    CPPUNIT_ASSERT_EQUAL(3, table.countLiberty(state, m71hi));
    CPPUNIT_ASSERT_EQUAL(4, table.countLiberty(state, m53ka));

    const Move m39hi(Position(3,9),ROOK,  WHITE);
    const Move m57ka(Position(5,7),BISHOP,WHITE);

    CPPUNIT_ASSERT_EQUAL(3, table.countLiberty(state, m39hi));
    CPPUNIT_ASSERT_EQUAL(4, table.countLiberty(state, m57ka));
  }
  {
    NumEffectState state(CsaString(
			   "P1 *  *  *  *  *  * -OU *  * \n"
			   "P2 *  *  *  *  *  *  * -FU * \n"
			   "P3 *  *  *  *  * +GI *  *  * \n"
			   "P4 *  *  * -FU+HI *  *  *  * \n"
			   "P5 *  * -KA+FU *  * +KA *  * \n"
			   "P6 *  *  *  * -HI+FU *  *  * \n"
			   "P7 *  *  * -GI *  *  *  *  * \n"
			   "P8 * +FU *  *  *  *  *  *  * \n"
			   "P9 *  * +OU *  *  *  *  *  * \n"
			   "P-00AL\n"
			   "+\n").getInitialState());
    const Move m42ng(Position(4,3),Position(4,2),PSILVER,PTYPE_EMPTY,true,BLACK);
    const Move m42gi(Position(4,3),Position(4,2),SILVER,PTYPE_EMPTY,false,BLACK);

    CPPUNIT_ASSERT_EQUAL(2, table.countLiberty(state, m42gi)); // 本当は3
    CPPUNIT_ASSERT_EQUAL(2, table.countLiberty(state, m42ng));

    const Move m68ng(Position(6,7),Position(6,8),PSILVER,PTYPE_EMPTY,true,WHITE);
    const Move m68gi(Position(6,7),Position(6,8),SILVER,PTYPE_EMPTY,false,WHITE);

    CPPUNIT_ASSERT_EQUAL(2, table.countLiberty(state, m68gi));
    CPPUNIT_ASSERT_EQUAL(2, table.countLiberty(state, m68ng));

    const Move m51ry(Position(5,4),Position(5,1),PROOK,PTYPE_EMPTY,true,BLACK);
    const Move m53um(Position(3,5),Position(5,3),PBISHOP,PTYPE_EMPTY,true,BLACK);

    CPPUNIT_ASSERT_EQUAL(2, table.countLiberty(state, m51ry));
    CPPUNIT_ASSERT_EQUAL(3, table.countLiberty(state, m53um));

    const Move m59ry(Position(5,6),Position(5,9),PROOK,PTYPE_EMPTY,true,WHITE);
    const Move m57um(Position(7,5),Position(5,7),PBISHOP,PTYPE_EMPTY,false,WHITE);

    CPPUNIT_ASSERT_EQUAL(2, table.countLiberty(state, m59ry));
    CPPUNIT_ASSERT_EQUAL(3, table.countLiberty(state, m57um));
  }
  {
    // 空き王手
    NumEffectState state(CsaString(
			   "P1 *  *  *  *  *  *  *  *  * \n"
			   "P2 *  *  *  * -OU *  * +GI+HI\n"
			   "P3 *  *  *  *  *  *  *  *  * \n"
			   "P4 *  *  *  *  *  *  *  *  * \n"
			   "P5 *  *  *  *  *  *  *  *  * \n"
			   "P6 *  *  *  *  *  *  *  *  * \n"
			   "P7 *  *  *  *  *  *  *  *  * \n"
			   "P8 *  *  *  *  *  *  *  *  * \n"
			   "P9 *  * +OU *  *  *  *  *  * \n"
			   "P-00AL\n"
			   "+\n").getInitialState());
    const Move m31ng(Position(2,2),Position(3,1),PSILVER,PTYPE_EMPTY,true,BLACK);
    const Move m13gi(Position(2,2),Position(2,3),SILVER,PTYPE_EMPTY,false,BLACK);

    CPPUNIT_ASSERT_EQUAL(7, table.countLiberty(state, m31ng));
    CPPUNIT_ASSERT_EQUAL(7, table.countLiberty(state, m13gi));
  }
  {
    // 空き王手 2マス離れ
    NumEffectState state(CsaString(
			   "P1 *  *  *  *  *  *  *  *  * \n"
			   "P2 *  *  *  * -OU * +KA * +HI\n"
			   "P3 *  *  *  *  *  *  *  *  * \n"
			   "P4 *  *  *  *  *  *  *  *  * \n"
			   "P5 *  *  *  *  *  *  *  *  * \n"
			   "P6 *  *  *  *  *  *  *  *  * \n"
			   "P7 *  *  *  *  *  *  *  *  * \n"
			   "P8 *  *  *  *  *  *  *  *  * \n"
			   "P9 *  * +OU *  *  *  *  *  * \n"
			   "P-00AL\n"
			   "+\n").getInitialState());
    const Move m54um(Position(3,2),Position(5,4),PBISHOP,PTYPE_EMPTY,true,BLACK);
    const Move m65um(Position(3,2),Position(6,5),PBISHOP,PTYPE_EMPTY,true,BLACK);

    CPPUNIT_ASSERT_EQUAL(4, table.countLiberty(state, m54um));
    CPPUNIT_ASSERT_EQUAL(5, table.countLiberty(state, m65um)); // XXX:今のliberty-1
  }
  {
    // 空き王手 2マス離れ 無関係
    NumEffectState state(CsaString(
			   "P1 *  *  *  *  *  *  *  *  * \n"
			   "P2 *  *  *  * -OU *  *  *  * \n"
			   "P3 *  *  *  *  *  *  *  *  * \n"
			   "P4 *  *  *  *  *  * +KY *  * \n"
			   "P5 *  *  *  *  *  *  *  *  * \n"
			   "P6 *  *  *  *  *  *  *  * +KA\n"
			   "P7 *  *  *  *  *  *  *  *  * \n"
			   "P8 *  *  *  *  *  *  *  *  * \n"
			   "P9 *  * +OU *  *  *  *  *  * \n"
			   "P-00AL\n"
			   "+\n").getInitialState());
    const Move m32ky(Position(3,4),Position(3,2),LANCE,PTYPE_EMPTY,false,BLACK);
    const Move m32ny(Position(3,4),Position(3,2),PLANCE,PTYPE_EMPTY,true,BLACK);

    CPPUNIT_ASSERT_EQUAL(8, table.countLiberty(state, m32ky));
    CPPUNIT_ASSERT_EQUAL(6, table.countLiberty(state, m32ny));
  }
  {
    // 追加利き
    NumEffectState state(CsaString(
			   "P1 *  *  *  *  *  *  *  *  * \n"
			   "P2 *  *  *  * -OU *  *  *  * \n"
			   "P3 *  *  *  * -FU *  *  *  * \n"
			   "P4 *  *  *  *  *  *  *  *  * \n"
			   "P5 *  *  *  * +HI *  *  *  * \n"
			   "P6 *  *  *  * +KY *  *  *  * \n"
			   "P7 *  *  *  *  *  *  *  *  * \n"
			   "P8 *  *  *  *  *  *  *  *  * \n"
			   "P9 *  * +OU *  *  *  *  *  * \n"
			   "P-00AL\n"
			   "+\n").getInitialState());
    const Move m53ry(Position(5,5),Position(5,3),PROOK,PAWN,true,BLACK);

    CPPUNIT_ASSERT_EQUAL(2, table.countLiberty(state, m53ry));
  }
}

void osl::checkmate::
ProofNumberTableTest::testLibertyAfterAllDrop()
{
  const ProofNumberTable& table = Proof_Number_Table;
  {
    NumEffectState state(CsaString(
			   "P1 *  *  *  *  *  *  *  *  * \n"
			   "P2 *  *  *  * -OU *  *  *  * \n"
			   "P3 *  *  *  *  *  *  *  *  * \n"
			   "P4 *  *  *  * +FU *  *  *  * \n"
			   "P5 *  *  *  *  *  *  *  *  * \n"
			   "P6 *  *  *  *  *  *  *  *  * \n"
			   "P7 *  *  *  *  *  *  *  *  * \n"
			   "P8 *  *  *  *  *  *  *  *  * \n"
			   "P9 *  * +OU *  *  *  *  *  * \n"
			   "P-00AL\n"
			   "+\n").getInitialState());
    CPPUNIT_ASSERT_EQUAL(6, table.libertyAfterAllDrop(state));
  }
  {
    NumEffectState state(CsaString(
			   "P1 *  *  *  *  *  *  *  *  * \n"
			   "P2 *  *  *  * -OU *  *  *  * \n"
			   "P3 *  *  *  *  *  *  *  *  * \n"
			   "P4 *  *  *  * +FU *  *  *  * \n"
			   "P5 *  *  *  *  *  *  *  *  * \n"
			   "P6 *  *  *  *  *  *  *  *  * \n"
			   "P7 *  *  *  *  *  *  *  *  * \n"
			   "P8 *  *  *  *  *  *  *  *  * \n"
			   "P9 *  * +OU *  *  *  *  *  * \n"
			   "P+00KI\n"
			   "P-00AL\n"
			   "+\n").getInitialState());
    CPPUNIT_ASSERT_EQUAL(3, table.libertyAfterAllDrop(state));
  }
  {
    NumEffectState state(CsaString(
			   "P1 *  *  *  *  *  *  *  *  * \n"
			   "P2 *  *  *  * -OU *  *  *  * \n"
			   "P3 *  *  *  *  *  *  *  *  * \n"
			   "P4 *  *  *  * +FU *  *  *  * \n"
			   "P5 *  *  *  *  *  *  *  *  * \n"
			   "P6 *  *  *  *  *  *  *  *  * \n"
			   "P7 *  *  *  *  *  *  *  *  * \n"
			   "P8 *  *  *  *  *  *  *  *  * \n"
			   "P9 *  * +OU *  *  *  *  *  * \n"
			   "P+00KI00GI\n"
			   "P-00AL\n"
			   "+\n").getInitialState());
    CPPUNIT_ASSERT_EQUAL(3, table.libertyAfterAllDrop(state));
  }
  {
    NumEffectState state(CsaString(
			   "P1 *  *  *  *  *  *  *  *  * \n"
			   "P2 *  *  *  * -OU *  *  *  * \n"
			   "P3 *  *  *  *  *  *  *  *  * \n"
			   "P4 *  *  *  * +FU *  *  *  * \n"
			   "P5 *  *  *  *  *  *  *  *  * \n"
			   "P6 *  *  *  *  *  *  *  *  * \n"
			   "P7 *  *  *  *  *  *  *  *  * \n"
			   "P8 *  *  *  *  *  *  *  *  * \n"
			   "P9 *  * +OU *  *  *  *  *  * \n"
			   "P+00GI\n"
			   "P-00AL\n"
			   "+\n").getInitialState());
    CPPUNIT_ASSERT_EQUAL(5, table.libertyAfterAllDrop(state));
  }
  {
    NumEffectState state(CsaString(
			   "P1 *  *  *  *  *  *  *  *  * \n"
			   "P2 *  *  *  * -OU *  *  *  * \n"
			   "P3 *  *  *  *  *  *  *  *  * \n"
			   "P4 *  *  *  * +FU *  *  *  * \n"
			   "P5 *  *  *  *  *  *  *  *  * \n"
			   "P6 *  *  *  *  *  *  *  *  * \n"
			   "P7 *  *  *  *  *  *  *  *  * \n"
			   "P8 *  *  *  *  *  *  *  *  * \n"
			   "P9 *  * +OU *  *  *  *  *  * \n"
			   "P+00HI\n"
			   "P-00AL\n"
			   "+\n").getInitialState());
    CPPUNIT_ASSERT_EQUAL(4, table.libertyAfterAllDrop(state));
  }
}

void osl::checkmate::
ProofNumberTableTest::testLibertyAfterAllMove()
{
  const ProofNumberTable& table = Proof_Number_Table;
  {
    NumEffectState state(CsaString(
			   "P1 *  *  *  *  *  *  *  *  * \n"
			   "P2 *  *  *  * -OU *  *  *  * \n"
			   "P3 *  *  *  *  *  *  *  *  * \n"
			   "P4 *  *  *  * +FU *  *  *  * \n"
			   "P5 *  *  *  *  *  *  *  *  * \n"
			   "P6 *  *  *  *  *  *  *  *  * \n"
			   "P7 *  *  *  *  *  *  *  *  * \n"
			   "P8 *  *  *  *  *  *  *  *  * \n"
			   "P9 *  * +OU *  *  *  *  *  * \n"
			   "P-00AL\n"
			   "+\n").getInitialState());
    CPPUNIT_ASSERT_EQUAL(6, table.libertyAfterAllMove(state));
  }
  {
    NumEffectState state(CsaString(
			   "P1 *  *  *  *  *  *  *  *  * \n"
			   "P2 *  *  *  * -OU *  *  *  * \n"
			   "P3 *  *  *  *  *  *  *  *  * \n"
			   "P4 *  *  *  * +FU *  *  *  * \n"
			   "P5 *  *  * +KE *  *  *  *  * \n"
			   "P6 *  *  *  *  *  *  *  *  * \n"
			   "P7 *  *  *  *  *  *  *  *  * \n"
			   "P8 *  *  *  *  *  *  *  *  * \n"
			   "P9 *  * +OU *  *  *  *  *  * \n"
			   "P-00AL\n"
			   "+\n").getInitialState());
    CPPUNIT_ASSERT_EQUAL(3, table.libertyAfterAllMove(state));
  }
  {
    NumEffectState state(CsaString(
			   "P1 *  *  *  *  *  *  *  *  * \n"
			   "P2 *  *  *  *  *  *  *  *  * \n"
			   "P3 *  *  *  * -OU *  *  *  * \n"
			   "P4 *  *  *  *  *  *  *  *  * \n"
			   "P5 *  *  *  * +FU *  *  *  * \n"
			   "P6 *  *  * +KE *  *  *  *  * \n"
			   "P7 *  *  *  *  *  *  *  *  * \n"
			   "P8 *  *  *  *  *  *  *  *  * \n"
			   "P9 *  * +OU *  *  *  *  *  * \n"
			   "P-00AL\n"
			   "+\n").getInitialState());
    CPPUNIT_ASSERT(3 < table.libertyAfterAllMove(state));
  }
  {
    NumEffectState state(CsaString(
			   "P1 *  *  *  *  *  *  *  *  * \n"
			   "P2 *  *  *  *  *  *  *  *  * \n"
			   "P3 *  *  *  * -OU *  *  *  * \n"
			   "P4 *  *  *  *  *  *  *  *  * \n"
			   "P5 *  *  * +KI+FU *  *  *  * \n"
			   "P6 *  *  *  *  *  *  *  *  * \n"
			   "P7 *  *  *  *  *  *  *  *  * \n"
			   "P8 *  *  *  *  *  *  *  *  * \n"
			   "P9 *  * +OU *  *  *  *  *  * \n"
			   "P-00AL\n"
			   "+\n").getInitialState());
    CPPUNIT_ASSERT_EQUAL(3, table.libertyAfterAllMove(state));
  }
  {
    NumEffectState state(CsaString(
			   "P1 *  *  *  *  *  *  *  *  * \n"
			   "P2 *  * -OU-FU *  * +HI *  * \n"
			   "P3 * -FU * +FU *  *  *  *  * \n"
			   "P4 *  *  *  *  *  *  *  *  * \n"
			   "P5 *  *  *  *  *  *  *  *  * \n"
			   "P6 *  *  *  *  *  *  *  *  * \n"
			   "P7 *  *  *  *  *  *  *  *  * \n"
			   "P8 *  *  *  *  *  *  *  *  * \n"
			   "P9 *  * +OU *  *  *  *  *  * \n"
			   "P-00AL\n"
			   "+\n").getInitialState());
    CPPUNIT_ASSERT_EQUAL(1, table.libertyAfterAllMove(state));
  }
  {
    NumEffectState state(CsaString(
			   "P1 *  *  *  *  *  *  *  *  * \n"
			   "P2 *  *  *  *  *  *  *  *  * \n"
			   "P3 *  *  *  *  *  *  *  *  * \n"
			   "P4 *  *  *  *  *  *  *  *  * \n"
			   "P5 *  * -OU-FU *  * +HI *  * \n"
			   "P6 * -FU * +FU *  *  *  *  * \n"
			   "P7 *  *  *  *  *  *  *  *  * \n"
			   "P8 *  *  *  *  *  *  *  *  * \n"
			   "P9 *  * +OU *  *  *  *  *  * \n"
			   "P-00AL\n"
			   "+\n").getInitialState());
    CPPUNIT_ASSERT_EQUAL(3, table.libertyAfterAllMove(state));
  }
}

/* ------------------------------------------------------------------------- */
// ;;; Local Variables:
// ;;; mode:c++
// ;;; c-basic-offset:2
// ;;; End:

